/** @file scim_backend.h
 *  @brief scim::BackEndBase Interface.
 *
 *  Provide an abstract interface class to
 *  manage a set of IMEngineFactory instances.
 */

/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2004 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2003 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_backend.h,v 1.16 2004/07/24 02:14:28 suzhe Exp $
 */

#ifndef __SCIM_BACKEND_H
#define __SCIM_BACKEND_H

namespace scim {

typedef Slot1<void, int>
        BackEndSlotVoid;

typedef Slot2<void, int, int>
        BackEndSlotInt;

typedef Slot2<void, int, bool>
        BackEndSlotBool;

typedef Slot2<void, int, const WideString&>
        BackEndSlotWideString;

typedef Slot2<void, int, const KeyEvent&>
        BackEndSlotKeyEvent;

typedef Slot2<void, int, const LookupTable&>
        BackEndSlotLookupTable;

typedef Slot2<void, int, const Property&>
        BackEndSlotProperty;

typedef Slot2<void, int, const PropertyList&>
        BackEndSlotPropertyList;

typedef Slot3<void, int, const WideString&,const AttributeList&>
        BackEndSlotWideStringAttributeList;

/**
 * @brief An exception class to hold BackEnd related errors.
 *
 * scim::BackEndBase and its derived classes must throw
 * scim::BackEndError object when error.
 */
class BackEndError: public Exception
{
public:
    BackEndError (const String& what_arg)
        : Exception (String("scim::BackEnd: ") + what_arg) { }
};

/**
 * @brief The interface class to manage a set of IMEngineFactory
 * and IMEngineInstance objects.
 *
 * This is mainly a helper interface class used by scim::FrontEndBase.
 * Its responsibility is to hold a set of IMEngineFactory instances
 * and manage the locales list supported by them. 
 * 
 * Most developer should just use the default implementation
 * scim::CommonBackEnd.
 */
class BackEndBase : public ReferencedObject
{
    class BackEndBaseImpl;

    BackEndBaseImpl *m_impl;

public:
    /**
     * @brief Default constructor.
     */
    BackEndBase ();

    /**
     * @brief Virtual destructor.
     */
    virtual ~BackEndBase ();

    /**
     * @brief Get a list of all locales supported by all FrontEnds.
     * @return A comma separated locales list.
     */
    String get_all_locales () const;

    /**
     * @brief Return the number of factories held by this BackEnd.
     */
    uint32 number_of_factories () const;

    /**
     * @return Return the pointer of a Factory.
     */
    IMEngineFactoryPointer get_factory_pointer (uint32 idx) const;

public:
    /**
     * @name Methods to manipulate IMEngine Factories.
     *
     * @{
     */

    /**
     * @brief Get the IMEngine factories list for specific encoding
     *
     * @param uuids    the vector to store the factories' uuids which
     *                 support the encoding.
     * @param encoding the encoding to be queried. If empty,
     *                 all IMEngine factories will be returned.
     *
     * @return the number of IMEngine factories found.
     */
    uint32 get_factory_list (std::vector<String> &uuids, const String &encoding = String ("")) const;

    /**
     * @brief get the name of an IMEngine factory.
     *
     * @param uuid the uuid of the IMEngine factory
     * @return the name of the IMEngine factory.
     */
    WideString get_factory_name (const String &uuid) const;

    /**
     * @brief get the authors info of an IMEngine factory.
     * @param uuid the uuid of the IMEngine factory
     * @return the authors info of the IMEngine factory.
     */
    WideString get_factory_authors (const String &uuid) const;

    /**
     * @brief get the credits info of an IMEngine factory.
     * @param uuid the uuid of the IMEngine factory
     * @return the credits info of the IMEngine factory.
     */
    WideString get_factory_credits (const String &uuid) const;

    /**
     * @brief get the help info of an IMEngine factory.
     * @param uuid the uuid of the IMEngine factory
     * @return the help info of the IMEngine factory.
     */
    WideString get_factory_help (const String &uuid) const;

    /**
     * @brief get the icon file of an IMEngine factory.
     * @param uuid the uuid of the IMEngine factory
     * @return the icon file name of the IMEngine factory.
     */
    String get_factory_icon_file (const String &uuid) const;

    /**
     * @brief get the supported locales of an IMEngine  factory.
     * @param uuid the uuid of the IMEngine factory
     * @return a comma separated list of the supported locales.
     */
    String get_factory_locales (const String &uuid) const;

    /**
     * @brief get the language of an IMEngine factory.
     * @param uuid the uuid of the IMEngine factory
     * @return the language of this IMEngine factory.
     */
    String get_factory_language (const String &uuid) const;

    /**
     * @}
     */

public:
    /**
     * @name Methods to manipulate IMEngine Instances.
     *
     * @{
     */

    /**
     * @brief create a new IMEngine instance for specific encoding.
     *
     * @param sf_uuid the IMEngineFactory UUID.
     * @param encoding the encoding to be used.
     *
     * @return the newly created IMEngine instance id, -1 means error occurred.
     */
    int  new_instance (const String &sf_uuid, const String &encoding);

    /**
     * @brief replace an IMEngine  instance by a new instance created by another factory.
     *
     * This function is used to change the input method for an input context on the fly.
     *
     * @param si_id the IMEngine instance to be replaced.
     * @param sf_uuid the new IMEngine factory to be used.
     */
    bool replace_instance (int si_id, const String &sf_uuid);

    /**
     * @brief delete an IMEngine instance according to its id.
     * @param id the id of the IMEngine instance to be deleted.
     * @return true if success, false if there is no such instance.
     */
    bool delete_instance (int id);

    /**
     * @brief delete all IMEngine instances.
     *
     * This function should be called just before quitting the FrontEnd.
     */
    void delete_all_instances ();

    /**
     * @brief get the factory uuid of this instance.
     * @param id the IMEngine instance id.
     * @return the factory uuid of this instance.
     */
    String get_instance_uuid (int id) const;

    /**
     * @brief get the working encoding of an IMEngine instance.
     * @param id the IMEngine instance id.
     * @return the working encoding of this IMEngine instance.
     */
    String get_instance_encoding (int id) const;

    /**
     * @brief get the name of an IMEngine instance.
     * @param id the IMEngine instance id.
     * @return the name of this IMEngine instance,
     *         aka. the name of its factory.
     */
    WideString get_instance_name (int id) const;

    /**
     * @brief get the authors info of an IMEngine instance.
     * @param id the IMEngine instance id.
     * @return the authors info of this IMEngine instance,
     *         aka. the authors of its factory.
     */
    WideString get_instance_authors (int id) const;

    /**
     * @brief get the credits info of an IMEngine instance.
     * @param id the IMEngine instance id.
     * @return the credits info of this IMEngine instance,
     *         aka. the credits of its factory.
     */
    WideString get_instance_credits (int id) const;

    /**
     * @brief get the help of an IMEngine instance.
     * @param id the IMEngine instance id.
     * @return the help of this IMEngine instance,
     *         aka. the help of its factory.
     */
    WideString get_instance_help (int id) const;

    /**
     * @brief get the icon file of an IMEngine instance.
     * @param id the IMEngine instance id.
     * @return the icon file name of this IMEngine instance.
     */
    String get_instance_icon_file (int id) const;

    /**
     * @brief process a key event using specific IMEngine instance.
     * @param id the IMEngine instance id.
     * @param key the key event to be processed.
     * @return true if the event was processed successfully,
     *         false if the event was not processed and should
     *         be forward to the client application.
     */
    bool process_key_event (int id, const KeyEvent& key) const;

    /**
     * @brief let a specific IMEngine instance move its preedit caret.
     * @param id the IMEngine instance id.
     * @param pos the new preedit caret position.
     */
    void move_preedit_caret (int id, unsigned int pos) const;

    /**
     * @brief let a specific IMEngine instance select a candidate in its current lookup table.
     * @param id the IMEngine instance id.
     * @param index the candidate index in current lookup table page to be selected.
     */
    void select_candidate (int id, unsigned int index) const;

    /**
     * @brief update the page size of a specific IMEngine instance's lookup table.
     * @param id the IMEngine instance id.
     * @param page_size the new page size to be used.
     */
    void update_lookup_table_page_size (int id, unsigned int page_size) const;

    /**
     * @brief Let a specific IMEngine instance flip its lookup table to the previous page.
     */
    void lookup_table_page_up (int id) const;

    /**
     * @brief Let a specific IMEngine instance flip its lookup table to the previous page.
     */
    void lookup_table_page_down (int id) const;

    /**
     * @brief reset a specific IMEngine instance.
     * @param id the id of the IMEngine instance to be reset.
     */
    void reset (int id) const;

    /**
     * @brief focus in a specific IMEngine instance.
     * @param id the id of the IMEngine instance to be focused in.
     */
    void focus_in (int id) const;

    /**
     * @brief focus out a specific IMEngine instance.
     * @param id the id of the IMEngine instance to be focused out.
     */
    void focus_out (int id) const;

    /**
     * @brief trigger a property of a specific IMEngine instance.
     * @param id the id of the IMEngine instance.
     * @param property the key of the property to be triggered.
     */
    void trigger_property (int id, const String &property) const;

    /**
     * @}
     */
public:
    /**
     * @name Signal connection methods.
     *
     * These functions are used by FrontEnds to connect their corresponding slots to
     * this BackEnd's signals.
     *
     * The first parameter of these methods are the id of corresponding instance.
     *
     * @{
     */
    Connection signal_connect_show_preedit_string   (BackEndSlotVoid *slot);
    Connection signal_connect_show_aux_string       (BackEndSlotVoid *slot);
    Connection signal_connect_show_lookup_table     (BackEndSlotVoid *slot);
    Connection signal_connect_hide_preedit_string   (BackEndSlotVoid *slot);
    Connection signal_connect_hide_aux_string       (BackEndSlotVoid *slot);
    Connection signal_connect_hide_lookup_table     (BackEndSlotVoid *slot);
    Connection signal_connect_update_preedit_caret  (BackEndSlotInt *slot);
    Connection signal_connect_update_preedit_string (BackEndSlotWideStringAttributeList *slot);
    Connection signal_connect_update_aux_string     (BackEndSlotWideStringAttributeList *slot);
    Connection signal_connect_update_lookup_table   (BackEndSlotLookupTable *slot);
    Connection signal_connect_commit_string         (BackEndSlotWideString *slot);
    Connection signal_connect_forward_key_event     (BackEndSlotKeyEvent *slot);
    Connection signal_connect_register_properties   (BackEndSlotPropertyList *slot);
    Connection signal_connect_update_property       (BackEndSlotProperty *slot);
    /** @} */

protected:
    /**
     * @name Methods used by derived classes.
     *
     * @{
     */

    /**
     * @brief Add an IMEngineFactory object into BackEnd.
     */
    bool add_factory (const IMEngineFactoryPointer &factory);

    /**
     * @brief Set the list of unicode locales to be supported.
     */
    void set_supported_unicode_locales (const String &locales);

    /**
     * @brief Destroy all factory instances.
     */
    void destroy_all_factories ();

    /**
     * @}
     */
};

/**
 * @typedef typedef Pointer <BackEndBase> BackEndPointer;
 *
 * A smart pointer for scim::BackEndBase and its derived classes.
 */
typedef Pointer <BackEndBase> BackEndPointer;

/**
 * @brief The default implementation of scim::BackEndBase interface.
 */
class CommonBackEnd : public BackEndBase
{
    class CommonBackEndImpl;
    CommonBackEndImpl *m_impl;

public:
    /**
     * @brief Constructor
     *
     * @param config The pointer to the Config object.
     * @param modules The list of the IMEngine modules to be loaded.
     */
    CommonBackEnd (const ConfigPointer       &config,
                   const std::vector<String> &modules);

    virtual ~CommonBackEnd ();
};

} // namespace scim

#endif //__SCIM_BACKEND_H

/*
vi:ts=4:nowrap:ai:expandtab
*/
