/**
 * @file scim_socket_transaction.h
 * @brief Socket transaction class.
 */

/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2004 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2003 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_socket_transaction.h,v 1.29 2004/07/05 06:23:58 suzhe Exp $
 */

#ifndef __SCIM_SOCKET_TRANSACTION_H
#define __SCIM_SOCKET_TRANSACTION_H

namespace scim {

/**
 * @addtogroup SocketCommunication
 * @{
 */

/**
 * @brief Signature of all valid data types which can be store into transaction.
 */
enum SocketTransactionDataType
{
    SCIM_TRANS_DATA_UNKNOWN,        //!< Unknown transaction data type.
    SCIM_TRANS_DATA_COMMAND,        //!< Send/Receive command.
    SCIM_TRANS_DATA_RAW,            //!< Send/Receive raw buffer.
    SCIM_TRANS_DATA_UINT32,         //!< Send/Receive uint32.
    SCIM_TRANS_DATA_STRING,         //!< Send/Receive String.
    SCIM_TRANS_DATA_WSTRING,        //!< Send/Receive WideString.
    SCIM_TRANS_DATA_KEYEVENT,       //!< Send/Receive KeyEvent.
    SCIM_TRANS_DATA_ATTRIBUTE_LIST, //!< Send/Receive AttributeList.
    SCIM_TRANS_DATA_LOOKUP_TABLE,   //!< Send/Receive LookupTable.
    SCIM_TRANS_DATA_PROPERTY,       //!< Send/Receive Property.
    SCIM_TRANS_DATA_PROPERTY_LIST,  //!< Send/Receive PropertyList.
    SCIM_TRANS_DATA_VECTOR_UINT32,  //!< Send/Receive vector<uint32>.
    SCIM_TRANS_DATA_VECTOR_STRING,  //!< Send/Receive vector<String>.
    SCIM_TRANS_DATA_VECTOR_WSTRING  //!< Send/Receive vector<WideString>.
};

/**
 * @brief Command types used by Socket Frontend/IMEngine/Config and Panel.
 * @addtogroup TransactionCommands
 * @{
 */
const int SCIM_TRANS_CMD_UNKNOWN                          = 0;

const int SCIM_TRANS_CMD_REQUEST                          = 1;
const int SCIM_TRANS_CMD_REPLY                            = 2;
const int SCIM_TRANS_CMD_OK                               = 3;
const int SCIM_TRANS_CMD_FAIL                             = 4;

const int SCIM_TRANS_CMD_OPEN_CONNECTION                  = 5;
const int SCIM_TRANS_CMD_CLOSE_CONNECTION                 = 6;

const int SCIM_TRANS_CMD_LOAD_FILE                        = 7;
const int SCIM_TRANS_CMD_EXIT                             = 99;

const int SCIM_TRANS_CMD_PROCESS_KEY_EVENT                = 100;
const int SCIM_TRANS_CMD_MOVE_PREEDIT_CARET               = 101;
const int SCIM_TRANS_CMD_SELECT_CANDIDATE                 = 102;
const int SCIM_TRANS_CMD_UPDATE_LOOKUP_TABLE_PAGE_SIZE    = 103;
const int SCIM_TRANS_CMD_LOOKUP_TABLE_PAGE_UP             = 104;
const int SCIM_TRANS_CMD_LOOKUP_TABLE_PAGE_DOWN           = 105;
const int SCIM_TRANS_CMD_RESET                            = 106;
const int SCIM_TRANS_CMD_FOCUS_IN                         = 107;
const int SCIM_TRANS_CMD_FOCUS_OUT                        = 108;
const int SCIM_TRANS_CMD_TRIGGER_PROPERTY                 = 109;

const int SCIM_TRANS_CMD_SHOW_PREEDIT_STRING              = 110;
const int SCIM_TRANS_CMD_SHOW_AUX_STRING                  = 111;
const int SCIM_TRANS_CMD_SHOW_LOOKUP_TABLE                = 112;
const int SCIM_TRANS_CMD_HIDE_PREEDIT_STRING              = 113;
const int SCIM_TRANS_CMD_HIDE_AUX_STRING                  = 114;
const int SCIM_TRANS_CMD_HIDE_LOOKUP_TABLE                = 115;
const int SCIM_TRANS_CMD_UPDATE_PREEDIT_CARET             = 116;
const int SCIM_TRANS_CMD_UPDATE_PREEDIT_STRING            = 117;
const int SCIM_TRANS_CMD_UPDATE_AUX_STRING                = 118;
const int SCIM_TRANS_CMD_UPDATE_LOOKUP_TABLE              = 119;
const int SCIM_TRANS_CMD_COMMIT_STRING                    = 120;
const int SCIM_TRANS_CMD_FORWARD_KEY_EVENT                = 121;
const int SCIM_TRANS_CMD_REGISTER_PROPERTIES              = 122;
const int SCIM_TRANS_CMD_UPDATE_PROPERTY                  = 123;

const int SCIM_TRANS_CMD_NEW_INSTANCE                     = 200;
const int SCIM_TRANS_CMD_DELETE_INSTANCE                  = 201;
const int SCIM_TRANS_CMD_DELETE_ALL_INSTANCES             = 202;

const int SCIM_TRANS_CMD_GET_FACTORY_LIST                 = 203;
const int SCIM_TRANS_CMD_GET_FACTORY_NAME                 = 204;
const int SCIM_TRANS_CMD_GET_FACTORY_AUTHORS              = 205;
const int SCIM_TRANS_CMD_GET_FACTORY_CREDITS              = 206;
const int SCIM_TRANS_CMD_GET_FACTORY_HELP                 = 207;
const int SCIM_TRANS_CMD_GET_FACTORY_LOCALES              = 208;
const int SCIM_TRANS_CMD_GET_FACTORY_ICON_FILE            = 209;
const int SCIM_TRANS_CMD_GET_FACTORY_LANGUAGE             = 210;

const int SCIM_TRANS_CMD_FLUSH_CONFIG                     = 300;
const int SCIM_TRANS_CMD_ERASE_CONFIG                     = 301;
const int SCIM_TRANS_CMD_GET_CONFIG_STRING                = 302;
const int SCIM_TRANS_CMD_SET_CONFIG_STRING                = 303;
const int SCIM_TRANS_CMD_GET_CONFIG_INT                   = 304;
const int SCIM_TRANS_CMD_SET_CONFIG_INT                   = 305;
const int SCIM_TRANS_CMD_GET_CONFIG_BOOL                  = 306;
const int SCIM_TRANS_CMD_SET_CONFIG_BOOL                  = 307;
const int SCIM_TRANS_CMD_GET_CONFIG_DOUBLE                = 308;
const int SCIM_TRANS_CMD_SET_CONFIG_DOUBLE                = 309;
const int SCIM_TRANS_CMD_GET_CONFIG_VECTOR_STRING         = 310;
const int SCIM_TRANS_CMD_SET_CONFIG_VECTOR_STRING         = 311;
const int SCIM_TRANS_CMD_GET_CONFIG_VECTOR_INT            = 312;
const int SCIM_TRANS_CMD_SET_CONFIG_VECTOR_INT            = 313;
const int SCIM_TRANS_CMD_RELOAD_CONFIG                    = 314;

const int SCIM_TRANS_CMD_PANEL_COMMAND_START              = 500;
const int SCIM_TRANS_CMD_PANEL_COMMAND_END                = 599;

const int SCIM_TRANS_CMD_USER_DEFINED                     = 10000;
/**
 * @}
 */


/**
 * @brief An exception class to hold SocketTransaction related errors.
 *
 * scim::SocketTransaction and its related classes must throw
 * scim::SocketTransactionError object when error.
 */
class SocketTransactionError: public Exception
{
public:
    SocketTransactionError (const String& what_arg)
        : Exception (String("scim::SocketTransaction: ") + what_arg) { }
};

/**
 * @brief This class is used to pack up many data and commands into one package
 *        and send them via socket.
 */
class SocketTransaction
{
    class SocketTransactionImpl;

    SocketTransactionImpl * m_impl;

    SocketTransaction (const SocketTransaction &);
    const SocketTransaction & operator = (const SocketTransaction &);

public:
    /**
     * @brief Constructor.
     *
     * @param bufsize the initial buffer size, maybe grow afterwards.
     */
    SocketTransaction (size_t bufsize = 512);

    /**
     * @brief Destructor.
     */
    ~SocketTransaction ();

    /**
     * @brief Check if the transaction is valid.
     *
     * @return true if this SocketTransaction object is valid and ready to use.
     */
    bool valid () const;

    /**
     * @brief Write the transaction to a socket.
     *
     * @param socket the socket to be written to.
     * @param signature the leading signature to be written
     *        into the socket before the transaction itself,
     *        this signature maybe missing when receive the transaction.
     *        It's useful to check the connection before receiving
     *        a transaction by reading this signature.
     *
     * @return true if success.
     */
    bool write_to_socket (const Socket &socket, uint32 signature = 0) const;

    /**
     * @brief Read a transaction from a socket.
     *
     * @param socket the socket to be read from.
     * @param timeout time out in millisecond (1/1000 second), -1 means infinity.
     *
     * @return true if success.
     */
    bool read_from_socket (const Socket &socket, int timeout = -1);

    /**
     * @brief Store a command into this transaction.
     *
     * @param cmd the command code, like SCIM_TRANS_CMD_FOCUS_IN etc.
     */
    void put_command (int cmd);

    /**
     * @brief Store a uint32 value into this transaction.
     */
    void put_data (uint32 val);

    /**
     * @brief Store a String object into this transaction.
     */
    void put_data (const String &str);

    /**
     * @brief Store a WideString object into this transaction.
     */
    void put_data (const WideString &str);

    /**
     * @brief Store a KeyEvent object into this transaction.
     */
    void put_data (const KeyEvent &key);

    /**
     * @brief Store an AttributeList object into this transaction.
     */
    void put_data (const AttributeList &attrs);

    /**
     * @brief Store a Property object into this transaction.
     */
    void put_data (const Property &property);

    /**
     * @brief Store a PropertyList object into this transaction.
     */
    void put_data (const PropertyList &properties);

    /**
     * @brief Store a LookupTable object into this transaction.
     */
    void put_data (const LookupTable &table);

    /**
     * @brief Store a std::vector<uint32> object into this transaction.
     */
    void put_data (const std::vector<uint32> &vec);

    /**
     * @brief Store a std::vector<String> object into this transaction.
     */
    void put_data (const std::vector<String> &vec);

    /**
     * @brief Store a std::vector<WideString> object into this transaction.
     */
    void put_data (const std::vector<WideString> &vec);

    /**
     * @brief Store a raw buffer into this transaction.
     */
    void put_data (const char *raw, size_t bufsize);

    /**
     * @brief Get the type of the data at current read position.
     *
     * @return The type of the data can be read currently.
     */
    SocketTransactionDataType get_data_type () const;

    /**
     * @brief Get a command from current read position.
     */
    bool get_command (int &cmd);

    /**
     * @brief Get an uint32 value from current read position.
     */
    bool get_data (uint32 &val);

    /**
     * @brief Get a String from current read position.
     */
    bool get_data (String &str);

    /**
     * @brief Get a WideString from current read position.
     */
    bool get_data (WideString &str);

    /**
     * @brief Get a KeyEvent from current read position.
     */
    bool get_data (KeyEvent &key);

    /**
     * @brief Get an AttributeList from current read position.
     */
    bool get_data (AttributeList &attrs);

    /**
     * @brief Get a Property from current read position.
     */
    bool get_data (Property &property);

    /**
     * @brief Get a PropertyList from current read position.
     */
    bool get_data (PropertyList &properties);

    /**
     * @brief Get a CommonLookupTable from current read position.
     */
    bool get_data (CommonLookupTable &table);

    /**
     * @brief Get a std::vector<uint32> from current read position.
     */
    bool get_data (std::vector<uint32> &vec);

    /**
     * @brief Get a std::vector<String> from current read position.
     */
    bool get_data (std::vector<String> &vec);

    /**
     * @brief Get a std::vector<WideString> from current read position.
     */
    bool get_data (std::vector<WideString> &vec);

    /**
     * @brief Get a raw buffer from current read position.
     * 
     * if raw == NULL then return the bufsize and skip this data.
     * *raw should be deleted afterwards (do not use free!).
     */
    bool get_data (char **raw, size_t &bufsize);

    /**
     * @brief Skip one data from current read position.
     */
    bool skip_data ();

    /**
     * @brief Rewind the current read position, then the data can be read again.
     */
    void rewind_read_pos ();

    /**
     * @brief Clear the transaction, all data in this transaction will be freed.
     */
    void clear ();
};

/**
 * @brief Helper function to open a connection to a socket server
 * with a standard hand shake protocol.
 *
 * This function is used by a socket client to establish a connection
 * between a socket server with a standard hand shake protocol.
 *
 * The communication between Panel and FrontEnd, SocketFrontEnd and SocketIMEngine,
 * SocketFrontEnd and SocketConfig all uses this hand shake protocol.
 *
 * @param key         A random magic key sent from the socket server
 *                    to identify this client in later communications.
 * @param client_type The type of this socket client, for example:
 *                    "FrontEnd", "GUI", "SocketIMEngine", "SocketConfig" etc.
 *                    If the type is "ConnectionTester" then this call just
 *                    test if the connection can be established. The client
 *                    should close this socket just after the call.
 * @param server_type The request socket server type, for example:
 *                    "Panel", "SocketFrontEnd" etc.
 * @param socket      The reference to the client socket which has been
 *                    connected to the socket server.
 * @param timeout     The socket read timeout in millisecond, -1 means unlimited.
 *
 * @return true if the connection was established successfully, otherwise
 *         return false, and the client should close the socket.
 */
bool   scim_socket_trans_open_connection   (uint32       &key,
                                            const String &client_type,
                                            const String &server_type,
                                            const Socket &socket,
                                            int           timeout = -1);

/**
 * @brief Helper function to accept a connection request from a socket client
 * with a standard hand shake protocol.
 *
 * This function is used by a socket server to accept a connection request
 * from a socket client which is calling scim_socket_trans_open_connection ().
 *
 * If a client with type "ConnectionTester" connected to this socket server, then
 * this function will return an empty string, but tell the client the connection
 * was established successfully.
 *
 * @param key          A random magic key to identify the socket client in later
 *                     communications.
 * @param server_types The type of this server, for example:
 *                     "SocketFrontEnd", "Panel" etc.
 *                     One server can have multiple types, separated by comma.
 * @param client_types A list of acceptable client types, separated by comma.
 *                     The client type maybe: "FrontEnd", "GUI", "SocketIMEngine" etc.
 * @param socket       The socket connected to the client.
 * @param timeout      the socket read timeout in millisecond, -1 means unlimited.
 *
 * @return The type of the accepted socket client, or an empty string if the
 *         connection could not be established.
 */
String scim_socket_trans_accept_connection (uint32       &key,
                                            const String &server_types,
                                            const String &client_types,
                                            const Socket &socket,
                                            int           timeout = -1);
/** @} */

} // namespace scim

#endif //__SCIM_SOCKET_TRANSACTION_H

/*
vi:ts=4:nowrap:ai:expandtab
*/

