# -*- coding: utf-8 -*-    
# $Id: usbThread.py 47 2011-06-13 10:20:14Z georgesk $	

licenceEn="""
    file usbThread.py
    this file is part of the project scolasync
    
    Copyright (C) 2010 Georges Khaznadar <georgesk@ofset.org>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

import subprocess, threading, re, os.path, time

_threadNumber=0

class ThreadRegister:
    """
    Une classe pour tenir un registre des threads concernant les baladeurs.
    """

    def __init__(self):
        """
        Le constructure met en place un dictionnaire
        """
        self.dico={}

    def __str__(self):
        return "ThreadRegister: %s" %self.dico
        
    def push(self, ud, thread):
        """
        @param ud un disque
        @param thread un thread
        Empile un thread pour le baladeur ud
        """
        if ud.owner not in self.dico.keys():
            self.dico[ud.owner]=[thread]
        else:
            self.dico[ud.owner].append(thread)

    def pop(self, ud, thread):
        """
        @param ud un disque
        @param thread un thread
        Dépile un thread pour le baladeur ud
        """
        self.dico[ud.owner].remove(thread)

    def busy(self, owner):
        """
        Indique si le disque est occupé par des threads
        @param owner le propriétaire du disque
        @return les données associées par le dictionnaire
        """
        if owner in self.dico.keys():
            return self.dico[owner]
        return []

globalThreads=ThreadRegister()

def _sanitizePath(path):
    """
    Évite d'avoir des <i>slashes</i> dans un nom de thread
    @return la fin du nom de chemin, après le dernier <i>slash</i> ;
    si le chemin ne finit pas bien, remplace les <i>slashes</i> par
    des sous-tirets "_".
    """
    pattern=re.compile(".*([^/]+)")
    m=pattern.match(str(path))
    if m:
        return m.group(1)
    else:
        return str(path).replace('/','_')

def _threadName(ud):
    """
    fabrique un nom de thread commençant par th_, suivi d'un nombre unique,
    suivi d'une chaîne relative à la clé USB
    @param ud une instance de uDisk
    @return un nom de thread unique
    """
    global _threadNumber
    name="th_%04d_%s" %(_threadNumber,_sanitizePath(ud.path))
    _threadNumber+=1
    return name

def _date():
    """
    Renvoie la date et l'heure dans un format court
    @return une chaîne donnée par strftime et le format %Y/%m/%d-%H:%M:%S
    """
    return time.strftime("%Y/%m/%d-%H:%M:%S")

def _call(cmd, logfile):
    """
    Lance une commande dans un shell, et journalise la réussite ou l'échec
    @param cmd la commande shell, si cette commande contient des points-virgules elle sera découpée en plusieurs sous-commandes, chacune traitée séparément.
    @param logfile le fichier de journalisation
    """
    for command in cmd.split(";"):
        okToLog="echo [%s] Success: %s >> %s" %(_date(), command, logfile)
        koToLog="echo [%s] Error: %s >> %s" %(_date(), command, logfile)
        cmd1="(%s && %s) || %s" %(command, okToLog, koToLog)
        subprocess.call(cmd1, shell=True)

class abstractThreadUSB(threading.Thread):
    """
    Une classe abstraite
     Cette classe sert de creuset pour les classe servant aux copies
     et aux effacement.
    """
    def __init__(self,ud, fileList, subdir, dest=None, logfile="/dev/null"):
        """
        Constructeur
        Crée un thread pour copier une liste de fichiers vers une clé USB.
        @param ud l'instance uDisk correspondant à une partition de clé USB
        @param fileList la liste des fichiers à traiter
        @param subdir un sous-répertoire de la clé USB
        @param dest un répertoire de destination si nécessaire, None par défaut
        @param logfile un fichier de journalisation, /dev/null par défaut
        """
        threading.Thread.__init__(self,target=self.toDo,
                                  args=(ud, fileList, subdir, dest, logfile),
                                  name=_threadName(ud))        
        self.cmd=u"echo This is an abstract method, don't call it"
        self.ud=ud
        ud.threadRunning=True
        self.fileList=fileList
        self.subdir=subdir
        self.dest=dest
        self.logfile=logfile

    def __str__(self):
        """
        Renvoie une chaîne informative sur le thread
        @return une chaine donnant des informations sur ce qui va
        se passer dans le thread qui a été créé.
        """
        result="%s(\n" %self.threadType()
        result+="  ud       = %s\n" %self.ud
        result+="  fileList = %s\n" %self.fileList
        result+="  subdir   = %s\n" %self.subdir
        result+="  dest     = %s\n" %self.dest
        result+="  logfile  = %s\n" %self.logfile
        result+="  cmd      = %s\n" %self.cmd
        result+="\n"
        return result

    def threadType(self):
        """
        @return une chaîne courte qui informe sur le type de thread
        """
        return "abstractThreadUSB"

    def toDo(self, ud, fileList, subdir, dest, logfile):
        """
        La fonction abstraite pour les choses à faire
        @param ud l'instance uDisk correspondant à une partition de clé USB
        @param fileList la liste des fichiers à traiter
        @param subdir un sous-répertoire de la clé USB
        @param dest un répertoire de destination
        @param logfile un fichier de journalisation
        """
        # ça ne fait rien du tout pour un thread abstrait
        pass
    
class threadCopyToUSB(abstractThreadUSB):
    """
    Classe pour les threads copiant vers les clés USB
    """
    def __init__(self,ud, fileList, subdir, logfile="/dev/null"):
        """
        Constructeur
        Crée un thread pour copier une liste de fichiers vers une clé USB.
        @param ud l'instance uDisk correspondant à une partition de clé USB
        @param fileList la liste des fichiers à copier
        @param subdir le sous-répertoire de la clé USB où faire la copie
        @param logfile un fichier de journalisation, /dev/null par défaut
        """
        abstractThreadUSB.__init__(self,ud, fileList, subdir, dest=None, logfile=logfile)
        self.cmd=u"mkdir -p '{toDir}'; cp -R '{fromFile}' '{toFile}'"

    def threadType(self):
        """
        @return une chaîne courte qui informe sur le type de thread
        """
        return "threadCopyToUSB"

    def toDo(self, ud, fileList, subdir, dest, logfile):
        """
        Copie une liste de fichiers vers une clé USB sous un répertoire donné.
         Ce répertoire est composé de ud.visibleDir() joint au
         sous-répertoire subdir.
         À chaque fichier ou répertoire copié, une ligne est journalisée dans le
         fichier de journal de l'application.
        @param ud l'instance uDisk correspondant à une partition de clé USB
        @param fileList la liste des fichiers à copier
        @param logfile un fichier de journalisation
        @param subdir le sous-répertoire de la clé USB où faire la copie
        """
        global globalThreads
        globalThreads.push(ud, self)
        while subdir[0]=='/':
            subdir=subdir[1:]
        destpath=os.path.join(ud.ensureMounted(),ud.visibleDir(),subdir)
        for f in fileList:
            fileName=os.path.basename(f)
            cmd=self.cmd.format(fromFile=f,
                                toDir=destpath,
                                toFile=os.path.join(destpath,fileName))
            _call(cmd,logfile)
        globalThreads.pop(ud, self)

class threadCopyFromUSB(abstractThreadUSB):
    """
    Classe pour les threads copiant depuis les clés USB
    """
    def __init__(self,ud, fileList, subdir=".", dest="/tmp",
                 rootPath="/", logfile="/dev/null"):
        """
        Constructeur
        Crée un thread pour copier une liste de fichiers depuis une clé USB
        vers un répertoire de disque.
        @param ud l'instance uDisk correspondant à une partition de clé USB
        @param fileList la liste des fichiers à copier
        @param subdir le sous-répertoire de la clé USB d'où faire la copie
        @param dest un répertoire de destination
        @param logfile un fichier de journalisation, /dev/null par défaut
        """
        abstractThreadUSB.__init__(self,ud, fileList, subdir, dest=dest,
                                   logfile=logfile)
        self.rootPath=rootPath
        self.cmd=u"mkdir -p '{toDir}'; cp -R '{fromPath}' '{toPath}'"

    def toDo(self, ud, fileList, subdir, dest, logfile):
        """
        Copie une liste de fichiers d'une clé USB sous un répertoire donné.
         À chaque fichier ou répertoire copié, une ligne est journalisée
         dans le fichier de journal de l'application.
        @param ud l'instance uDisk correspondant à une partition de clé USB
        @param fileList la liste des fichiers à copier
        @param dest un répertoire de destination
        @param logfile un fichier de journalisation
        @param subdir le sous-répertoire de la clé USB où faire la copie
        """
        global globalThreads
        globalThreads.push(ud, self)
        for f in fileList:
            ## prend le fichier ou le répertoire sur le disque courant
            fromPath=os.path.join(ud.ensureMounted(), f)
            owner=ud.ownerByDb()
            ## personnalise le nom de la destination
            newName=u"%s_%s" %(owner,f)
            ## calcule le point de copie et le répertoire à créer s'il le faut
            toPath=os.path.join(dest,newName)
            toDir=os.path.dirname(toPath)
            cmd=self.cmd.format(fromPath=fromPath, toPath=toPath, toDir=toDir)
            _call(cmd,logfile)
        globalThreads.pop(ud, self)
            
class threadDeleteInUSB(abstractThreadUSB):
    """
    Classe pour les threads effaçant des sous-arbres dans les clés USB
    """
    def __init__(self,ud, fileList, subdir, logfile="/dev/null"):
        """
        Constructeur
         Crée un thread pour supprimer une liste de fichiers dans une clé USB.
        @param ud l'instance uDisk correspondant à une partition de clé USB
        @param fileList la liste des fichiers à supprimer
        @param subdir le sous-répertoire de la clé USB où faire les suppressions
        @param logfile un fichier de journalisation, /dev/null par défaut
        """
        abstractThreadUSB.__init__(self,ud, fileList, subdir, dest=None, logfile=logfile)
        self.cmd=u"rm -rf '{toDel}'"

    def toDo(self, ud, fileList, subdir, dest, logfile):
        """
        Supprime une liste de fichiers dans une clé USB.
         La liste est prise sous un répertoire donné. Le répertoire visible
         qui dépend du constructuer d ela clé est pris en compte.
         À chaque fichier ou répertoire supprimé, une ligne est
         journalisée dans le fichier de journal de l'application.
        @param l'instance uDisk correspondant à une partition de clé USB
        @param fileList la liste des fichiers à copier
        @param dest un répertoire de destination
        @param logfile un fichier de journalisation
        @param subdir le sous-répertoire de la clé USB où faire la copie
        """
        global globalThreads
        globalThreads.push(ud, self)
        for f in fileList:
            toDel=os.path.join(ud.ensureMounted(), f)
            cmd=self.cmd.format(toDel=toDel)
            _call(cmd,logfile)
        globalThreads.pop(ud, self)
