/*******************************************************************************
 * Score Reading Trainer                                                       *
 * Copyright (C) 2004 by Jos Pablo Ezequiel Fernndez <pupeno@pupeno.com>     *
 *                                                                             *
 * This program is free software; you can redistribute it and/or               *
 * modify it under the terms of the GNU General Public License                 *
 * version 2 as published by the Free Software Foundation.                     *
 *                                                                             *
 * This program is distributed in the hope that it will be useful,             *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 * GNU General Public License for more details.                                *
 *                                                                             *
 * You should have received a copy of the GNU General Public License           *
 * along with this program; if not, write to the Free Software                 *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *******************************************************************************/


#include <qcanvas.h>
#include <qpoint.h>
#include <qwmatrix.h>

#include <kstandarddirs.h>
#include <kdebug.h>
#include <kapplication.h>
#include <kmessagebox.h>
#include <klocale.h>

#include "Score.h"

#define NOTE_MARGIN 25
#define ACCIDENTAL_MARGIN 5

Score::Score(QWidget *parent, const char *name, WFlags f) : QCanvasView(parent, name, f), mClef(Score::ClefG2), mKeySignature(Score::KeySignatureC) {
	//kdDebug() << "Running Score::Score(QWidget *parent, const char *name, WFlags f)" << endl;

	// Centers (to be loaded from a file or something like that).
	center[Score::ClefG2] = 122;
	center[Score::ClefC3] = 80;
	center[Score::ClefC4] = 80;
	center[Score::ClefF4] = 30;
	center[Score::Up] = 140;
	center[Score::Down] = 20;

	center[Score::Natural] = 30;
	center[Score::Sharp] = 30;
	center[Score::DoubleSharp] = 20;
	center[Score::Flat] = 40;
	center[Score::DoubleFlat] = 40;

	// Positions (to be loaded from a file or something like that).
	position[Score::ClefG2] = 2;
	position[Score::ClefC3] = 4;
	position[Score::ClefC4] = 6;
	position[Score::ClefF4] = 6;
	position[Score::C]=-2;
	position[Score::D]=-1;
	position[Score::E]=0;
	position[Score::F]=1;
	position[Score::G]=2;
	position[Score::A]=3;
	position[Score::B]=4;
	position[Score::KeySignatureG] = 8;
	position[Score::KeySignatureD] = 5;
	position[Score::KeySignatureA] = 9;
	position[Score::KeySignatureE] = 6;
	position[Score::KeySignatureB] = 3;
	position[Score::KeySignatureFSharp] = 7;
	position[Score::KeySignatureCSharp] = 4;
	position[Score::KeySignatureF] = 4;
	position[Score::KeySignatureBFlat] = 7;
	position[Score::KeySignatureEFlat] = 3;
	position[Score::KeySignatureAFlat] = 6;
	position[Score::KeySignatureDFlat] = 2;
	position[Score::KeySignatureGFlat] = 5;
	position[Score::KeySignatureCFlat] = 1;

	// Don't ever show the horizontal scrollbar
	setHScrollBarMode(QScrollView::AlwaysOff);
	setVScrollBarMode(QScrollView::AlwaysOff);

	// Create the main canvas and set it as the canvas for 'this' QCanvasView
	mMainCanvas = new QCanvas(this, "canvas");
	mMainCanvas->resize(1500, 500);
	setCanvas(mMainCanvas);

	// Create the lines that will be the pentagram
	for(int index = 0; index < 5; index++){
		QCanvasLine *line = new QCanvasLine(mMainCanvas);
		line->setPoints(0, score2pixels(index*2), mMainCanvas->size().width(), score2pixels(index*2));
		line->show();
	}

	// Create the sprites for the diferent objects
	mClefSprite = new QCanvasSprite(new QCanvasPixmapArray(), mMainCanvas);

	setPaletteBackgroundColor(QColor("Red"));
}

Score::~Score(){
	//kdDebug() << "Running Score::~Score()" << endl;
}

Score::Clefs Score::clef(){
	//kdDebug() << "Running Score::clef()" << endl;
	return mClef;
}

void Score::setClef(Clefs pClef){
	//kdDebug() << "Running Score::setClef(Clefs pClef)" << endl;
	if(pClef >= Score::ClefG2 and pClef <= Score::ClefF4){
		mClef = pClef;

		QString clef;
		// Load the right pixmap and position it at the right place
		switch(pClef){
			case Score::ClefG2:{
				clef = "g";
				break;
			}
			case Score::ClefC3:{
				clef = "c";
				break;
			}
			case Score::ClefC4:{
				clef = "c";
				break;
			}
			case Score::ClefF4:{
				clef = "f";
				break;
			}
		}
		clef = locate("data", QString("ScoreReadingTrainer/pics/%1clef.png").arg(clef));
		if(clef.isEmpty()){
			KMessageBox::error(this, i18n("Could not find file \"%1clef.png\", your installation might be corrupt, if not, this may be a bug, please, report it.").arg(clef));
			kapp->quit();
			return;
		}
		mClefSprite->setSequence(new QCanvasPixmapArray(clef));
		mClefSprite->move(10, score2pixels(position[pClef])-center[pClef]);
		mClefSprite->show();
		mMainCanvas->update();
	}
}

Score::KeySignatures Score::keySignature(){
	//kdDebug() << "Running Score::keySignature()" << endl;
	return mKeySignature;
}

void Score::setKeySignature(KeySignatures pKeySignature){
	//kdDebug() << "Running Score::setKeySignature(KeySignatures pKeySignature)" << endl;
	if(pKeySignature >= Score::KeySignatureCFlat and pKeySignature <= Score::KeySignatureCSharp ){
		mKeySignature = pKeySignature;
		for(QCanvasItemList::Iterator it = mKeySignatureSprites.begin(); it != mKeySignatureSprites.end(); ++it){
			(*it)->hide();
		}
		mKeySignatureSprites.clear();

		if(mKeySignature > Score::KeySignatureC) {
			QString ks = locate("data", "ScoreReadingTrainer/pics/sharp.png");
			if(ks.isEmpty()){
				KMessageBox::error(this, i18n("Could not find file \"sharp.png\", your installation might be corrupt, if not, this may be a bug, please, report it."));
				kapp->quit();
				return;
			}
			for(int it = Score::KeySignatureG ; it <= pKeySignature ; ++it){
				int pos = position[it];
				QCanvasSprite *keySignatureSprite = new QCanvasSprite(new QCanvasPixmapArray(ks), mMainCanvas);
				keySignatureSprite->setSequence(new QCanvasPixmapArray(ks));
				if(mKeySignatureSprites.empty()){
					keySignatureSprite->move(mClefSprite->boundingRect().right() + 15, score2pixels(clefCompensate(pos))-center[Score::Sharp]);
				} else {
					keySignatureSprite->move(mKeySignatureSprites.last()->boundingRect().right() + 5, score2pixels(clefCompensate(pos))-center[Score::Sharp]);
				}
				keySignatureSprite->show();
				mKeySignatureSprites.append(keySignatureSprite);
			}
		} else if(mKeySignature < Score::KeySignatureC) {
			QString ks = locate("data", "ScoreReadingTrainer/pics/flat.png");
			if(ks.isEmpty()){
				KMessageBox::error(this, i18n("Could not find file \"flat.png\", your installation might be corrupt, if not, this may be a bug, please, report it."));
				kapp->quit();
				return;
			}

			for(int it = Score::KeySignatureF ; it >= pKeySignature ; --it){
				int pos = position[it];
				QCanvasSprite *keySignatureSprite = new QCanvasSprite(new QCanvasPixmapArray(ks), mMainCanvas);
				keySignatureSprite->setSequence(new QCanvasPixmapArray(ks));
				if(mKeySignatureSprites.empty()){
					keySignatureSprite->move(mClefSprite->boundingRect().right() + 15, score2pixels(clefCompensate(pos))-center[Score::Flat]);
				} else {
					keySignatureSprite->move(mKeySignatureSprites.last()->boundingRect().right() + 5, score2pixels(clefCompensate(pos))-center[Score::Flat]);
				}
				keySignatureSprite->show();
				mKeySignatureSprites.append(keySignatureSprite);
			}
		}
		mMainCanvas->update();
	}
}

unsigned int Score::enqueueNote(Note note){
	//kdDebug() << "Running Score::enqueueNote(Note note)" << endl;

	// Calculate the vertical positioning in score steps including clef compensation.
	int vScorePos = clefCompensate(position[note.noteValue()] + 7 * note.octave());

	// Let's see if we'll use up or down note, Score::B is the fist up note in G.
	bool down = vScorePos > position[Score::B];

	// Translate the vertical position from score steps to real pixels.
	int vPos = score2pixels(vScorePos);

	// For the names of the file to load.
	QString file = QString::null;

	// Put the name of the needed accidental into file.
	switch(note.accidental()){
		case(Score::Sharp):{
			file = "sharp";
			break;
		}
		case(Score::DoubleSharp):{
			file = "doublesharp";
			break;
		}
		case(Score::Flat):{
			file = "flat";
			break;
		}
		case(Score::DoubleFlat):{
			file = "doubleflat";
			break;
		}
		case(Score::Natural):{
			file = "natural";
			break;
		}
		case(Score::None):{
			file = QString::null;
			break;
		}
	}
	// Load the right image for accidental
	QCanvasSprite *accidentalSprite = 0L;

	// If file is set, we have to load an accidental, if not, nothing.
	if(file){
		QString completeFile=locate("data", QString("ScoreReadingTrainer/pics/%1.png").arg(file));
		if(!completeFile){
			KMessageBox::error(this, i18n("Could not find file \"%1\", your installation might be corrupt, if not, this may be a bug, please, report it.").arg(file));
			kapp->quit();
			return mNotes.count();
		}
		file=completeFile;
		accidentalSprite = new QCanvasSprite(new QCanvasPixmapArray(file), mMainCanvas);
	}

	// Put the file name of the needed note (up or down) on file
	if(down){
		file=locate("data", "ScoreReadingTrainer/pics/notedown.png");
		if(!file){
			KMessageBox::error(this, i18n("Could not find file \"notedown.png\", your installation might be corrupt, if not, this may be a bug, please, report it."));
			kapp->quit();
			return mNotes.count();
		}
	} else {
		file=locate("data", "ScoreReadingTrainer/pics/noteup.png");
		if(!file){
			KMessageBox::error(this, i18n("Could not find file \"noteup.png\", your installation might be corrupt, if not, this may be a bug, please, report it."));
			kapp->quit();
			return mNotes.count();
		}
	}
	// Load the right image for note
	QCanvasSprite *noteSprite = new QCanvasSprite(new QCanvasPixmapArray(file), mMainCanvas);

	// hNotePos to hold the horizontal position of the note.
	int hNotePos = 0;

	// Find which is the lattest object (form left to right) being draw to draw after it.
	if(!mNotesSprites.isEmpty()){
		// If there are previous notes, calculate the horizontal position to the last note.
		hNotePos += mNotesSprites.last()->boundingRect().right();
	} else if(!mKeySignatureSprites.isEmpty()){
		// If there's a key signature and no previous notes, calculate the horizontal position to the key signature.
		hNotePos += mKeySignatureSprites.last()->boundingRect().right();
	} else {
		// If there's isn't anything (notes nor key signature), use the clef to calculate the horizontal offset (there's always a clef).
		hNotePos += mClefSprite->boundingRect().right();
	}

	// We add 1, because we want to start counting on the first empty pixel after the previous object, not in the last pixel of the previous object.
	hNotePos += 1;

	// Add the margin to the previous object to the horizontal position.
	hNotePos += NOTE_MARGIN;
	
	// If there's an accidental, put it in the right place and add the horizontal offset.
	if(accidentalSprite){
		// Put the accidental where the note should have been
		accidentalSprite->move(hNotePos, vPos - center[note.accidental()]);
		// show it
		accidentalSprite->show();
		// and add it to the list.
		mAccidentalsSprites.push_back(accidentalSprite);

		// Compensate horizontally the position of the note to not overlap with the accidental
		hNotePos += accidentalSprite->boundingRect().width() + ACCIDENTAL_MARGIN;
	}

	// Where the extra line, if needed, will be started to drawn.
	int extraLineStartPoint = hNotePos - 10;
	
	// Draw the note.
	if(down){
		// If it's down.
		noteSprite->move(hNotePos, vPos-center[Score::Down]);
	} else {
		// If it's up.
		noteSprite->move(hNotePos, vPos-center[Score::Up]);
	}

	// Now that the note was added, we know where the extra line, if needed, will be ended.
	int extraLineEndPoint = noteSprite->boundingRect().right()+10;
	
	// Draw the ledger lines
	if(vScorePos < -1 or vScorePos > 9){
		// We'll draw lines from init
		int init;
		// to stop
		int stop;

		// Chech if the note is up the or down
		if(vScorePos > 0){
			// If it's up, start drawing at line 10 (because 9 is the last statically drawed line
			init = 10;
			// and end drawing at the note.
			stop = vScorePos;
		} else {
			// If it's down, start drawing at the note
			init = (vScorePos % 2 != 0) ? vScorePos + 1 : vScorePos;
			// and end drawing before the first line
			stop = -2;
		}

		// This loops draws all the lines that the note needs when it's outside the five lines.
		for(int index = init; index <= stop; index+=2){
			// Create the line
			QCanvasLine *line = new QCanvasLine(mMainCanvas);
			
			// Vertical position of this extra line
			int extraLineVerticalPos = score2pixels(index);
			
			// Draw the extra line in the calculated dimensions and...
			line->setPoints(extraLineStartPoint, extraLineVerticalPos, extraLineEndPoint, extraLineVerticalPos);
			// show it.
			line->show();

			// Add it to the list.
			mLinesSprites.push_back(line);
		}
	}

	// Show the note.
	noteSprite->show();

	// Add the note to the list of notes.
	mNotesSprites.push_back(noteSprite);

	mMainCanvas->update();

	mNotes.push_back(note);
	return mNotes.count();
}

Note Score::note(){
	//kdDebug() << "Running Score::note()" << endl;
	return mNotes.first();
}

void Score::resizeEvent(QResizeEvent *e){
	//kdDebug() << "Score::resizeEvent(QResizeEvent *)" << endl;
	//kdDebug() << "Trying to scale " << e->size().height() << "/" << mMainCanvas->height() << endl;
	//kdDebug() << "Trying to scale " << 1.0*e->size().height()/mMainCanvas->height() << endl;
	QWMatrix matrix;
	matrix.scale( 1.0*e->size().height()/mMainCanvas->height(), 1.0*e->size().height()/mMainCanvas->height());
	if(!setWorldMatrix(matrix)){
		kdDebug() << "Setting the world matrix failed" << endl;
	}
	//mMainCanvas->resize(e->size().width(), mMainCanvas->height());
	QCanvasView::resizeEvent(e);
}

unsigned int Score::dequeueNote(){
	//kdDebug() << "Running Score::dequeueNote()" << endl;
	// If there's no note, don't remove anything.
	if(!mNotes.isEmpty() and !mNotesSprites.isEmpty()){
		// moveBy will hold how much we have to move everything to the left.
		int moveBy = 0;

		// vScorePos will have the vertical postion of the note, only (by now) to check if there's a line asosiated to remove
		int vScorePos = clefCompensate(position[mNotes[0].noteValue()] + 7 * mNotes[0].octave());

		// If the note isn't on five lines, the extra line should be removed
		if(vScorePos < -1 or vScorePos > 9){
			// We'll remove lines from init
			int init;
			// to stop
			int stop;

			// Chech if the note is up the or down
			if(vScorePos > 0){
				// If it's up, start drawing at line 10 (because 9 is the last statically drawed line
				init = 10;
				// and end drawing at the note
				stop = vScorePos;
			} else {
				// If it's down, start drawing at the note
				init = vScorePos;
				// and end drawing before the first line
				stop = -1;
			}
			for(int index = init; index <= stop; index+=2){
				// Hide the about-to-be-removed little line.
				mLinesSprites[0]->hide();

				// Remove the line of the note
				mLinesSprites.pop_front();
			}
		}

		// If the note contains an accidental, remove it and add the accidental width (plus the margin) to the moveBy variable (how much will move everything)
		if(mNotes.first().accidental() != Score::None){
			// Calculate moveBy
			moveBy += mAccidentalsSprites[0]->boundingRect().width() + ACCIDENTAL_MARGIN;

			// Hide it
			mAccidentalsSprites[0]->hide();

			// Remove it
			mAccidentalsSprites.pop_front();
		}

		// Add to moveBy the width of the note.
		moveBy += mNotesSprites[0]->boundingRect().width() + NOTE_MARGIN;

		// Hide the note.
		mNotesSprites[0]->hide();

		// Remove the (graphical) note.
		mNotesSprites.pop_front();

		// Remove the (data) note.
		mNotes.pop_front();

		// Move everything to the left by moveBy.
		for( QCanvasItemList::iterator it = mNotesSprites.begin(); it != mNotesSprites.end(); ++it ){
			(*it)->moveBy(-moveBy, 0);
		}
		for( QCanvasItemList::iterator it = mAccidentalsSprites.begin(); it != mAccidentalsSprites.end(); ++it ){
			(*it)->moveBy(-moveBy, 0);
		}
		for( QCanvasItemList::iterator it = mLinesSprites.begin(); it != mLinesSprites.end(); ++it ){
			(*it)->moveBy(-moveBy, 0);
		}

		// Let's rock and roll
		mMainCanvas->update();
	}
	return mNotes.count();
}

void Score::clearNotes(){
	//kdDebug() << "Running Score::clearNotes()" << endl;

	// Remove all the notes
	for( QCanvasItemList::iterator it = mNotesSprites.begin(); it != mNotesSprites.end(); ++it ){
		(*it)->hide();
	}
	mNotesSprites.clear();

	// REmove all the accidentals
	for( QCanvasItemList::iterator it = mAccidentalsSprites.begin(); it != mAccidentalsSprites.end(); ++it ){
		(*it)->hide();
	}
	mAccidentalsSprites.clear();

	// Remove all the extra lines
	for( QCanvasItemList::iterator it = mLinesSprites.begin(); it != mLinesSprites.end(); ++it ){
		(*it)->hide();
	}
	mLinesSprites.clear();

	mNotes.clear();
	mMainCanvas->update();
}

void Score::setClefG2(){
	//kdDebug() << "Running Score::setClefG2()" << endl;
	setClef(Score::ClefG2);
}

void Score::setClefC3(){
	//kdDebug() << "Running Score::setClefC3()" << endl;
	setClef(Score::ClefC3);
}

void Score::setClefC4(){
	//kdDebug() << "Running Score::setClefC4()" << endl;
	setClef(Score::ClefC4);
}

void Score::setClefF4(){
	//kdDebug() << "Running Score::setClefF4()" << endl;
	setClef(Score::ClefF4);
}


void Score::setKeySignatureC(){
	//kdDebug() << "Running Score::setKeySignatureC()" << endl;
	setKeySignature(Score::KeySignatureC);
}

void Score::setKeySignatureG(){
	//kdDebug() << "Running Score::setKeySignatureG()" << endl;
	setKeySignature(Score::KeySignatureG);
}

void Score::setKeySignatureD(){
	//kdDebug() << "Running Score::setKeySignatureD()" << endl;
	setKeySignature(Score::KeySignatureD);
}

void Score::setKeySignatureA(){
	//kdDebug() << "Running Score::setKeySignatureA()" << endl;
	setKeySignature(Score::KeySignatureA);
}

void Score::setKeySignatureE(){
	//kdDebug() << "Running Score::setKeySignatureE()" << endl;
	setKeySignature(Score::KeySignatureE);
}

void Score::setKeySignatureB(){
	//kdDebug() << "Running Score::setKeySignatureB()" << endl;
	setKeySignature(Score::KeySignatureB);
}

void Score::setKeySignatureFSharp(){
	//kdDebug() << "Running Score::setKeySignatureFsharp()" << endl;
	setKeySignature(Score::KeySignatureFSharp);
}

void Score::setKeySignatureCSharp(){
	//kdDebug() << "Running Score::setKeySignatureCSharp()" << endl;
	setKeySignature(Score::KeySignatureCSharp);
}

void Score::setKeySignatureF(){
	//kdDebug() << "Running Score::setKeySignatureF()" << endl;
	setKeySignature(Score::KeySignatureF);
}

void Score::setKeySignatureBFlat(){
	//kdDebug() << "Running Score::setKeySignatureBFlat()" << endl;
	setKeySignature(Score::KeySignatureBFlat);
}

void Score::setKeySignatureEFlat(){
	//kdDebug() << "Running Score::setKeySignatureEFlat()" << endl;
	setKeySignature(Score::KeySignatureEFlat);
}

void Score::setKeySignatureAFlat(){
	//kdDebug() << "Running Score::setKeySignatureAFlat()" << endl;
	setKeySignature(Score::KeySignatureAFlat);
}

void Score::setKeySignatureDFlat(){
	//kdDebug() << "Running Score::setKeySignatureDFlat()" << endl;
	setKeySignature(Score::KeySignatureDFlat);
}

void Score::setKeySignatureGFlat(){
	//kdDebug() << "Running Score::setKeySignatureGFlat()" << endl;
	setKeySignature(Score::KeySignatureGFlat);
}

void Score::setKeySignatureCFlat(){
	//kdDebug() << "Running Score::setKeySignatureCFlat()" << endl;
	setKeySignature(Score::KeySignatureCFlat);
}

int Score::score2pixels(int pPos){
	//kdDebug() << "Running Score::score2pixels()" << endl;
	// To turn score steps into pixels,
	// 1) we get the middle of the height of the whole canvas.
	// 2) we remove from the score position (pPos) the position of Score::B which is the B that corresponds to the middle line that we want in the middle
	// 3) we multiply the resultin (from second) score position to get it in pixels relative to the middle of the screen
	// 4) we remove the position relative (the coordinates are upside down) to the middle of the screen
	return mMainCanvas->size().height()/2 - (pPos - position[Score::B]) * 20;
}

int Score::clefCompensate(int pPos){
	if(mClef == Score::ClefC3){
		return pPos -1;
	} else if(mClef == Score::ClefC4){
		return pPos + 1;
	} else if(mClef == Score::ClefF4){
		return pPos -2;
	}
	return pPos;
}

Note::Note(Score::NoteValues pNoteValue, Score::Accidentals pAccidental, int pOctave){
	//kdDebug() << "Running Note::Note(Score::NoteValues pNoteValue, Score::Accidentals pAccidental, int pOctave)" << endl;

	// Check the values, if not, create defaults
	if(pNoteValue >= Score::C and pNoteValue <= Score::B){
		mNoteValue = pNoteValue;
	} else {
		mNoteValue = Score::C;
	}

	if(pAccidental >= Score::None and pAccidental <= Score::DoubleFlat){
		mAccidental = pAccidental;
	} else {
		mAccidental = Score::None;
	}

	mOctave = pOctave;
}

Score::NoteValues Note::noteValue(){
	//kdDebug() << "Running Note::noteValue()" << endl;
	return mNoteValue;
}

bool Note::setNoteValue(Score::NoteValues pNoteValue){
	//kdDebug() << "Running Note::setNoteValue(Score::NoteValues pNoteValue)" << endl;
	if(pNoteValue >= Score::C and pNoteValue <= Score::B){
		mNoteValue = pNoteValue;
		return true;
	}
	return false;
}

Score::Accidentals Note::accidental(){
	//kdDebug() << "Running Note::accidental()" << endl;
	return mAccidental;
}

bool Note::setAccidental(Score::Accidentals pAccidental){
	//kdDebug() << "Running Note::setAccidental(Score::Accidentals pAccidental)" << endl;
	if(pAccidental >= Score::None and pAccidental <= Score::DoubleFlat){
		mAccidental = pAccidental;
		return true;
	}
	return false;
}

int Note::octave(){
	//kdDebug() << "Running Note::octave()" << endl;
	return mOctave;
}

void Note::setOctave(int pOctave){
	//kdDebug() << "Running Note::setOctave(int pOctave)" << endl;
	mOctave = pOctave;
}

#include "Score.moc"
