/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef SRKVIDEOCONTAINER_H
#define SRKVIDEOCONTAINER_H

/**
 * The container of the videoCanvas that implements the commentbox moving-functionality.
 * (resizing is implemented in the commentbox itself)
 *
 * @author Bram Biesbrouck <b@beligum.org>
 */

#include <string>

#include <qscrollview.h>

#include <libinstrudeo/isdcommentbox.h>

#include "srkerror.h"

using namespace std;

#define DEFAULT_COMMENT_TEXT "doubleclick here to set text"
#define VIDEO_MENU_POPUP_NAME "videoPopup"
#define COMMENT_MENU_POPUP_NAME "commentPopup"

class ISDRecording;
class QMouseEvent;
class SRKVideoCanvas;
class SRKCommentbox;
class QPoint;
class QGLContext;
class SRKVideoContainer : public QScrollView
{
    Q_OBJECT

 public:
    //-----CONSTRUCTORS-----
    SRKVideoContainer(QWidget* parent = NULL, const char* name = 0);

    virtual ~SRKVideoContainer();

    //-----METHODS-----
    /**
     * Loads a recording-file from disk and extracts initial data.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode loadRecording(ISDRecording* recording);

    /**
     * Closes the current recording.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode closeRecording();

    /**
     * Change the zoom factor of the displayed video.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode changeZoomFactor(int factor);

    /**
     * Change the language in which the comments are made.
     *
     * @param lang The new language.
     */
    SRKError::SRKErrorCode changeCommentsLanguage(QString& lang);

    /**
     * Adds a commentbox to the recording at this position.
     *
     * @param type The type of the commentbox.
     * @param position The current position of the video.
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode addComment(ISDCommentbox::commentboxType type, int position);
    
    /**
     * Deletes the currently active commentbox. If none is active, nothing happens.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode deleteActiveComment();

    /**
     * Edit the properties of the currently active commentbox. If none is active, nothing happens.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode editActiveCommentProperties();

    /**
     * Ends the active commentbox at the specified position. If none is active, nothing happens.
     * If the position if before the start of the active commentbox, nothing happens (shouldn't be possible).
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode endActiveCommentAtPosition(int position);

    /**
     * Tell the canvas something has changed and it needs to update itself.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode updateCanvas();

    /**
     * Update the comment-wrappers of the comments.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode updateComments();

 signals:
    void firePopupMenu(QString& name, QPoint& pos);
    void fireMarkDirty();

 public slots:
     /**
      * Use this function to aggregate back a popup-call to the main window.
      *
      * @param name The name of the menu in the XMLUI file.
      * @param pos The global position of the mouseclick.
      */
     void popupMenu(QString& name, QPoint& pos);

     /**
      * Callback function for the other widgets to signal a dirty-action.
      */
     void markDirty();

 protected:
    //-----OVERLOADED METHODS-----
    virtual void resizeEvent(QResizeEvent*);
    virtual void contentsMouseMoveEvent(QMouseEvent* e);
    virtual void contentsMousePressEvent(QMouseEvent* e);
    virtual void contentsMouseReleaseEvent(QMouseEvent* e);
    virtual void contentsMouseDoubleClickEvent(QMouseEvent* e);
    
    //-----METHODS-----
    /**
     * Centers the video to the container.
     */
    void recenterVideo();

    /**
     * Stretches the video to the bounds of the container.
     */
    void maximizeVideo();

    /**
     * Is called when the dragging of a commentbox ends.
     */
    void commentDraggingFinished();

    /**
     * Is called when the creation of a commentbox ends.
     */
    void commentCreationFinished();

    /**
     * Toggles the currently active comment.
     *
     * @param comment The comment to activate or NULL to deactivate all comments.
     */
    void setActiveComment(SRKCommentbox* comment);

    /**
     * Creates a commentbox-wrapper from the ISDCombobox object.
     *
     * @param isdBox The Instrudeo commentbox.
     * @return The newly created and allocated wrapper-combobox or NULL if an error occurred.
     */
    SRKCommentbox* createCommentbox(ISDCommentbox* isdBox);

    //-----VARIABLES-----
    SRKVideoCanvas* videoCanvas;
    bool creatingComment;
    bool draggingComment;
    SRKCommentbox* activeComment;
    QPoint commentOffsetPoint;
    ISDRecording* recording;
    QString currentCommentsLanguage;
};

#endif
