/*
 * Seahorse
 *
 * Copyright (C) 2003 Jacob Perkins
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 * Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef __SEAHORSE_KEY_H__
#define __SEAHORSE_KEY_H__

#include <gtk/gtk.h>
#include <gpgme.h>

#include "seahorse-validity.h"

#define SEAHORSE_TYPE_KEY		(seahorse_key_get_type ())
#define SEAHORSE_KEY(obj)		(GTK_CHECK_CAST ((obj), SEAHORSE_TYPE_KEY, SeahorseKey))
#define SEAHORSE_KEY_CLASS(klass)	(GTK_CHECK_CLASS_CAST ((klass), SEAHORSE_TYPE_KEY, SeahorseKeyClass))
#define SEAHORSE_IS_KEY(obj)		(GTK_CHECK_TYPE ((obj), SEAHORSE_TYPE_KEY))
#define SEAHORSE_IS_KEY_CLASS(klass)	(GTK_CHECK_CLASS_TYPE ((klass), SEAHORSE_TYPE_KEY))
#define SEAHORSE_KEY_GET_CLASS(obj)	(GTK_CHECK_GET_CLASS ((obj), SEAHORSE_TYPE_KEY, SeahorseKeyClass))

/* Key info loaded */
typedef enum {
    SKEY_INFO_NONE,
    SKEY_INFO_REMOTE,
    SKEY_INFO_NORMAL,
    SKEY_INFO_COMPLETE
} SeahorseKeyInfo;

/* Possible key changes */
typedef enum {
    SKEY_CHANGE_ALL,
	SKEY_CHANGE_SIGNERS,
	SKEY_CHANGE_PASS,
	SKEY_CHANGE_TRUST,
	SKEY_CHANGE_DISABLED,
	SKEY_CHANGE_EXPIRES,
	SKEY_CHANGE_REVOKERS,
	SKEY_CHANGE_UIDS,
	SKEY_CHANGE_SUBKEYS
} SeahorseKeyChange;

/* Forward declaration */
struct _SeahorseKeySource;

typedef struct _SeahorseKey SeahorseKey;
typedef struct _SeahorseKeyClass SeahorseKeyClass;
	
struct _SeahorseKey
{
	GtkObject		parent;
	
	/*< public >*/
	gpgme_key_t		  key;
    struct _SeahorseKeySource *key_source;
};

struct _SeahorseKeyClass
{
	GtkObjectClass		parent_class;
	
	/* Signal emitted when one of the key's attributes has changed */
	void 			(* changed)		(SeahorseKey		*skey,
							 SeahorseKeyChange	change);
};

GType           seahorse_key_get_type (void);

SeahorseKey*    seahorse_key_new            (struct _SeahorseKeySource *sksrc,
                                             gpgme_key_t        key);

void			seahorse_key_destroy		(SeahorseKey		*skey);

void			seahorse_key_changed		(SeahorseKey		*skey,
							 SeahorseKeyChange	change);

gint			seahorse_key_get_num_uids	(const SeahorseKey	*skey);

gint			seahorse_key_get_num_subkeys	(const SeahorseKey	*skey);

gpgme_subkey_t		seahorse_key_get_nth_subkey	(const SeahorseKey	*skey,
							 const guint		index);

const gchar*		seahorse_key_get_keyid		(const SeahorseKey	*skey,
							 const guint		index);

gchar*          seahorse_key_get_userid	        (const SeahorseKey	*skey,
                                                 const guint        index);

gchar*          seahorse_key_get_userid_name    (const SeahorseKey	*skey,
                                                 const guint        index);

gchar*          seahorse_key_get_userid_email   (const SeahorseKey	*skey,
                                                 const guint        index);

gchar*          seahorse_key_get_userid_comment (const SeahorseKey	*skey,
                                                 const guint        index);

gpgme_user_id_t seahorse_key_get_nth_userid (const SeahorseKey *skey,
                             const guint        index);                            

gchar*			seahorse_key_get_fingerprint	(const SeahorseKey	*skey);

const gchar*		seahorse_key_get_id		(gpgme_key_t		key);

gboolean		seahorse_key_is_valid		(const SeahorseKey	*skey);

gboolean		seahorse_key_can_encrypt	(const SeahorseKey	*skey);

SeahorseValidity	seahorse_key_get_validity	(const SeahorseKey	*skey);

SeahorseValidity	seahorse_key_get_trust		(const SeahorseKey	*skey);

struct _SeahorseKeySource*  seahorse_key_get_source    (const SeahorseKey *skey);

SeahorseKeyInfo seahorse_key_get_loaded_info (SeahorseKey *skey);

#endif /* __SEAHORSE_KEY_H__ */
