#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef __MSW__
# include <windows.h>
#endif

#include <GL/gl.h>

#include "../include/string.h"
#include "../include/disk.h"
#include "../include/tga.h"

#include "gw.h"
#include "stategl.h"

#include "image.h"

#include "sar.h"


sar_image_struct *SARImageLoadFromFile(const char *file);
void SARImageDestroy(sar_image_struct *image);

void SARImageDraw(
	gw_display_struct *display,
	const sar_image_struct *image,
	int x, int y,
	int width, int height
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define IS_STRING_EMPTY(s)      (((s) != NULL) ? ((s) == '\0') : True)

#define RADTODEG(r)     ((r) * 180 / PI)
#define DEGTORAD(d)     ((d) * PI / 180)


/*
 *	Loads and allocates a new image structure, returns the
 *	newly allocated structure or NULL on error.
 *
 *	If the specified file path is not absolute then the
 *	file will be looked for in the local and global data directories.
 */
sar_image_struct *SARImageLoadFromFile(const char *file)
{
	const char *cstrptr;
        int width, height;
        u_int8_t *data;
	sar_image_struct *image;
	struct stat stat_buf;
	char tmp_path[PATH_MAX + NAME_MAX];


	if((file != NULL) ? (*file == '\0') : True)
	    return(NULL);

	if(ISPATHABSOLUTE(file))
	{
	    strncpy(tmp_path, file, PATH_MAX + NAME_MAX);
	}
	else
	{
	    cstrptr = PrefixPaths(dname.local_data, file);
	    if(cstrptr != NULL)
	    {
		if(stat(cstrptr, &stat_buf))
		    cstrptr = PrefixPaths(dname.global_data, file);
	    }

	    strncpy(
		tmp_path,
		(cstrptr != NULL) ? cstrptr : file,
		PATH_MAX + NAME_MAX
	    );
	}
	tmp_path[PATH_MAX + NAME_MAX - 1] = '\0';


	/* File exists? */
#ifndef __MSW__
	if(stat(tmp_path, &stat_buf))
	{
            fprintf(stderr,
                "%s: No such file.\n",
                file
            );
            return(NULL);
	}
	if(!S_ISREG(stat_buf.st_mode))
	{
	    fprintf(stderr,
                "%s: Not a file.\n",
		file
            );
            return(NULL);
	}
#endif

	/* Load data from tga image file. */
	data = TgaReadFromFileFastRGBA(
	    tmp_path, &width, &height, 0x00000000
	);
	if(data == NULL)
	    return(NULL);

	/* Allocate image structure. */
	image = (sar_image_struct *)calloc(1, sizeof(sar_image_struct));
	if(image == NULL)
	{
	    free(data);
	    return(NULL);
	}

	/* Get image structure values. */
	image->type = SAR_IMAGE_TYPE_RGBA;
	image->width = width;
	image->height = height;

	image->data = data;	/* Transfer allocated image data. */
	data = NULL;

	return(image);
}

/*
 *	Deallocates the resources of the image and the image structure
 *	itself.
 */
void SARImageDestroy(sar_image_struct *image)
{
	if(image == NULL)
	    return;

	/* Deallocate image data. */
	free(image->data);

	/* Deallocate image structure itself. */
	free(image);
}

/*
 *	Draws the image to the screen at the specified width and
 *	height.
 */
void SARImageDraw(
	gw_display_struct *display,
	const sar_image_struct *image,
	int x, int y,
	int width, int height
)
{
	StateGLBoolean alpha_test;
	GLint x2, y2;
	GLfloat x_zoom_factor, y_zoom_factor;
	int win_width, win_height;


	if((display == NULL) || (image == NULL))
	    return;

	if((image->width < 1) || (image->height < 1) ||
           (image->data == NULL)
	)
	    return;

        GWContextGet(
            display, GWContextCurrent(display),
            NULL, NULL,
            NULL, NULL,
            &win_width, &win_height
        );

	/* Calculate GL converted coordinate positions. */
	x2 = (GLint)x;
	y2 = (GLint)(win_height - y - 1);

	/* Sanitize positions. */
	if(x2 < 0)
	    x2 = 0;
	if(y2 < 0)
	    y2 = 0;

	/* Calculate zoom factors. */
        x_zoom_factor = (GLfloat)width / (GLfloat)image->width;
        y_zoom_factor = -(GLfloat)height / (GLfloat)image->height;
 

	/* Set starting drawing position. */
	glRasterPos2i(
	    CLIP(x2, 0, win_width - 1),
	    CLIP(y2, 0, win_height - 1)
	);

	/* Set zoom factor. */
	glPixelZoom(x_zoom_factor, y_zoom_factor);

	/* Enable GL alpha testing (some images have transparent pixels). */
	alpha_test = display->state_gl.alpha_test;
	StateGLEnable(&display->state_gl, GL_ALPHA_TEST);
	StateGLAlphaFunc(&display->state_gl, GL_GREATER, 0.5);

	/* Draw the image. */
	glDrawPixels(
	    image->width, image->height,
	    GL_RGBA, GL_UNSIGNED_BYTE,
	    (const GLvoid *)image->data
	);

	/* Disable GL alpha testing. */
	if(!alpha_test)
	    StateGLDisable(&display->state_gl, GL_ALPHA_TEST);

	/* Restore zoom. */
        glPixelZoom(1.0f, 1.0f);
}
