#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "../include/fio.h"
#include "../include/disk.h"
#include "../include/string.h"

#include "sound.h"
#include "musiclistio.h"
#include "sar.h"
#include "config.h"


sar_music_ref_struct *SARMusicMatchPtr(
        sar_music_ref_struct **ml, int total,
        int id, int *music_ref_num
);

void SARMusicListDeleteAll(
        sar_music_ref_struct ***ml, int *total
);
int SARMusicListLoadFromFile(
        const char *filename,
	sar_music_ref_struct ***ml, int *total
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	Matches the music reference structure in the given list with
 *	the given id code.
 */
sar_music_ref_struct *SARMusicMatchPtr(
        sar_music_ref_struct **ml, int total,
	int id, int *music_ref_num
)
{
	int i;
	sar_music_ref_struct *mr_ptr;


	if(music_ref_num != NULL)
	    (*music_ref_num) = -1;

	if((ml == NULL) || (total < 1) || (id < 0))
	    return(NULL);

	for(i = 0; i < total; i++)
	{
	    mr_ptr = ml[i];
	    if(mr_ptr == NULL)
		continue;

	    if(mr_ptr->id == id)
	    {
		if(music_ref_num != NULL)
		    (*music_ref_num) = i;
		return(mr_ptr);
	    }
	}

	return(NULL);
}


/*
 *      Deletes music files reference list.
 */
void SARMusicListDeleteAll(
        sar_music_ref_struct ***ml, int *total
)
{
        int i;
	sar_music_ref_struct *mr_ptr;

        if((ml == NULL) || (total == NULL))
            return;

        for(i = 0; i < (*total); i++)
        {
            mr_ptr = (*ml)[i];
            if(mr_ptr == NULL)
                continue;

            free(mr_ptr->filename);
            free(mr_ptr);
        }

        if(*ml != NULL)
        {
            free(*ml);
            *ml = NULL;
        }
        *total = 0;
}

/*
 *      Loads music file references list from file.
 *
 *      Given pointer array should be set to valid values. It will be
 *      deallocated first before the file is loaded.
 */
int SARMusicListLoadFromFile(
        const char *filename,
        sar_music_ref_struct ***ml, int *total
)
{
        int i;
        FILE *fp;
	char *buf = NULL;
        struct stat stat_buf;  
        double value[10];
	sar_music_ref_struct *mr_ptr = NULL;

        if((filename == NULL) || (ml == NULL) || (total == NULL))
            return(-1);

        /* Deallocate any existing music references in the list. */
        SARMusicListDeleteAll(ml, total);

        if(stat(filename, &stat_buf))
        {
            fprintf(stderr, "%s: No such file.\n", filename);
            return(-1);
        }
#ifdef S_ISREG
        if(!S_ISREG(stat_buf.st_mode))
        {
            fprintf(stderr,
                "%s: Not a file.\n",
                filename
            );
            return(-1);
        }
#endif  /* S_ISREG */

        /* Open file. */
        fp = FOpen(filename, "rb");
        if(fp == NULL)   
        {
            fprintf(stderr, "%s: Cannot open.\n", filename);
            return(-1);
        }

        do
        {
            buf = FSeekNextParm(
                fp,
                buf,
                SAR_COMMENT_CHAR,
                SAR_CFG_DELIM_CHAR
            );
            if(buf == NULL)
                break;

            if(!strcasecmp(buf, "Version"))
            {
                FGetValuesF(fp, value, 2);

            }

            else if(!strcasecmp(buf, "MusicAdd"))
            {
                char *strptr = FGetString(fp);

                mr_ptr = NULL;


                i = *total;
                *total = i + 1;
                *ml = (sar_music_ref_struct **)realloc(
                    *ml,
                    (*total) * sizeof(sar_music_ref_struct *)
                );
                if(*ml == NULL)
                {
                    *total = 0;
                }
                else
                {
                    mr_ptr = (sar_music_ref_struct *)calloc(
                        1, sizeof(sar_music_ref_struct)
                    );
                    (*ml)[i] = mr_ptr;
                }


                /* Set msuic id code. */
                if(mr_ptr != NULL)
		{
		    mr_ptr->id = atoi(strptr);
		}

		free(strptr);
            }
            else if(!strcasecmp(buf, "MusicFileName"))
            {
                char *strptr = FGetString(fp);

                if(mr_ptr != NULL)
                {
                    free(mr_ptr->filename);
                    mr_ptr->filename = strptr;
                }
                else
                {
                    free(strptr);
                }
            }
            else if(!strcasecmp(buf, "MusicFlags"))
            {
		char *strptr = FGetString(fp);
		const char *cstrptr;


		/* Get start of value string as cstrptr. */
		cstrptr = (const char *)strptr;

		/* Itterate through value string, checking each flag. */
                while((cstrptr != NULL) ? ((*cstrptr) != '\0') : 0)
                {
                    if(strcasepfx(cstrptr, "repeating"))
			mr_ptr->flags |= SAR_MUSIC_REF_FLAGS_REPEAT;
		    else if(strcasepfx(cstrptr, "fade_in") ||
                            strcasepfx(cstrptr, "fadein")
		    )
                        mr_ptr->flags |= SAR_MUSIC_REF_FLAGS_FADE_IN;
                    else if(strcasepfx(cstrptr, "fade_out") || 
                            strcasepfx(cstrptr, "fadeout")
                    )
                        mr_ptr->flags |= SAR_MUSIC_REF_FLAGS_FADE_OUT;

		    while(!ISBLANK(*cstrptr) && ((*cstrptr) != '\0'))
			cstrptr++;
		    while(ISBLANK(*cstrptr))
			cstrptr++;
		}

		free(strptr);
            }

            else
            {
                fprintf(
		    stderr,
                    "%s: Unsupported parameter `%s'.\n",
                    filename, buf
                );
                FSeekNextLine(fp);
            }

        } while(1);

        /* Close file. */
        FClose(fp);

	return(0);
}
