/*
                    SAR Objects and Scene Structures

            All coordinates and vectors use left hand rule.

 */

#ifndef OBJ_H
#define OBJ_H

#include <sys/types.h>

#include "sfm.h"
#include "v3dtex.h"
#include "sound.h"


/*
 *	Engine flags type:
 */
#define sar_obj_flags_t	unsigned long

/*
 *	Gradient animation counter type:
 */
#define sar_grad_anim_t	u_int16_t
#define SAR_GRAD_ANIM_COEFF(x)	((float)(x) / (float)((sar_grad_anim_t)-1))

/*
 *	DMS units type:
 */
#define sar_dms_t	float


/*
 *	Air worthy states:
 */
#define SAR_AIR_WORTHY_NOT_FLYABLE	0	/* Pile of junk. */
#define SAR_AIR_WORTHY_OUT_OF_CONTROL	1
#define SAR_AIR_WORTHY_FLYABLE		2

/*
 *	Engine state:
 */
#define SAR_ENGINE_STATE_OFF		0
#define SAR_ENGINE_STATE_INIT		1
#define SAR_ENGINE_STATE_ON		2

/*
 *	Human color palette index and max:
 */
#define SAR_HUMAN_COLOR_FACE            0
#define SAR_HUMAN_COLOR_HAIR            1
#define SAR_HUMAN_COLOR_TORSO           2
#define SAR_HUMAN_COLOR_HIPS            3
#define SAR_HUMAN_COLOR_LEGS            4
#define SAR_HUMAN_COLOR_FEET            5
#define SAR_HUMAN_COLOR_ARMS            6
#define SAR_HUMAN_COLOR_HANDS           7

#define SAR_HUMAN_COLORS_MAX            8


/*
 *	Color structure:
 */
typedef struct {
	float	a, r, g, b;	/* 0.0 to 1.0 */
} sar_color_struct;
#define SAR_COLOR(p)	((sar_color_struct *)(p))


/*
 *	Visual model structure:
 */
typedef struct {

#define SAR_VISUAL_MODEL_NOT_LOADED	0
#define SAR_VISUAL_MODEL_LOADING	1
#define SAR_VISUAL_MODEL_LOADED		2

	int load_state;		/* One of SAR_VISUAL_MODEL_*. */

	int ref_count;		/* Reference count. */

	char *filename;		/* Can be NULL. */
	char *name;		/* Can be NULL. */

	void *data;		/* A GLuint referencing a GL list. */

} sar_visual_model_struct;
#define SAR_VISUAL_MODEL(p)	((sar_visual_model_struct *)(p))


/*
 *	Position structure:
 */
typedef struct {
	float	x, y, z;		/* In meters. */
} sar_position_struct;
#define SAR_POSITION(p)		((sar_position_struct *)(p))

/*
 *	Direction structure:
 */
typedef struct {
	float	heading, pitch, bank;	/* In radians. */
} sar_direction_struct;
#define SAR_DIRECTION(p)	((sar_direction_struct *)(p))

/*
 *	Scale structure:
 */
typedef struct {
	float	x, y, z;		/* Coefficient. */
} sar_scale_struct;
#define SAR_SCALE(p)		((sar_scale_struct *)(p))


/*
 *	Intercept coordinate structure:
 */
typedef struct {

	sar_obj_flags_t	flags;
	float	x, y, z;	/* Center of intercept, in meters. */
	float	radius;		/* Cylendrical size of intercept, in meters. */
	float	urgency;	/* Specifies the "urgency" of which this
				 * intercept must be reached (used for a
				 * variety of purposes).
				 */
	char	*name;		/* Intercept name/label. */

} sar_obj_intercept_struct;

#define SAR_OBJ_INTERCEPT(p)	((sar_obj_intercept_struct *)(p))


/*
 *	Light structure:
 */
typedef struct {

#define SAR_LIGHT_FLAG_ON		(1 << 0)
#define SAR_LIGHT_FLAG_STROBE		(1 << 1)
#define SAR_LIGHT_FLAG_ATTENUATE	(1 << 2)

	sar_obj_flags_t flags;

	sar_position_struct pos;
	sar_direction_struct dir;

	int radius;	/* In pixels. */

	sar_color_struct color;			/* Apperance color. */
	sar_color_struct attenuate_color;	/* Attenuate color. */

	/* On/off timers (if the SAR_LIGHT_FLAG_STROBE flag is set)
	 * in milliseconds.
	 */
	time_t	next_on, int_on,
		next_off, int_off;
	time_t	int_delay_on;		/* Additional interval to wait
					 * before turning light on when
					 * timer is reset.
					 */
} sar_light_struct;
#define SAR_LIGHT(p)	((sar_light_struct *)(p))


/*
 *	Physical contact and crash boundaries structure:
 *
 *	Note that any object can have this contact bounds, but will be
 *	have to follow some special rules if they do.
 *
 *	First if the object is a human or aircraft, it should have only
 *	the SAR_CRASH_FLAG_CRASH_OTHER flag set and it's contact
 *	shape should be SAR_CONTACT_SHAPE_CYLENDRICAL.
 *
 *	Buildings and other objects that you want to set
 *	SAR_CRASH_FLAG_CRASH_CAUSE can have any shape (recommend
 *	SAR_CONTACT_SHAPE_CYLENDRICAL or SAR_CONTACT_SHAPE_RECTANGULAR).
 *
 *	Now objects that you want to set landable or walkable (with
 *	crash_flags set to SAR_CRASH_FLAG_SUPPORT_SURFACE) may not be
 *	human or aircrafts (objects that themselves need to be checked
 *	if on top of something).
 *
 *	If an object is set SAR_CRASH_FLAG_SUPPORT_SURFACE then the object
 *	can only have its heading rotated.
 *
 *	Note that object's of type SAR_OBJ_TYPE_GROUND are always
 *	landable/walkable and their shape is always cylendrical.
 */
typedef struct {

        /* Crash detection flags, specifies if this object can collide
         * with other objects, if other objects can collide with
         * this object, and if this object produces a flat support
	 * surface that is landable/walkable (respectivly with defined
	 * flags).
	 *
	 * If no flags are set it can be considered a quick way
	 * (ie if(crash_flags == 0)) to determine this object has no
         * crash interaction with other objects.
         */
#define SAR_CRASH_FLAG_CRASH_OTHER      (1 << 0)
#define SAR_CRASH_FLAG_CRASH_CAUSE      (1 << 1)
#define SAR_CRASH_FLAG_SUPPORT_SURFACE	(1 << 2)
        sar_obj_flags_t crash_flags;

        /* This value used if and only if the SAR_CRASH_FLAG_CRASH_CAUSE
	 * flag is set in crash_flags. This value specifies what type of
         * crash it would be if some other object crashed into this
	 * object.
         */
#define SAR_CRASH_TYPE_OBSTRUCTION      0       /* General structure. */
#define SAR_CRASH_TYPE_GROUND           1
#define SAR_CRASH_TYPE_MOUNTAIN         2
#define SAR_CRASH_TYPE_BUILDING         3
#define SAR_CRASH_TYPE_AIRCRAFT         4
        int crash_type;

        /* Body contact shape, one of SAR_CONTACT_SHAPE_*. */
#define SAR_CONTACT_SHAPE_SPHERICAL     0   
#define SAR_CONTACT_SHAPE_CYLENDRICAL   1
#define SAR_CONTACT_SHAPE_RECTANGULAR	2
        int contact_shape;

        /* Radius value in meters for contact shapes
	 * SAR_CONTACT_SHAPE_SPHERICAL or SAR_CONTACT_SHAPE_CYLENDRICAL.
	 */
        float contact_radius;

        /* Cylendrical height value in meters for contact shape
	 * SAR_CONTACT_SHAPE_CYLENDRICAL.
         */
        float contact_h_min, contact_h_max;

	/* Rectangular contact shape in meters for contact shape
	 * SAR_CONTACT_SHAPE_RECTANGULAR (left hand rule).
	 */
	float	contact_x_min, contact_x_max,
		contact_y_min, contact_y_max,
		contact_z_min, contact_z_max;

	/* Value records for the object's inversed heading for the trig
	 * functions, this is used to speed up rotations about heading
	 * for SAR_CONTACT_SHAPE_RECTANGULAR shape contact.
	 *
	 * Ie cos_heading = cos(-heading) and sin_heading = sin(-heading)
	 */
	float	cos_heading,
		sin_heading;

} sar_contact_bounds_struct;

#define SAR_CONTACT_BOUNDS(p)	((sar_contact_bounds_struct *)(p))


/*
 *      Sound source structure:
 *
 *      Specifies a location (and optional direction) of a sound
 *      to be played.
 */
typedef struct {

	char	*name;		/* Arbitary name to identify this sound
				 * source from a list of sound sources.
				 */

	char	*filename,	/* Full path to sound object. */
		*filename_far;

	float	range,		/* Maximum range, in meters. */
		range_far;	/* Play far sound if beyond this range,
				 * in meters.
				 */

        sar_position_struct pos;        /* Offset from center of object. */

	float	cutoff;		/* Cutoff angle in radians, can be 0.0 to
				 * specify "in all directions".
				 */
	sar_direction_struct dir;	/* Direction, used only if cutoff
					 * is positive.
					 */

        /* Sample rate limit, the amount that the sample rate
         * can increase to. For example, if the sound object's sample
         * rate is 11025 hz and sample_rate_limit is 30000 hz, then
         * the sound object's sample rate can be increased from
         * 11025 to 30000 hz.
         */
	int	sample_rate_limit;

} sar_sound_source_struct;

#define SAR_SOUND_SOURCE(p)	((sar_sound_source_struct *)(p))


/*
 *	Rotor/propellar structure:
 */
typedef struct {

/* Can pitch forward for duel propulsion direction aircraft like the
 * V-22 Ospray.
 */
#define SAR_ROTOR_FLAG_CAN_PITCH	(1 << 1)

/* Is in `forward' prop motion position (pitched forwards if flag set).
 * Note that this has no affect on actual flight physics, the member
 * flight_model_type on the aircraft structure will determine the
 * correct flight model type.
 */
#define SAR_ROTOR_FLAG_PITCH_STATE	(1 << 2)

/* Cannot pitch rotors when landed on ground if this flag is set. */
#define SAR_ROTOR_FLAG_NO_PITCH_LANDED	(1 << 3)

/* Does not rotate (spin) about its Z axis, used for stationary
 * rotor housing.
 */
#define SAR_ROTOR_FLAG_NO_ROTATE	(1 << 4)

/* Pitches and banks along with control positions if any. */
#define SAR_ROTOR_FLAG_FOLLOW_PB	(1 << 5)

/* Blades will blur using the blades_blur_color if spinning too fast. */
#define SAR_ROTOR_FLAG_BLADES_BLUR_FAST		(1 << 6)

/* Blades always blur (overrides SAR_ROTOR_FLAG_BLADES_BLUR_FAST). */
#define SAR_ROTOR_FLAG_BLADES_BLUR_ALWAYS	(1 << 7)



	sar_obj_flags_t flags;

	/* Offset relative to center of object. */
        sar_position_struct pos;

	/* Rotation (always spins on Y axis). */
        sar_direction_struct dir;

	/* Radius in meters, for knowing size to make rotor wash. */
	float radius;

	/* Offset to blades of rotor (for drawing blured rotor at the
	 * proper distance from the center).
	 */
	float blades_offset;

	/* Spin animation position. */
	sar_grad_anim_t	anim_pos;

        sar_visual_model_struct *visual_model;

	/* Rotor pitch rotate position, 0 to (sar_grad_anim_t)-1,
	 * where (sar_grad_anim_t)-1 is pitched forwards
	 * (when SAR_ROTOR_FLAG_PITCH_STATE is set).
	 */
	sar_grad_anim_t pitch_anim_pos;

	/* Rotor wash animation position. */
	sar_grad_anim_t rotor_wash_anim_pos;

	/* Rotor wash texture number on scene, can be -1. */
	int rotor_wash_tex_num;

	/* Control position coefficients, used if
	 * SAR_ROTOR_FLAG_FOLLOW_PB is set.
	 */
	float control_coeff_pitch, control_coeff_bank;

	/* Blades blur color, used if spinning too fast and
	 *
	 */
	sar_color_struct blades_blur_color;

} sar_obj_rotor_struct;

#define SAR_OBJ_ROTOR(p)	((sar_obj_rotor_struct *)(p))


/*
 *	Generic moveable part (flaps, ailerons, elevator, rudder, air
 *	brakes, landing gears, canopies, and doors):
 */
typedef struct {

#define SAR_OBJ_PART_FLAG_STATE		(1 << 0)	/* Set if opened/extended. */
#define SAR_OBJ_PART_FLAG_HIDE_MIN	(1 << 1)	/* Hide when animation value is at min. */
#define SAR_OBJ_PART_FLAG_HIDE_MAX	(1 << 2)	/* Hide when animation value is at max. */

/* Following flags are object part type specific, some may have bits
 * that are identical to other flags of this domain but there will be
 * no conflicts because one type will never have flags set from another.
 * For example if the part is of type door, then flags will only be
 * door flags and never say landing gear flags.
 */
/* Doors, rescue doors, and canopies. */
#define SAR_OBJ_PART_FLAG_DOOR_FIXED	(1 << 3)	/* True if fixed (always closed). */
#define SAR_OBJ_PART_FLAG_DOOR_LOCKED	(1 << 4)	/* True if locked. */
#define SAR_OBJ_PART_FLAG_DOOR_STAY_OPEN	(1 << 5)	/* Do not close door on
								 * next loop check,
								 * ie if door was opened by player explicitly.
								 */
/* Landing gears. */
#define SAR_OBJ_PART_FLAG_LGEAR_FIXED	(1 << 3)	/* True if fixed (always down). */
#define SAR_OBJ_PART_FLAG_LGEAR_DAMAGED	(1 << 4)	/* True if damaged. */
#define SAR_OBJ_PART_FLAG_LGEAR_MISSING	(1 << 5)	/* Non-existant. */
#define SAR_OBJ_PART_FLAG_LGEAR_SKI	(1 << 6)	
#define SAR_OBJ_PART_FLAG_LGEAR_FLOATS	(1 << 7)	/* Can land on water. */


/* Object part types. */
#define SAR_OBJ_PART_TYPE_AILERON_LEFT		0
#define SAR_OBJ_PART_TYPE_AILERON_RIGHT		1
#define SAR_OBJ_PART_TYPE_RUDDER_TOP		2
#define SAR_OBJ_PART_TYPE_RUDDER_BOTTOM		3
#define SAR_OBJ_PART_TYPE_ELEVATOR		4
#define SAR_OBJ_PART_TYPE_CANNARD		5	/* Forward elevator. */
#define SAR_OBJ_PART_TYPE_AILERON_ELEVATOR_LEFT		6	/* Combo. */
#define SAR_OBJ_PART_TYPE_AILERON_ELEVATOR_RIGHT	7	/* Combo. */
#define SAR_OBJ_PART_TYPE_FLAP			8
#define SAR_OBJ_PART_TYPE_AIR_BRAKE		9
#define SAR_OBJ_PART_TYPE_DOOR			10
#define SAR_OBJ_PART_TYPE_DOOR_RESCUE		11
#define SAR_OBJ_PART_TYPE_CANOPY		12
#define SAR_OBJ_PART_TYPE_LANDING_GEAR		13

	int type;		/* One of SAR_MOVEABLE_PART_TYPE_*. */

        sar_obj_flags_t flags;

        /* Offset relative to center of object, pos_min and pos_max
	 * are deltas relative to pos_cen
	 *
	 * Note that some part types interprite these values differently
	 */
        sar_position_struct pos_min, pos_cen, pos_max;

        /* Direction, dir_min and dir_max are deltas relative to 
	 * dir_cen
	 *
	 * Note that some part types interprite these values differently
	 */
        sar_direction_struct dir_min, dir_cen, dir_max;

        sar_grad_anim_t anim_pos,       /* 0 to (sar_grad_anim_t)-1. */
                        anim_rate;      /* Units per cycle. */

        sar_visual_model_struct *visual_model;

} sar_obj_part_struct;

#define SAR_OBJ_PART(p)		((sar_obj_part_struct *)(p))


/*
 *	External fuel tank structure (a fuel tank on an object):
 *
 *	This is a structure that is a member of the
 *	sar_object_aircraft_struct structure and not to be confused with
 *	the sar_object_fueltank_struct which is an object itself
 *	(a fuel tank dropped from an aircraft as an independant object).
 */
typedef struct {

#define SAR_EXTERNAL_FUELTANK_FLAG_FIXED	(1 << 0)	/* Not droppable. */
#define SAR_EXTERNAL_FUELTANK_FLAG_ONBOARD	(1 << 1)	/* Not jettesoned. */

	sar_obj_flags_t flags;

	/* Offset from center of object in meters. */
	sar_position_struct offset_pos;

	/* Spherical contact bounds radius in meters. */
	float	radius;

	/* Mass, all in kg. */
	float	dry_mass, fuel, fuel_max;

	/* Visual model. */
	sar_visual_model_struct *visual_model;

} sar_external_fueltank_struct;

#define SAR_EXTERNAL_FUELTANK(p)	((sar_external_fueltank_struct *)(p))


/*
 *	Hoist (rope with basket or rescuer at the end) structure:
 */
typedef struct {

	/* Offset from center of object. */
	sar_position_struct offset;

	/* Position (not offset) of basket. */
	sar_position_struct pos;

	/* Direction of basket. */
	sar_direction_struct dir;

	/* Rope extention (if <= 0.0 then implies it is fully retracted)
         * all units in meters.
	 */
	float	rope_cur,
		rope_max;
	float	rope_rate;	/* In meters per cycle. */

	/* Rope visual extension, since rope_cur may be longer than the
	 * distance to the landable ground (in meters) we need to use
	 * this value when drawing.
	 */
	float rope_cur_vis;

	/* Indicates rope end is in contact with ground if true. */
	int on_ground;

	/* Hoist deployment type, determines what's on the end of the
	 * hoist. Can be either SAR_HOIST_DEPLOYMENT_BASKET or
	 * SAR_HOIST_DEPLOYMENT_DIVER. Note that contact bounds are always
	 * checked the same way regardless of what is on the end of the 
	 * rope.
	 */
#define SAR_HOIST_DEPLOYMENT_BASKET	0
#define SAR_HOIST_DEPLOYMENT_DIVER	1
	int deployment;

	/* Cylendrical contact area of rescue basket, in meters. */
	float contact_radius;
	float contact_z_min, contact_z_max;

	/* Load capacity, in kg. */
	float capacity;

	/* Reference to occupant (people) object index numbers.
	 * This does not include the diver.
	 */
	int	*occupant;
	int	total_occupants;

	/* Total mass of occupant object(s) in kg, this is to speed up
	 * calculations. The value is updated when a new occupant object
	 * is added to the hoist and reset to 0 when the hoist is deployed
	 * the next time.
	 */
	float	occupants_mass;

	/* Texture reference numbers on scene structure. */
	int	side_tex_num, end_tex_num, bottom_tex_num,
		water_ripple_tex_num;

	/* Diver color palette. */
	sar_color_struct diver_color[SAR_HUMAN_COLORS_MAX];

	/* Animation position and rate. */
	sar_grad_anim_t anim_pos, anim_rate;

} sar_obj_hoist_struct;

#define SAR_OBJ_HOIST(p)	((sar_obj_hoist_struct *)(p))


/*
 *	Object type codes:
 */
#define SAR_OBJ_TYPE_GARBAGE		0
#define SAR_OBJ_TYPE_STATIC		1
#define SAR_OBJ_TYPE_AUTOMOBILE		2
#define SAR_OBJ_TYPE_WATERCRAFT		3
#define SAR_OBJ_TYPE_AIRCRAFT		4
/* Note type 5 used to be airplane, it's now changed to be part of
 * type 4 (which was helicopter) so both 4 and 5 are all flying things
 * now.
 */

/* Cylendrical ground base with (optional) touchable heightfield. */
#define SAR_OBJ_TYPE_GROUND		6
/* Runway. */
#define SAR_OBJ_TYPE_RUNWAY		7
/* Helipad. */
#define SAR_OBJ_TYPE_HELIPAD		8

/* Human. */
#define SAR_OBJ_TYPE_HUMAN		9

/* Smoke trail. */
#define SAR_OBJ_TYPE_SMOKE		10
/* Fire. */
#define SAR_OBJ_TYPE_FIRE		11
/* Explosion. */
#define SAR_OBJ_TYPE_EXPLOSION		12
/* Chemical spray (water, fire-retardant, etc). */
#define SAR_OBJ_TYPE_CHEMICAL_SPRAY	13
/* Aircraft fuel tank (dropping from the sky, not a fuel tank that is on
 * an aircraft).
 */
#define SAR_OBJ_TYPE_FUELTANK		14

/* Premodeled object. */
#define SAR_OBJ_TYPE_PREMODELED		20

/*
 *	Object type check macros:
 */
#define SAR_OBJ_IS_STATIC(p)	\
(((p) != NULL) ? ((*(int *)(p)) == SAR_OBJ_TYPE_STATIC) : 0)

#define SAR_OBJ_IS_AUTOMOBILE(p)	\
(((p) != NULL) ? ((*(int *)(p)) == SAR_OBJ_TYPE_AUTOMOBILE) : 0)

#define SAR_OBJ_IS_WATERCRAFT(p)	\
(((p) != NULL) ? ((*(int *)(p)) == SAR_OBJ_TYPE_WATERCRAFT) : 0)

#define SAR_OBJ_IS_AIRCRAFT(p)	\
(((p) != NULL) ? ((*(int *)(p)) == SAR_OBJ_TYPE_AIRCRAFT) : 0)

#define SAR_OBJ_IS_GROUND(p)	\
(((p) != NULL) ? ((*(int *)(p)) == SAR_OBJ_TYPE_GROUND) : 0)

#define SAR_OBJ_IS_RUNWAY(p)	\
(((p) != NULL) ? ((*(int *)(p)) == SAR_OBJ_TYPE_RUNWAY) : 0)

#define SAR_OBJ_IS_HELIPAD(p)	\
(((p) != NULL) ? ((*(int *)(p)) == SAR_OBJ_TYPE_HELIPAD) : 0)

#define SAR_OBJ_IS_HUMAN(p)	\
(((p) != NULL) ? ((*(int *)(p)) == SAR_OBJ_TYPE_HUMAN) : 0)

#define SAR_OBJ_IS_SMOKE(p)	\
(((p) != NULL) ? ((*(int *)(p)) == SAR_OBJ_TYPE_SMOKE) : 0)

#define SAR_OBJ_IS_FIRE(p)	\
(((p) != NULL) ? ((*(int *)(p)) == SAR_OBJ_TYPE_FIRE) : 0)

#define SAR_OBJ_IS_EXPLOSION(p)	\
(((p) != NULL) ? ((*(int *)(p)) == SAR_OBJ_TYPE_EXPLOSION) : 0)

#define SAR_OBJ_IS_CHEMICAL_SPRAY(p)	\
(((p) != NULL) ? ((*(int *)(p)) == SAR_OBJ_TYPE_CHEMICAL_SPRAY) : 0)

#define SAR_OBJ_IS_FUELTANK(p)	\
(((p) != NULL) ? ((*(int *)(p)) == SAR_OBJ_TYPE_FUELTANK) : 0)

#define SAR_OBJ_IS_PREMODELED(p)	\
(((p) != NULL) ? ((*(int *)(p)) == SAR_OBJ_TYPE_PREMODELED) : 0)

/*
 *	Object type substructure pointer get macros, these macros will
 *	return a pointer to the substructure (member data) with cast or
 *	NULL if the given object pointer is NULL or not of the
 *	corresponding type.
 */
#define SAR_OBJ_GET_STATIC(p)	\
(sar_object_static_struct *)((SAR_OBJ_IS_STATIC(p)) ? \
 ((sar_object_struct *)(p))->data : NULL \
)

#define SAR_OBJ_GET_AIRCRAFT(p)	\
(sar_object_aircraft_struct *)((SAR_OBJ_IS_AIRCRAFT(p)) ? \
 ((sar_object_struct *)(p))->data : NULL \
)

#define SAR_OBJ_GET_GROUND(p)	\
(sar_object_ground_struct *)((SAR_OBJ_IS_GROUND(p)) ? \
 ((sar_object_struct *)(p))->data : NULL \
)

#define SAR_OBJ_GET_RUNWAY(p)	\
(sar_object_runway_struct *)((SAR_OBJ_IS_RUNWAY(p)) ? \
 ((sar_object_struct *)(p))->data : NULL \
)

#define SAR_OBJ_GET_HELIPAD(p)	\
(sar_object_helipad_struct *)((SAR_OBJ_IS_HELIPAD(p)) ? \
 ((sar_object_struct *)(p))->data : NULL \
)

#define SAR_OBJ_GET_HUMAN(p)	\
(sar_object_human_struct *)((SAR_OBJ_IS_HUMAN(p)) ? \
 ((sar_object_struct *)(p))->data : NULL \
)

#define SAR_OBJ_GET_SMOKE(p)	\
(sar_object_smoke_struct *)((SAR_OBJ_IS_SMOKE(p)) ? \
 ((sar_object_struct *)(p))->data : NULL \
)

#define SAR_OBJ_GET_FIRE(p)	\
(sar_object_fire_struct *)((SAR_OBJ_IS_FIRE(p)) ? \
 ((sar_object_struct *)(p))->data : NULL \
)

#define SAR_OBJ_GET_EXPLOSION(p)	\
(sar_object_explosion_struct *)((SAR_OBJ_IS_EXPLOSION(p)) ? \
 ((sar_object_struct *)(p))->data : NULL \
)

#define SAR_OBJ_GET_CHEMICAL_SPRAY(p)	\
(sar_object_chemical_spray_struct *)((SAR_OBJ_IS_CHEMICAL_SPRAY(p)) ? \
 ((sar_object_struct *)(p))->data : NULL \
)

#define SAR_OBJ_GET_FUELTANK(p)	\
(sar_object_fueltank_struct *)((SAR_OBJ_IS_FUELTANK(p)) ? \
 ((sar_object_struct *)(p))->data : NULL \
)

#define SAR_OBJ_GET_PREMODELED(p)	\
(sar_object_premodeled_struct *)((SAR_OBJ_IS_PREMODELED(p)) ? \
 ((sar_object_struct *)(p))->data : NULL \
)


/*
 *	Flight model codes, used in member flight_model_type of
 *	aircraft structure sar_object_aircraft_struct.
 */
#define SAR_FLIGHT_MODEL_HELICOPTER	0
#define SAR_FLIGHT_MODEL_AIRPLANE	1
#define SAR_FLIGHT_MODEL_SLEW		2


/*
 *	SAR aircraft (helicopter/airplane) object sub structure:
 *
 *	This structure (and all object substructures) can be pointed to
 *	from member `data' in the sar_object_struct, determined by member
 *	`type'.
 */
typedef struct {

	/* Flight model type, one of SAR_FLIGHT_MODEL_*. */
	int flight_model_type;

	/* Previously set flight model type (for switching back and forth
	 * between flight model types.
	 */
	int last_flight_model_type;

	/* Sar flight model flight dynamics model structure. */
	SFMModelStruct *fdm;

        /* Air worthy state, one of SAR_AIR_WORTHY_*. */
        int air_worthy_state;

        /* Current speed, in meters per cycle. */
        float speed;

	/* Stall speed, the speed at which a controllable stall begins.
	 * Loss of lift can still occure twice beyond this value, in
	 * meters per cycle.
	 */
	float speed_stall, stall_coeff;

	/* Maximum speed (this is typically *several* times the rated
	 * maximum speed), in meters per cycle.
	 */
	float speed_max;

	/* overspeed_expected is the overspeed value according to the
	 * specifications of the aircraft. If the speed reaches or
	 * exceeds this value then overspeed warnings and affects are
	 * performed, units in meters per cycle.
	 *
	 * overspeed is the speed at which the aircraft will incure
	 * damage, in meters per cycle.
	 */
	float	overspeed_expected,
		overspeed;

	/* Current velocity vector representing velocity in object
	 * relative vector direction, in meters per cycle.
	 */
        sar_position_struct vel;

	/* Minimum drag in meters per cycle, must be non-negative. */
	float min_drag;

	/* Acceleration response for when flight_model_type is set to
	 * SAR_FLIGHT_MODEL_HELICOPTER. Higher values produce less
	 * responsiveness).
	 */
	sar_position_struct accel_responsiveness;

	/* Acceleration response for when flight_model_type is set to
	 * SAR_FLIGHT_MODEL_AIRPLANE. Higher values produce less
	 * responsiveness).
	 */
	sar_position_struct airplane_accel_responsiveness;


	/* Attitude change rates, in radians per cycle */
	sar_direction_struct attitude_change_rate;

	/* Pitch and bank leveling, in radians per cycle */
	float pitch_leveling, bank_leveling;

	/* Ground pitch offset in radians (can be negative) */
	float ground_pitch_offset;

        /* Cockpit offset position in object relative vector
	 * direction, in meters.
	 */
        sar_position_struct cockpit_offset_pos;

	/* Cockpit visual model. */
	sar_visual_model_struct *visual_model_cockpit;

	/* Belly to center of object in meters. */
        float belly_height;

	/* Height of landing gear in meters. */
        float gear_height;

	/* Mass. */
	float	dry_mass;	/* In kg. */
	float	fuel_rate;	/* In kg consumed per cycle. */
	float	fuel, fuel_max;	/* In kg. */
	int	crew,
		passengers,
		passengers_max;
	float	passengers_mass;	/* Total mass of current passengers,
					 * in kg.
					 */

	/* Passengers pending to leave or drop from this aircraft,
	 * 0 means none and any positive value means this many passengers
	 * need to go when the aircraft's door is fully opened and its
	 * conditions are right (ie landed at the right helipad).
	 */
	int passengers_leave_pending, passengers_drop_pending;

	/* External fuel tanks. */
	sar_external_fueltank_struct **external_fueltank;
	int total_external_fueltanks;

	/* Engine state and control positions. */
	int engine_state;		/* One of SAR_ENGINE_STATE_*. */
	time_t next_engine_on;		/* Time till engine starts up (in ms). */

	/* Throttle control position. In aircraft flight model this is 
	 * the actual throttle control position, in helicopter flight
	 * model this is the speed at which the rotors are spinning.
	 * Range is from 0.0 to 1.0.
	 */
        float throttle;

	/* Current collective value and collective range. The current
	 * collective value is always 1.0 in airplane flight model. In
	 * helicopter flight model, the current collective value matches
	 * the game controller's throttle position.
	 *
	 * Both the collective and collective_range are in values from
	 * 0.0 to 1.0.
	 */
	float collective, collective_range;

	/* Engine data. */
        float	engine_power;		/* In kg * m / cycle^2. */
	char	engine_can_pitch;	/* 1 if engines can pitch (for tilt
					 * rotors like the V22 Ospray).
					 */

	/* Engine sounds. */
	int engine_inside_sndsrc;
	void *engine_inside_sndplay;
	int engine_outside_sndsrc;
	void *engine_outside_sndplay;

	/* Repeating warning sounds. */
	int stall_sndsrc;
	void *stall_sndplay;
	int overspeed_sndsrc;
	void *overspeed_sndplay;


	/* Visual controller positions, these values only dictate
	 * the displayed controller positions.
	 */
	float	control_heading,	/* Rudder, -1.0 to 1.0. */
		control_pitch,		/* Elevator, -1.0 to 1.0. */
		control_bank;		/* Ailerons, -1.0 to 1.0. */

        /* Effective controller positions, these values will be passed
         * to the FDM and control the actual attitude changing.
         */
	float	control_effective_heading,	/* Rudder, -1.0 to 1.0. */
		control_effective_pitch,	/* Elevator, -1.0 to 1.0. */
		control_effective_bank;		/* Ailerons, -1.0 to 1.0. */

	float	elevator_trim;		/* -1.0 to 1.0 */

	/* Flaps. */
	float	flaps_position;		/* 0.0 (retracted) to 1.0 (fully deployed). */
	float	flaps_stall_offset;	/* Moves the speed_stall ahead by this
					 * many meters per cycle.
					 */

	/* Service ceiling, in meters. */
	float	service_ceiling;

	/* Distance from object's center to touchable ground in
	 * meters (note that this is usually negative unless the
	 * object is underground).
	 */
	float	center_to_ground_height;


	/* Moveable parts. */
	sar_obj_part_struct **part;
	int total_parts;

	/* Rotors. */
	sar_obj_rotor_struct **rotor;
	int total_rotors;
	int rotor_wash_tex_num;	/* Texture number on scene struct. */

	/* Landed states. */
	char	landed;		/* Indicates object is on ground if 1. */
	char	on_water;	/* If landed is 1 and this is 1, then
				 * this aircraft is on water.
				 */

	/* Landing gear state:
	 * -1	non-existant
	 * 0    up/retracted/in
	 * 1    down/deployed/out
	 */
	int landing_gear_state;

	/* Ground turning. */
	float	gturn_radius;	/* Distance from farthest non-turning gear to
				 * turning gear in meters, can be negative.
				 * If 0 then it implies it cannot turn.
				 */
	/* Ground turning velocity optimul and maximum in meters per
	 * cycle.
	 */
	float	gturn_vel_opt,
		gturn_vel_max;

	/* Air brakes and wheel brakes:
	 * -1   non-existant
	 * 0    off
	 * 1    on
	 * 2    locked (parking brakes on)
	 */
	int	air_brakes_state,
		wheel_brakes_state;

	/* Air brake affect on speed, in units of meters per cycle.
	 * Can be 0.0 if this aircraft has no air brakes.
	 * The affect of the air brakes will be at its maximum
	 * (the exact value of air_brakes_rate) when the speed
	 * approaches speed_max_expected.  As the speed reduces to
	 * speed_stall the affect of air_brakes_rate will be reduced
	 * to 10% and as the speed reduces to 0 the affect of
	 * air_brakes_rate will be none.
	 */
	float	air_brakes_rate;

	/* Spot light direction. */
	sar_direction_struct spotlight_dir;

	/* Rescue hoist. */
	sar_obj_hoist_struct hoist;

	/* Intercept way points. */
	sar_obj_intercept_struct **intercept;
	int total_intercepts, cur_intercept;

	/* Autopilot data. */
	int autopilot_state;		/* 0 = off, 1 = on. */
	float autopilot_altitude;	/* Target height in meters above sea
					 * level.
					 */

} sar_object_aircraft_struct;

#define SAR_OBJECT_AIRCRAFT(p)	((sar_object_aircraft_struct *)(p))


/*
 *	SAR ground elevation object sub structure:
 *
 *      This structure can be pointed to from member `data' in the
 *      sar_object_struct, determined by member `type'.
 */
typedef struct {

	/* Elevation from (above) local MSL in meters. If MSL was
	 * set to 10 meters and this elevation was set to 3 meters then
	 * the resulting elevation would be 13 meters to any object within
	 * contact of the ground object.
	 *
	 * Note that the ground object itself completely ignores the
	 * scene structure defined ground_elevation_msl value.
	 */
	float	elevation;

	/* Translation of heightfield from center of object. */
	float	x_trans, y_trans, z_trans;

	/* Heightfield data (only used if z_point_value is not NULL). */
	float	x_len,		/* grid_points_x * grid_x_spacing */
		y_len;		/* grid_points_y * grid_y_spacing */
	int	grid_points_x,	/* Number of grid points. */
		grid_points_y,
		grid_points_total;	/* grid_points_x * grid_points_y */
	float	grid_x_spacing,	/* Size of each grid in meters. */
		grid_y_spacing,
		grid_z_spacing;
	double	*z_point_value;	/* Heightfield z height map, each point
				 * value came from (image_data_pixel) /
				 * 0xff * grid_z_spacing.
				 */

        /* Value records for this ground object's inversed heading for the
	 * trig functions, this is used to speed up rotations of heading
         * for checking an object over this ground object's heightfield
	 * surface. It also limits ground objects to only have its
	 * heading rotated.
         *
         * Ie cos_heading = cos(-heading) and sin_heading = sin(-heading)
         */
	float	cos_heading,
		sin_heading;

} sar_object_ground_struct;

#define SAR_OBJECT_GROUND(p)	((sar_object_ground_struct *)(p))


/*
 *	Unit structures used in the sar_object_smoke_struct structure
 *	so that the sar_object_smoke_struct structure can represent
 *	a set of unit structure (bunch of smoke puffs).
 */
typedef struct {

	/* Position in world coordinates. */
	sar_position_struct pos;

	sar_position_struct vel;

	/* Color (if smoke type is set to SAR_OBJ_SMOKE_TYPE_SPARK). */
	sar_color_struct color;

	/* Current size in meters. */
	float	radius;

	/* Current visibility of smoke unit, in range of 0.0 to 1.0.
	 * If 0.0 then it will not be drawn at all and is considered
	 * "available for use".
	 */
	float	visibility;

} sar_object_smoke_unit_struct;
/*
 *	SAR smoke/spark/debris trails sub structure:
 */
typedef struct {

#define SAR_OBJ_SMOKE_TYPE_SMOKE	0
#define SAR_OBJ_SMOKE_TYPE_SPARKS	1
#define SAR_OBJ_SMOKE_TYPE_DEBRIS	2
	int type;		/* One of SAR_OBJ_SMOKE_TYPE_*. */

	/* Spawn offset position, this offset is applied to the actual
	 * location of the smoke trail object where each unit will be
	 * spawned at.
	 */
	sar_position_struct respawn_offset;

	/* Starting and ending radius values for all units.
	 *
	 * If the smoke type is SAR_OBJ_SMOKE_TYPE_SPARKS then radius_start
	 * is ignored and radius_max is how far sparks should fly.
	 */
	float	radius_start, radius_max;

	/* Radius increase rate in meters per cycle (must be positive).
	 *
	 * If the smoke type is SAR_OBJ_SMOKE_TYPE_SPARKS then this value
	 * is ignored.
	 */
	float	radius_rate;

	/* If this is true then when a smoke unit has reached its 
	 * maximum size its visiblity will be reset to 0.0 (thus marking
	 * it as "available for use").
	 */
	int hide_at_max;

	/* If this is true then this object will be marked for deletion
	 * (its life span set to 1) when all of its units are no longer
	 * visible (visibility = 0.0).
	 */
	int delete_when_no_units;

	/* Note that if respawn_int is 0 then no respawning will take
	 * place. This is often used to stop respawning but keep the
	 * already visible smoke units around.
	 */
	time_t respawn_int;	/* Respawn smoke unit interval in ms. */
	time_t respawn_next;	/* Next time to respawn smoke unit. */

        /* Texture number on scene structure. */
        int tex_num;

        /* Reference object that this smoke trail is to follow (can be -1
	 * for none or do not follow).
         */
        int ref_object;

	/* Each smoke unit forming this smoke trail. */
	sar_object_smoke_unit_struct *unit;
	int total_units;

} sar_object_smoke_struct;

#define SAR_OBJECT_SMOKE_UNIT(p)	((sar_object_smoke_unit_struct *)(p))
#define SAR_OBJECT_SMOKE(p)		((sar_object_smoke_struct *)(p))


/*
 *	SAR explosion sub structure:
 *
 *	Note this is also used for splashes.
 */
typedef struct {

	/* Spherical size of explosion in meters, this is also used to
	 * calculate the displayed explosion billboard.
	 */
	float	radius;

	/* Explosion color emission, one of SAR_EXPLOSION_COLOR_EMISSION_*. */
#define SAR_EXPLOSION_COLOR_EMISSION_NONE	0	/* Ie splashes, interacts with light. */
#define SAR_EXPLOSION_COLOR_EMISSION_IS_LIGHT	1	/* Does not interact with light. */	
#define SAR_EXPLOSION_COLOR_EMISSION_EMIT_LIGHT	2	/* Same as IS_LIGHT and gives off light too. */
	int color_emission;

	/* Center gravity offset, one of SAR_EXPLOSION_GRAVITY_*.
	 * All gravity values are `upright' oriented.
	 */
#define SAR_EXPLOSION_GRAVITY_CENTER	0
#define SAR_EXPLOSION_GRAVITY_BASE	1
	int gravity_offset;

	time_t frame_inc_int;   /* Frame increment interval (in ms). */
	time_t next_frame_inc;	/* Time to increment next frame (in ms). */

	int cur_frame;          /* Current frame. */
	int frame_repeats;	/* Number of times animation has cycled. */

	/* Number of frame repeats, 0 or less to repeat forever (or
	 * when life span has exceeded.
	 */
	int total_frame_repeats;

	/* Texture number on scene structure. */
	int tex_num;

	/* Reference object that this explosion is to follow (can be -1
	 * for none/do not follow).
	 */
	int ref_object;

} sar_object_explosion_struct;

#define SAR_OBJECT_EXPLOSION(p)	((sar_object_explosion_struct *)(p))


/* 
 *      SAR fire sub structure:
 */
typedef struct {

        /* Cylendrical size of fire in meters. This is also used to 
	 * calculate the fire billboard. Center is at base of fire.
	 */
        float	radius, height;

        time_t frame_inc_int;   /* Frame increment interval (in ms). */
        time_t next_frame_inc;  /* Time to increment next frame (in ms). */

        int cur_frame;          /* Current frame. */
        int frame_repeats;      /* Number of times animation has cycled. */

        /* Number of frame repeats, 0 or less to repeat forever (or
         * when life span has exceeded.
         */
        int total_frame_repeats;

        /* Texture number on scene structure. */
        int tex_num;

	/* Reference object that this fire is to follow (can be -1 for
	 * none/do not follow).
	 */
        int ref_object; 

} sar_object_fire_struct;

#define SAR_OBJECT_FIRE(p)	((sar_object_fire_struct *)(p))


/*
 *	Chemical spray object:
 *
 *	A single puff of chemical spray (ie water, fire-retardant, etc).
 */
typedef struct {

#define SAR_CHEMICAL_WATER		0
#define SAR_CHEMICAL_FIRE_RETARDANT	1

	int chemical_type;	/* One of SAR_CHEMICAL_*. */

	int owner;		/* Object that created this spray or -1 
				 * for none.
				 */

        /* Texture number on scene structure. */
	int tex_num;

} sar_object_chemical_spray_struct;

#define SAR_OBJECT_CHEMICAL_SPRAY(p)	((sar_object_chemical_spray_struct *)(p))


/*
 *	SAR aircraft fuel tank droped from an aircraft:
 *
 *	Note that this is not a fuel tank that is currently on an 
 *	aircraft, do not confuse this with the sar_external_fueltank_struct
 *	structure.
 */
typedef struct {

#define SAR_FUELTANK_FLAG_ON_GROUND	(1 << 1)

	sar_obj_flags_t flags;

        /* Current speed in meters per cycle. */
        float	speed;

	/* Maximum negative vertical velocity in meters per cycle. */
	float	vel_z_max;

        /* Current velocity vector representing velocity in object
         * relative vector direction, in meters per cycle.
         */
        sar_position_struct vel;

	/* Index of object of which this fuel tank fell off of
	 * (can be -1 for unknown).
	 */
	int ref_object;

	/* Mass, all in kg. */
	float	dry_mass,
		fuel, fuel_max;

} sar_object_fueltank_struct;

#define SAR_OBJECT_FUELTANK(p)	((sar_object_fueltank_struct *)(p))


/*
 *      SAR runway sub structure:
 */
typedef struct {

#define SAR_RUNWAY_FLAG_THRESHOLDS	(1 << 0)	/* Not displaced thresholds. */
#define SAR_RUNWAY_FLAG_BORDERS		(1 << 1)	/* Borders on sides. */
#define SAR_RUNWAY_FLAG_TD_MARKERS	(1 << 2)	/* Touch down markers. */
#define SAR_RUNWAY_FLAG_MIDWAY_MARKERS	(1 << 3)	/* Midway markers */
#define SAR_RUNWAY_FLAG_NORTH_GS	(1 << 4)	/* Glide slope. */
#define SAR_RUNWAY_FLAG_SOUTH_GS	(1 << 5)	/* Glide slope. */

	sar_obj_flags_t flags;

	/* Size in meters. */
	float	length,
		width;

	/* Surface type. */
#define SAR_RUNWAY_SURFACE_PAVED	0
#define SAR_RUNWAY_SURFACE_GRAVEL	1
#define SAR_RUNWAY_SURFACE_CONCRETE	2
#define SAR_RUNWAY_SURFACE_GROVED	3
        int     surface;

	/* End labels. */
	char	*north_label,
		*south_label;
	sar_visual_model_struct	*north_label_vmodel,
				*south_label_vmodel;
	float	north_label_width,
		south_label_width;

	/* Number of dashes (0 for none). */
	int	dashes;

	/* Edge lighting in meters (0.0 for none). */
	float	edge_light_spacing;

	/* Approach lighting flags. */
#define SAR_RUNWAY_APPROACH_LIGHTING_END	(1 << 0)
#define SAR_RUNWAY_APPROACH_LIGHTING_TRACER	(1 << 1)
#define SAR_RUNWAY_APPROACH_LIGHTING_ALIGN	(1 << 2)
#define SAR_RUNWAY_APPROACH_LIGHTING_ILS_GLIDE	(1 << 3)
	sar_obj_flags_t	north_approach_lighting_flags,
			south_approach_lighting_flags;
	sar_grad_anim_t	tracer_anim_pos, tracer_anim_rate;

	/* Displaced threshold offset from respective edges in meters
	 * (can be 0.0 for none).
	 */
	float	north_displaced_threshold,
		south_displaced_threshold;

	sar_visual_model_struct	*threshold_vmodel,
				*td_marker_vmodel,
				*midway_marker_vmodel,
				*north_displaced_threshold_vmodel,
				*south_displaced_threshold_vmodel;

        /* Runway background texture number (on scene structure). */
        int tex_num;

} sar_object_runway_struct;
#define SAR_OBJECT_RUNWAY(p)	((sar_object_runway_struct *)(p))


/*
 *	SAR helipad sub structure:
 */
typedef struct {

#define SAR_HELIPAD_FLAG_LABEL			(1 << 1)
#define SAR_HELIPAD_FLAG_EDGE_LIGHTING		(1 << 2)
#define SAR_HELIPAD_FLAG_FUEL			(1 << 3)
#define SAR_HELIPAD_FLAG_REPAIR			(1 << 4)
#define SAR_HELIPAD_FLAG_DROPOFF		(1 << 5)	/* Drop off passengers. */
#define SAR_HELIPAD_FLAG_REF_OBJECT		(1 << 6)	/* Has ref object defined. */
#define SAR_HELIPAD_FLAG_FOLLOW_REF_OBJECT	(1 << 7)
#define SAR_HELIPAD_FLAG_RESTART_POINT		(1 << 8)	/* Player restarting point. */

        sar_obj_flags_t flags;

	/* Helipad style code, one of SAR_HELIPAD_STYLE_*. */
#define SAR_HELIPAD_STYLE_GROUND_PAVED	0	/* Default. */
#define SAR_HELIPAD_STYLE_GROUND_BARE	1	/* Unpaved. */
#define SAR_HELIPAD_STYLE_BUILDING	2	/* Ie roof top. */
#define SAR_HELIPAD_STYLE_VEHICLE	3	/* Ie helipad on a ship. */

	int style;

        /* Size in meters, specifying the landable area. */
        float	length,
		width;

	/* Downward recession through helipad's landable ground, can be
	 * 0.0 or any positive value in meters. A value of 2.0 means 2
	 * meters down.
	 */
	float	recession;

        /* Label. */
        char	*label;
	sar_visual_model_struct *label_vmodel;
	float	label_width;

        /* Lighting edge spacing in meters. */
        float	light_spacing;

        /* Texture number on scene structure, specifying the texture
	 * for the helipad's main landable area.
	 */
        int tex_num;

	/* If SAR_HELIPAD_FLAG_REF_OBJECT is set then these members
	 * have affect. Note that ref_offset is applied first then
	 * ref_dir.
	 */
	int ref_object;		/* Object that this helipad `follows'. */
	/* These members are used if SAR_HELIPAD_FLAG_FOLLOW_REF_OBJECT
	 * is set. Note that ref_dir is applied first then ref_offset.
	 */
	sar_position_struct	ref_offset;	/* Rel to ref object. */
	sar_direction_struct	ref_dir;	/* Rel to ref object. */

} sar_object_helipad_struct;

#define SAR_OBJECT_HELIPAD(p)	((sar_object_helipad_struct *)(p))


/*
 *	SAR human structure:
 */
typedef struct {

#define SAR_HUMAN_FLAG_NEED_RESCUE	(1 << 1)
#define SAR_HUMAN_FLAG_SIT		(1 << 2)	/* Base at tush. */
#define SAR_HUMAN_FLAG_SIT_DOWN		(1 << 3)	/* Base at tush, feet out. */
#define SAR_HUMAN_FLAG_SIT_UP		(1 << 4)	/* Base at feet. */
#define SAR_HUMAN_FLAG_LYING		(1 << 5)
#define SAR_HUMAN_FLAG_ALERT		(1 << 6)	/* Is awake. */
#define SAR_HUMAN_FLAG_AWARE		(1 << 7)	/* Knows of surroundings. */
#define SAR_HUMAN_FLAG_IN_WATER		(1 << 8)
#define SAR_HUMAN_FLAG_ON_STREATCHER	(1 << 9)
#define SAR_HUMAN_FLAG_RUN		(1 << 10)	/* Animate as running. */
#define SAR_HUMAN_FLAG_RUN_TOWARDS	(1 << 11)	/* Intends to run towards,
							 * does not imply currently running.
							 */
#define SAR_HUMAN_FLAG_RUN_AWAY		(1 << 12)	/* Intends to run away,
							 * does not imply currently running.
							 */
#define SAR_HUMAN_FLAG_PUSHING		(1 << 13)
#define SAR_HUMAN_FLAG_GRIPPED		(1 << 14)	/* In rescue basket for example. */
#define SAR_HUMAN_FLAG_DIVER_CATCHER	(1 << 15)

	sar_obj_flags_t flags;

	float	mass;	/* In kg. */

	float	height;	/* In meters. */

	sar_grad_anim_t anim_pos, anim_rate;

	/* Colors, see definations for SAR_HUMAN_COLOR_* and
	 * SAR_HUMAN_COLORS_MAX for index positions and maximum colors
	 * (respectivly).
	 */
	sar_color_struct color[SAR_HUMAN_COLORS_MAX];

	/* Water ripples texture number on scene structure. */
	int water_ripple_tex_num;

	/* Reference to object running towards or away from, the following
	 * values have special meaning:
	 *
	 *	-1	No intercepting
	 *	-2	Intercept player
	 *
	 * Works when flags SAR_HUMAN_FLAG_RUN_TOWARDS xor 
	 * SAR_HUMAN_FLAG_RUN_AWAY is set and intercepting_object is 
	 * valid and not the human object itself.
	 */
	int intercepting_object;

	/* Distance to intercepting object in meters (may be ignored if
	 * intercepting_object is -1.
	 */
	float	intercepting_object_distance2d,
		intercepting_object_distance3d;

	/* Number of assisting humans following this human (0 for none).
	 * these are not other objects but rather groups of humans drawn
	 * with this human object to make it look like multiple
	 * humans.
	 */
	int assisting_humans;
        sar_color_struct assisting_human_color[SAR_HUMAN_COLORS_MAX];

	/* Messages. */
	char *mesg_enter;	/* Entering into aircraft or vehicle. */

} sar_object_human_struct;

#define SAR_OBJECT_HUMAN(p)	((sar_object_human_struct *)(p))


/*
 *	SAR premodeled object:
 */
typedef struct {

#define SAR_OBJ_PREMODELED_BUILDING		1
#define SAR_OBJ_PREMODELED_CONTROL_TOWER	2
#define SAR_OBJ_PREMODELED_HANGAR		3
#define SAR_OBJ_PREMODELED_POWER_TRANSMISSION_TOWER	4
#define SAR_OBJ_PREMODELED_TOWER		5
#define SAR_OBJ_PREMODELED_RADIO_TOWER		6

	int type;	/* One of SAR_OBJ_PREMODELED_*. */

	/* Values, depending on type, not all may be applicateable. */
	float	width,
		length,
		height;

	/* Animation. */
	sar_grad_anim_t anim_pos, anim_rate;

        /* Texture number on scene structure. */
#define SAR_OBJ_PREMODEL_MAX_TEXTURES	10
        int tex_num[SAR_OBJ_PREMODEL_MAX_TEXTURES];

} sar_object_premodeled_struct;

#define SAR_OBJECT_PREMODELED(p)	((sar_object_premodeled_struct *)(p))


/*
 *	SAR core object structure:
 */
typedef struct {

	int type;	/* Must be first member, one of SAR_OBJ_TYPE_*. */

#define SAR_OBJ_FLAG_NO_DEPTH_TEST	(1 << 1)
#define SAR_OBJ_FLAG_HIDE_DAY_MODEL	(1 << 2)	/* Hide day model when not day. */
#define SAR_OBJ_FLAG_HIDE_DAWN_MODEL	(1 << 3)	/* Hide dawn model when not dawn. */
#define SAR_OBJ_FLAG_HIDE_DUSK_MODEL	(1 << 4)	/* Hide dusk model when not dusk. */
#define SAR_OBJ_FLAG_HIDE_NIGHT_MODEL	(1 << 5)	/* Hide night model when not night. */

/* These two only work if SAR_OBJ_FLAG_HIDE_NIGHT_MODEL is set. */
#define SAR_OBJ_FLAG_NIGHT_MODEL_AT_DAWN	(1 << 6)	/* Show night modem at dawn. */
#define SAR_OBJ_FLAG_NIGHT_MODEL_AT_DUSK	(1 << 7)	/* Show night modem at dusk. */

#define SAR_OBJ_FLAG_SHADE_MODEL_SMOOTH	(1 << 8)	/* Use smooth shading. */

#define SAR_OBJ_FLAG_FAR_MODEL_DAY_ONLY	(1 << 9)	/* Display far model during
							 * time only.
							 */
#define SAR_OBJ_FLAG_POLYGON_OFFSET	(1 << 10)	/* Enable polygon offset. */
#define SAR_OBJ_FLAG_POLYGON_OFFSET_REVERSE	(1 << 11)	/* Same as SAR_OBJ_FLAG_POLYGON_OFFSET
								 * cept offsets in
								 * the other direction.
								 */
#define SAR_OBJ_FLAG_POLYGON_OFFSET_WRITE_DEPTH	(1 << 12)	/* Write depth if
								 * polygon offsetting.
								 */

	sar_obj_flags_t flags;

	/* Name of object (can be NULL). */
	char *name;

        /* Current position and direction. */
        sar_position_struct pos;
        sar_direction_struct dir;

	/* Visible range of this object in meters. */
	float	range;

	/* Visible range for displaying the far model in meters,
	 * if and only if the visual_model_far is not NULL and
	 * the range is beyond this value is the far model
	 * displayed instead of the standard model.
	 */
	float	range_far;


	/* Distance from sea level to touchable ground under
	 * object, in meters. (Not applicateable to object type
	 * SAR_OBJ_TYPE_GROUND).
	 */
	float	ground_elevation_msl;


	/* Contact and crash bounds. */
	sar_contact_bounds_struct *contact_bounds;


	/* Time stamp of when this object was created in milliseconds
	 * and in systime seconds (respectivly). The value in milliseconds 
	 * may not be accurate when timmers are reset.
	 */
	time_t	birth_time_ms,
		birth_time_sec;

	/* When this object `dies' (0 if lives forever). */
	time_t life_span;

	/* Hit points. */
	float	hit_points, hit_points_max;

	/* Visual models for day, dusk, night. */
	sar_visual_model_struct *visual_model;		/* Day. */
	sar_visual_model_struct *visual_model_far;	/* Far. */
	sar_visual_model_struct *visual_model_dawn;	/* Dawn. */
	sar_visual_model_struct *visual_model_dusk;	/* Dusk. */
	sar_visual_model_struct *visual_model_night;	/* Night. */
	sar_visual_model_struct *visual_model_shadow;	/* Shadow. */

	/* Lights. */
	sar_light_struct **light;
	int total_lights;

	/* Sound sources. */
	sar_sound_source_struct **sndsrc;
	int total_sndsrcs;

	/* Pointer to additional data specific to the object type,
	 * the structure's type is specified by this structure's member
	 * type.
	 */
	void *data;

} sar_object_struct;
#define SAR_OBJECT(p)	((sar_object_struct *)(p))



/*
 *	Scene ground base structure, contains visual models and tiling
 *	values for displaying of the ground base.
 */
typedef struct {

	/* Note, ground base moves with camera in modulous increments
	 * of the tiled width and height.
	 */

	/* Tiling size in meters. */
        int	tile_width,
		tile_height;

	/* Tiling limited to this range in meters from origin of
	 * tiling.
	 */
	float	close_range;


	/* Simple (farther away) solid color base plane and visual
	 * model.
	 */
	sar_color_struct color;
        sar_visual_model_struct *visual_model_simple;

	/* Close up texture tiled base plane visual model. */
	sar_visual_model_struct *visual_model_close;

} sar_scene_base_struct;
#define SAR_SCENE_BASE(p)	((sar_scene_base_struct *)(p))

/*
 *	Scene horizon structure:
 */
typedef struct {

	/* Records the last time of day since midnight from
	 * the scene structure in units of 5 minute intervals as a whole 
	 * number for horizon texture regeneration, see
	 * SARSimUpdateScene() in simmanage.c for when and how this value
	 * is updated.
	 */
	int last_tod;

	/* Gradient textures, 0 is most highlighted and
	 * total_textures - 1 is darkest.
	 */
	v3d_texture_ref_struct **texture;
	int total_textures;

} sar_scene_horizon_struct;
#define SAR_SCENE_HORIZON(p)	((sar_scene_horizon_struct *)(p))

/*
 *	Scene cloud layer structure:
 *
 *	Flat tiled layer of clouds as tiled textured quads.
 */
typedef struct {

        /* Note, cloud layer moves with camera in modulous           
         * of the tiled width and height on the xy plane.
         */

        /* Tiling size in meters. */
	int	tile_width,
		tile_height;

        /* Tiling limited to this range in meters from origin of
         * tiling.
         */
        float	range;

	/* Name of texture on scene structure. */
	char	*tex_name;

	sar_visual_model_struct *visual_model;

	/* Height from MSL in meters. */
	float	z;

} sar_cloud_layer_struct;
#define SAR_CLOUD_LAYER(p)	((sar_cloud_layer_struct *)(p))

/*
 *      Scene cloud `billboard' structure:
 *
 *      `Billboard' cloud objects placed across the scene.
 */
typedef struct {

        /* Note, cloud billboard moves with camera in modulous
         * of the tiled width and height on the xy plane.
         */

        /* Tiling size in meters. */
        int tile_width, tile_height;

        /* Name of texture on scene structure. */
        char *tex_name;
	/* Matched texture on scene structure. */
	int tex_num;

	/* Position of billboard object relative to tiled center. The
	 * z value is relative to the ground ofcourse (in meters).
	 */
	float	x, y, z;

	/* Size of cloud object in meters (note height not in feet). */
	float	width, height;


	/* Lightning min interval (in ms), 0 for no lightning. */
	time_t lightning_min_int;

	/* Lightning on interval (in ms), how long does the lightning
	 * stay on.
	 */
	time_t lightning_on_int;

	/* When this value is greater than 0 it implies lightning
	 * is on and will stay on for the remaining duration (in ms).
	 */
	time_t lightning_on_left;

	/* Last time lightning was on in ms. */
	time_t lightning_last;

	/* Lightning points, in meters relative to cloud bb center. */
#define SAR_LIGHTNING_POINTS_MAX	6
	sar_position_struct lightning_point[SAR_LIGHTNING_POINTS_MAX];

} sar_cloud_bb_struct;
#define SAR_CLOUD_BB(p)		((sar_cloud_bb_struct *)(p))

/*
 *	SAR scene structure:
 */
typedef struct {

	/* Title of scenery. */
	char	*title;

	/* Time of day (in seconds) since midnight. */
	float	tod;

	/* Time of day code, one of SARSceneTODCode*. This is
	 * automatically set based on the tod in SARSimUpdateScene().
	 */
#define SAR_SCENE_TOD_DAY		1
#define SAR_SCENE_TOD_DAWN		2
#define SAR_SCENE_TOD_NIGHT		3
#define SAR_SCENE_TOD_DUSK		4
	int	tod_code;

        /* Base flags, determines the attributes of the landable ground
	 * base.
         */
#define SAR_SCENE_BASE_FLAG_IS_WATER	(1 << 1)	/* Opposite of land. */
	sar_obj_flags_t	base_flags;

	/* CANT angle in radians, this value is only applied on readouts
	 * added to the internal angle value in question.
	 */
	float	cant_angle;

	/* MSL elevation in meters. */
	float	msl_elevation;

	/* GPS information. */
	sar_dms_t	dms_x_offset,	/* In degrees, -140.0 to 140.0. */
			dms_y_offset;	/* In degrees, -90.0 to 90.0. */
	float	planet_radius;		/* In meters. */

	/* List of visual models (structures containing GL list
	 * references) each object or resource that has a visual
	 * model (has a GL list basically) will point to one of
	 * these structures. Only these structures may be deallocated.
	 */
	sar_visual_model_struct	**visual_model;
	int			total_visual_models;

	/* Sky and horizon gradient colors. */
	sar_color_struct	sky_nominal_color,
				sky_brighten_color,
				sky_darken_color;
	/* Celestial object colors, these colors define the low and high
	 * tint colors of the celestial object (where low means it is near
	 * the horizon).
	 */
	sar_color_struct	star_low_color,
				star_high_color,
				sun_low_color,
				sun_high_color,
				moon_low_color,
				moon_high_color;

	/* Position of sun is dirived from light_pos. */

	/* Moon position. */
	sar_position_struct	moon_pos;
	/* Moon visibility hint, 1 = anywhere above horizon, 0 = hidden. */
	char			moon_visibility_hint;


	/* Atmosphere (when enabled). */
	float	atmosphere_dist_coeff,	/* Coeff * max visiblity. */
		atmosphere_density_coeff;

	/* Rain density coefficient, 0.0 for no rain, 0.5 for moderate,
	 * 1.0 for densest rain.
	 */
	float	rain_density;

	/* Reference to player object. */
	int			player_obj_num;
	sar_object_struct	*player_obj_ptr;
	char			player_has_crashed;	/* 1 if player object has crashed. */

	/* Pointers to SAR_OBJ_TYPE_GROUND objects, you may free
	 * the pointer array but not the pointer to structures!!
	 */
	sar_object_struct	**ground_object;
	int			total_ground_objects;

	/* Pointers to SAR_OBJ_TYPE_HUMAN objects that *need rescue*.
	 * You may free the pointer array but not the pointers to structures!!
	 */
	sar_object_struct	**human_need_rescue_object;
        int			total_human_need_rescue_objects;


	/* Camera reference. */
#define SAR_CAMERA_REF_COCKPIT	0
#define SAR_CAMERA_REF_SPOT	1
#define SAR_CAMERA_REF_TOWER	2
#define SAR_CAMERA_REF_MAP	3
#define SAR_CAMERA_REF_HOIST	4	/* Rescue hoist view. */
	int	camera_ref;

	/* Camera field of view, in radians along Z axis (or "about the
	 * X axis").
	 */
	float	camera_fovz;

	/* Direction camera is looking while set to
	 * SAR_CAMERA_REF_COCKPIT. The bank member is ignored.
	 */
	sar_direction_struct camera_cockpit_dir;

	/* Direction from object to camera while set to
	 * SAR_CAMERA_REF_SPOT. The bank member is ignored.
	 */
	sar_direction_struct camera_spot_dir;
	float	camera_spot_dist;

	/* Distance for SAR_CAMERA_REF_HOIST. */
	sar_direction_struct camera_hoist_dir;
	float	camera_hoist_dist;

	/* Position of camera when set to SAR_CAMERA_REF_MAP. */
	sar_position_struct camera_map_pos;

	/* Position of camera when set to SAR_CAMERA_REF_TOWER. */
	sar_position_struct camera_tower_pos;

	/* Target object that camera is tracking (can be -1). */
	int camera_target;

	/* Stack of camera rotation matrixes (set in sardraw.c) to
	 * rotate any vertex relative to the camera's rotation, each
	 * a 3 * 3 matrix.
	 * Member camera_rotmatrix_count indicates the actual number
	 * of matrixes set, which is equal or less than 
	 * SAR_CAMERA_ROTMATRIX_MAX.
	 */
#define SAR_CAMERA_ROTMATRIX_MAX	5
	double camera_rotmatrix[SAR_CAMERA_ROTMATRIX_MAX][3 * 3];
	int camera_rotmatrix_count;

	/* Ear position, this is updated when the scene is drawn and
	 * the camera set.
	 */
	sar_position_struct ear_pos;


	/* Global primary light position offset, this specifies a unit 
	 * offset relative to be applied to the camera position.
	 */
	sar_position_struct light_pos;

	/* Global primary light intensity/color, this is not the color
	 * of the sun. The sun color is found in sun_low_color and
	 * sun_high_color,
	 */
	sar_color_struct light_color;
	char light_visibility_hint;	/* 1 if anywhere above horizon. */


	/* Base ground. */
	sar_scene_base_struct base;

	/* Horizon. */
	sar_scene_horizon_struct horizon;

	/* Cloud layers. */
	sar_cloud_layer_struct **cloud_layer;
	int total_cloud_layers;

	/* Cloud `billboard' objects. */
	sar_cloud_bb_struct **cloud_bb;
	int total_cloud_bbs;

	/* Loaded textures list. */
	v3d_texture_ref_struct **texture_ref;
	int total_texture_refs;

	/* Index references to specific textures (can be -1 if the texture
	 * in question was not loaded in the above texture_ref list.
	 */
	int texnum_sun;
	int texnum_moon;
	int texnum_spotlightcast;

        /* Sound sources. */
        sar_sound_source_struct **sndsrc;
        int total_sndsrcs;


	/* Player control panel (ControlPanel *). */
	void *player_control_panel;


	/* Messages list. */
	char **message;
	int total_messages;

	/* Display message until this time runs out (in milliseconds), a
	 * value of 0 implies message not shown.
	 */
	time_t message_display_until;

	/* Sticky banner message, for displaying crash reason or mission
	 * failed reason whenever it is not NULL.
	 */
	char **sticky_banner_message;
	int total_sticky_banner_messages;

	/* Name of object and/or description of what the camera is
	 * showing.
	 */
	char *camera_ref_title;
	time_t camera_ref_title_display_until;

	/* Flight dynamics model realm structure, used by the SFM
	 * library as global reference while simulating all flight 
	 * dynamics models.
	 */
	SFMRealmStruct *realm;

} sar_scene_struct;
#define SAR_SCENE(p)	((sar_scene_struct *)(p))
#define SAR_IS_CAMERA_IN_COCKPIT(s)	(((s) != NULL) ? \
 ((s)->camera_ref == SAR_CAMERA_REF_COCKPIT) : 0 \
)
#define SAR_IS_EAR_IN_COCKPIT(s)	(((s) != NULL) ? \
 ((s)->camera_ref == SAR_CAMERA_REF_COCKPIT) : 0 \
)



/*
 *	Code segment to get typed pointer to object's extended
 *	data.
 */
/*
	sar_object_struct *obj_ptr;
        sar_object_aircraft_struct *obj_aircraft_ptr = NULL;
	sar_object_ground_struct *obj_ground_ptr = NULL;
	sar_object_runway_struct *obj_runway_ptr = NULL;
        sar_object_helipad_struct *obj_helipad_ptr = NULL;
	sar_object_smoke_struct *obj_smoke_ptr = NULL;
	sar_object_explosion_struct *obj_explosion_ptr = NULL;
	sar_object_human_struct *obj_human_ptr = NULL;
 */
#define SAR_GET_EXT_DATA_PTR	{ 				\
switch(obj_ptr->type)						\
{								\
 case SAR_OBJ_TYPE_AIRCRAFT:					\
  obj_aircraft_ptr = SAR_OBJECT_AIRCRAFT(obj_ptr->data);	\
  break;							\
 case SAR_OBJ_TYPE_GROUND:					\
  obj_ground_ptr = SAR_OBJECT_GROUND(obj_ptr->data);		\
  break;							\
 case SAR_OBJ_TYPE_RUNWAY:					\
  obj_runway_ptr = SAR_OBJECT_RUNWAY(obj_ptr->data);		\
  break;							\
 case SAR_OBJ_TYPE_HELIPAD:					\
  obj_helipad_ptr = SAR_OBJECT_HELIPAD(obj_ptr->data);		\
  break;							\
 case SAR_OBJ_TYPE_SMOKE:					\
  obj_smoke_ptr = SAR_OBJECT_SMOKE(obj_ptr->data);		\
  break;							\
 case SAR_OBJ_TYPE_FIRE:					\
  obj_fire_ptr = SAR_OBJECT_FIRE(obj_ptr->data);		\
  break;							\
 case SAR_OBJ_TYPE_EXPLOSION:					\
  obj_explosion_ptr = SAR_OBJECT_EXPLOSION(obj_ptr->data);	\
  break;							\
 case SAR_OBJ_TYPE_HUMAN:					\
  obj_human_ptr = SAR_OBJECT_HUMAN(obj_ptr->data);		\
  break;							\
 case SAR_OBJ_FUELTANK:						\
  obj_fueltank_ptr = SAR_OBJECT_FUELTANK(obj_ptr->data);	\
  break;							\
 case SAR_OBJ_CHEMICAL_SPRAY:					\
  obj_chemical_spray_ptr = SAR_OBJECT_CHEMICAL_SPRAY(obj_ptr->data); \
  break;							\
} }


#endif	/* OBJ_H */
