#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/stat.h>
#include <math.h>

#ifdef __MSW__
# include <windows.h>
#endif

#include "../include/fio.h"
#include "../include/string.h"
#include "../include/strexp.h"
#include "../include/disk.h"

#include "gw.h"
#include "x3d.h"
#include "text3d.h"
#include "sfm.h"
#include "v3dhf.h"
#include "v3dtex.h"
#include "v3dmh.h"
#include "v3dmp.h"
#include "v3dmodel.h"
#include "v3dfio.h"

#include "cp.h"
#include "cpfio.h"
#include "sarreality.h"
#include "obj.h"
#include "objsound.h"
#include "objutils.h"
#include "objio.h"
#include "human.h"
#include "smoke.h"
#include "fire.h"
#include "weather.h"
#include "sar.h"
#include "sartime.h"
#include "sarfio.h"
#include "simop.h"
#include "simmanage.h"
#include "simcb.h"
#include "simutils.h"
#include "config.h"

#include "runway/runway_displaced_threshold.x3d"
#include "runway/runway_midway_markers.x3d"
#include "runway/runway_td_markers.x3d"
#include "runway/runway_threshold.x3d"


/* Utilities */
static const char *NEXT_ARG(const char *s);
#if 0
static void CHOP_STR_BLANK(char *s);
#endif

/* Visual Model Loading */
sar_visual_model_struct *SARObjLoadX3DDataVisualModel(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        char **x3d_data,
        const sar_scale_struct *scale
);
sar_visual_model_struct *SARObjLoadTextVisualModel(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
	float font_width, float font_height,	/* In meters. */
	float character_spacing,		/* In meters. */
	const char *s,
	float *string_width			/* In meters. */
);

/* Parameter Loading */
int SARObjLoadTranslate(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_object_struct *obj_ptr, sar_parm_translate_struct *p_translate
);
int SARObjLoadTranslateRandom(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_parm_translate_random_struct *p_translate_random
);

static int SARObjLoadSoundSource(
	sar_scene_struct *scene,
	sar_sound_source_struct ***list, int *total,
	const char *arg
);
int SARObjLoadTexture(
	sar_core_struct *core_ptr, sar_scene_struct *scene,
	sar_parm_texture_load_struct *p_texture_load
);
int SARObjLoadHelipad(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
	sar_parm_new_helipad_struct *p_new_helipad
);
int SARObjLoadRunway(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_parm_new_runway_struct *p_new_runway
);
int SARObjLoadHuman(
	sar_core_struct *core_ptr, sar_scene_struct *scene,
	sar_parm_new_human_struct *p_new_human
);
int SARObjLoadFire(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_parm_new_fire_struct *p_new_fire
);
int SARObjLoadSmoke(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_parm_new_smoke_struct *p_new_smoke
);
int SARObjLoadHeightField(
        sar_core_struct *core_ptr,
        int obj_num, sar_object_struct *obj_ptr,   
        void *p, const char *filename, int line_num,
	GLuint list
);
static int SARObjLoadVisualPrimitive(
	sar_core_struct *core_ptr,
        int obj_num, sar_object_struct *obj_ptr,
        void *p, const char *filename, int line_num
);
static int SARObjLoadLine(
	sar_core_struct *core_ptr,
        int obj_num, sar_object_struct *obj_ptr,
        const char *line, const char *filename, int line_num
);
int SARObjLoadFromFile(
	sar_core_struct *core_ptr, int obj_num, const char *filename
);


#ifndef SAR_COMMENT_CHAR
# define SAR_COMMENT_CHAR       '#'
#endif

#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? (float)atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))

#define RADTODEG(r)     ((r) * 180.0 / PI)
#define DEGTORAD(d)     ((d) * PI / 180.0)

#define ISCOMMENT(c)	((c) == SAR_COMMENT_CHAR)
#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))
#define ISSTREMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : True)
#define STRLEN(s)	(((s) != NULL) ? ((int)strlen(s)) : 0)

static int last_begin_primitive_type;


/*
 *	Texture plane orientation codes:
 */
#define TEX_ORIENT_NONE	0
#define TEX_ORIENT_XY	1
#define TEX_ORIENT_YZ	2
#define TEX_ORIENT_XZ	3

static Boolean tex_on;
static int tex_orient;	/* One of TEX_ORIENT_* */

typedef struct {
	float i, j;	/* Position of `upper left' in meters */
	float w, h;	/* Size `to lower right' in meters */
} tex_coord_struct;
static tex_coord_struct tex_coord;


/*
 *      Seeks s to next argument.
 *
 *      If s is "red green blue" then return will be "green blue".
 */
static const char *NEXT_ARG(const char *s)
{
        if(s == NULL)
            return(s);

        /* Seek past current arg till next blank or end of string. */
        while(!ISBLANK(*s) && (*s != '\0'))
            s++;

        /* Seek past spaces to next arg. */
        while(ISBLANK(*s))
            s++;

        return(s);
}

#if 0
/*
 *      Terminates the string on the first blank character encountered.
 */
static void CHOP_STR_BLANK(char *s)
{
        if(s == NULL)
            return;

        while(*s != '\0')
        {
            if(ISBLANK(*s))
            {
                *s = '\0';
                break;
            }
            else
            {
                s++;
            }
        }
}
#endif


/*
 *	Loads a visual model from the given X3D data.
 */
sar_visual_model_struct *SARObjLoadX3DDataVisualModel(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
	char **x3d_data,
	const sar_scale_struct *scale
)
{
        sar_visual_model_struct *vmodel;
	X3DInterpValues v;

	if(x3d_data == NULL)
	    return(NULL);

	/* Set up X3D data values. */
	v.flags =	X3D_VALUE_FLAG_COORDINATE_SYSTEM |
			((scale != NULL) ? X3D_VALUE_FLAG_SCALE : 0) |
			X3D_VALUE_FLAG_SKIP_TRANSLATES |
			X3D_VALUE_FLAG_SKIP_ROTATES;
	v.coordinate_system = X3D_COORDINATE_SYSTEM_XYZ;
	if(scale != NULL)
	{
	    v.scale_x = scale->x;
	    v.scale_y = scale->y;
	    v.scale_z = scale->z;
	}

        /* Create a new SAR visual model */
        vmodel = SARVisualModelNew(
            scene,
            NULL, NULL	/* No file or model name (so not shared) */
        );
        /* New visual model must not be shared */
        if(SARVisualModelGetRefCount(vmodel) == 1)
        {
            GLuint list = (GLuint)SARVisualModelNewList(vmodel);
            if(list != 0)
            {
                vmodel->load_state = SAR_VISUAL_MODEL_LOADING;
                glNewList(list, GL_COMPILE);
		X3DOpenDataGLOutput(
		    x3d_data, &v, NULL, 0
		);
                glEndList();
                vmodel->load_state = SAR_VISUAL_MODEL_LOADED;
            }
        }

        return(vmodel);
}

/*
 *	Loads a SAR visual model displaying the string s in 3d.
 *
 *	Origin is in lower left corner of 3d string.
 */
sar_visual_model_struct *SARObjLoadTextVisualModel(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        float font_width, float font_height,	/* In meters. */
        float character_spacing,		/* In meters. */
        const char *s,
        float *string_width			/* In meters. */
)
{
	sar_visual_model_struct *vmodel;

	if(string_width != NULL)
	    *string_width = 0.0f;

	if((scene == NULL) || ISSTREMPTY(s))
	    return(NULL);

	/* Create a new SAR visual model. */
	vmodel = SARVisualModelNew(
	    scene,
	    NULL, NULL	/* No filename or model name, so never shared. */
	);
	/* Really a new (not shared) visual model? */
	if(SARVisualModelGetRefCount(vmodel) == 1)
	{
	    GLuint list = (GLuint)SARVisualModelNewList(vmodel);
	    if(list != 0)
	    {
		vmodel->load_state = SAR_VISUAL_MODEL_LOADING;
		glNewList(list, GL_COMPILE);
		Text3DStringOutput(
		    font_width, font_height,
		    character_spacing,
		    s,
		    string_width
		);
		glEndList();
		vmodel->load_state = SAR_VISUAL_MODEL_LOADED;
	    }
	}

	return(vmodel);
}

/*
 *	Handles a translate parameter with respect to the given inputs.
 *
 *	Returns non-zero on error.
 */
int SARObjLoadTranslate(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_object_struct *obj_ptr, sar_parm_translate_struct *p_translate
)
{
	sar_position_struct new_pos;

	if((obj_ptr == NULL) || (p_translate == NULL))
	    return(-1);

	memcpy(&new_pos, &p_translate->translate, sizeof(sar_position_struct));

	/* Check certain objects that need their position offsetted
	 * internally.
	 */
	if(obj_ptr->type == SAR_OBJ_TYPE_HUMAN)
	{
	    /* Human objects fall at a constant rate, so we need to move
	     * it up twice the fall rate cycle to ensure that it
	     * `stays up' any hollow contact surfaces.
	     */
	    new_pos.z -= (float)(2.0 * SAR_DEF_HUMAN_FALL_RATE);
	}


	/* Update the ground elevation (for more information on usage,
	 * see SARSimApplyArtificialForce() on how it uses 
	 * SARSimFindGround().
	 */

        /* Check objects of types that need to know about ground
         * elevation.
         */
        if((obj_ptr->type == SAR_OBJ_TYPE_AIRCRAFT) ||
           (obj_ptr->type == SAR_OBJ_TYPE_HUMAN) ||
           (obj_ptr->type == SAR_OBJ_TYPE_FUELTANK)
        )
        {
	    float ground_elevation = 0.0f;

	    ground_elevation += SARSimFindGround(
		scene,
		core_ptr->object, core_ptr->total_objects,
		&new_pos	/* Position of our object. */
	    );

	    obj_ptr->ground_elevation_msl = ground_elevation;
	}

	/* Realize the new position. */
	SARSimWarpObject(
	    scene, obj_ptr,
	    &new_pos, NULL
	);

	return(0);
}

/*
 *	Translates the object to a random position.
 *
 *	Returns non-zero on error.
 */
int SARObjLoadTranslateRandom(
        sar_core_struct *core_ptr, sar_scene_struct *scene,  
        sar_object_struct *obj_ptr,
        sar_parm_translate_random_struct *p_translate_random
)
{
	float rand_coeff = SARRandomCoeff(0);
	float r = p_translate_random->radius_bound;
	float z = p_translate_random->z_bound;
	float rand_theta;
	sar_position_struct *pos;


	/* Calculate a random direction based on the random coefficient. */
	rand_theta = (float)SFMSanitizeRadians(rand_coeff * (2.0 * PI));


        if((obj_ptr == NULL) || (p_translate_random == NULL))
            return(-1);

	pos = &obj_ptr->pos;

	pos->x += (float)(sin(rand_theta) * rand_coeff * r);
	pos->y += (float)(cos(rand_theta) * rand_coeff * r);
	pos->z += (float)(rand_coeff * z);

	/* Realize position. */
	SARSimWarpObject(scene, obj_ptr, pos, NULL);

	return(0);
}

/*
 *	Loads an object sound source.
 */
static int SARObjLoadSoundSource(
        sar_scene_struct *scene,
        sar_sound_source_struct ***list, int *total,
        const char *arg
)
{
	int i;
	const char *cstrptr;
	float range, range_far, cutoff;
	sar_sound_source_struct *sndobj;
	char *strptr;
	char name[256];
	char filename[PATH_MAX + NAME_MAX];
        char filename_far[PATH_MAX + NAME_MAX];
	sar_position_struct pos;
	sar_direction_struct dir;
	int sample_rate_limit;


	if(ISSTREMPTY(arg))
	    return(-1);

#define SEEK_NEXT_ARG					\
{							\
 while(!ISBLANK(*cstrptr) && (*cstrptr != '\0'))	\
  cstrptr++;						\
 while(ISBLANK(*cstrptr))				\
  cstrptr++;						\
}

	/* Begin parsing argument. */
        cstrptr = arg;
        while(ISBLANK(*cstrptr))
            cstrptr++;

	/* First argument, name. */
	strncpy(name, cstrptr, 256);
	name[256 - 1] = '\0';
	strptr = strchr(name, ' ');
	if(strptr != NULL)
	    *strptr = '\0';
	strptr = strchr(name, '\t');
        if(strptr != NULL)
            *strptr = '\0';
	SEEK_NEXT_ARG

	/* Range */
	range = ATOF(cstrptr);
	SEEK_NEXT_ARG

	/* Range Far */
        range_far = ATOF(cstrptr);
	SEEK_NEXT_ARG

        /* Position */
	pos.x = ATOF(cstrptr);
	SEEK_NEXT_ARG
        pos.y = ATOF(cstrptr);
        SEEK_NEXT_ARG
        pos.z = ATOF(cstrptr);
        SEEK_NEXT_ARG

        /* Cutoff */
	cutoff = (float)DEGTORAD(ATOF(cstrptr));
        SEEK_NEXT_ARG

        /* Direction */
        dir.heading = (float)DEGTORAD(ATOF(cstrptr));
        SEEK_NEXT_ARG
        dir.pitch = (float)DEGTORAD(ATOF(cstrptr));
        SEEK_NEXT_ARG
        dir.bank = (float)DEGTORAD(ATOF(cstrptr));	/* Not used */
        SEEK_NEXT_ARG

	/* Sample Rate Limit */
	sample_rate_limit = ATOI(cstrptr);
	SEEK_NEXT_ARG

	/* File name */
        strncpy(filename, cstrptr, PATH_MAX + NAME_MAX);
        filename[PATH_MAX + NAME_MAX - 1] = '\0';
        strptr = strchr(filename, ' ');
        if(strptr != NULL)
            *strptr = '\0';
        strptr = strchr(filename, '\t');
        if(strptr != NULL)
            *strptr = '\0';
	SEEK_NEXT_ARG

        /* File name far */
        strncpy(filename_far, cstrptr, PATH_MAX + NAME_MAX);
        filename_far[PATH_MAX + NAME_MAX - 1] = '\0';
        strptr = strchr(filename_far, ' ');
        if(strptr != NULL)
            *strptr = '\0';
        strptr = strchr(filename_far, '\t');
        if(strptr != NULL)
            *strptr = '\0';
        SEEK_NEXT_ARG


	/* Must have a file name */
	if(ISSTREMPTY(filename))
	    return(-1);

	/* Complete file names */
        if(!ISPATHABSOLUTE(filename))
        {
	    const char *s = PrefixPaths(dname.local_data, filename);
            struct stat stat_buf;

	    /* If file does not exist locally then use global */
	    if((s != NULL) ? stat(cstrptr, &stat_buf) : True)
		s = PrefixPaths(dname.global_data, filename);
            if(s != NULL)
                strncpy(filename, s, PATH_MAX + NAME_MAX);
            filename[PATH_MAX + NAME_MAX - 1] = '\0';
        }
        if(!ISPATHABSOLUTE(filename_far))
        {
            const char *s = PrefixPaths(dname.local_data, filename_far);
            struct stat stat_buf;

	    /* If file does not exist locally then use global */
	    if((s != NULL) ? stat(cstrptr, &stat_buf) : True)
		s = PrefixPaths(dname.global_data, filename_far);
            if(s != NULL)
                strncpy(filename_far, s, PATH_MAX + NAME_MAX);
            filename_far[PATH_MAX + NAME_MAX - 1] = '\0';
        }


	/* Allocate more pointers */
	i = *total;
	*total = i + 1;
	*list = (sar_sound_source_struct **)realloc(
	    *list,
	    (*total) * sizeof(sar_sound_source_struct *)
	);
	if(*list == NULL)
	{
	    *total = 0;
	    return(-1);
	}
	/* Create new sound source */
	(*list)[i] = sndobj = SARSoundSourceNew(
	    name,
	    filename,
	    filename_far,
	    range,
	    range_far,
	    &pos, cutoff, &dir,
	    sample_rate_limit
	);

#undef SEEK_NEXT_ARG

	return(0);
}


/*
 *	Loads a new texture specified by the values in p_texture_load
 *	and stores it on the scene structure.
 *
 *	If the scene structure already has a texture loaded that matches
 *	the texture reference name specified in p_texture_load then 
 *	nothing will be done and 0 is returned.
 *
 *	If the texture reference name specified in p_texture_load is
 *	NULL or an empty string then a new texture will be loaded
 *	implicitly.
 *
 *	Returns non-zero on error.
 */
int SARObjLoadTexture(
	sar_core_struct *core_ptr, sar_scene_struct *scene,
	sar_parm_texture_load_struct *p_texture_load
)
{
	v3d_texture_ref_struct *t;
#define len	1024
        char lname[len];
	char lpath[len];


        if((scene == NULL) || (p_texture_load == NULL))
            return(-1);

	*lname = '\0';
	*lpath = '\0';

	if(p_texture_load->name != NULL)
	    strncpy(lname, p_texture_load->name, len);
        if(p_texture_load->file != NULL)
            strncpy(lpath, p_texture_load->file, len);

	/* No path? */
	if(ISSTREMPTY(lpath))
	    return(-1);

	/* Check if a texture with he same reference name is already
	 * loaded on the scene, if it is then do not load this texture.
	 */
	if(ISSTREMPTY(lname))
	    t = NULL;
	else
            t = SARGetTextureRefByName(scene, lname);
	/* Is there a texture with the same reference name loaded? */
	if(t != NULL)
	    return(0);

	/* Complete texture file path */
	if(!ISPATHABSOLUTE(lpath))
	{
	    const char *s = PrefixPaths(dname.local_data, lpath);
	    struct stat stat_buf;

	    /* If file does not exist locally then use global */
	    if((s != NULL) ? stat(s, &stat_buf) : True)
		s = PrefixPaths(dname.global_data, lpath);
	    if(s != NULL)
		strncpy(lpath, s, len);
	    lpath[len - 1] = '\0';
	}

	/* Load new texture */
	t = V3DTextureLoadFromFile2DPreempt(
	    lpath, lname, V3D_TEX_FORMAT_RGBA
	);
	if(t != NULL)
	{
	    /* Add this texture to the scene's list of textures */
	    int i = MAX(scene->total_texture_refs, 0);
	    scene->total_texture_refs = i + 1;

	    scene->texture_ref = (v3d_texture_ref_struct **)realloc(
		scene->texture_ref,
		scene->total_texture_refs * sizeof(v3d_texture_ref_struct *)
	    );
	    if(scene->texture_ref == NULL)
	    {
		V3DTextureDestroy(t);
		return(-1);
	    }
	    scene->texture_ref[i] = t;

	    /* Set texture priority. */
	    V3DTexturePriority(t, p_texture_load->priority);

	    return(0);
	}
	else
	{
	    /* Could not load texture */
	    return(-1);
	}
#undef len
}

/*
 *      Loads a helipad object from the given argument line string.
 *
 *      Returns the newly created human object number or -1 on error.
 */
int SARObjLoadHelipad(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_parm_new_helipad_struct *p_new_helipad
)
{
	const char *s;
        int obj_num = -1;
	sar_object_struct *obj_ptr;
	sar_object_helipad_struct *obj_helipad_ptr;
	const char *style, *ref_obj_name;
	const char *tex_name = SAR_STD_TEXNAME_HELIPAD_PAVED;
	int ref_obj_num = -1;
	sar_object_struct *ref_obj_ptr = NULL;


        if((scene == NULL) || (p_new_helipad == NULL))
            return(obj_num);

	/* Create a new helipad object */
	obj_num = SARObjCreate(
	    scene, &core_ptr->object, &core_ptr->total_objects,
	    SAR_OBJ_TYPE_HELIPAD
	);
	obj_ptr = (obj_num > -1) ? core_ptr->object[obj_num] : NULL;
	if(obj_ptr == NULL)
	{
	    obj_num = -1;
	    return(obj_num);
	}
	obj_helipad_ptr = SAR_OBJ_GET_HELIPAD(obj_ptr);
	if(obj_helipad_ptr == NULL)
	    return(obj_num);

	/* Begin setting values */

	/* Helipad flags */
	obj_helipad_ptr->flags = p_new_helipad->flags;

	/* Range */
	obj_ptr->range = SAR_HELIPAD_DEF_RANGE;

	/* Style */
	style = p_new_helipad->style;
	if(style == NULL)
	{
	    /* Set default style */
	    obj_helipad_ptr->style = SAR_HELIPAD_STYLE_GROUND_PAVED;
	    tex_name = SAR_STD_TEXNAME_HELIPAD_PAVED;
	}
	else
	{
	    if(!strcasecmp(style, "ground_paved") ||
               !strcasecmp(style, "standard") ||
               !strcasecmp(style, "default")
	    )
	    {
		obj_helipad_ptr->style = SAR_HELIPAD_STYLE_GROUND_PAVED;
		tex_name = SAR_STD_TEXNAME_HELIPAD_PAVED;
	    }
	    else if(!strcasecmp(style, "ground_bare"))
	    {
		obj_helipad_ptr->style = SAR_HELIPAD_STYLE_GROUND_BARE;
		tex_name = SAR_STD_TEXNAME_HELIPAD_BARE;
	    }
            else if(!strcasecmp(style, "building"))
	    {
                obj_helipad_ptr->style = SAR_HELIPAD_STYLE_BUILDING;
		tex_name = SAR_STD_TEXNAME_HELIPAD_BUILDING;
	    }
            else if(!strcasecmp(style, "vehicle"))
	    {
                obj_helipad_ptr->style = SAR_HELIPAD_STYLE_VEHICLE;
		tex_name = SAR_STD_TEXNAME_HELIPAD_VEHICLE;
	    }
	    else
	    {
		fprintf(
		    stderr,
		    "Unsupported helipad style type `%s'.\n",
		    style
		);
		obj_helipad_ptr->style = SAR_HELIPAD_STYLE_GROUND_PAVED;
		tex_name = SAR_STD_TEXNAME_HELIPAD_PAVED;
	    }
	}


	/* Size */
	obj_helipad_ptr->length = (float)MAX(p_new_helipad->length, 1.0);
        obj_helipad_ptr->width = (float)MAX(p_new_helipad->width, 1.0);
	obj_helipad_ptr->recession = (float)MAX(p_new_helipad->recession, 0.0);

	/* Specify contact bounds so that the helipad is landable */
	SARObjAddContactBoundsRectangular(
	    obj_ptr, SAR_CRASH_FLAG_SUPPORT_SURFACE, 0,
	    -(obj_helipad_ptr->width / 2),
	    (obj_helipad_ptr->width / 2),
	    -(obj_helipad_ptr->length / 2),
	    (obj_helipad_ptr->length / 2),
	    0.0f, 0.0f
	);

	/* Label */
	s = p_new_helipad->label;
	obj_helipad_ptr->label = STRDUP(s);
        if(s != NULL)
        {
	    /* Generate visual model for label using 3d text */
            int len = MAX(STRLEN(s), 1);
            float font_height = (float)(MIN(
		obj_helipad_ptr->width, obj_helipad_ptr->length
	    ) * 0.7 / len);
            float font_width = (float)(font_height * 0.8);
            obj_helipad_ptr->label_vmodel = SARObjLoadTextVisualModel(
                core_ptr, scene,
                font_width, font_height,
                (float)(font_width * 0.05),
                s,
                &obj_helipad_ptr->label_width
            );
        }
        else
        {
	    /* No label */
            obj_helipad_ptr->label_vmodel = NULL;
            obj_helipad_ptr->label_width = 0;
        }

	/* Light spacing */
	obj_helipad_ptr->light_spacing = obj_helipad_ptr->length / 10;

	/* Texture reference number from scene */
	obj_helipad_ptr->tex_num = SARGetTextureRefNumberByName(
	    scene, tex_name
	);

	/* Match reference object? */
	ref_obj_name = p_new_helipad->ref_obj_name;
	if((ref_obj_name != NULL) ? (*ref_obj_name != '\0') : False)
	{
	    ref_obj_ptr = SARObjMatchPointerByName(
		scene, core_ptr->object, core_ptr->total_objects,
		ref_obj_name, &ref_obj_num
	    );
	    if(ref_obj_ptr != NULL)
	    {

	    }
	    else
	    {
		fprintf(
		    stderr,
 "Could not match reference object `%s' for helipad object #%i.\n",
		    ref_obj_name, obj_num
		);
	    }
	}
	/* Set reference object */
	obj_helipad_ptr->ref_object = ref_obj_num;
	/* Make sure reference object is not this object */
	if(ref_obj_ptr == obj_ptr)
	{
	    ref_obj_num = -1;
	    ref_obj_ptr = NULL;

	    obj_helipad_ptr->flags &= ~SAR_HELIPAD_FLAG_REF_OBJECT;
	    obj_helipad_ptr->flags &= ~SAR_HELIPAD_FLAG_FOLLOW_REF_OBJECT;

	    obj_helipad_ptr->ref_object = -1;
	}

	/* Set reference offset relative to the reference object */
	memcpy(
	    &obj_helipad_ptr->ref_offset,
	    &p_new_helipad->ref_offset,
	    sizeof(sar_position_struct)
	);

	/* Set reference direction relative to the reference object */
	memcpy(
	    &obj_helipad_ptr->ref_dir,
	    &p_new_helipad->ref_dir, 
	    sizeof(sar_direction_struct)
	);

	/* Realize new position if there is a reference object */
	if((obj_helipad_ptr->flags & SAR_HELIPAD_FLAG_REF_OBJECT) &&
           (obj_helipad_ptr->flags & SAR_HELIPAD_FLAG_FOLLOW_REF_OBJECT)
	)
	    SARSimWarpObjectRelative(
		scene, obj_ptr,
		core_ptr->object, core_ptr->total_objects, ref_obj_num,
		&obj_helipad_ptr->ref_offset,
		&obj_helipad_ptr->ref_dir
	    );

	return(obj_num);
}

/*
 *	Loads a runway object from the given argument line string.
 *
 *      Returns the newly created human object number or -1 on error.
 */
int SARObjLoadRunway(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_parm_new_runway_struct *p_new_runway
)
{
	const char *s;
        int obj_num = -1;
        sar_object_struct *obj_ptr;
        sar_object_runway_struct *obj_runway_ptr;


        if((scene == NULL) || (p_new_runway == NULL))
            return(obj_num);

        /* Create a new runway object */
        obj_num = SARObjCreate(
            scene, &core_ptr->object, &core_ptr->total_objects,
            SAR_OBJ_TYPE_RUNWAY
        );
        obj_ptr = (obj_num > -1) ? core_ptr->object[obj_num] : NULL;
        if(obj_ptr == NULL)
        {
            obj_num = -1;
            return(obj_num);
        }
        obj_runway_ptr = SAR_OBJ_GET_RUNWAY(obj_ptr);
        if(obj_runway_ptr == NULL)
            return(obj_num);

        /* Begin setting values */

	/* Flags */
	obj_runway_ptr->flags |= p_new_runway->flags;

	/* Range */
	obj_ptr->range = p_new_runway->range;

	/* Size */
	obj_runway_ptr->length = p_new_runway->length;
	obj_runway_ptr->width = p_new_runway->width;

	/* Surface type */
	obj_runway_ptr->surface = p_new_runway->surface;

	/* Set contact bounds to ensure that the object is landable */
	SARObjAddContactBoundsRectangular(
	    obj_ptr,
	    SAR_CRASH_FLAG_SUPPORT_SURFACE, 0,
	    -(obj_runway_ptr->width / 2),
	    (obj_runway_ptr->width / 2),
	    -(obj_runway_ptr->length / 2),
	    (obj_runway_ptr->length / 2),
	    0.0f, 0.0f
	);

	/* Label */
	obj_runway_ptr->north_label = STRDUP(p_new_runway->north_label);
	obj_runway_ptr->south_label = STRDUP(p_new_runway->south_label);

	/* Create visual models for labels using 3d text */
	s = p_new_runway->north_label;
	if(s != NULL)
	{
	    int len = MAX(STRLEN(s), 1);
	    float font_height = (float)(obj_runway_ptr->width * 0.8 / len);
 	    float font_width = (float)(font_height * 0.8);

	    obj_runway_ptr->north_label_vmodel = SARObjLoadTextVisualModel(
		core_ptr, scene,
		font_width, font_height,
		(float)(font_width * 0.15),
		s,
		&obj_runway_ptr->north_label_width
	    );
	}
	else
	{
	    obj_runway_ptr->north_label_vmodel = NULL;
	    obj_runway_ptr->north_label_width = 0;
	}
        s = p_new_runway->south_label;
        if(s != NULL)
        {
            int len = STRLEN(s);
            float font_height = (float)(obj_runway_ptr->width * 0.8 / len);
            float font_width = (float)(font_height * 0.8);

            obj_runway_ptr->south_label_vmodel = SARObjLoadTextVisualModel(
                core_ptr, scene,
                font_width, font_height,
                (float)(font_width * 0.15),
                s,
                &obj_runway_ptr->south_label_width
            );
        }
        else
        {
            obj_runway_ptr->south_label_vmodel = NULL;
            obj_runway_ptr->south_label_width = 0;
        }

	/* Dashes */
	obj_runway_ptr->dashes = (p_new_runway->dashes > 0) ?
	    p_new_runway->dashes : 10;

	/* Displaced thresholds */
	obj_runway_ptr->north_displaced_threshold =
	    p_new_runway->north_displaced_threshold;
        obj_runway_ptr->south_displaced_threshold =
            p_new_runway->south_displaced_threshold;

	/* Edge lighting */
	obj_runway_ptr->edge_light_spacing = (float)(
	    (p_new_runway->edge_light_spacing > 0.0f) ?
		p_new_runway->edge_light_spacing :
		(obj_runway_ptr->length / 50)
	);

	/* Approach lighting */
	obj_runway_ptr->north_approach_lighting_flags = 0;
	obj_runway_ptr->south_approach_lighting_flags = 0;

	obj_runway_ptr->tracer_anim_pos = 0;
	obj_runway_ptr->tracer_anim_rate = (sar_grad_anim_t)(
	    ((sar_grad_anim_t)-1) * 0.5
	);

	/* Get texture number from scene */
	obj_runway_ptr->tex_num = SARGetTextureRefNumberByName(
	    scene, SAR_STD_TEXNAME_RUNWAY
	);

	/* Begin loading visual models for runway decorations */
	/* Threshold */
	if(obj_runway_ptr->flags & SAR_RUNWAY_FLAG_THRESHOLDS)
	{
	    sar_scale_struct scale;
	    scale.x = obj_runway_ptr->width;
	    scale.y = 50.0f;
	    scale.z = 1.0f;
	    obj_runway_ptr->threshold_vmodel = SARObjLoadX3DDataVisualModel(
		core_ptr, scene,
		runway_threshold_x3d,
		&scale
	    );
	}
        /* Touchdown markers */
        if(obj_runway_ptr->flags & SAR_RUNWAY_FLAG_TD_MARKERS)
        {
            sar_scale_struct scale;
            scale.x = obj_runway_ptr->width;
            scale.y = 30.0f;
            scale.z = 1.0f;
            obj_runway_ptr->td_marker_vmodel = SARObjLoadX3DDataVisualModel(
                core_ptr, scene,
                runway_td_markers_x3d,
                &scale
            );
        }
        /* Midway markers */
        if(obj_runway_ptr->flags & SAR_RUNWAY_FLAG_MIDWAY_MARKERS)
        {
            sar_scale_struct scale;
            scale.x = obj_runway_ptr->width;
            scale.y = (float)(obj_runway_ptr->length * 0.05);
            scale.z = 1.0f;
            obj_runway_ptr->midway_marker_vmodel = SARObjLoadX3DDataVisualModel(
                core_ptr, scene,
                runway_midway_markers_x3d,
                &scale
            );
        }
        /* North displaced threshold */
        if(obj_runway_ptr->north_displaced_threshold > 0.0f)
        {
            sar_scale_struct scale;
            scale.x = obj_runway_ptr->width;
            scale.y = obj_runway_ptr->north_displaced_threshold;
            scale.z = 1.0f;
            obj_runway_ptr->north_displaced_threshold_vmodel =
		SARObjLoadX3DDataVisualModel(
		    core_ptr, scene,
		    runway_displaced_threshold_x3d,
		    &scale
		);
        }
        /* South displaced threshold */
        if(obj_runway_ptr->south_displaced_threshold > 0.0f)
        {
            sar_scale_struct scale;
            scale.x = obj_runway_ptr->width;
            scale.y = obj_runway_ptr->south_displaced_threshold;
            scale.z = 1.0f;
            obj_runway_ptr->south_displaced_threshold_vmodel =
                SARObjLoadX3DDataVisualModel(
                    core_ptr, scene,
                    runway_displaced_threshold_x3d,
                    &scale
                );
        }

	return(obj_num);
}

/*
 *	Loads a human object from the given argument line string.
 *
 *	Returns the newly created human object number or -1 on error.
 */
int SARObjLoadHuman(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_parm_new_human_struct *p_new_human
)
{
	int obj_num = -1;

        if((scene == NULL) || (p_new_human == NULL))
            return(obj_num);

	/* Create new human object */
	obj_num = SARHumanCreate(
	    core_ptr->human_data,
	    scene, &core_ptr->object, &core_ptr->total_objects,
	    p_new_human->flags,
	    p_new_human->type_name
	);

	return(obj_num);
}

/*
 *      Loads a fire object from the given argument line string.
 *
 *      Returns the newly created fire object number or -1 on error.
 */
int SARObjLoadFire( 
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_parm_new_fire_struct *p_new_fire 
)
{
	int obj_num = -1;
	sar_object_struct *obj_ptr;
	sar_object_fire_struct *obj_fire_ptr;
	sar_position_struct pos;


        if((scene == NULL) || (p_new_fire == NULL))
            return(obj_num);

	memset(&pos, 0x00, sizeof(sar_position_struct));

        /* Create new fire object */
        obj_num = FireCreate(
	    core_ptr, scene,
	    &core_ptr->object, &core_ptr->total_objects,
	    &pos, p_new_fire->radius, p_new_fire->height,
	    -1,
	    SAR_STD_TEXNAME_FIRE
        );
        obj_ptr = (obj_num > -1) ? core_ptr->object[obj_num] : NULL;
        if(obj_ptr == NULL)
        {
            obj_num = -1;
            return(obj_num);
        }
	obj_fire_ptr = SAR_OBJ_GET_FIRE(obj_ptr);
        if(obj_fire_ptr == NULL)
            return(obj_num);

	/* Begin setting values */

	/* Animation repeats */
	obj_fire_ptr->total_frame_repeats = -1;	/* Infinate */

	obj_ptr->life_span = 0;

        return(obj_num);
}

/*
 *      Loads a smoke object from the given argument line string.
 *
 *      Returns the newly created fire object number or -1 on error.
 */
int SARObjLoadSmoke(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_parm_new_smoke_struct *p_new_smoke
)
{
        int obj_num = -1;
	sar_object_struct *obj_ptr;
        sar_position_struct pos;
	const char *tex_name = NULL;


        if((scene == NULL) || (p_new_smoke == NULL))
            return(obj_num);

        memset(&pos, 0x00, sizeof(sar_position_struct));

	/* Get name of smoke texture to use from the color code */
	switch(p_new_smoke->color_code)
	{
	  case 0:	/* Light */
	    tex_name = SAR_STD_TEXNAME_SMOKE_LIGHT;
	    break;
	  case 1:	/* Medium */
	    tex_name = SAR_STD_TEXNAME_SMOKE_MEDIUM;
            break;
          case 2:	/* Dark */
            tex_name = SAR_STD_TEXNAME_SMOKE_DARK;
            break;
	}

        /* Create new smoke trail object */
	obj_num = SmokeCreate(
	    scene, &core_ptr->object, &core_ptr->total_objects,
	    SAR_OBJ_SMOKE_TYPE_SMOKE,	/* Smoke type */
            &pos, &p_new_smoke->offset,	/* Position and unit spawn offset */
	    p_new_smoke->radius_start,
	    p_new_smoke->radius_max,
	    p_new_smoke->radius_rate,	/* May be -1.0 for autocalc */
	    p_new_smoke->hide_at_max,
	    p_new_smoke->total_units,
	    p_new_smoke->respawn_int,
	    tex_name,
	    -1,				/* No reference object */
	    0				/* No limit on life span */
	);
        obj_ptr = (obj_num > -1) ? core_ptr->object[obj_num] : NULL;
        if(obj_ptr == NULL)
        {
            obj_num = -1;
            return(obj_num);
        }

	/* Begin setting values */



        return(obj_num);
}

/*
 *	Loads heightfield primitive p for the given object.
 *
 *	If list is 0 then no gl commands will be issued and only the
 *	z height points will be recorded.
 */
int SARObjLoadHeightField(
        sar_core_struct *core_ptr,
        int obj_num, sar_object_struct *obj_ptr,
        void *p, const char *filename, int line_num,
	GLuint list			/* Can be 0. */
)
{
	int status;
	float x_length, y_length, z_length;
	int num_grids_x, num_grids_y;
	double grid_space_x, grid_space_y;
	double *zpoints = NULL;

        sar_object_ground_struct *obj_ground_ptr = NULL;
	mp_heightfield_load_struct *mp_heightfield_load =
	    (mp_heightfield_load_struct *)p;
	v3d_hf_options_struct hfopt;
	char path[PATH_MAX + NAME_MAX];


        if((obj_ptr == NULL) || (mp_heightfield_load == NULL))
            return(-1);

	if(ISSTREMPTY(mp_heightfield_load->path))
	    return(-1);

        obj_ground_ptr = SAR_OBJ_GET_GROUND(obj_ptr);

	/* Get heightfield file */
	strncpy(path, mp_heightfield_load->path, PATH_MAX + NAME_MAX);
	path[PATH_MAX + NAME_MAX - 1] = '\0';

        if(!ISPATHABSOLUTE(path))
        {
	    const char *s = PrefixPaths(dname.local_data, path);
	    struct stat stat_buf;

	    /* If file does not exist locally then use global */
	    if((s != NULL) ? stat(s, &stat_buf) : True)
		s = PrefixPaths(dname.global_data, path);
            if(s != NULL)
                strncpy(path, s, PATH_MAX + NAME_MAX);
            path[PATH_MAX + NAME_MAX - 1] = '\0';
        }

	/* Begin loading heightfield */

	/* Get size of heightfield from heightfield load primitive */
	x_length = (float)mp_heightfield_load->x_length;
        y_length = (float)mp_heightfield_load->y_length;
        z_length = (float)mp_heightfield_load->z_length;
	if((x_length <= 0.0f) ||
           (y_length <= 0.0f) ||
           (z_length <= 0.0f)
	)
	    return(-2);

	glPushMatrix();
	{
	    /* Translate */
	    glTranslated(
		mp_heightfield_load->x,
		mp_heightfield_load->z,
		-mp_heightfield_load->y
	    );

	    /* No rotations */

	    /* Set up heightfield options */
	    hfopt.flags = (V3D_HF_OPT_FLAG_WINDING |
		V3D_HF_OPT_FLAG_SET_NORMAL | V3D_HF_OPT_FLAG_SET_TEXCOORD
	    );
	    hfopt.winding = V3D_HF_WIND_CCW;
	    hfopt.set_normal = V3D_HF_SET_NORMAL_STREATCHED;
	    hfopt.set_texcoord = V3D_HF_SET_TEXCOORD_ALWAYS;

	    /* Load heightfield */
	    status = V3DHFLoadFromFile(
		path,
		x_length, y_length, z_length,	/* Scaling in meters. */
		&num_grids_x, &num_grids_y,	/* Number of grids. */
		&grid_space_x, &grid_space_y,	/* Grid spacing in meters. */
		&zpoints,			/* Heightfield points as u_int8_t. */
		(void *)list,			/* Our GL list in context. */
		&hfopt
	    );
	    if(status)
	    {
		/* Load error */
		free(zpoints);
		return(-1);
	    }
	}
	glPopMatrix();

	/* Set newly loaded values to object */
	if(obj_ground_ptr != NULL)
	{
	    obj_ground_ptr->x_trans = (float)mp_heightfield_load->x;
	    obj_ground_ptr->y_trans = (float)mp_heightfield_load->y;
	    obj_ground_ptr->z_trans = (float)mp_heightfield_load->z;

	    obj_ground_ptr->x_len = (float)x_length;
	    obj_ground_ptr->y_len = (float)y_length;

	    obj_ground_ptr->grid_points_x = num_grids_x;
            obj_ground_ptr->grid_points_y = num_grids_y;
            obj_ground_ptr->grid_points_total = num_grids_x * num_grids_y;

	    obj_ground_ptr->grid_x_spacing = (float)grid_space_x;
            obj_ground_ptr->grid_y_spacing = (float)grid_space_y;
            obj_ground_ptr->grid_z_spacing = (float)z_length;

	    /* Replace old heightfield z point data on ground object
	     * structure with the newly allocated one
	     */
	    free(obj_ground_ptr->z_point_value);
	    obj_ground_ptr->z_point_value = zpoints;
	    zpoints = NULL;	/* Reset zpoints to mark it as transfered */
	}

	/* Free allocated heightfield z point data if it was not 
	 * transfered to the ground object structure
	 */
	free(zpoints);
	zpoints = NULL;

	return(0);
}

/*
 *	Called by SARObjLoadFromFile() to parse and handle the given
 *	V3D primitive p with respect to the given inputs.
 */
static int SARObjLoadVisualPrimitive(
	sar_core_struct *core_ptr,
        int obj_num, sar_object_struct *obj_ptr,
        void *p, const char *filename, int line_num
)
{
	int i, ptype, v_total = 0;
	Boolean need_end_primitive = False;
	mp_vertex_struct *ns = NULL, **nd = NULL;
	mp_vertex_struct *vs = NULL, **vd = NULL;
	mp_vertex_struct *tcs = NULL, **tcd = NULL;

	mp_point_struct *mp_point;
	mp_line_struct *mp_line;
	mp_line_strip_struct *mp_line_strip;
	mp_line_loop_struct *mp_line_loop;
	mp_triangle_struct *mp_triangle;
	mp_triangle_strip_struct *mp_triangle_strip;
	mp_triangle_fan_struct *mp_triangle_fan;
	mp_quad_struct *mp_quad;
	mp_quad_strip_struct *mp_quad_strip;
	mp_polygon_struct *mp_polygon;
	mp_vertex_struct *first_normal_ptr = NULL, *n_ptr;

	float x = 0.0f, y = 0.0f, z = 0.0f;
	float tx = 0.0f, ty = 0.0f;


	if((obj_ptr == NULL) || (p == NULL))
	    return(-1);

	ptype = *(int *)p;	/* Primitive type */

	/* Handle by primitive type */
	switch(ptype)
	{
	  case V3DMP_TYPE_POINT:
	    mp_point = (mp_point_struct *)p;
	    if(last_begin_primitive_type != ptype)
	    {
		glBegin(GL_POINTS);
		last_begin_primitive_type = ptype;
	    }
	    ns = &mp_point->n[0];
	    vs = &mp_point->v[0];
	    tcs = &mp_point->tc[0];
	    v_total = V3DMP_POINT_NVERTEX;
	    break;

          case V3DMP_TYPE_LINE:
            mp_line = (mp_line_struct *)p;
            if(last_begin_primitive_type != ptype)
            {
                glBegin(GL_LINES);
                last_begin_primitive_type = ptype;
            }
	    ns = &mp_line->n[0];
            vs = &mp_line->v[0];
	    tcs = &mp_line->tc[0];
            v_total = V3DMP_LINE_NVERTEX;
            break;

          case V3DMP_TYPE_LINE_STRIP:
            mp_line_strip = (mp_line_strip_struct *)p;
            glBegin(GL_LINE_STRIP); need_end_primitive = True;
	    nd = mp_line_strip->n;
            vd = mp_line_strip->v;
	    tcd = mp_line_strip->tc;
            v_total = mp_line_strip->total;
            break;

          case V3DMP_TYPE_LINE_LOOP:
            mp_line_loop = (mp_line_loop_struct *)p;
            glBegin(GL_LINE_LOOP); need_end_primitive = True;
	    nd = mp_line_loop->n;
            vd = mp_line_loop->v;
            tcd = mp_line_loop->tc;
            v_total = mp_line_loop->total;
            break;

          case V3DMP_TYPE_TRIANGLE:
            mp_triangle = (mp_triangle_struct *)p;
            if(last_begin_primitive_type != ptype)
            {
                glBegin(GL_TRIANGLES);
                last_begin_primitive_type = ptype;
            }
            ns = &mp_triangle->n[0];
            vs = &mp_triangle->v[0];
            tcs = &mp_triangle->tc[0];
            v_total = V3DMP_TRIANGLE_NVERTEX;
            break;

          case V3DMP_TYPE_TRIANGLE_STRIP:
            mp_triangle_strip = (mp_triangle_strip_struct *)p;
            glBegin(GL_TRIANGLE_STRIP); need_end_primitive = True;
            nd = mp_triangle_strip->n;
            vd = mp_triangle_strip->v;
            tcd = mp_triangle_strip->tc;
            v_total = mp_triangle_strip->total;
            break;

          case V3DMP_TYPE_TRIANGLE_FAN:
            mp_triangle_fan = (mp_triangle_fan_struct *)p;
            glBegin(GL_TRIANGLE_FAN); need_end_primitive = True;
            nd = mp_triangle_fan->n;
            vd = mp_triangle_fan->v;
            tcd = mp_triangle_fan->tc;
            v_total = mp_triangle_fan->total;
            break;

          case V3DMP_TYPE_QUAD:
            mp_quad = (mp_quad_struct *)p;
            if(last_begin_primitive_type != ptype)
            {
                glBegin(GL_QUADS);
                last_begin_primitive_type = ptype;
            }
            ns = &mp_quad->n[0];
            vs = &mp_quad->v[0];
            tcs = &mp_quad->tc[0];
            v_total = V3DMP_QUAD_NVERTEX;
            break;

          case V3DMP_TYPE_QUAD_STRIP:
            mp_quad_strip = (mp_quad_strip_struct *)p;
            glBegin(GL_QUAD_STRIP); need_end_primitive = True;
            nd = mp_quad_strip->n;
            vd = mp_quad_strip->v;   
            tcd = mp_quad_strip->tc;
            v_total = mp_quad_strip->total;
            break;

          case V3DMP_TYPE_POLYGON:
            mp_polygon = (mp_polygon_struct *)p;
            glBegin(GL_POLYGON); need_end_primitive = True;
            nd = mp_polygon->n;
            vd = mp_polygon->v;
            tcd = mp_polygon->tc;
            v_total = mp_polygon->total;
            break;
	}


	/* Get pointer to first normal */
	i = 0;
	if(nd != NULL)
	    first_normal_ptr = nd[i];
	else if(ns != NULL)
	    first_normal_ptr = &ns[i];
	else
	    first_normal_ptr = NULL;

	/* Iterate through each vertex start from last to first,
	 * becuase the winding stored on V3D model file is clockwise
	 * and we need to handle it counter-clockwise
	 */
	for(i = v_total - 1; i >= 0; i--)
	{
	    /* Get vertex values but do not set just yet, we only need
	     * the coordinates for now in order to set the texcoord
	     * first (in case the texture is being plane oriented)
	     */
	    if((vd != NULL) ? (vd[i] != NULL) : False)
	    {
		const mp_vertex_struct *v = vd[i];
		x = (float)v->x;
		y = (float)v->y;
		z = (float)v->z;
	    }
	    else if(vs != NULL)
	    {
		const mp_vertex_struct *v = &vs[i];
                x = (float)v->x;
                y = (float)v->y;
                z = (float)v->z;
            }

	    /* Get normal and make sure it is not a zero vector, if it
	     * is then use the first normal (if any).
	     */
	    if(nd != NULL)
	    {
		n_ptr = nd[i];
                if((n_ptr->x == 0.0) && (n_ptr->y == 0.0) && (n_ptr->z == 0.0))
                    n_ptr = first_normal_ptr;
	    }
	    else if(ns != NULL)
	    {
		n_ptr = &ns[i];
                if((n_ptr->x == 0.0) && (n_ptr->y == 0.0) && (n_ptr->z == 0.0))
                    n_ptr = first_normal_ptr;
	    }
	    else
	    {
		n_ptr = first_normal_ptr;	/* All elese use first normal */
	    }

	    /* Got valid normal? */
	    if(n_ptr != NULL)
		glNormal3d(n_ptr->x, n_ptr->z, -n_ptr->y);

	    /* Texture enabled? */
	    if(tex_on)
	    {
		/* Set texture coordinates */
                switch(tex_orient)
                {
                  case TEX_ORIENT_XY:
                    if(tex_coord.w > 0.0f)
                        tx = (x - tex_coord.i) / tex_coord.w;
                    else
                        tx = 0.0f;
                    if(tex_coord.h > 0.0f)
                        ty = (y - tex_coord.j) / tex_coord.h;
                    else
                        ty = 0.0f;
                    glTexCoord2f(tx, 1.0f - ty);
                    break;

                  case TEX_ORIENT_YZ:
                    if(tex_coord.w > 0.0f)
                        tx = -(y - tex_coord.i) / tex_coord.w;
                    else
                        tx = 0.0f;
                    if(tex_coord.h > 0.0f)
                        ty = (z - tex_coord.j) / tex_coord.h;
                    else
                        ty = 0.0f;
                    glTexCoord2f(tx, 1.0f - ty);
                    break;

                  case TEX_ORIENT_XZ:
                    if(tex_coord.w > 0.0f)
                        tx = (x - tex_coord.i) / tex_coord.w;
                    else
                        tx = 0.0f;
                    if(tex_coord.h > 0.0f)
                        ty = (z - tex_coord.j) / tex_coord.h;
                    else
                        ty = 0.0f;
                    glTexCoord2f(tx, 1.0f - ty);
                    break;

		  default:
		    if(tcd != NULL)
		    {
			if(tcd[i] != NULL)
			{
			    tx = (float)tcd[i]->x;
                            ty = (float)tcd[i]->y;
			}
		    }
		    else if(tcs != NULL)
		    {
			tx = (float)tcs[i].x;
			ty = (float)tcs[i].y;
		    }

                    glTexCoord2f(tx, 1.0f - ty);
		    break;
		}
	    }

	    /* Set vertex */
            glVertex3d(x, z, -y);
	}

	/* End primitive as needed */
	if(need_end_primitive)
	{
	    glEnd();
	    need_end_primitive = False;
	}

	return(0);
} 



/*
 *	Loads the line which should have come from an object file.
 *	This function is designed to be called by SARObjLoadFromFile()
 *	and inputs are assumed valid.
 *
 *	Returns 0 if the line was parsed and handled, -2 if the line
 *	is not supported or a comment, and -1 on general error.
 */
static int SARObjLoadLine(
	sar_core_struct *core_ptr,
	int obj_num, sar_object_struct *obj_ptr,
	const char *line, const char *filename, int line_num
)
{
	char *s;
	int sc, sr, st;
        double value[16];
        char parm[256];
        const char *arg;

        sar_object_aircraft_struct *obj_aircraft_ptr = NULL;
        sar_object_ground_struct *obj_ground_ptr = NULL;
	sar_obj_rotor_struct *rotor_ptr = NULL;
	sar_obj_part_struct	*aileron_left_ptr = NULL,
				*aileron_right_ptr = NULL,
                                *rudder_top_ptr = NULL,
                                *rudder_bottom_ptr = NULL,
                                *elevator_ptr = NULL,
                                *cannard_ptr = NULL,
                                *aileron_elevator_left_ptr = NULL,
                                *aileron_elevator_right_ptr = NULL,
				*flap_ptr = NULL,
				*abrake_ptr = NULL,
				*door_ptr = NULL,
				*lgear_ptr = NULL;
	sar_external_fueltank_struct *eft_ptr = NULL;

	sar_position_struct *pos = &obj_ptr->pos;
	sar_direction_struct *dir = &obj_ptr->dir;
	sar_scene_struct *scene = core_ptr->scene;
/*
	sar_object_struct ***ptr = &core_ptr->object;
	int *total = &core_ptr->total_objects;
 */
	Boolean is_player = (scene->player_obj_ptr == obj_ptr) ? True : False;

	if(ISSTREMPTY(line))
	    return(-2);

        /* Seek past spaces */
        while(ISBLANK(*line))
            line++;

        /* Skip comments */
        if(ISCOMMENT(*line))
            return(-2);

	/* Get object type specific values */
	switch(obj_ptr->type)
	{
	  case SAR_OBJ_TYPE_AIRCRAFT:
	    obj_aircraft_ptr = SAR_OBJ_GET_AIRCRAFT(obj_ptr);
	    if(obj_aircraft_ptr != NULL)
            {
                int i = obj_aircraft_ptr->total_rotors;
                if(i > 0)
                    rotor_ptr = obj_aircraft_ptr->rotor[i - 1];
            }
	    break;

	  case SAR_OBJ_TYPE_GROUND:
            obj_ground_ptr = SAR_OBJ_GET_GROUND(obj_ptr);
	    break;
	}
	/* Get object parts */
	if(True)
	{
	    int i, part_type;
	    sar_obj_part_struct *part_ptr;

#define DO_GET_PART	{				\
 sar_obj_part_struct *p;				\
 i = 0; part_ptr = NULL;				\
 while(1) {						\
  p = SARObjGetPartPtr(obj_ptr, part_type, i);		\
  if(p != NULL)						\
  { part_ptr = p; i++; }				\
  else							\
  { break; }						\
} }
            part_type = SAR_OBJ_PART_TYPE_AILERON_LEFT;
            DO_GET_PART
            aileron_left_ptr = part_ptr;

            part_type = SAR_OBJ_PART_TYPE_AILERON_RIGHT;
            DO_GET_PART
            aileron_right_ptr = part_ptr;

            part_type = SAR_OBJ_PART_TYPE_RUDDER_TOP;
            DO_GET_PART
            rudder_top_ptr = part_ptr;

            part_type = SAR_OBJ_PART_TYPE_RUDDER_BOTTOM;
            DO_GET_PART
            rudder_bottom_ptr = part_ptr;

            part_type = SAR_OBJ_PART_TYPE_ELEVATOR;
            DO_GET_PART
            elevator_ptr = part_ptr;

            part_type = SAR_OBJ_PART_TYPE_CANNARD;
            DO_GET_PART
            cannard_ptr = part_ptr;

            part_type = SAR_OBJ_PART_TYPE_AILERON_ELEVATOR_LEFT;
            DO_GET_PART
            aileron_elevator_left_ptr = part_ptr;

            part_type = SAR_OBJ_PART_TYPE_AILERON_ELEVATOR_RIGHT;
            DO_GET_PART
            aileron_elevator_right_ptr = part_ptr;

	    part_type = SAR_OBJ_PART_TYPE_FLAP;
	    DO_GET_PART
            flap_ptr = part_ptr;

	    part_type = SAR_OBJ_PART_TYPE_LANDING_GEAR;
	    DO_GET_PART
	    lgear_ptr = part_ptr;

	    part_type = SAR_OBJ_PART_TYPE_DOOR_RESCUE;
	    DO_GET_PART
	    door_ptr = part_ptr;

#undef DO_GET_PART
	}


	/* Begin parsing line */

	/* Get parameter */
	strncpy(parm, line, 256);
	parm[256 - 1] = '\0';
	s = parm;
	while(!ISBLANK(*s) && (*s != '\0'))
	    s++;
	*s = '\0';

	/* Seek to start of arguments in the line */
	arg = line;
	while(!ISBLANK(*arg) && (*arg != '\0'))
	    arg++;
	while(ISBLANK(*arg))
	    arg++;

	/* Begin handling parameter */
	if(True)
	{
	    /* Texture Base Directory */
            if(!strcasecmp(parm, "texture_base_directory") ||
	       !strcasecmp(parm, "texture_base_dir")
            )
            {
                /* Ignore since it is loaded from the V3D header */
	    }
            /* Texture Load */
            else if(!strcasecmp(parm, "texture_load"))
            {
		/* Ignore since it is loaded from the V3D header */
            }
	    /* Version */
	    else if(!strcasecmp(parm, "version"))
	    {
		int major, minor, release;
		const char *s = arg;
		major = ATOI(s);
		s = NEXT_ARG(s);
                minor = ATOI(s);
                s = NEXT_ARG(s);
                release = ATOI(s);

                if((major > PROG_VERSION_MAJOR) ||
                   (minor > PROG_VERSION_MINOR) ||
                   (release > PROG_VERSION_RELEASE)
                )
                {
                    int need_warn = 0;
                    if(major > PROG_VERSION_MAJOR)
                        need_warn = 1;
                    else if((major == PROG_VERSION_MAJOR) &&
                            (minor > PROG_VERSION_MINOR)
                    )
                        need_warn = 1;
                    else if((major == PROG_VERSION_MAJOR) &&
                            (minor == PROG_VERSION_MINOR) &&
                            (release == PROG_VERSION_RELEASE)
                    )
                        need_warn = 1;
                    if(need_warn)
                        fprintf(
                            stderr,
"%s: Warning: File format version %i.%i.%i is newer than program\
 version %i.%i.%i.",
                            filename,
                            major, minor,
                            release, PROG_VERSION_MAJOR,
                            PROG_VERSION_MINOR, PROG_VERSION_RELEASE
                        );
                }


	    }
            /* Name */
            else if(!strcasecmp(parm, "name"))
            {
		free(obj_ptr->name);
		obj_ptr->name = STRDUP(arg);
	    }
            /* Description */
            else if(!strcasecmp(parm, "desc") ||
		    !strcasecmp(parm, "description")
	    )
            {
		/* Ignore */
            }
            /* Type */
            else if(!strcasecmp(parm, "type"))
            {
		/* Ignore since the object type should already be set
		 * prior to calling this function
		 */
	    }
            /* Range */
            else if(!strcasecmp(parm, "range"))
            {
		st = 1;
                sr = sscanf(arg, "%lf", &value[0]);
		for(sc = sr; sc < st; sc++)
		    value[sc] = 0.0;

		obj_ptr->range = (float)value[0];
		if(value[0] < 0.0f)
		    fprintf(stderr,
 "%s: Line %i: Warning: range should not be negative.\n",
                        filename, line_num
                    );
	    }
            /* Range Far */
            else if(!strcasecmp(parm, "range_far"))
            {
                st = 1;
                sr = sscanf(arg, "%lf", &value[0]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                obj_ptr->range_far = (float)value[0];
                if(value[0] < 0.0f)
                    fprintf(stderr,
 "%s: Line %i: Warning: range_far should not be negative.\n",
                        filename, line_num
                    );
            }
            /* No Depth Test */
            else if(!strcasecmp(parm, "no_depth_test"))
	    {
		obj_ptr->flags |= SAR_OBJ_FLAG_NO_DEPTH_TEST;
	    }
	    /* Smooth Shading */
	    else if(!strcasecmp(parm, "shade_model_smooth"))
	    {
		obj_ptr->flags |= SAR_OBJ_FLAG_SHADE_MODEL_SMOOTH;
	    }
            /* Flat Shading */
            else if(!strcasecmp(parm, "shade_model_flat"))
            {
		obj_ptr->flags &= ~SAR_OBJ_FLAG_SHADE_MODEL_SMOOTH;
            }
            /* Offset Polygons */
            else if(!strcasecmp(parm, "offset_polygons"))
            {
                obj_ptr->flags |= SAR_OBJ_FLAG_POLYGON_OFFSET;
            }
	    /* Show Night Model At Dawn */
	    else if(!strcasecmp(parm, "show_night_model_at_dawn"))
            {
		obj_ptr->flags |= SAR_OBJ_FLAG_NIGHT_MODEL_AT_DAWN;
	    }
            /* Show Night Model At Dusk */
            else if(!strcasecmp(parm, "show_night_model_at_dusk"))
            {
                obj_ptr->flags |= SAR_OBJ_FLAG_NIGHT_MODEL_AT_DUSK;
            }
	    /* Show Far Model Day Only */
	    else if(!strcasecmp(parm, "show_far_day_only"))
            {
                obj_ptr->flags |= SAR_OBJ_FLAG_FAR_MODEL_DAY_ONLY;
            }
            /* Crash Flags */
            else if(!strcasecmp(parm, "crash_flags"))
            {
                /* Arguments are: <crash_into_other> <causes_crash>
                 * <support_surface> <crash_type>
                 */
		sar_contact_bounds_struct *cb;
                st = 4;
                sr = sscanf(arg, "%lf %lf %lf %lf",
		    &value[0], &value[1], &value[2], &value[3]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

		/* Create contact bounds as needed */
		if(obj_ptr->contact_bounds == NULL)
		    obj_ptr->contact_bounds = SAR_CONTACT_BOUNDS(
			calloc(1, sizeof(sar_contact_bounds_struct))
		    );
		cb = obj_ptr->contact_bounds;
		if(cb != NULL)
		{
		    cb->crash_flags = 0;	/* Reset crash flags. */

		    /* Crash into other objects? */
		    if((int)value[0])
			cb->crash_flags |= SAR_CRASH_FLAG_CRASH_OTHER;

		    /* Other objects can crash into this object? */
		    if((int)value[1])
			cb->crash_flags |= SAR_CRASH_FLAG_CRASH_CAUSE;

                    /* Other objects can land or walk on top of object? */
                    if((int)value[2])
                        cb->crash_flags |= SAR_CRASH_FLAG_SUPPORT_SURFACE;

		    /* Crash type code */
		    cb->crash_type = (int)value[3];
		}
            }
            /* Contact Bounds Spherical */
            else if(!strcasecmp(parm, "contact_spherical"))
            {
                /* Arguments are: <radius>
                 */
                sar_contact_bounds_struct *cb = obj_ptr->contact_bounds;
                st = 1;
                sr = sscanf(arg, "%lf", &value[0]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                SARObjAddContactBoundsSpherical(
                    obj_ptr,
                    (cb != NULL) ? cb->crash_flags : 0,
                    (cb != NULL) ? cb->crash_type : 0,
                    (float)value[0]		/* Radius. */
                );
	    }
            /* Contact Bounds Cylendical */
            else if(!strcasecmp(parm, "contact_cylendrical"))
            {
                /* Arguments are: <radius> <height_min> <height_max>
                 */
                sar_contact_bounds_struct *cb = obj_ptr->contact_bounds;
                st = 3; 
                sr = sscanf(arg, "%lf %lf %lf",
                    &value[0], &value[1], &value[2]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0; 

		SARObjAddContactBoundsCylendrical(
		    obj_ptr,
		    (cb != NULL) ? cb->crash_flags : 0,
		    (cb != NULL) ? cb->crash_type : 0,
		    (float)value[0],			/* Radius. */
		    (float)value[1], (float)value[2]	/* Height min/max. */
		);
            }
            /* Contact Bounds Rectangular */
            else if(!strcasecmp(parm, "contact_rectangular"))
            {
                /* Arguments are: <x_min> <x_max> <y_min> <y_max>
		 * <z_min> <z_max>
                 */
                sar_contact_bounds_struct *cb = obj_ptr->contact_bounds;
                st = 6;
                sr = sscanf(arg, "%lf %lf %lf %lf %lf %lf",
                    &value[0], &value[1], &value[2],
		    &value[3], &value[4], &value[5]
		);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                SARObjAddContactBoundsRectangular(
                    obj_ptr,
                    (cb != NULL) ? cb->crash_flags : 0,
                    (cb != NULL) ? cb->crash_type : 0,
		    (float)value[0], (float)value[1],		/* X min/max. */
		    (float)value[2], (float)value[3],		/* Y min/max. */
		    (float)value[4], (float)value[5]		/* Z min/max. */
                );
            }
            /* Speed */
            else if(!strcasecmp(parm, "speed"))
            {
                /* Arguments are: <speed_stall> <speed_max>
                 * <min_drag> <overspeed_expected> <overspeed>
		 * All units from miles per hour.
                 */
                st = 5;
                sr = sscanf(arg, "%lf %lf %lf %lf %lf",
                    &value[0], &value[1], &value[2], &value[3], &value[4]
		);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                {
		    obj_aircraft_ptr->speed_stall =
			(float)SFMMPHToMPC(value[0]);
		    obj_aircraft_ptr->speed_max =
			(float)SFMMPHToMPC(value[1]);
		    obj_aircraft_ptr->min_drag =
			(float)SFMMPHToMPC(value[2]);
		    obj_aircraft_ptr->overspeed_expected =
			(float)SFMMPHToMPC(value[3]);
		    obj_aircraft_ptr->overspeed =
			(float)SFMMPHToMPC(value[4]);
		}
            }
	    /* Air Brakes */
            else if(!strcasecmp(parm, "air_brakes"))
            {
                /* Arguments are: <rate>
                 * All units from miles per hour.
                 */
                st = 1;
                sr = sscanf(arg, "%lf",
                    &value[0]
		);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                {
                    obj_aircraft_ptr->air_brakes_rate =
                        (float)SFMMPHToMPC(value[0]);
		    if(obj_aircraft_ptr->air_brakes_rate <= 0.0f)
			obj_aircraft_ptr->air_brakes_state = -1;
		    else
			obj_aircraft_ptr->air_brakes_state = 0;
                }
            }

            /* Helicopter Acceleration Responsiveness */
            else if(!strcasecmp(parm, "helicopter_accelresp") ||
                    !strcasecmp(parm, "helicopter_acceleration_responsiveness")
            )
            {
                /* Arguments are: <i> <j> <k>
                 */
                sar_position_struct *ar = NULL;

                st = 3;
                sr = sscanf(arg, "%lf %lf %lf",
                    &value[0], &value[1], &value[2]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

		if(obj_aircraft_ptr != NULL)
		    ar = &obj_aircraft_ptr->accel_responsiveness;

		if(ar != NULL)
                {
                    ar->x = (float)value[0];
                    ar->y = (float)value[1];
                    ar->z = (float)value[2];
                }
            }
            /* Airplane Acceleration Responsiveness */
            else if(!strcasecmp(parm, "airplane_accelresp") ||
                    !strcasecmp(parm, "airplane_acceleration_responsiveness")
            )
            {
                /* Arguments are: <i> <j> <k>
                 */
                sar_position_struct *ar = NULL;

                st = 3;
                sr = sscanf(arg, "%lf %lf %lf",
                    &value[0], &value[1], &value[2]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

		if(obj_aircraft_ptr != NULL)
                    ar = &obj_aircraft_ptr->airplane_accel_responsiveness;

		if(ar != NULL)
                {
                    ar->x = (float)value[0];
                    ar->y = (float)value[1];
                    ar->z = (float)value[2];
                }
            }
            /* Cockpit Offset */
            else if(!strcasecmp(parm, "cockpit_offset"))
            {
                /* Arguments are: <x> <y> <z>
                 */
                st = 3;
                sr = sscanf(arg, "%lf %lf %lf",
                    &value[0], &value[1], &value[2]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                {
                    obj_aircraft_ptr->cockpit_offset_pos.x = (float)value[0];
                    obj_aircraft_ptr->cockpit_offset_pos.y = (float)value[1];
                    obj_aircraft_ptr->cockpit_offset_pos.z = (float)value[2];
                }
            }
	    /* Control Panel */
	    else if(!strcasecmp(parm, "control_panel") &&
		    is_player
	    )
            {
		/* <x> <y> <z>
		 * <heading> <pitch> <bank>
		 * <width> <height>
		 * <path>
		 *
		 * Note that position and size units are in centimeters.
		 * The <path> specifies the control panel directory (not
		 * the instruments file).
		 */
		float x, y, z, heading, pitch, bank, width, height;
		const char *path;

		const char *cstrptr = arg;
		x = ATOF(cstrptr);
		cstrptr = NEXT_ARG(cstrptr);
                y = ATOF(cstrptr);
                cstrptr = NEXT_ARG(cstrptr);
                z = ATOF(cstrptr);

                cstrptr = NEXT_ARG(cstrptr);
                heading = (float)DEGTORAD(ATOF(cstrptr));
                cstrptr = NEXT_ARG(cstrptr);
                pitch = (float)DEGTORAD(ATOF(cstrptr));
                cstrptr = NEXT_ARG(cstrptr);
                bank = (float)DEGTORAD(ATOF(cstrptr));
                cstrptr = NEXT_ARG(cstrptr);

                width = ATOF(cstrptr);
                cstrptr = NEXT_ARG(cstrptr);
                height = ATOF(cstrptr);
                cstrptr = NEXT_ARG(cstrptr);

		path = cstrptr;

		if(*path != '\0')
		{
		    ControlPanel *cp = CPNew(core_ptr->display);
		    CPDelete((ControlPanel *)scene->player_control_panel);
		    scene->player_control_panel = cp;
		    if(cp != NULL)
		    {
			CPLoadFromFile(cp, path);
			CPSetPosition(cp, x, y, z);
			CPSetDirection(cp, heading, pitch, bank);
			CPSetSize(cp, width, height);
		    }
		}
	    }
            /* Belly Height */
            else if(!strcasecmp(parm, "belly_height"))
            {
                /* Arguments are: <height> */
                st = 1;
                sr = sscanf(arg, "%lf", &value[0]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                    obj_aircraft_ptr->belly_height = (float)value[0];
            }
            /* Landing Gear Height */
            else if(!strcasecmp(parm, "gear_height"))
            {
                /* Arguments are: <height> */
                st = 1;
                sr = sscanf(arg, "%lf", &value[0]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

		if(value[0] < 0.0f)
                    fprintf(stderr,
 "%s: Line %i: Warning: gear_height should not be negative.\n",
                        filename, line_num
                    );

                if(obj_aircraft_ptr != NULL)
                    obj_aircraft_ptr->gear_height = (float)value[0];
            }
            /* Ground Turning */
            else if(!strcasecmp(parm, "ground_turning"))
            {
                /* Arguments are: <turn_rad> <turn_vel_opt>
		 * <turn_vel_max>
		 * Units from miles per hour.
		 */
                st = 3;
                sr = sscanf(arg, "%lf %lf %lf",
                    &value[0], &value[1], &value[2]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                {
                    /* Turn radius in meters, distance from farthest
                     * non-turnable wheel to turnable wheel.
                     * Can be negative or 0.0 for no turning.
                     */
                    obj_aircraft_ptr->gturn_radius = (float)value[0];

                    /* Optimul ground turning velocity along
                     * aircraft's y axis in meters per cycle.
                     */
                    obj_aircraft_ptr->gturn_vel_opt =
                        (float)SFMMPHToMPC(value[1]);

                    /* Maximum ground turning velocity along
                     * aircraft's y axis in meters per cycle.
                     */
                    obj_aircraft_ptr->gturn_vel_max =
                        (float)SFMMPHToMPC(value[2]);
                }
            }
            /* Dry Mass */
            else if(!strcasecmp(parm, "dry_mass"))
            {
                /* Arguments are: <mass>
		 * Units from kg.
		 */
                st = 1;
                sr = sscanf(arg, "%lf", &value[0]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(value[0] < 0)
                    fprintf(stderr,
 "%s: Line %i: Warning: dry_mass value is negative.\n",
                        filename, line_num
                    );
                if(obj_aircraft_ptr != NULL)
                    obj_aircraft_ptr->dry_mass = (float)value[0];
            }
            /* Fuel */
            else if(!strcasecmp(parm, "fuel"))
            {
                /* Arguments are: <consumption_rate> <fuel_init> 
                 * <fuel_max>
                 * Units from kg per second, kg, and kg.
                 */
                st = 3;
                sr = sscanf(arg, "%lf %lf %lf",
                    &value[0], &value[1], &value[2]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                {
                    /* Convert fuel rate from (kg / sec) to (kg / cycle). */
                    obj_aircraft_ptr->fuel_rate = (float)MAX(
			value[0] * SAR_SEC_TO_CYCLE_COEFF, 0.0
		    );

                    /* Fuel in kg. */
                    obj_aircraft_ptr->fuel = (float)MAX(value[1], 0.0);
                    obj_aircraft_ptr->fuel_max = (float)MAX(value[2], 0.0);
                }
            }
            /* Crew */
            else if(!strcasecmp(parm, "crew"))
            {
                /* Arguments are: <crew> <passengers>
                 * <passengers_max>
                 */
                st = 3;
                sr = sscanf(arg, "%lf %lf %lf",
                    &value[0], &value[1], &value[2]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                {
                    obj_aircraft_ptr->crew = (int)value[0];
                    obj_aircraft_ptr->passengers = (int)value[1];
                    obj_aircraft_ptr->passengers_max = (int)value[2];
                }
            }
            /* Engine */
            else if(!strcasecmp(parm, "engine"))
            {
                /* Arguments are: <can_pitch> <init_pitch>
                 * <power> <collective_range>
		 * Power from units of kg * m / cycle^2.
                 */
                st = 4;
                sr = sscanf(arg, "%lf %lf %lf %lf",
                    &value[0], &value[1], &value[2], &value[3]
		);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                {
                    /* Can pitch? */
                    obj_aircraft_ptr->engine_can_pitch = (((int)value[0]) ?
			1 : 0
		    );

                    /* Initial rotor pitch state, this determines the
		     * the value for member flight_model_type.
		     */
		    switch((int)value[1])
		    {
		      case 1:
			obj_aircraft_ptr->flight_model_type =
			    SAR_FLIGHT_MODEL_AIRPLANE;
			break;

		      /* All else assume helicopter. */
		      default:
			obj_aircraft_ptr->flight_model_type =
                            SAR_FLIGHT_MODEL_HELICOPTER;
                        break;
		    }
		    /* Explicitly set previous flight model type to the
		     * same type as the current one since this is the
		     * first time this is being set for this object.
		     */
		    obj_aircraft_ptr->last_flight_model_type =
			obj_aircraft_ptr->flight_model_type;

                    /* Engine power (in kg * m / cycle^2). */
                    obj_aircraft_ptr->engine_power = (float)value[2];

		    /* Collective range coefficient (from 0.0 to 1.0). */
                    obj_aircraft_ptr->collective_range = (float)value[3];
                }
            }
            /* Service Ceiling */  
            else if(!strcasecmp(parm, "service_ceiling"))
            {
                /* Arguments are: <altitude>
                 * Units from feet.
                 */
                st = 1;
                sr = sscanf(arg, "%lf", &value[0]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                    obj_aircraft_ptr->service_ceiling =
                        (float)SFMFeetToMeters(value[0]);
            }
            /* Attitude Change Rates */
            else if(!strcasecmp(parm, "attitude_change_rate"))
            {
                /* Arguments are: <heading> <pitch> <bank>
                 * Units from degrees per second.
		 */
		sar_direction_struct *acr = NULL;
                st = 3;
                sr = sscanf(arg, "%lf %lf %lf",
                    &value[0], &value[1], &value[2]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                    acr = &obj_aircraft_ptr->attitude_change_rate;

                if(acr != NULL)
                {
                    /* Do not call SFMDegreesToRadians() since we do not
                     * want sanitizing, values are from degrees per
                     * second.
                     */
                    acr->heading = (float)(DEGTORAD(value[0]) *
                        SAR_SEC_TO_CYCLE_COEFF);
                    acr->pitch = (float)(DEGTORAD(value[1]) *   
                        SAR_SEC_TO_CYCLE_COEFF);
                    acr->bank = (float)(DEGTORAD(value[2]) *   
                        SAR_SEC_TO_CYCLE_COEFF);
                }
            }
            /* Attitude Leveling */
            else if(!strcasecmp(parm, "attitude_leveling"))
            {
                /* Arguments are: <heading> <pitch> <bank>
                 * Units from degrees per second.
                 */
                st = 3;
                sr = sscanf(arg, "%lf %lf %lf",
                    &value[0], &value[1], &value[2]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                {
                    /* Do not call SFMDegreesToRadians() since we do not
                     * want sanitizing, values are from degrees per
                     * second
                     */
                    obj_aircraft_ptr->pitch_leveling = (float)(
                        DEGTORAD(value[1]) * SAR_SEC_TO_CYCLE_COEFF);
                    obj_aircraft_ptr->bank_leveling = (float)(
                        DEGTORAD(value[2]) * SAR_SEC_TO_CYCLE_COEFF);
                }
            }
	    /* Ground Pitch Offset */
	    else if(!strcasecmp(parm, "ground_pitch_offset"))
            {
                /* Arguments are: <ground_pitch_offset>
                 */
                st = 1;
                sr = sscanf(arg, "%lf",
                    &value[0]
		);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                {
                    /* Do not call SFMDegreesToRadians() since we do not
                     * want sanitizing, values are in degrees
                     */
                    obj_aircraft_ptr->ground_pitch_offset = (float)(
			DEGTORAD(value[0])
		    );
		}
	    }
            /* New Light */
            else if(!strcasecmp(parm, "light_new"))
            {
                /* Arguments are: <x> <y> <z> <r> <g> <b> <a>
		 * <radius> <init_on?> <type> <on_int> <off_int>
		 * <delay_on_int>
                 * Units from radius are in pixels and intervals are
		 * in milliseconds.
                 */
                sar_light_struct *light;
                st = 13;
                sr = sscanf(arg,
 "%lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf",
                    &value[0], &value[1], &value[2],
                    &value[3], &value[4], &value[5],
                    &value[6], &value[7], &value[8],
                    &value[9], &value[10], &value[11],
		    &value[12]
		);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

		light = SARObjCreateLight(
		    scene,
		    &obj_ptr->light, &obj_ptr->total_lights
		);
		if(light != NULL)
		{
                    /* Position (values 0 to 2). */
                    light->pos.x = (float)value[0];
                    light->pos.y = (float)value[1];
                    light->pos.z = (float)value[2];

                    /* Color (values 3 to 6). */
                    light->color.r = (float)value[3];
                    light->color.g = (float)value[4];
                    light->color.b = (float)value[5];
                    light->color.a = (float)value[6];

                    /* Radius (in pixels, value 7). */
                    light->radius = (int)MAX(value[7], 0.0);

                    /* Initially on (value 8)? */
                    if((int)value[8])  
                        light->flags |= SAR_LIGHT_FLAG_ON;

                    /* Type of light (value 9). */
                    switch((int)value[9])
                    {
                      case 2:	/* Attenuate/spot light. */
                        light->flags |= SAR_LIGHT_FLAG_ATTENUATE;
                        break;

                      case 1:	/* Strobe. */
                        light->flags |= SAR_LIGHT_FLAG_STROBE;
                        break;

		      /* All else assume regular light. */
                    }

                    /* Strobe on and off intervals (values 10 to 11). */
                    light->int_on = (time_t)value[10];
                    light->int_off = (time_t)value[11];
		    light->int_delay_on = (time_t)value[12];
                }
	    }
	    /* New Sound Source */
	    else if(!strcasecmp(parm, "sound_source_new"))
	    {
		SARObjLoadSoundSource(
		    scene,
		    &obj_ptr->sndsrc, &obj_ptr->total_sndsrcs,
		    arg
		);
	    }
            /* New Rotor */
            else if(!strcasecmp(parm, "rotor_new") ||
                    !strcasecmp(parm, "propellar_new")
            )
            {
                /* Arguments are: <x> <y> <z>
		 * <heading> <pitch> <bank> <radius> <has_prop_wash?>
		 * <follow_control_pitch_bank?> <blurs_when_fast?>
		 * <can_pitch?> <no_pitch_landed?> <no_rotate?>
		 * <blades_offset>
                 */
                st = 14;
                sr = sscanf(arg,
 "%lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf",
                    &value[0], &value[1], &value[2],
                    &value[3], &value[4], &value[5],
                    &value[6], &value[7], &value[8],
		    &value[9], &value[10], &value[11],
		    &value[12], &value[13]
		);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

		rotor_ptr = NULL;

                if(obj_aircraft_ptr != NULL)
                {
                    int n = SARObjCreateRotor(
                        scene,
                        &obj_aircraft_ptr->rotor,
                        &obj_aircraft_ptr->total_rotors
                    );
                    rotor_ptr = ((n < 0) ? NULL :
                        obj_aircraft_ptr->rotor[n]
                    );
                }
                if(rotor_ptr == NULL)
                {
                    fprintf(stderr,
 "%s: Line %i: Error: Cannot create rotor/propellar (check object type).\n",
                        filename, line_num
                    );
                }
                else
		{
		    pos = &rotor_ptr->pos;
                    dir = &rotor_ptr->dir;      

                    pos->x = (float)value[0];
                    pos->y = (float)value[1];
                    pos->z = (float)value[2];

                    dir->heading = (float)SFMDegreesToRadians(value[3]);
                    dir->pitch = (float)SFMDegreesToRadians(value[4]);
                    dir->bank = (float)SFMDegreesToRadians(value[5]);

                    rotor_ptr->radius = (float)MAX(value[6], 0.0);

                    /* Has rotor wash? */
                    if((int)value[7])
                    {
                        rotor_ptr->rotor_wash_tex_num =
                            SARGetTextureRefNumberByName(
                                scene,
                                SAR_STD_TEXNAME_ROTOR_WASH
                            );
                    } 
                    else
                    {
                        rotor_ptr->rotor_wash_tex_num = -1;
                    }

		    /* Follow controls for pitch and bank? */
		    if((int)value[8])
		    {
			rotor_ptr->flags |= SAR_ROTOR_FLAG_FOLLOW_PB;
		    }

		    /* Blurs when fast? */
                    if((int)value[9] == 1)
                    {
                        rotor_ptr->flags |= SAR_ROTOR_FLAG_BLADES_BLUR_FAST;
                    }
		    /* Blurs always? */
                    else if((int)value[9] == 2)
                    {
                        rotor_ptr->flags |= SAR_ROTOR_FLAG_BLADES_BLUR_ALWAYS;
                    }

                    /* Can pitch? */
                    if((int)value[10] == 1)
                    {
                        rotor_ptr->flags |= SAR_ROTOR_FLAG_CAN_PITCH;
                    }

		    /* No pitching of rotors when landed? */
		    if((int)value[11] == 1)
                    {
                        rotor_ptr->flags |= SAR_ROTOR_FLAG_NO_PITCH_LANDED;
                    }

                    /* No rotate? */
                    if((int)value[12] == 1)
                    {
                        rotor_ptr->flags |= SAR_ROTOR_FLAG_NO_ROTATE;
                    }

                    rotor_ptr->blades_offset = (float)MAX(value[13], 0.0);
                }
	    }
            /* Rotor Blur Color */
            else if(!strcasecmp(parm, "rotor_blur_color") ||
                    !strcasecmp(parm, "propellar_blur_color")
            )
            {
                /* Arguments are: <r> <b> <g> <a> */
                st = 4;
                sr = sscanf(arg,
		    "%lf %lf %lf %lf",
		    &value[0], &value[1], &value[2], &value[3]
                );
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

		if(rotor_ptr != NULL)
		{
		    if(!(rotor_ptr->flags & SAR_ROTOR_FLAG_BLADES_BLUR_FAST) &&
		       !(rotor_ptr->flags & SAR_ROTOR_FLAG_BLADES_BLUR_ALWAYS)
		    )
		    {
			fprintf(
			    stderr,
 "%s: Line %i: Warning: Rotor is not set to blur at all, blur color not set.\n",
			    filename, line_num
			);
		    }
		    else
                    {
			sar_color_struct *c = &rotor_ptr->blades_blur_color;

			c->r = (float)CLIP(value[0], 0.0, 1.0);
                        c->g = (float)CLIP(value[1], 0.0, 1.0); 
                        c->b = (float)CLIP(value[2], 0.0, 1.0); 
                        c->a = (float)CLIP(value[3], 0.0, 1.0); 
                    }
		}
	    }
	    /* New Air Brake */
	    else if(!strcasecmp(parm, "air_brake_new"))
            {
		/* Arguments are:
		 * <x> <y> <z>
		 * <heading> <pitch> <bank>
		 * <dep_dheading> <dep_dpitch> <dep_dbank>
		 * <deployed?> <anim_rate>
		 * <visible_when_retracted?>
                 */
                st = 12;
                sr = sscanf(arg,
 "%lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf",
                    &value[0], &value[1], &value[2],
                    &value[3], &value[4], &value[5],
                    &value[6], &value[7], &value[8],
                    &value[9], &value[10], &value[11]
                );
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                abrake_ptr = NULL;

                if(obj_aircraft_ptr != NULL)
                {
                    abrake_ptr = SARObjCreateAirBrake(
                        scene,
                        &obj_aircraft_ptr->part,
                        &obj_aircraft_ptr->total_parts
                    );
                }
                /* Was an air brake created and/or found? */
                if(abrake_ptr == NULL)
                {
                    fprintf(stderr,
 "%s: Line %i: Error: Cannot create air brake (check object type).\n",
                        filename, line_num
                    );
                }
                else
                {
                    pos = &abrake_ptr->pos_min;
                    pos->x = (float)value[0];
                    pos->y = (float)value[1];
                    pos->z = (float)value[2];

                    pos = &abrake_ptr->pos_cen;
                    pos->x = (float)value[0];
                    pos->y = (float)value[1];
                    pos->z = (float)value[2];

                    pos = &abrake_ptr->pos_max;
                    pos->x = (float)value[0];
                    pos->y = (float)value[1];
                    pos->z = (float)value[2];

                    /* Do not call SFMDegreesToRadians() when setting
		     * directions since we may want delta values instead
		     * of the sanitized direction.
                     */

		    dir = &abrake_ptr->dir_min;
                    dir->heading = (float)DEGTORAD(value[3]);
                    dir->pitch = (float)DEGTORAD(value[4]);
                    dir->bank = (float)DEGTORAD(value[5]);

                    dir = &abrake_ptr->dir_cen;
                    dir->heading = (float)DEGTORAD(value[3]);
                    dir->pitch = (float)DEGTORAD(value[4]);
                    dir->bank = (float)DEGTORAD(value[5]);

		    dir = &abrake_ptr->dir_max;
                    dir->heading = (float)DEGTORAD(value[6]);
                    dir->pitch = (float)DEGTORAD(value[7]);
                    dir->bank = (float)DEGTORAD(value[8]);

                    abrake_ptr->anim_rate = (sar_grad_anim_t)value[10];
                    abrake_ptr->flags = 0;

                    /* Air brake initially deployed? */
                    if((int)value[9])
                    {
                        /* Air brake initially deployed. */
                        abrake_ptr->anim_pos = (sar_grad_anim_t)-1;
                        abrake_ptr->flags |= SAR_OBJ_PART_FLAG_STATE;
                    }
                    else
                    {
                        /* Air brake initially retracted. */
                        abrake_ptr->anim_pos = 0;
                        abrake_ptr->flags &= ~SAR_OBJ_PART_FLAG_STATE;
                    }

		    /* Air brake visible when retracted (hide air brakes
		     * when retracted)?
		     */
		    if(!(int)value[11])
			abrake_ptr->flags |= SAR_OBJ_PART_FLAG_HIDE_MIN;
                }
            }
            /* New Landing Gear */
            else if(!strcasecmp(parm, "landing_gear_new"))
            {
                /* Arguments are: <x> <y> <z>
                 * <heading> <pitch> <bank> <anim_rate> <up?>
		 * <fixed?> <ski?> <floats?>
                 */
                st = 11;
                sr = sscanf(arg,
 "%lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf",
                    &value[0], &value[1], &value[2],
                    &value[3], &value[4], &value[5],
                    &value[6], &value[7], &value[8],
		    &value[9], &value[10]
                );
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                    lgear_ptr = SARObjCreateLandingGear(
                        scene,
                        &obj_aircraft_ptr->part,
                        &obj_aircraft_ptr->total_parts
                    );
		else
		    lgear_ptr = NULL;
                if(lgear_ptr == NULL)
                {
                    fprintf(
			stderr,
 "%s: Line %i: Error: Unable to create landing gear (check object type).\n",
                        filename, line_num
                    );
                }
                else
                {
                    pos = &lgear_ptr->pos_min;
                    pos->x = (float)value[0];
                    pos->y = (float)value[1];
                    pos->z = (float)value[2];

                    pos = &lgear_ptr->pos_cen;
                    pos->x = (float)value[0];
                    pos->y = (float)value[1];
                    pos->z = (float)value[2];

                    pos = &lgear_ptr->pos_max;
                    pos->x = (float)value[0];
                    pos->y = (float)value[1];
                    pos->z = (float)value[2];

                    /* Do not call SFMDegreesToRadians() when setting
                     * directions since we may want delta values instead
                     * of the sanitized direction.
                     */

                    dir = &lgear_ptr->dir_min;
                    dir->heading = (float)(0.0 * PI);
                    dir->pitch = (float)(0.0 * PI);
                    dir->bank = (float)(0.0 * PI);

                    dir = &lgear_ptr->dir_cen;
                    dir->heading = (float)(0.0 * PI);
                    dir->pitch = (float)(0.0 * PI);
                    dir->bank = (float)(0.0 * PI);

                    dir = &lgear_ptr->dir_max;
                    dir->heading = (float)DEGTORAD(value[3]);
                    dir->pitch = (float)DEGTORAD(value[4]);
                    dir->bank = (float)DEGTORAD(value[5]);

                    lgear_ptr->anim_rate = (sar_grad_anim_t)value[6];
                    lgear_ptr->flags = 0;

                    /* Landing gear up or down? */
                    if((int)value[7])
                    {
                        /* Landing gear initially up. */
                        lgear_ptr->anim_pos = (sar_grad_anim_t)-1;
                        lgear_ptr->flags &= ~SAR_OBJ_PART_FLAG_STATE;
                    }
		    else
                    {
                        /* Landing gear initially down. */
                        lgear_ptr->anim_pos = 0;
                        lgear_ptr->flags |= SAR_OBJ_PART_FLAG_STATE;
                    }
                    /* Landing gear fixed? */
                    if((int)value[8])
                    {
                        /* Fixed */
                        lgear_ptr->anim_pos = 0;
                        lgear_ptr->flags |= SAR_OBJ_PART_FLAG_STATE;
                        lgear_ptr->flags |= SAR_OBJ_PART_FLAG_LGEAR_FIXED;
                    }
                    else
                    {
                        /* Retractable */
                        lgear_ptr->flags &= ~SAR_OBJ_PART_FLAG_LGEAR_FIXED;
                    }  
                    /* Ski? */
                    if((int)value[9])
                        lgear_ptr->flags |= SAR_OBJ_PART_FLAG_LGEAR_SKI;
                    else
                        lgear_ptr->flags &= ~SAR_OBJ_PART_FLAG_LGEAR_SKI;

		    /* Floats? */
                    if((int)value[10])
                        lgear_ptr->flags |= SAR_OBJ_PART_FLAG_LGEAR_FLOATS;
                    else
                        lgear_ptr->flags &= ~SAR_OBJ_PART_FLAG_LGEAR_FLOATS;

		    /* Landing gears are always hidden when retracted,
		     * note that maximum position is considered retracted
		     */
		    lgear_ptr->flags |= SAR_OBJ_PART_FLAG_HIDE_MAX;

		    /* Need to update landing gear state on aircraft */
		    if(lgear_ptr->flags & SAR_OBJ_PART_FLAG_LGEAR_FIXED)
			obj_aircraft_ptr->landing_gear_state = 2;
		    else
			obj_aircraft_ptr->landing_gear_state = 1;
		    if(!(lgear_ptr->flags & SAR_OBJ_PART_FLAG_LGEAR_SKI))
			obj_aircraft_ptr->wheel_brakes_state = 0;
                }
	    }
	    /* Aileron left, aileron right, rudder top, rudder bottom,
	     * elevator, cannard, aileron/elevator left, or
	     * aileron/elevator right.
	     */
            else if(!strcasecmp(parm, "aileron_left_new") ||
	            !strcasecmp(parm, "aileron_right_new") ||
                    !strcasecmp(parm, "rudder_top_new") ||
                    !strcasecmp(parm, "rudder_bottom_new") ||
                    !strcasecmp(parm, "elevator_new") ||
                    !strcasecmp(parm, "cannard_new") ||
                    !strcasecmp(parm, "aileron_elevator_left_new") ||
                    !strcasecmp(parm, "aileron_elevator_right_new") ||
		    !strcasecmp(parm, "flap_new")
	    )
            {
		int part_type = -1;
		sar_obj_part_struct **part_ptr = NULL;

                /* Arguments are:
		 *
		 * <x> <y> <z> <h> <p> <b> <t_min> <t_max>
                 */
                st = 8;
                sr = sscanf(arg,
 "%lf %lf %lf %lf %lf %lf %lf %lf",
                    &value[0], &value[1], &value[2],
                    &value[3], &value[4], &value[5],
		    &value[6], &value[7]
                );
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

		/* Determine the part type and get part pointer */
		if(!strcasecmp(parm, "aileron_left_new"))
		{
		    part_type = SAR_OBJ_PART_TYPE_AILERON_LEFT;
		    part_ptr = &aileron_left_ptr;
		}
		else if(!strcasecmp(parm, "aileron_right_new"))
                {
                    part_type = SAR_OBJ_PART_TYPE_AILERON_RIGHT;
		    part_ptr = &aileron_right_ptr;
		}
                else if(!strcasecmp(parm, "rudder_top_new"))
                {
                    part_type = SAR_OBJ_PART_TYPE_RUDDER_TOP;
                    part_ptr = &rudder_top_ptr;
		}
                else if(!strcasecmp(parm, "rudder_bottom_new"))
                {
                    part_type = SAR_OBJ_PART_TYPE_RUDDER_BOTTOM;
                    part_ptr = &rudder_bottom_ptr;
		}
                else if(!strcasecmp(parm, "elevator_new"))
                {
                    part_type = SAR_OBJ_PART_TYPE_ELEVATOR;
                    part_ptr = &elevator_ptr;
		}
                else if(!strcasecmp(parm, "cannard_new"))
                {
                    part_type = SAR_OBJ_PART_TYPE_CANNARD;
                    part_ptr = &cannard_ptr;
		}
                else if(!strcasecmp(parm, "aileron_elevator_left_new"))
                {
                    part_type = SAR_OBJ_PART_TYPE_AILERON_ELEVATOR_LEFT;
                    part_ptr = &aileron_elevator_left_ptr;
		}
                else if(!strcasecmp(parm, "aileron_elevator_right_new"))
                {
                    part_type = SAR_OBJ_PART_TYPE_AILERON_ELEVATOR_RIGHT;
                    part_ptr = &aileron_elevator_right_ptr;
		}
                else if(!strcasecmp(parm, "flap_new"))
                {
                    part_type = SAR_OBJ_PART_TYPE_FLAP;
                    part_ptr = &flap_ptr;
                }

		/* Got valid part type? */
		if(part_ptr != NULL)
		{
		    /* Create new part */
		    if(obj_aircraft_ptr != NULL)
			*part_ptr = SARObjCreatePart(
			    scene,
			    &obj_aircraft_ptr->part,
			    &obj_aircraft_ptr->total_parts,
			    part_type
			);
		    if(*part_ptr != NULL)
		    {
			float	t_min = (float)value[6],
				t_max = (float)value[7];
			sar_direction_struct	*dir_min,
						*dir_max;

			/* Minimum position */
			pos = &(*part_ptr)->pos_min;
			pos->x = 0.0f;
			pos->y = 0.0f;
			pos->z = 0.0f;

			/* Center position */
                        pos = &(*part_ptr)->pos_cen;
                        pos->x = (float)value[0];
                        pos->y = (float)value[1];
                        pos->z = (float)value[2];

			/* Maximum position */
                        pos = &(*part_ptr)->pos_max;
                        pos->x = 0.0f;
                        pos->y = 0.0f;
                        pos->z = 0.0f;

			/* Do not call SFMDegreesToRadians() when setting
			 * directions since we may want delta values
			 * instead of the sanitized direction
			 */

			/* Minimum direction */
			dir_min = dir = &(*part_ptr)->dir_min;
                        dir->heading = (float)(0.0 * PI);
                        dir->pitch = (float)(0.0 * PI);
                        dir->bank = (float)(0.0 * PI);

			/* Center direction */
                        dir = &(*part_ptr)->dir_cen;
                        dir->heading = (float)DEGTORAD(value[3]);
                        dir->pitch = (float)DEGTORAD(value[4]);
                        dir->bank = (float)DEGTORAD(value[5]);

			/* Maximum direction */
                        dir_max = dir = &(*part_ptr)->dir_max;
                        dir->heading = (float)(0.0 * PI);
                        dir->pitch = (float)(0.0 * PI);
                        dir->bank = (float)(0.0 * PI);

			/* Set values specific to the part's type */
			switch(part_type)
			{
			  case SAR_OBJ_PART_TYPE_AILERON_LEFT:
                          case SAR_OBJ_PART_TYPE_AILERON_RIGHT:
                            dir_min->pitch += (float)DEGTORAD(t_min);
                            dir_max->pitch += (float)DEGTORAD(t_max);
			    break;
                          case SAR_OBJ_PART_TYPE_RUDDER_TOP:
			  case SAR_OBJ_PART_TYPE_RUDDER_BOTTOM:
                            dir_min->heading += (float)DEGTORAD(t_min);
                            dir_max->heading += (float)DEGTORAD(t_max);
                            break;
                          case SAR_OBJ_PART_TYPE_ELEVATOR:
                          case SAR_OBJ_PART_TYPE_CANNARD:
                            dir_min->pitch += (float)DEGTORAD(t_min);
                            dir_max->pitch += (float)DEGTORAD(t_max);
                            break;
			  case SAR_OBJ_PART_TYPE_AILERON_ELEVATOR_LEFT:
                          case SAR_OBJ_PART_TYPE_AILERON_ELEVATOR_RIGHT:
                            dir_min->pitch += (float)DEGTORAD(t_min);
                            dir_max->pitch += (float)DEGTORAD(t_max);
                            break;
			  case SAR_OBJ_PART_TYPE_FLAP:
                            dir_min->pitch += (float)DEGTORAD(t_min);
                            dir_max->pitch += (float)DEGTORAD(t_max);
                            break;
			}
		    }
		    else
		    {
                        fprintf(
                            stderr,
"%s: Line %i: Error: Unable to create part type %i (check object type).\n",
                            filename, line_num, part_type
                        );
                    }
		}
	    }

            /* New External Fuel Tank */
            else if(!strcasecmp(parm, "fueltank_new") ||
                    !strcasecmp(parm, "fuel_tank_new")
	    )
            {
                /* Arguments are: <x> <y> <z> <radius>
                 * <dry_mass> <fuel> <fuel_max> <droppable>
                 */
                st = 8;
                sr = sscanf(arg,
 "%lf %lf %lf %lf %lf %lf %lf %lf",
                    &value[0], &value[1], &value[2],
                    &value[3], &value[4], &value[5],
                    &value[6], &value[7]
                );
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                eft_ptr = NULL;

                if(obj_aircraft_ptr != NULL)
                {
                    int n = SARObjCreateExternalFuelTanks(
                        scene,
                        &obj_aircraft_ptr->external_fueltank,
                        &obj_aircraft_ptr->total_external_fueltanks
                    );
                    eft_ptr = ((n < 0) ? NULL :
                        obj_aircraft_ptr->external_fueltank[n]
                    );
		}
		if(eft_ptr == NULL)
		{
                    fprintf(
			stderr,
 "%s: Line %i: Error: Unable to create fuel tank (check object type).\n",
                        filename, line_num
                    );
                }
                else
                {
                    pos = &eft_ptr->offset_pos;

		    /* Position */
                    pos->x = (float)value[0];
                    pos->y = (float)value[1];
                    pos->z = (float)value[2];

		    /* Size */
		    eft_ptr->radius = (float)MAX(value[3], 0.0);

		    /* Mass and fuel */
		    eft_ptr->dry_mass = (float)MAX(value[4], 0.0);
                    eft_ptr->fuel_max = (float)MAX(value[6], 0.0);
                    eft_ptr->fuel = (float)CLIP(value[5], 0.0, eft_ptr->fuel_max);

		    /* Fixed? */
		    if((int)value[7])
			eft_ptr->flags |= SAR_EXTERNAL_FUELTANK_FLAG_FIXED;
		    else
                        eft_ptr->flags &= ~SAR_EXTERNAL_FUELTANK_FLAG_FIXED;

		    /* Mark as initially on board */
		    eft_ptr->flags |= SAR_EXTERNAL_FUELTANK_FLAG_ONBOARD;
		}
	    }
            /* Rescue Door */
            else if(!strcasecmp(parm, "rescue_door_new"))
            {
                /* Arguments are:
		 * <x_closed> <y_closed> <z_closed>
                 * <x_opened> <y_opened> <z_opened>
		 * <h_opened> <p_opened> <b_opened>
		 * <x_thres> <y_thres> <z_thres>
		 * <anim_rate> <opened?>
                 */
                st = 14;
                sr = sscanf(arg,
 "%lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf",
                    &value[0], &value[1], &value[2],
                    &value[3], &value[4], &value[5],
                    &value[6], &value[7], &value[8],
                    &value[9], &value[10], &value[11],
		    &value[12], &value[13]
                );
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                    door_ptr = SARObjCreateDoorRescue(
                        scene,
                        &obj_aircraft_ptr->part,
                        &obj_aircraft_ptr->total_parts
                    );
		else
		    door_ptr = NULL;
                if(door_ptr == NULL)
                {
                    fprintf(
			stderr,
 "%s: Line %i: Error: Unable to create rescue door (check object type).\n",
                        filename, line_num
                    );
                }
                else
                {
                    /* Closed position */
                    pos = &door_ptr->pos_min;
                    pos->x = (float)value[0];
                    pos->y = (float)value[1];  
                    pos->z = (float)value[2];

                    /* Opened position */
                    pos = &door_ptr->pos_max;
                    pos->x = (float)value[3];
                    pos->y = (float)value[4];
                    pos->z = (float)value[5];

                    /* Do not call SFMDegreesToRadians() since we do not
                     * want sanitizing
                     */
                    dir = &door_ptr->dir_max;
                    dir->heading = (float)DEGTORAD(value[6]);
                    dir->pitch = (float)DEGTORAD(value[7]);
                    dir->bank = (float)DEGTORAD(value[8]);

		    /* Threshold position */
                    pos = &door_ptr->pos_cen;
                    pos->x = (float)value[9];
                    pos->y = (float)value[10];
                    pos->z = (float)value[11];

		    /* Animation rate */
                    door_ptr->anim_rate = (sar_grad_anim_t)value[12];

		    /* Flags */
                    door_ptr->flags = 0;
                    /* Initially opened? */
                    if((int)value[13])
                    {
                        /* Door initially open */
                        door_ptr->anim_pos = (sar_grad_anim_t)-1;
                        door_ptr->flags |= SAR_OBJ_PART_FLAG_STATE;
			door_ptr->flags |= SAR_OBJ_PART_FLAG_DOOR_STAY_OPEN;
                    }
                    else
                    {
                        /* Door initially closed */
                        door_ptr->anim_pos = 0;
                        door_ptr->flags &= ~SAR_OBJ_PART_FLAG_STATE;
                    }
                }
            }
            /* Hoist */
            else if(!strcasecmp(parm, "hoist"))
            {
                /* Arguments are: <x> <y> <z>
                 * <rope_max> <rope_rate> <capacity>
		 * <radius> <z_min> <z_max>
                 */
                sar_obj_hoist_struct *hoist_ptr;
                st = 9;
                sr = sscanf(arg,
 "%lf %lf %lf %lf %lf %lf %lf %lf %lf",
                    &value[0], &value[1], &value[2],
                    &value[3], &value[4], &value[5],
                    &value[6], &value[7], &value[8]
                );  
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

		hoist_ptr = SARObjGetHoistPtr(obj_ptr, 0, NULL);
		if(hoist_ptr == NULL)
		{

		}
		else
		{
		    sar_human_data_entry_struct *hde_ptr;

		    /* Position */
                    hoist_ptr->offset.x = (float)value[0];
                    hoist_ptr->offset.y = (float)value[1];
                    hoist_ptr->offset.z = (float)value[2];

		    /* Rope */
                    hoist_ptr->rope_max = (float)value[3];
                    hoist_ptr->rope_rate = (float)value[4];

		    /* Capacity */
                    hoist_ptr->capacity = (float)value[5];

		    /* Contact size */
                    hoist_ptr->contact_radius = (float)value[6];
                    hoist_ptr->contact_z_min = (float)value[7];
                    hoist_ptr->contact_z_max = (float)value[8];

		    /* Deployment type */
		    hoist_ptr->deployment = SAR_HOIST_DEPLOYMENT_BASKET;

		    /* Get hoist texture reference numbers from scene */
                    hoist_ptr->side_tex_num =
                        SARGetTextureRefNumberByName(
			    scene, SAR_STD_TEXNAME_BASKET_SIDE
			);
                    hoist_ptr->end_tex_num =
                        SARGetTextureRefNumberByName(
			    scene, SAR_STD_TEXNAME_BASKET_END
			);
                    hoist_ptr->bottom_tex_num =
                        SARGetTextureRefNumberByName(
			    scene, SAR_STD_TEXNAME_BASKET_BOTTOM
			);
		    hoist_ptr->water_ripple_tex_num =
			SARGetTextureRefNumberByName(
			    scene, SAR_STD_TEXNAME_WATER_RIPPLE
			);

		    /* Diver color */
		    hde_ptr = SARHumanMatchEntryByName(
			core_ptr->human_data, SAR_HUMAN_PRESET_NAME_DIVER
		    );
		    if(hde_ptr == NULL)
			hde_ptr = SARHumanMatchEntryByName(
			    core_ptr->human_data, SAR_HUMAN_PRESET_NAME_STANDARD
			);
		    if(hde_ptr != NULL)
		    {
			memcpy(
			    hoist_ptr->diver_color,
			    hde_ptr->color,
			    SAR_HUMAN_COLORS_MAX * sizeof(sar_color_struct)
			);
		    }

		    /* Animation position */
		    hoist_ptr->anim_pos = 0;
		    hoist_ptr->anim_rate = SAR_HUMAN_ANIM_RATE;
/* TODO (using numan animation rate for now) */
		}
	    }
            /* Ground Elevation */
            else if(!strcasecmp(parm, "ground_elevation"))
            {
                /* Arguments are: <altitude>
		 * Units from feet.
                 */
                st = 1;
                sr = sscanf(arg, "%lf", &value[0]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_ground_ptr == NULL)
                {
                    fprintf(stderr, 
 "%s: Line %i: Warning: Setting ground_elevation for object not type `%i'.\n",
                        filename, line_num, SAR_OBJ_TYPE_GROUND
                    );
                }
                else
                {
                    if(value[0] < 0)
                        fprintf(stderr,
 "%s: Line %i: Warning: ground_elevation value is negative.\n",
                            filename, line_num
                        );

                    obj_ground_ptr->elevation =
                        (float)SFMFeetToMeters(value[0]);
                }
            }   




	    /* Unsupported Parameter */
	    else
	    {
		/* Ignore */
	    }
	}


	return(0);
}



/*
 *      Loads object model data from file. Object can be already allocated  
 *      if n is not -1. Otherwise a new object will be appended to the given
 *      objects pointer array.
 */
int SARObjLoadFromFile(
        sar_core_struct *core_ptr, int obj_num, const char *filename
)
{
        FILE *fp;
        const char *v3d_model_name, *line_ptr;
        int status, line_num, *total;

	int pn, ptype;
	void *p;

	void **v3d_h = NULL;
	void *h;
	int hn, htype, total_v3d_h = 0;

	v3d_model_struct *v3d_model_ptr, **v3d_model = NULL;
	int v3d_model_num, total_v3d_models = 0;

	gw_display_struct *display = core_ptr->display;
        sar_scene_struct *scene = core_ptr->scene;
        sar_object_struct ***ptr, *obj_ptr;
        sar_object_aircraft_struct *obj_aircraft_ptr = NULL;
        sar_object_ground_struct *obj_ground_ptr = NULL;

	int total_rotors = 0;
        sar_obj_rotor_struct *rotor_ptr = NULL;

        int total_aileron_lefts = 0;
        sar_obj_part_struct *aileron_left_ptr = NULL;
        int total_aileron_rights = 0;
        sar_obj_part_struct *aileron_right_ptr = NULL;
        int total_rudder_tops = 0;
        sar_obj_part_struct *rudder_top_ptr = NULL;
        int total_rudder_bottoms = 0;
        sar_obj_part_struct *rudder_bottom_ptr = NULL;
        int total_elevators = 0;
        sar_obj_part_struct *elevator_ptr = NULL;
        int total_cannards = 0;
        sar_obj_part_struct *cannard_ptr = NULL;
        int total_aileron_elevator_lefts = 0;
        sar_obj_part_struct *aileron_elevator_left_ptr = NULL;
        int total_aileron_elevator_rights = 0;
        sar_obj_part_struct *aileron_elevator_right_ptr = NULL;
        int total_flaps = 0;
        sar_obj_part_struct *flap_ptr = NULL;
	int total_abrakes = 0;
	sar_obj_part_struct *abrake_ptr = NULL;
	int total_lgears = 0;
        sar_obj_part_struct *lgear_ptr = NULL;
	int total_doors = 0;
        sar_obj_part_struct *door_ptr = NULL;
	int total_external_fueltanks = 0;
	sar_external_fueltank_struct *eft_ptr = NULL;

	sar_direction_struct *dir;
	sar_position_struct *pos;
	char *sar_vmodel_name;
	sar_visual_model_struct **sar_vmodel;	/* SAR visual model ptr */

	struct stat stat_buf;
	char tmp_path[PATH_MAX + NAME_MAX];


        if(ISSTREMPTY(filename) || (display == NULL) || (scene == NULL))
            return(-1);

        ptr = &core_ptr->object;
        total = &core_ptr->total_objects;

        /* Complete filename as needed */
        if(ISPATHABSOLUTE(filename))
        {
            strncpy(tmp_path, filename, PATH_MAX + NAME_MAX);
        }
        else
        {
            const char *s = PrefixPaths(dname.local_data, filename);
	    if((s != NULL) ? stat(s, &stat_buf) : True)
		s = PrefixPaths(dname.global_data, filename);
            if(s != NULL)
                strncpy(tmp_path, s, PATH_MAX + NAME_MAX);
        }
        tmp_path[PATH_MAX + NAME_MAX - 1] = '\0';
	filename = tmp_path;

	/* File does not exist or is a directory? */
        if(stat(filename, &stat_buf))
        {
            fprintf(
		stderr,
		"%s: No such file.\n",
                filename
            );
            return(-1);
        }
#ifdef S_ISDIR
	if(S_ISDIR(stat_buf.st_mode))
	{
            fprintf(
		stderr,
		"%s: Is a directory.\n",
                filename
            );
            return(-1);
        }
#endif	/* S_ISDIR */

	/* Open V3D model file */
        fp = FOpen(filename, "rb");
        if(fp == NULL)
            return(-1);

        /* Object not created yet? */
        if(obj_num < 0)
        {
            /* Create new object (the type will be set later) */
            obj_num = SARObjCreate(scene, ptr, total, SAR_OBJ_TYPE_STATIC);
        }
	/* Ensure object was created */
        if(SARObjIsAllocated(*ptr, *total, obj_num))
        {
            obj_ptr = (*ptr)[obj_num];

	    /* Set default values on the object */

	    /* Flags */
	    obj_ptr->flags |= (	SAR_OBJ_FLAG_HIDE_DAWN_MODEL |
				SAR_OBJ_FLAG_HIDE_DUSK_MODEL |
				SAR_OBJ_FLAG_HIDE_NIGHT_MODEL
	    );

            /* Get pointer to substructure and set default values
	     * depending on the object's type
	     */
	    /* Aircraft? */
	    obj_aircraft_ptr = SAR_OBJ_GET_AIRCRAFT(obj_ptr);
	    if(obj_aircraft_ptr != NULL)
	    {
                obj_aircraft_ptr->air_worthy_state = SAR_AIR_WORTHY_FLYABLE;
                obj_aircraft_ptr->engine_state = SAR_ENGINE_STATE_ON;
                obj_aircraft_ptr->next_engine_on = 0;

		obj_aircraft_ptr->ground_pitch_offset = (float)(0.0 * PI);

                obj_aircraft_ptr->rotor_wash_tex_num =
		    SARGetTextureRefNumberByName(
			scene, SAR_STD_TEXNAME_ROTOR_WASH
		    );

                obj_aircraft_ptr->spotlight_dir.heading = (float)DEGTORAD(0.0);
                obj_aircraft_ptr->spotlight_dir.pitch = (float)DEGTORAD(80.0);
                obj_aircraft_ptr->spotlight_dir.bank = (float)DEGTORAD(0.0);

		obj_aircraft_ptr->landed = 1;

		obj_aircraft_ptr->landing_gear_state = -1;
		obj_aircraft_ptr->air_brakes_state = -1;
		obj_aircraft_ptr->wheel_brakes_state = -1;
	    }
	    /* Ground? */
	    obj_ground_ptr = SAR_OBJ_GET_GROUND(obj_ptr);
	    if(obj_ground_ptr != NULL)
	    {


            }

	    pos = &obj_ptr->pos;
	    dir = &obj_ptr->dir;
        }
        else    
        {
	    /* Failed to create object, close V3D file and return 
	     * indicating error
	     */
            FClose(fp);
            return(-1);
        }

        /* Begin reading V3D model file */
	status = V3DLoadModel(
	    NULL, fp,
	    &v3d_h, &total_v3d_h,
	    &v3d_model, &total_v3d_models,
	    NULL, NULL
        );

        /* Close V3D model file, it is no longer needed */
        FClose(fp);
	fp = NULL;

	/* Error loading visual model file? */
	if(status)
	{
	    /* Error encountered while loading visual model file,
	     * delete V3D header items, model primitives, and return
	     * indicating error
	     */
	    V3DMHListDeleteAll(&v3d_h, &total_v3d_h);
	    V3DModelListDeleteAll(&v3d_model, &total_v3d_models);
	    return(-1);
	}


	/* Iterate through V3D header items */
	for(hn = 0; hn < total_v3d_h; hn++)
	{
	    h = v3d_h[hn];
	    if(h == NULL)
		continue;

	    /* Handle by header item type */
	    htype = V3DMHGetType(h);
	    switch(htype)
	    {
		const mh_comment_struct *mh_comment;
		const mh_texture_base_directory_struct *mh_tbd;
		const mh_texture_load_struct *mh_texture_load;
		const mh_color_specification_struct *mh_color_spec;
		sar_parm_texture_load_struct *p_texture_load;

	      case V3DMH_TYPE_COMMENT:
		mh_comment = (mh_comment_struct *)h;
		/* Ignore comments */
		break;

	      case V3DMH_TYPE_TEXTURE_BASE_DIRECTORY:
		mh_tbd = (mh_texture_base_directory_struct *)h;
		/* Ignore */
		break;

	      case V3DMH_TYPE_TEXTURE_LOAD:
		mh_texture_load = (mh_texture_load_struct *)h;
		p_texture_load = (sar_parm_texture_load_struct *)SARParmNew(
		    SAR_PARM_TEXTURE_LOAD
		);
		if(p_texture_load != NULL)
		{
		    /* Copy data from model header item to sar parm. */
		    free(p_texture_load->name);
		    p_texture_load->name = STRDUP(mh_texture_load->name);
		    free(p_texture_load->file);
		    p_texture_load->file = STRDUP(mh_texture_load->path);
		    p_texture_load->priority = (float)mh_texture_load->priority;
		    /* Load texture using the sar parm. */
		    SARObjLoadTexture(
			core_ptr, scene,
			p_texture_load
		    );
		    /* Deallocate sar parm for loading texture, not needed
		     * anymore.
		     */
		    SARParmDelete(p_texture_load);
		}
		break;

	      case V3DMH_TYPE_COLOR_SPECIFICATION:
		mh_color_spec = (mh_color_specification_struct *)h;
		/* Ignore */
		break;
	    }
	}


	/* Iterate through V3D models */
	for(v3d_model_num = 0; v3d_model_num < total_v3d_models; v3d_model_num++)
	{
	    v3d_model_ptr = v3d_model[v3d_model_num];
	    if(v3d_model_ptr == NULL)
		continue;

	    /* Get pointers to the object's parts for use with this
	     * V3D model, the pointers must be fetched for each V3D
	     * model since each V3D model may create a new object part
	     */
	    if(obj_aircraft_ptr != NULL)
	    {
		/* Rotors */
		total_rotors = obj_aircraft_ptr->total_rotors;
		rotor_ptr = ((total_rotors > 0) ?
		    obj_aircraft_ptr->rotor[total_rotors - 1] : NULL
		);

		/* External fuel tanks */
		total_external_fueltanks = obj_aircraft_ptr->total_external_fueltanks;
                eft_ptr = ((total_external_fueltanks > 0) ?
		    obj_aircraft_ptr->external_fueltank[total_external_fueltanks - 1]
		    : NULL
                );
	    }
/* Add fetching of substructure pointers support for other object
 * types here
 */
	    else
	    {
		/* Rotors */
		total_rotors = 0;
		rotor_ptr = NULL;

		/* External fuel tanks */
		total_external_fueltanks = 0;
		eft_ptr = NULL;
	    }

	    /* Get pointers to object part structures */
	    if(True)
	    {
                int i, part_type;
                sar_obj_part_struct *part_ptr;

#define DO_GET_PART				\
{						\
 sar_obj_part_struct *p;			\
 i = 0; part_ptr = NULL;			\
 while(1) {					\
  p = SARObjGetPartPtr(obj_ptr, part_type, i);	\
  if(p != NULL)					\
  { part_ptr = p; i++; }			\
  else						\
  { break; }					\
 }						\
}

                part_type = SAR_OBJ_PART_TYPE_AILERON_LEFT;
                DO_GET_PART
                total_aileron_lefts = i;
                aileron_left_ptr = part_ptr;

                part_type = SAR_OBJ_PART_TYPE_AILERON_RIGHT;
                DO_GET_PART
                total_aileron_rights = i;
                aileron_right_ptr = part_ptr;

                part_type = SAR_OBJ_PART_TYPE_RUDDER_TOP;
                DO_GET_PART
                total_rudder_tops = i;
                rudder_top_ptr = part_ptr;

                part_type = SAR_OBJ_PART_TYPE_RUDDER_BOTTOM;
                DO_GET_PART
                total_rudder_bottoms = i;
                rudder_bottom_ptr = part_ptr;

                part_type = SAR_OBJ_PART_TYPE_ELEVATOR;
                DO_GET_PART
                total_elevators = i;
                elevator_ptr = part_ptr;

                part_type = SAR_OBJ_PART_TYPE_CANNARD;
                DO_GET_PART
                total_cannards = i;
                cannard_ptr = part_ptr;

                part_type = SAR_OBJ_PART_TYPE_AILERON_ELEVATOR_LEFT;
                DO_GET_PART
                total_aileron_elevator_lefts = i;
                aileron_elevator_left_ptr = part_ptr;

                part_type = SAR_OBJ_PART_TYPE_AILERON_ELEVATOR_RIGHT;
                DO_GET_PART
                total_aileron_elevator_rights = i;
                aileron_elevator_right_ptr = part_ptr;

                part_type = SAR_OBJ_PART_TYPE_FLAP;
                DO_GET_PART
                total_flaps = i;
                flap_ptr = part_ptr;

                part_type = SAR_OBJ_PART_TYPE_AIR_BRAKE;
                DO_GET_PART
                total_abrakes = i;
                abrake_ptr = part_ptr;

                part_type = SAR_OBJ_PART_TYPE_LANDING_GEAR;
                DO_GET_PART
                total_lgears = i;
                lgear_ptr = part_ptr;

                part_type = SAR_OBJ_PART_TYPE_DOOR_RESCUE;
                DO_GET_PART
                total_doors = i;
                door_ptr = part_ptr;

#undef DO_GET_PART
	    }


	    /* Iterate through each "other data" line on this V3D
	     * model, these "other data" lines will specify values
	     * specific for SAR objects
	     */
 	    for(line_num = 0;
                line_num < v3d_model_ptr->total_other_data_lines;
                line_num++
	    )
	    {
		line_ptr = v3d_model_ptr->other_data_line[line_num];
		if(line_ptr == NULL)
		    continue;

		SARObjLoadLine(
		    core_ptr, obj_num, obj_ptr,
		    line_ptr, filename, line_num
		);
	    }

	    /* Do not continue the handling of this V3D model if it
	     * does not contain primitives (if the V3D model is not
	     * a standard V3D model)
	     */
	    if(v3d_model_ptr->type != V3D_MODEL_TYPE_STANDARD)
		continue;

	    /* Create model by the V3D model's name, the name must
	     * be defined or else we will not be able to know what
	     * this V3D model is suppose to represent
	     */
	    v3d_model_name = v3d_model_ptr->name;
	    if(ISSTREMPTY(v3d_model_name))
		continue;

	    /* Reset the SAR visual model and name pointer to NULL */
	    sar_vmodel = NULL;
	    sar_vmodel_name = NULL;

	    /* Handle by SAR visual model type */
	    /* Standard model */
	    if(!strcasecmp(v3d_model_name, "standard"))
	    {
		sar_vmodel = &obj_ptr->visual_model;
		free(sar_vmodel_name);
		sar_vmodel_name = STRDUP("standard");
	    }
            /* Standard Far model */
            else if(!strcasecmp(v3d_model_name, "standard_far"))
            {
                sar_vmodel = &obj_ptr->visual_model_far;
                free(sar_vmodel_name);
                sar_vmodel_name = STRDUP("standard_far");
            }
            /* Standard Dusk model */ 
            else if(!strcasecmp(v3d_model_name, "standard_dusk"))
            {
                sar_vmodel = &obj_ptr->visual_model_dusk;
                free(sar_vmodel_name);
                sar_vmodel_name = STRDUP("standard_dusk");
            }
            /* Standard Night model */
            else if(!strcasecmp(v3d_model_name, "standard_night"))
            {
                sar_vmodel = &obj_ptr->visual_model_night;
                free(sar_vmodel_name);
                sar_vmodel_name = STRDUP("standard_night");
            }
            /* Standard Dawn model */
            else if(!strcasecmp(v3d_model_name, "standard_dawn"))
            {
                sar_vmodel = &obj_ptr->visual_model_dawn;
                free(sar_vmodel_name);
                sar_vmodel_name = STRDUP("standard_dawn");
            }
            /* Rotor model */
            else if(!strcasecmp(v3d_model_name, "rotor") ||
                    !strcasecmp(v3d_model_name, "propellar")
            )
            {
                if(rotor_ptr != NULL)
		{
                    sar_vmodel = &rotor_ptr->visual_model;

		    free(sar_vmodel_name);
		    sar_vmodel_name = (char *)malloc(80 * sizeof(char));
		    sprintf(
			sar_vmodel_name,
			"rotor%i",
			total_rotors - 1
		    );
		}
            }
            /* Aileron Left model */
            else if(!strcasecmp(v3d_model_name, "aileron_left"))
            {
                if(aileron_left_ptr != NULL)
                {
                    sar_vmodel = &aileron_left_ptr->visual_model;

                    free(sar_vmodel_name);
                    sar_vmodel_name = (char *)malloc(80 * sizeof(char));
		    sprintf(
			sar_vmodel_name,
			"aileron_left%i",
			total_aileron_lefts - 1
		    );
                }
            }
            /* Aileron Right model */
            else if(!strcasecmp(v3d_model_name, "aileron_right"))
            {
                if(aileron_right_ptr != NULL)
                {
                    sar_vmodel = &aileron_right_ptr->visual_model;

                    free(sar_vmodel_name);
                    sar_vmodel_name = (char *)malloc(80 * sizeof(char));
		    sprintf(
			sar_vmodel_name,
			"aileron_right%i",
			total_aileron_rights - 1
		    );
                }
            }
            /* Rudder Top model */
            else if(!strcasecmp(v3d_model_name, "rudder_top"))
            {
                if(rudder_top_ptr != NULL)
                {
                    sar_vmodel = &rudder_top_ptr->visual_model;

                    free(sar_vmodel_name);
                    sar_vmodel_name = (char *)malloc(80 * sizeof(char));
		    sprintf(
			sar_vmodel_name,
			"rudder_top%i",
			total_rudder_tops - 1
		    );
                }
            }
            /* Rudder Bottom model */
            else if(!strcasecmp(v3d_model_name, "rudder_bottom"))
            {
                if(rudder_bottom_ptr != NULL)
                {
                    sar_vmodel = &rudder_bottom_ptr->visual_model;

                    free(sar_vmodel_name);
                    sar_vmodel_name = (char *)malloc(80 * sizeof(char));
		    sprintf(
			sar_vmodel_name,
			"rudder_bottom%i",
			total_rudder_bottoms - 1
		    );
                }
            }
            /* Elevator model */
            else if(!strcasecmp(v3d_model_name, "elevator"))
            {
                if(elevator_ptr != NULL)
                {
                    sar_vmodel = &elevator_ptr->visual_model;

                    free(sar_vmodel_name);
                    sar_vmodel_name = (char *)malloc(80 * sizeof(char));
		    sprintf(
			sar_vmodel_name,
			"elevator%i",
			total_elevators - 1
		    );
                }
            }
            /* Cannard model */
            else if(!strcasecmp(v3d_model_name, "cannard"))
            {
                if(cannard_ptr != NULL)
                {
                    sar_vmodel = &cannard_ptr->visual_model;

                    free(sar_vmodel_name);
                    sar_vmodel_name = (char *)malloc(80 * sizeof(char));
                    sprintf(
                        sar_vmodel_name,
                        "cannard%i",
                        total_cannards - 1
                    );
                }
            }
            /* Aileron Elevator Left model */
            else if(!strcasecmp(v3d_model_name, "aileron_elevator_left"))
            {
                if(aileron_elevator_left_ptr != NULL)
                {
                    sar_vmodel = &aileron_elevator_left_ptr->visual_model;

                    free(sar_vmodel_name);
                    sar_vmodel_name = (char *)malloc(80 * sizeof(char));
                    sprintf(
                        sar_vmodel_name,
                        "aileron_elevator_left%i",
                        total_aileron_elevator_lefts - 1
                    );
                }
            }
            /* Aileron Elevator Right model */
            else if(!strcasecmp(v3d_model_name, "aileron_elevator_right"))
            {
                if(aileron_elevator_right_ptr != NULL)
                {
                    sar_vmodel = &aileron_elevator_right_ptr->visual_model;

                    free(sar_vmodel_name);
                    sar_vmodel_name = (char *)malloc(80 * sizeof(char));
                    sprintf(
                        sar_vmodel_name,
                        "aileron_elevator_right%i",
                        total_aileron_elevator_rights - 1
                    );
                }
            }
            /* Flap model */
            else if(!strcasecmp(v3d_model_name, "flap"))
            {
                if(flap_ptr != NULL)
                {
                    sar_vmodel = &flap_ptr->visual_model;

                    free(sar_vmodel_name);
                    sar_vmodel_name = (char *)malloc(80 * sizeof(char));
                    sprintf(
                        sar_vmodel_name,
                        "flap%i",
                        total_flaps - 1
                    );
                }
            }
            /* Air Brake model */
            else if(!strcasecmp(v3d_model_name, "air_brake"))
            {
                if(abrake_ptr != NULL)
                {
                    sar_vmodel = &abrake_ptr->visual_model;

                    free(sar_vmodel_name);
                    sar_vmodel_name = (char *)malloc(80 * sizeof(char));
                    sprintf(
			sar_vmodel_name,
			"air_brake%i",
                        total_abrakes - 1
                    );
                }
            }
            /* Landing Gear model */
            else if(!strcasecmp(v3d_model_name, "landing_gear"))
            {
                if(lgear_ptr != NULL)
		{
                    sar_vmodel = &lgear_ptr->visual_model;

		    free(sar_vmodel_name);
                    sar_vmodel_name = (char *)malloc(80 * sizeof(char));
                    sprintf(
			sar_vmodel_name,
			"landing_gear%i",
                        total_lgears - 1
                    );
		}
            }
            /* Door model */
            else if(!strcasecmp(v3d_model_name, "door"))
            {
                if(door_ptr != NULL)
		{
                    sar_vmodel = &door_ptr->visual_model;

		    free(sar_vmodel_name);
                    sar_vmodel_name = (char *)malloc(80 * sizeof(char));
                    sprintf(
			sar_vmodel_name,
			"door%i",
			total_doors - 1
		    );
		}
            }
            /* External Fuel Tank model */
            else if(!strcasecmp(v3d_model_name, "fueltank") ||
                    !strcasecmp(v3d_model_name, "fuel_tank")
            )
            {
                if(eft_ptr != NULL)
                {
                    sar_vmodel = &eft_ptr->visual_model;

                    free(sar_vmodel_name);
                    sar_vmodel_name = (char *)malloc(80 * sizeof(char));
                    sprintf(
			sar_vmodel_name,
			"fueltank%i",
                        total_external_fueltanks - 1
                    );
                }
            }
            /* Cockpit model */
            else if(!strcasecmp(v3d_model_name, "cockpit"))
            {
                if(obj_aircraft_ptr != NULL)
		{
                    sar_vmodel = &obj_aircraft_ptr->visual_model_cockpit;

                    free(sar_vmodel_name);
                    sar_vmodel_name = STRDUP("cockpit");
		}
            }
            /* Shadow model */
            else if(!strcasecmp(v3d_model_name, "shadow"))
            {
		sar_vmodel = &obj_ptr->visual_model_shadow;
		free(sar_vmodel_name);
		sar_vmodel_name = STRDUP("shadow");
            }

	    /* Check if sar_vmodel is NULL, if it is then it means that
	     * the V3D model's name did not match a SAR visual model
	     * that is supported here
	     */
	    if(sar_vmodel == NULL)
	    {
		/* The V3D model's name is not one that we support */
		fprintf(
		    stderr,
"%s: Warning: V3D model type \"%s\" is not supported for this object.\n",
		    filename, v3d_model_name
		);

		/* Delete SAR visual model name */
		free(sar_vmodel_name);
		sar_vmodel_name = NULL;

		/* Do not continue processing this V3D model */
		continue;
	    }

	    /* Is the SAR visual model pointer pointing to an existing
	     * SAR visual model?
	     */
            if(*sar_vmodel != NULL)
            {
		/* This implies that the SAR visual model has already
		 * been created and that we are recreating it
		 */
                fprintf(
		    stderr,
 "%s: Warning: V3D model type \"%s\" redefined.\n",
                    filename, v3d_model_name
                );

		/* Unref this SAR visual model */
		SARVisualModelUnref(scene, *sar_vmodel);
		*sar_vmodel = NULL;
            }

            /* Create new SAR visual model or return a pointer to an
	     * existing visual model that matches the given filename
	     * and SAR visual model name.
	     */
            *sar_vmodel = SARVisualModelNew(
		scene, filename, sar_vmodel_name
	    );

	    /* Delete SAR visual model name (it is not needed anymore) */
	    free(sar_vmodel_name);
	    sar_vmodel_name = NULL;

	    /* If there is exactly one reference count on the "new"
	     * SAR visual model then that implies that the SAR visual
	     * model is not an existing visual model (one that has
	     * already been created and is now thus `shared'), so in
	     * which case here we need to generate a new GL list for
	     * the SAR visual model
	     */
	    if(SARVisualModelGetRefCount(*sar_vmodel) == 1)
	    {
		/* (Re)generate GL list on visual model structure. */
		GLuint list = (GLuint)SARVisualModelNewList(*sar_vmodel);
		if(list != 0)
		{
		    const char *texture_name;
		    StateGLBoolean blend_state = False;

                    /* Mark SAR visual model as loading */
                    (*sar_vmodel)->load_state = SAR_VISUAL_MODEL_LOADING;

		    /* Begin recording GL list */
		    glNewList(list, GL_COMPILE);

		    /* Reset GL states */
		    V3DTextureSelect(NULL);
		    tex_on = False;
		    tex_orient = TEX_ORIENT_NONE;
/* Do not reset color, model file should set it, if not then it means
 * it relys on the code to set it.
 *
		    glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
 */

		    /* Reset last begin primitive type */
		    last_begin_primitive_type = -1;

		    /* Iterate through each V3D model primitive */
		    for(pn = 0; pn < v3d_model_ptr->total_primitives; pn++)
		    {
			p = v3d_model_ptr->primitive[pn];
			if(p == NULL)
			    continue;

			/* Get primitive type */
                        ptype = V3DMPGetType(p);

			/* Check if last begin primitive differs from this
			 * one. If it does then glEnd() needs to be called
			 * to end a prior glBegin() that was not ended
			 */
			if((last_begin_primitive_type != ptype) &&
			   (last_begin_primitive_type > -1)
			)
			{
			    glEnd();
			    last_begin_primitive_type = -1;
			}

			switch(ptype)
			{
			    const mp_comment_struct *mp_comment;
			    const mp_color_struct *mp_color;
			    const mp_texture_select_struct *mp_texture_select;
			    const mp_texture_orient_xy_struct *mp_texture_xy;
			    const mp_texture_orient_yz_struct *mp_texture_yz;
			    const mp_texture_orient_xz_struct *mp_texture_xz;
			    const mp_heightfield_load_struct *mp_heightfield_load;

			  case V3DMP_TYPE_COMMENT:
			    mp_comment = (mp_comment_struct *)p;
#if 0
/* Ignore the processing of each line in comment primitives that are
 * in V3D models
 */
			    if(mp_comment->total_lines > 0)
			    {
				int cline_num;
				const char *cline_ptr;

				for(cline_num = 0;
				    cline_num < mp_comment->total_lines;
				    cline_num++
				)
				{
				    cline_ptr = mp_comment->line[cline_num];
				    if(cline_ptr == NULL)
					continue;

				    SARObjLoadLine(
					core_ptr, obj_num, obj_ptr,
					cline_ptr, filename, 0
				    );
				}
			    }
#endif
			    break;

			  case V3DMP_TYPE_POINT:
	                  case V3DMP_TYPE_LINE:
	                  case V3DMP_TYPE_LINE_STRIP:
	                  case V3DMP_TYPE_LINE_LOOP:
	                  case V3DMP_TYPE_TRIANGLE:
	                  case V3DMP_TYPE_TRIANGLE_STRIP:
        	          case V3DMP_TYPE_TRIANGLE_FAN:
	                  case V3DMP_TYPE_QUAD:
	                  case V3DMP_TYPE_QUAD_STRIP:
	                  case V3DMP_TYPE_POLYGON:
			    SARObjLoadVisualPrimitive(
	                        core_ptr, obj_num, obj_ptr,
	                        p, filename, 0
	                    );
			    break;

			  case V3DMP_TYPE_COLOR:
			    mp_color = (mp_color_struct *)p;
			    glColor4f(
				(GLfloat)mp_color->r,
				(GLfloat)mp_color->g,
				(GLfloat)mp_color->b,
				(GLfloat)mp_color->a
			    );
			    /* Enable GL_BLEND if alpha is less than 1.0 */
			    if(mp_color->a < 1.0f)
			    {
				if(!blend_state)
				{
				    StateGLEnableF(&display->state_gl, GL_BLEND, GL_TRUE);
				    glBlendFunc(
					GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA
				    );
				    StateGLDisableF(&display->state_gl, GL_ALPHA_TEST, GL_TRUE);
				    blend_state = True;
				}
			    }
			    else
			    {
				if(blend_state)
				{
				    StateGLDisableF(&display->state_gl, GL_BLEND, GL_TRUE);
				    StateGLEnableF(&display->state_gl, GL_ALPHA_TEST, GL_TRUE);
				    blend_state = False;
				}
			    }
			    break;

	                  case V3DMP_TYPE_TEXTURE_SELECT:
	                    mp_texture_select = (mp_texture_select_struct *)p;
			    texture_name = mp_texture_select->name;

			    if(ISSTREMPTY(texture_name))
	                    {
	                        /* Empty string implies unselect texture */
	                        V3DTextureSelect(NULL);
				tex_on = False;
	                        tex_orient = TEX_ORIENT_NONE;
	                    }
	                    else
	                    {
	                        /* Select texture */
	                        v3d_texture_ref_struct *t = SARGetTextureRefByName(
				    scene, texture_name
				);
	                        if(t == NULL)
	                        {
	                            fprintf(
					stderr,
 "%s: Warning: Texture \"%s\" not defined (on model or globally).\n",
	                                filename, texture_name
	                            );
				    tex_on = False;
	                            tex_orient = TEX_ORIENT_NONE;
	                        }
	                        else
				{
				    tex_on = True;
	                            V3DTextureSelect(t);
				}
			    }
	                    break;

			  case V3DMP_TYPE_TEXTURE_ORIENT_XY:
			    mp_texture_xy = (mp_texture_orient_xy_struct *)p;
	                    tex_orient = TEX_ORIENT_XY;
	                    tex_coord.i = (float)mp_texture_xy->x;  
	                    tex_coord.j = (float)mp_texture_xy->y;
	                    tex_coord.w = (float)mp_texture_xy->dx;
	                    tex_coord.h = (float)mp_texture_xy->dy;
			    break;

			  case V3DMP_TYPE_TEXTURE_ORIENT_YZ:
			    mp_texture_yz = (mp_texture_orient_yz_struct *)p;
	                    tex_orient = TEX_ORIENT_YZ;
	                    tex_coord.i = (float)mp_texture_yz->y;
	                    tex_coord.j = (float)mp_texture_yz->z;
	                    tex_coord.w = (float)mp_texture_yz->dy;
	                    tex_coord.h = (float)mp_texture_yz->dz;
			    break;

			  case V3DMP_TYPE_TEXTURE_ORIENT_XZ:
			    mp_texture_xz = (mp_texture_orient_xz_struct *)p;
	                    tex_orient = TEX_ORIENT_XZ;
	                    tex_coord.i = (float)mp_texture_xz->x;
	                    tex_coord.j = (float)mp_texture_xz->z;
	                    tex_coord.w = (float)mp_texture_xz->dx;
	                    tex_coord.h = (float)mp_texture_xz->dz;
			    break;

			  case V3DMP_TYPE_TEXTURE_OFF:
	                    V3DTextureSelect(NULL);
			    tex_on = False;
	                    tex_orient = TEX_ORIENT_NONE;
			    break;

			  case V3DMP_TYPE_HEIGHTFIELD_LOAD:
			    mp_heightfield_load = (mp_heightfield_load_struct *)p;
			    SARObjLoadHeightField(
	                        core_ptr, obj_num, obj_ptr,
	                        p, filename, 0,
				list
			    );
			    break;
			}
                    }	/* Iterate through each V3D model primitive */

		    /* Check if last_begin_primitive_type is valid,
		     * if it is then that means we need to call glEnd()
		     * to end a prior glBegin().
		     */
		    if(last_begin_primitive_type > -1)
		    {
			glEnd();
			last_begin_primitive_type = -1;
		    }

		    /* Check if blending was still enabled while 
		     * generating the list.
		     */
		    if(blend_state)
		    {
			StateGLDisableF(&display->state_gl, GL_BLEND, GL_TRUE);
			StateGLEnableF(&display->state_gl, GL_ALPHA_TEST, GL_TRUE);
			blend_state = False;
		    }


		    /* End gl list */
	            glEndList();

		    /* Mark SAR Visual Model as finished loading */
		    (*sar_vmodel)->load_state = SAR_VISUAL_MODEL_LOADED;

		}	/* (Re)generate GL list on SAR visual model structure */
	    }
	    else
	    {
		/* The SAR visual model already has more than one
		 * reference count which implies it is shared
		 */

		/* Need to check for certain model data that we still
		 * need to load (even if the visual model has more than
		 * one ref count), all of the model data loaded here
		 * should not generate any GL commands when they are
		 * loaded
		 */

		if(True)
		{
                    /* Iterate through each V3D model primitive */
                    for(pn = 0; pn < v3d_model_ptr->total_primitives; pn++)
                    {
                        p = v3d_model_ptr->primitive[pn];
                        if(p == NULL)
                            continue;

                        /* Get primitive type */
                        ptype = V3DMPGetType(p);

			switch(ptype)
			{
			    const mp_heightfield_load_struct *mp_heightfield_load;

                          case V3DMP_TYPE_HEIGHTFIELD_LOAD:
                            mp_heightfield_load = (mp_heightfield_load_struct *)p;
			    /* Need to load z points for height field but
			     * not issue any GL commands for it (by
			     * passing the GL list as 0)
			     */
                            SARObjLoadHeightField(
                                core_ptr, obj_num, obj_ptr,
                                p, filename, 0,
                                0		/* No gl list. */
                            );
                            break;


                        }
                    }   /* Iterate through each V3D model primitive */
		}
	    }
	}


	/* Delete V3D models and V3D header items (no longer needed) */
	V3DMHListDeleteAll(&v3d_h, &total_v3d_h);
	V3DModelListDeleteAll(&v3d_model, &total_v3d_models);


        /* Create flight dynamics model if object is an aircraft */
        if(obj_aircraft_ptr != NULL)
        {
	    if(scene->realm == NULL)
	    {
		fprintf(
		    stderr,
 "%s: Warning: Cannot create SFM for NULL realm structure on scene\n",
		    filename
		);
	    }
	    else
	    {
		/* Create new flight dynamics model and reset its
		 * values to defaults
		 */
		SFMModelStruct *fdm = SFMModelAllocate();
		if(fdm != NULL)
		{
		    fdm->landed_state = True;

		    SFMModelAdd(scene->realm, fdm);
		}
		obj_aircraft_ptr->fdm = fdm;
	    }
        }


	/* Need to realize translation and rotation values on object */
	SARSimWarpObject(
	    scene, obj_ptr,
	    &obj_ptr->pos, &obj_ptr->dir
	);

	return(0);
}
