#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#ifdef __MSW__
# include <windows.h>
#endif

#include "v3dtex.h"

#include "obj.h"
#include "objutils.h"
#include "smoke.h"


int SmokeCreate(
	sar_scene_struct *scene,
	sar_object_struct ***ptr, int *total,
	int smoke_type,			/* One of SAR_OBJ_SMOKE_TYPE_*. */
	const sar_position_struct *pos,
	const sar_position_struct *respawn_offset,
	float radius_start,		/* In meters. */
	float radius_max,		/* In meters. */
	float radius_rate,		/* In meters per second. */
	int hide_at_max,
	int total_units,		/* Max smoke units in trail. */
        time_t respawn_int,             /* In ms. */
	const char *tex_name,
	int ref_object,			/* Can be -1 for none. */
	time_t life_span
);

int SmokeCreateSparks(
        sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
	const sar_position_struct *pos,
	const sar_position_struct *respawn_offset,
	float sparks_distance,		/* How far should sparks fly. */
	int ref_object,
	time_t life_span
);


/*
 *	Creates a smoke trail object, returning the new object's number
 *	or -1 on error.
 */
int SmokeCreate(
        sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
        int smoke_type,                 /* One of SAR_OBJ_SMOKE_TYPE_*. */
	const sar_position_struct *pos,
	const sar_position_struct *respawn_offset,
        float radius_start,            /* In meters. */
        float radius_max,              /* In meters. */
        float radius_rate,             /* In meters per second. */
        int hide_at_max,
        int total_units,                /* Max smoke units in trail. */
        time_t respawn_int,             /* In ms. */
        const char *tex_name,
	int ref_object,			/* Can be -1 for none. */
        time_t life_span
)
{
	int i, obj_num;
	sar_object_struct *obj_ptr;
	sar_object_smoke_struct *obj_smoke_ptr;


	if((scene == NULL) || (pos == NULL))
	    return(-1);

	/* Create smoke trail object. */
	obj_num = SARObjCreate(
	    scene, ptr, total,
	    SAR_OBJ_TYPE_SMOKE
	);
	if(obj_num < 0)
	    return(-1);

	obj_ptr = (*ptr)[obj_num];
	if(obj_ptr == NULL)
	    return(-1);


	/* Need to automatically calculate radius_rate? */
	if(radius_rate < 0.0f)
	{
	    float dt = (float)(total_units * respawn_int / 1000.0f);

	    if(dt > 0.0f)
		radius_rate = (float)((radius_max - radius_start) / dt);
	    else
		radius_rate = 0.0f;
	}

	/* Begin setting values */

	/* Position */
	memcpy(
	    &obj_ptr->pos,
	    pos,
	    sizeof(sar_position_struct)
	);
	/* Set visible range based on maximum size. */
	if(radius_max >= 400.0f)
	    obj_ptr->range = (float)SFMMilesToMeters(18.0);
        else if(radius_max >= 200.0f)
            obj_ptr->range = (float)SFMMilesToMeters(16.0);
	else if(radius_max >= 100.0f)
            obj_ptr->range = (float)SFMMilesToMeters(13.0);
	else
	    obj_ptr->range = (float)SFMMilesToMeters(10.0);

	SARObjAddContactBoundsSpherical(
	    obj_ptr,
	    0, 0,		/* Crash flags and crash type. */
	    radius_start	/* Use starting radius size as contact bounds. */
	);

	obj_ptr->life_span = life_span;


        /* Get pointer to smoke trail sub structure. */
        obj_smoke_ptr = SAR_OBJ_GET_SMOKE(obj_ptr);
        if(obj_smoke_ptr == NULL)
            return(-1);

	/* Set smoke trail type to smoke. */
	obj_smoke_ptr->type = smoke_type;

	/* Set smoke trail specific values. */
	if(respawn_offset != NULL)
	    memcpy(
		&obj_smoke_ptr->respawn_offset,
		respawn_offset,
		sizeof(sar_position_struct)
	    );

	obj_smoke_ptr->radius_start = radius_start;
        obj_smoke_ptr->radius_max = radius_max;
	obj_smoke_ptr->radius_rate = radius_rate;

	obj_smoke_ptr->hide_at_max = hide_at_max;
	obj_smoke_ptr->delete_when_no_units = 0;

	obj_smoke_ptr->respawn_int = respawn_int;
	obj_smoke_ptr->respawn_next = 0;

	obj_smoke_ptr->tex_num = SARGetTextureRefNumberByName(
            scene, tex_name
        );

        obj_smoke_ptr->ref_object = ref_object;

	obj_smoke_ptr->total_units = total_units;
	if(obj_smoke_ptr->total_units > 0)
	{
	    obj_smoke_ptr->unit = (sar_object_smoke_unit_struct *)realloc(
		obj_smoke_ptr->unit,
		obj_smoke_ptr->total_units * sizeof(sar_object_smoke_unit_struct)
	    );
	    if(obj_smoke_ptr->unit == NULL)
	    {
		obj_smoke_ptr->total_units = total_units = 0;
	    }
	}
	for(i = 0; i < obj_smoke_ptr->total_units; i++)
	{
	    sar_object_smoke_unit_struct *smoke_unit_ptr = &obj_smoke_ptr->unit[i];

	    memset(&smoke_unit_ptr->pos, 0x00, sizeof(sar_position_struct));
	    smoke_unit_ptr->radius = 0.0f;
	    smoke_unit_ptr->visibility = 0.0f;
	}


	return(obj_num);
}


/*
 *	Creates a smoke trail object to display as sparks, returning the
 *	new object's number or -1 on error.
 */
int SmokeCreateSparks(
        sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
	const sar_position_struct *pos,
	const sar_position_struct *respawn_offset,
	float sparks_distance,		/* How far should sparks fly */
	int ref_object,
	time_t life_span
)
{
        int obj_num;
        sar_object_struct *obj_ptr;
        sar_object_smoke_struct *obj_smoke_ptr;
	time_t respawn_int = 500;


        if((scene == NULL) || (pos == NULL))
            return(-1);

	/* Note about respawn interval: The respawning of sparks per
	 * respawn interval is a group of sparks (instead of just one
	 * unit as with the smoke puffs).
	 */

	obj_num = SmokeCreate(
	    scene, ptr, total,
	    SAR_OBJ_SMOKE_TYPE_SPARKS,
	    pos, respawn_offset,
	    1.0f,		/* Radius start (ignored for sparks). */
	    sparks_distance,	/* Radius max (how far sparks should fly). */
	    1.0f,		/* Radius growth rate in meters per cycle
				 * (ignored for sparks).
				 */
	    0,			/* No hide at max. */
	    10,			/* Total units. */
	    respawn_int,	/* Respawn interval in ms (*). */
	    NULL,		/* No texture. */
	    ref_object,
	    life_span
	);
        if(obj_num < 0)
            return(-1);

        obj_ptr = (*ptr)[obj_num];
        if(obj_ptr == NULL)
            return(-1);

        /* Get pointer to smoke trail sub structure. */
        obj_smoke_ptr = SAR_OBJ_GET_SMOKE(obj_ptr);
        if(obj_smoke_ptr == NULL)
            return(-1);

	return(obj_num);
}
