/*
 * Copyright (C) 2005 Alex Murray <murray.alex@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef SENSORS_APPLET_H
#define SENSORS_APPLET_H
#include <gtk/gtk.h>
#include <panel-applet.h>

typedef struct _SensorsApplet SensorsApplet;
typedef	struct _PrefsDialog PrefsDialog;

#include "prefs-dialog.h"

typedef enum {UNUSED = 0, /* as a flag to test against later */
	      ACPI,
	      HDDTEMP,
	      I2C_SYS,
	      I2C_PROC,
	      I8K,
	      OMNIBOOK,
	      PMU_SYS,
	      N_SENSOR_INTERFACES
} SensorInterface;

static const gchar *sensor_interface[] = {"unused",
					  "acpi",
					  "hddtemp",
					  "i2c-sys",
					  "i2c-proc",
					  "i8k",
					  "omnibook",
					  "pmu-sys",
};

/* enumeration used to identify columns in the GtkTreeStore data
 * structure and to access specific gconf keys too.
 */
enum {
	PATH_COLUMN = 0,
	ID_COLUMN,
	LABEL_COLUMN,
	INTERFACE_COLUMN,
	SENSOR_TYPE_COLUMN,
	ENABLE_COLUMN,
	VISIBLE_COLUMN,
       	ALARM_VALUE_COLUMN,
	ALARM_TYPE_COLUMN, 
	ALARM_ENABLE_COLUMN,
	ALARM_COMMAND_COLUMN,
	ALARM_TIMEOUT_COLUMN,
	ALARM_TIMEOUT_INDEX_COLUMN,
	MULTIPLIER_COLUMN,
	OFFSET_COLUMN,
	N_COLUMNS
};


typedef enum {
	CPU_VID_SENSOR = 0,
	CURRENT_SENSOR,
	FAN_SENSOR,
	TEMP_SENSOR,
	VOLTAGE_SENSOR
} SensorType;

/* alarm types */
typedef enum {
	ALARM_WHEN_VALUE_GREATER_THAN_THRESHOLD = 0,
	ALARM_WHEN_VALUE_LESS_THAN_THRESHOLD
} AlarmType;


/* for identifying font sizes */
typedef enum {
	XX_LARGE = 0,
	X_LARGE,
	LARGE,
	MEDIUM,
	SMALL,
	X_SMALL,
	XX_SMALL
} FontSize;

/* should always return the value in degrees celcius, revolutions per
 * minute, volts, amps, etc */
typedef float (*GetSensorValueFunction)(const gchar *path, 
					const gchar *id, 
					SensorType type,
					GError **error);

struct _SensorsApplet {
	/* the actual applet for this instance */
	PanelApplet* applet;

	GtkTreeStore *sensors;

	GetSensorValueFunction get_sensor_value[N_SENSOR_INTERFACES];

	guint timeout_id;
	/* preferences and about windows (if Gtk < 2.6)*/
#ifndef HAVE_GTK_26
	GtkWidget *about_dialog;
#endif
	PrefsDialog *prefs_dialog;

	/* primary table to contain the panel dispay - we pack the
	 * list of labels and sensor values into this container */
	GtkWidget *table;
	GSList *labels;
	GSList *values;
	GtkTooltips *tooltips;

	/* list of gchar* to the alarm commands, which have been
	 * allocated for each active alarm - these are then freed when
	 * the alarm is turned off */
	GSList *alarm_commands;
};


#define XX_LARGE_TEXT "xx-large"
#define X_LARGE_TEXT "x-large"
#define LARGE_TEXT "large"
#define MEDIUM_TEXT "medium"
#define SMALL_TEXT "small"
#define X_SMALL_TEXT "x-small"
#define XX_SMALL_TEXT "xx-small"


/* non-static function prototypes */
void sensors_applet_init(SensorsApplet *sensors_applet);
void sensors_applet_update_display(SensorsApplet *sensors_applet);
void sensors_applet_register_sensors_interface(SensorsApplet *sensors_applet,
					       SensorInterface interface,
					       GetSensorValueFunction get_sensor_value_function);

/* path should be the full path to a file representing the sensor (eg
 * /dev/hda or /sys/devices/platform/i2c-0/0-0290/temp1_input) - this
 * value is tested when loaded from gconf to see if it is a real path
 * and will fail if is not and thus values will not be restored from
 * gconf */
gboolean sensors_applet_add_sensor_full_details(SensorsApplet *sensors_applet,
						const gchar *path, 
						const gchar *id, 
						const gchar *label, 
						SensorInterface interface, 
						SensorType type, 
						gboolean enable,
						gdouble alarm_value,
						AlarmType alarm_type,
						gboolean alarm_enable,
						const gchar *alarm_command,
						gint alarm_timeout,
						gdouble multiplier,
						gdouble offset);


gboolean sensors_applet_add_sensor(SensorsApplet *sensors_applet,
				   const gchar *path, 
				   const gchar *id, 
				   const gchar *label, 
				   SensorInterface interface, 
				   gboolean enable,
				   SensorType type); 
void sensors_applet_alarm_off(SensorsApplet *sensors_applet, GtkTreeIter *iter);
#endif /* SENSORS_APPLET_H */
