# Copyright (C) 2004 Tiago Cogumbreiro <cogumbreiro@users.sf.net>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
#
# Authors: Tiago Cogumbreiro <cogumbreiro@users.sf.net>

"""Uses totem.plparser to try and load playlists."""

from totem import plparser

from serpentine.mastering import HintsFilter
from serpentine.errors import UnsupportedLocationError

class PlparserFilter (HintsFilter):
    def __init__ (self):
        super(PlparserFilter, self).__init__()
        self.priority = 10
        
    def __on_pl_entry (self, parser, uri, title, genre, hints_list):
        hints = {'location': uri}
        if title is not None:
            hints['title'] = title
        hints_list.append(hints)
        
    def _on_entry_parsed(self, parser, uri, data, hints_list):
        hints = {'location': uri}

        for src_key, dst_key in (('title', 'title'), ('author', 'artist')):
            val = data[src_key]
            if val is not None:
                data[dst_key] = val

        hints_list.append(hints)

    def filter_location (self, location):
        hints_list = []
        p = plparser.Parser()
        try:
            p.connect("entry", self.__on_pl_entry, hints_list)
        except TypeError:
            p.connect('entry-parsed', self._on_entry_parsed, hints_list)
            
        if p.parse(location, False) == plparser.PARSER_RESULT_SUCCESS:
            return hints_list
        raise UnsupportedLocationError()
    
def create_plugin(serp):
    serp.register_playlist_file_pattern ("PLS Audio Playlist", "*.pls")
    serp.register_music_file_pattern ("PLS Audio Playlist", "*.pls")
    
    serp.register_playlist_file_pattern ("MP3 Playlist File", "*.m3u")
    serp.register_music_file_pattern ("MP3 Playlist File", "*.m3u")
    
    serp.add_hints_filter (PlparserFilter ())
    
