/* @(#)find.c	1.13 04/10/04 Copyright 2004 J. Schilling */
#ifndef lint
static	char sccsid[] =
	"@(#)find.c	1.13 04/10/04 Copyright 2004 J. Schilling";
#endif
/*
 *	Another find implementation...
 *
 *	Copyright (c) 2004 J. Schilling
 */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <mconfig.h>
#include <stdio.h>
#include <unixstd.h>
#include <stdxlib.h>
#ifdef	HAVE_FCHDIR
#include <fctldefs.h>
#else
#include <maxpath.h>
#endif
#include <statdefs.h>
#include <timedefs.h>
#include <waitdefs.h>
#include <strdefs.h>
#include <utypes.h>		/* incl. limits.h (_POSIX_ARG_MAX/ARG_MAX) */
#ifdef	HAVE_SYS_PARAM_H
#include <sys/param.h>		/* #defines NCARGS on old systems */
#endif
#include <btorder.h>
#include <getcwd.h>
#include <patmatch.h>
#if	defined(HAVE_FNMATCH_H) & defined(HAVE_FNMATCH)
#include <fnmatch.h>
#endif
#include <standard.h>
#include <schily.h>
#include <pwd.h>
#include <grp.h>

#include "list.h"
#include "mem.h"
#include "walk.h"
#include "gettnum.h"

char	strvers[] = "1.0";	/* The pure version string	*/

typedef struct {
	char	*left;
	char	*right;
	char	*this;
	int	op;
	union {
		int	i;
		long	l;
		dev_t	dev;
		ino_t	ino;
		mode_t	mode;
		nlink_t	nlink;
		uid_t	uid;
		gid_t	gid;
		size_t	size;
		time_t	time;
	} val;
	int	val2;
} node;

LOCAL	char	*tokennames[] = {
#define	OPEN	0
	"(",
#define	CLOSE	1
	")",
#define	LNOT	2
	"!",
#define	AND	3
	"a",
#define	LOR	4
	"o",
#define	ATIME	5
	"atime",
#define	CTIME	6
	"ctime",
#define	DEPTH	7
	"depth",
#define	EXEC	8
	"exec",
#define	FOLLOW	9	/* POSIX Extension */
	"follow",
#define	FSTYPE	10	/* POSIX Extension */
	"fstype",
#define	GROUP	11
	"group",
#define	INUM	12	/* POSIX Extension */
	"inum",
#define	LINKS	13
	"links",
#define	LOCL	14	/* POSIX Extension */
	"local",
#define	LS	15	/* POSIX Extension */
	"ls",
#define	MODE	16	/* POSIX Extension */
	"mode",
#define	MOUNT	17	/* POSIX Extension */
	"mount",
#define	MTIME	18
	"mtime",
#define	NAME	19
	"name",
#define	NEWER	20
	"newer",
#define	NOGRP	21
	"nogroup",
#define	NOUSER	22
	"nouser",
#define	OK_EXEC	23
	"ok",
#define	PERM	24
	"perm",
#define	PRINT	25
	"print",
#define	PRINTNNL 26	/* POSIX Extension */
	"printnnl",
#define	PRUNE	27
	"prune",
#define	SIZE	28
	"size",
#define	TIME	29	/* POSIX Extension */
	"time",
#define	TYPE	30
	"type",
#define	USER	31
	"user",
#define	XDEV	32
	"xdev",
#define	PATH	33
	"path",
#define	LNAME	34
	"lname",
#define	PAT	35
	"pat",
#define	PPAT	36
	"ppat",
#define	LPAT	37
	"lpat",
#define	HELP	38
	"help",
#define	ENDARGS	39	/* Found End of Arg Vector */
	0,
#define	EXECPLUS 40
	"exec",
	0
};
#define	NTOK	((sizeof (tokennames) / sizeof (tokennames[0])) - 1)

/*
 *	---------------------------------
 *	| Other struct plusargs fields	|	Don't count against ARG_MAX
 *	---------------------------------
 *	---------------------------------
 *	| 	New Arg vector[0]	|	Space for ARG_MAX starts here
 *	---------------------------------
 *	|		.		|
 *	|		.		|	Arg space grows upwards
 *	|		V		|
 *	---------------------------------
 *	|	 Arg vector end		|
 *	---------------------------------
 *	---------------------------------
 *	| Space for first arg string	|
 *	---------------------------------
 *	|		^		|
 *	|		.		|	String space "grows" downwards
 *	|		.		|
 *	---------------------------------
 *	| Space for first arg string	|	Space for ARG_MAX ends here
 *	---------------------------------
 */
struct plusargs {
	struct plusargs	*next;		/* Next in list for flushing	*/
	char		*endp;		/* Points to end of block	*/
	char		**nextarg;	/* Points to next av[] entry	*/
	char		*laststr;	/* points to last used string	*/
	int		ac;		/* The argc for our command	*/
	char		*av[1];		/* The argv for our command	*/
};
LOCAL struct plusargs *plusp;

#define	MINSECS		(60)
#define	HOURSECS	(60 * MINSECS)
#define	DAYSECS		(24 * HOURSECS)
#define	YEARSECS	(365 * DAYSECS)

EXPORT	time_t	sixmonth;		/* 6 months before limit (ls)	*/
EXPORT	time_t	now;			/* now limit (ls)		*/

LOCAL	node	*Tree = (node *)0;
LOCAL	int	walkflags = WALK_CHDIR | WALK_PHYS;
LOCAL	int	Argc = 0;
LOCAL	char	**Argv = (char **)0;
LOCAL	int	primtype = 0;
LOCAL	int	err = 0;
LOCAL	int	patlen = 0;
LOCAL	int	*pstate;
LOCAL	BOOL	have_print = FALSE;
#ifdef	HAVE_FCHDIR
LOCAL	int	FHome = -1;
#else
LOCAL	char	FHome[MAXPATHNAME+1];
#endif

#ifndef	__GNUC__
#define	inline
#endif

LOCAL	node	*allocnode	__PR((void));
LOCAL	void	nexttoken	__PR((void));
LOCAL	char	*nextarg	__PR((node *t));
LOCAL	node	*parse		__PR((void));
LOCAL	node	*parseand	__PR((void));
LOCAL	node	*parseprim	__PR((void));
LOCAL	void	getperm		__PR((node *t));
LOCAL	char	*getsperm	__PR((char *p, node *t));
LOCAL	mode_t	iswho		__PR((int c));
LOCAL	int	isop		__PR((int c));
LOCAL	mode_t	isperm		__PR((int c));
LOCAL	int	walkfunc	__PR((char *nm, struct stat *fs, int type, struct WALK *state));
LOCAL	inline BOOL	expr	__PR((char *f, char *ff, struct stat *fs, struct WALK *state, node *t));
LOCAL	BOOL	doexec		__PR((char *f, int ac, char **av));
LOCAL	int	argsize		__PR((void));
LOCAL	void	pluscreate	__PR((int ac, char **av));
LOCAL	BOOL	plusexec	__PR((char *f, node *t));
LOCAL	void	usage		__PR((int ret));
LOCAL	int	getflg		__PR((char *optstr, long *argp));
EXPORT	int	main		__PR((int ac, char **av));


LOCAL node *
allocnode()
{
	node *n;

	n = __malloc(sizeof (node), "allocnode");
	n->left = 0;
	n->right = 0;
	n->this = 0;
	n->op = 0;
	n->val.l = 0;
	n->val2 = 0;
	return (n);
}

LOCAL void
nexttoken()
{
	char *word;
	char *xword;
	char *tail;
	char *xtail;
	char **tp;

	if (Argc <= 0) {
		primtype = ENDARGS;
		return;
	}
	xword = word = *Argv;
	if ((xtail = tail = strchr(word, '=')) != NULL) {
		*tail++ = '\0';
			*Argv = tail;
	} else {
		Argv++; Argc--;
	}
	if (*word == '-') {
		/*
		 * Do not allow -(, -), -!
		 */
		if (word[1] == '\0' || !strchr("()!", word[1]))
			word++;
	} else if (!strchr("()!", word[0]) && (!xtail || xtail[1] == '\0')) {
		goto bad;
	}
	for (primtype = 0, tp = tokennames; *tp; tp++, primtype++) {
		if (streql(*tp, word))
			return;
	}
bad:
	if (xtail)
		*xtail = '=';
	errmsgno(EX_BAD, "Bad Option: '%s'.\n", xword);
	usage(EX_BAD);
}

LOCAL char *
nextarg(t)
	node	*t;
{
	if (Argc-- <= 0) {
		char	*prim	= NULL;
		int	pt	= t->op;

		if (pt >= 0 && pt < NTOK)
			prim = tokennames[pt];
		if (prim) {
			comerrno(EX_BAD, "Missing arg for '%s%s'.\n",
				pt > LNOT ? "-":"", prim);
		} else {
			comerrno(EX_BAD, "Missing arg.\n");
		}
		/* NOTREACHED */
		return ((char *)0);
	} else {
		return (*Argv++);
	}
}

LOCAL node *
parse()
{
	node	*n;

	n = parseand();
	if (primtype == LOR) {
		node	*l = allocnode();

		l->left = (char *)n;
		l->op = primtype;
		nexttoken();
		l->right = (char *)parse();
		return (l);
	}
	return (n);
}

LOCAL node *
parseand()
{
	node	*n;

	n = parseprim();
	if ((primtype == AND) ||
	    (primtype != LOR && primtype != CLOSE && primtype != ENDARGS)) {
		node	*l = allocnode();

		l->left = (char *)n;
		l->op = AND;		/* If no Operator, default to AND -a */
		if (primtype == AND)	/* Fetch next Operator for next node */
			nexttoken();
		l->right = (char *)parseand();
		return (l);
	}
	return (n);
}

LOCAL node *
parseprim()
{
	register node	*n;
	register char	*p;
		Llong	ll;

	n = allocnode();
	switch (n->op = primtype) {

	/*
	 * Use simple to old (historic) shell globbing.
	 */
	case NAME:
	case PATH:
	case LNAME:
#if	defined(HAVE_FNMATCH_H) & defined(HAVE_FNMATCH)
		n->this = nextarg(n);
		nexttoken();
		return (n);
#endif
		/* FALLTHROUGH */
		/* Implement "fallback" to patmatch() if we have no fnmatch() */

	/*
	 * Use patmatch() which is a regular expression matcher that implements
	 * extensions that are compatible to old (historic) shell globbing.
	 */
	case PAT:
	case PPAT:
	case LPAT: {
		int	plen;

		plen = strlen(n->this = nextarg(n));
		if (plen > patlen)
			patlen = plen;
		n->right = __malloc(sizeof (int)*plen, "space for pattern");

		if ((n->val.i = patcompile((Uchar *)n->this, plen, (int *)n->right)) == 0)
			comerrno(EX_BAD, "Bad pattern in '-%s %s'.\n",
						tokennames[n->op], n->this);
		nexttoken();
		return (n);
	}

	case SIZE:
		p = n->left = nextarg(n);
		if (p[0] == '-' || p[0] == '+')
			p++;
		p = astoll(p, &ll);
		if (p[0] == 'c' && p[1] == '\0')
			n->this = p;
		else if (*p)
			comerrno(EX_BAD,
				"Non numeric character '%c' in '-size %s'.\n",
				*p, n->left);
		n->val.size = ll;
		nexttoken();
		return (n);

	case LINKS:
		p = n->left = nextarg(n);
		if (p[0] == '-' || p[0] == '+')
			p++;
		p = astoll(p, &ll);
		if (*p)
			comerrno(EX_BAD,
				"Non numeric character '%c' in '-links %s'.\n",
				*p, n->left);
		n->val.nlink = ll;
		nexttoken();
		return (n);

	case INUM:
		p = n->left = nextarg(n);
		if (p[0] == '-' || p[0] == '+')
			p++;
		p = astoll(p, &ll);
		if (*p)
			comerrno(EX_BAD,
				"Non numeric character '%c' in '-inum %s'.\n",
				*p, n->left);
		n->val.ino = ll;
		nexttoken();
		return (n);

	case TIME:
	case ATIME:
	case CTIME:
	case MTIME: {
		int	len;

		p = n->left = nextarg(n);
		if (p[0] == '-' || p[0] == '+')
			p++;
		if (gettnum(p, &n->val.time) != 1) {
			comerrno(EX_BAD,
				"Bad timespec in '-%s %s'.\n",
				tokennames[n->op], n->left);
		}
		len = strlen(p);
		if (len > 0) {
			len = (Uchar)p[len-1];
			if (!(len >= '0' && len <= '9'))
				n->val2 = 1;
		}
		nexttoken();
		return (n);
	}
	case NEWER: {
		struct stat ns;

		if (stat(n->left = nextarg(n), &ns) < 0)
			comerr("Cannot stat '%s'.\n", n->left);

		n->val.time = ns.st_mtime;
		nexttoken();
		return (n);
	}

	case TYPE:
		n->this = (char *)nextarg(n);
		switch (*(n->this)) {

		case 'b': case 'c': case 'd': case 'D':
		case 'e': case 'f': case 'l': case 'p':
		case 's':
			if ((n->this)[1] == '\0') {
				nexttoken();
				return (n);
			}
		}
		comerrno(EX_BAD, "Bad type '%c' in '-type %s'.\n",
			*n->this, n->this);
		break;

	case FSTYPE:
#ifdef	HAVE_ST_FSTYPE
		n->this = (char *)nextarg(n);
#else
		comerrno(EX_BAD, "-fstype not supported by this OS.\n");
#endif
		nexttoken();
		return (n);

	case LOCL:
#ifndef	HAVE_ST_FSTYPE
		comerrno(EX_BAD, "-local not supported by this OS.\n");
#endif
		nexttoken();
		return (n);

	case USER: {
		struct  passwd  *pw;
		char		*u;

		u = n->left = nextarg(n);
		if (u[0] == '-' || u[0] == '+')
			u++;
		if ((pw = getpwnam(u)) != NULL) {
			n->val.uid = pw->pw_uid;
		} else {
			if (*astoll(n->left, &ll))
				comerrno(EX_BAD,
				"User '%s' not in passwd in database.\n",
				n->left);
			n->val.uid = ll;
		}
		nexttoken();
		return (n);
	}

	case GROUP: {
		struct  group	*gr;
		char		*g;

		g = n->left = nextarg(n);
		if (g[0] == '-' || g[0] == '+')
			g++;
		if ((gr = getgrnam(g)) != NULL) {
			n->val.gid = gr->gr_gid;
		} else {
			if (*astoll(n->left, &ll))
				comerrno(EX_BAD,
				"Group '%s' not in group in database.\n",
				n->left);
			n->val.gid = ll;
		}
		nexttoken();
		return (n);
	}

	case PERM:
		n->left = nextarg(n);
		getperm(n);
		nexttoken();
		return (n);

	case MODE:
		comerrno(EX_BAD, "-mode not yet implemented.\n");
		nexttoken();
		return (n);

	case XDEV:
	case MOUNT:
		walkflags |= WALK_MOUNT;
		nexttoken();
		return (n);
	case DEPTH:
		walkflags |= WALK_DEPTH;
		nexttoken();
		return (n);
	case FOLLOW:
		walkflags &= ~WALK_PHYS;
		nexttoken();
		return (n);

	case NOUSER:
	case NOGRP:
	case PRUNE:
		nexttoken();
		return (n);

	case OK_EXEC:
	case EXEC: {
		int	i = 1;

		n->this = (char *)Argv;	/* Cheat: Pointer is pointer	*/
		nextarg(n);		/* Eat up cmd name		*/
		while ((p = nextarg(n)) != NULL) {
			if (streql(p, "{}"))
				Argv[-1] = NULL;
			else if (streql(p, ";"))
				break;
			else if (streql(p, "+") && Argv[-2] == NULL) {
				n->op = primtype = EXECPLUS;
				pluscreate(--i, (char **)n->this);
				n->this = (char *)plusp;
				break;
			}
			i++;
		}
		n->val.i = i;
#ifdef	PLUS_DEBUG
		if (0) {
			char **pp = (char **)n->this;
			for (i = 0; i < n->val.i; i++, pp++)
				printf("ARG %d '%s'\n", i, *pp);
		}
#endif
	}
	/* FALLTHRU */

	case PRINT:
	case PRINTNNL:
	case LS:
		have_print = TRUE;
		nexttoken();
		return (n);

	case ENDARGS:
#ifdef	DEBUG
		errmsgno(EX_BAD, "ENDARGS in parseprim()\n");
#endif
		comerrno(EX_BAD, "Incomplete expression.\n");
		/* NOTREACHED */

	case OPEN:
		nexttoken();
		n->this = (char *)parse();
		if (primtype != CLOSE) {
			comerrno(EX_BAD,
				"Found '%s', but ')' expected.\n", *Argv);
		} else {
			nexttoken();
			return (n);
		}
		break;

	case CLOSE:
		comerrno(EX_BAD, "Missing '('.\n");
		/* NOTREACHED */

	case LNOT:
		nexttoken();
		n->this = (char *)parseprim();
		return (n);

	case AND:
	case LOR:
		comerrno(EX_BAD, "Invalid expression with -%s.\n", tokennames[n->op]);
		/* NOTREACHED */

	case HELP:
		usage(0);

	default:
		comerrno(EX_BAD, "Internal malfunction.\n");
		/* NOTREACHED */
	}
	return (0);
}

/*
 * This is the mode bit translation code stolen from star...
 */
#define	TSUID		04000	/* Set UID on execution */
#define	TSGID		02000	/* Set GID on execution */
#define	TSVTX		01000	/* On directories, restricted deletion flag */
#define	TUREAD		00400	/* Read by owner */
#define	TUWRITE		00200	/* Write by owner special */
#define	TUEXEC		00100	/* Execute/search by owner */
#define	TGREAD		00040	/* Read by group */
#define	TGWRITE		00020	/* Write by group */
#define	TGEXEC		00010	/* Execute/search by group */
#define	TOREAD		00004	/* Read by other */
#define	TOWRITE		00002	/* Write by other */
#define	TOEXEC		00001	/* Execute/search by other */

#define	TALLMODES	07777	/* The low 12 bits mentioned in the standard */

#if	S_ISUID == TSUID && S_ISGID == TSGID && S_ISVTX == TSVTX && \
	S_IRUSR == TUREAD && S_IWUSR == TUWRITE && S_IXUSR == TUEXEC && \
	S_IRGRP == TGREAD && S_IWGRP == TGWRITE && S_IXGRP == TGEXEC && \
	S_IROTH == TOREAD && S_IWOTH == TOWRITE && S_IXOTH == TOEXEC

#define	HAVE_POSIX_MODE_BITS	/* st_mode bits are equal to TAR mode bits */
#endif

#ifdef	HAVE_POSIX_MODE_BITS	/* st_mode bits are equal to TAR mode bits */
#define	OSMODE(xmode)	    (xmode)
#else
#define	OSMODE(xmode)	    ((xmode & TSUID   ? S_ISUID : 0)  \
			    | (xmode & TSGID   ? S_ISGID : 0) \
			    | (xmode & TSVTX   ? S_ISVTX : 0) \
			    | (xmode & TUREAD  ? S_IRUSR : 0) \
			    | (xmode & TUWRITE ? S_IWUSR : 0) \
			    | (xmode & TUEXEC  ? S_IXUSR : 0) \
			    | (xmode & TGREAD  ? S_IRGRP : 0) \
			    | (xmode & TGWRITE ? S_IWGRP : 0) \
			    | (xmode & TGEXEC  ? S_IXGRP : 0) \
			    | (xmode & TOREAD  ? S_IROTH : 0) \
			    | (xmode & TOWRITE ? S_IWOTH : 0) \
			    | (xmode & TOEXEC  ? S_IXOTH : 0))
#endif

LOCAL void
getperm(t)
	node	*t;
{
	char	*p;
	Llong	ll;
	mode_t	mm;

	p = t->left;
	if (*p == '-')
		p++;

	if (*p >= '0' && *p <= '7') {
		p = astollb(p, &ll, 8);
		if (*p) {
			comerrno(EX_BAD,
				"Non octal character '%c' in '-%s %s'.\n",
				*p, tokennames[t->op], t->left);
		}
		mm = ll & TALLMODES;
		t->val.mode = OSMODE(mm);
		return;
	}
	p = getsperm(p, t);
	if (p && *p != '\0') {
		comerrno(EX_BAD,
			"Bad perm character '%c' found in '-%s %s'.\n",
			*p, tokennames[t->op], t->left);
	}
#ifdef	PERM_DEBUG
	error("GETPERM (%c) %4.4lo\n", *t->left, (long)t->val.mode);
#endif
}

LOCAL char *
getsperm(p, t)
	char	*p;
	node	*t;
{
	mode_t	permval = 0;	/* POSIX start value for "find" */
	mode_t	who;
	mode_t	m;
	int	op;
	mode_t	perms;
	mode_t	pm;

nextclause:
#ifdef	PERM_DEBUG
	error("getsperm(%s)\n", p);
#endif
	who = 0;
	while ((m = iswho(*p)) != 0) {
		p++;
		who |= m;
	}
	if (who == 0) {
		mode_t	mask = umask(0);

		umask(mask);
		who = ~mask;
		who &= (S_ISUID|S_ISGID|S_ISVTX|S_IRWXU|S_IRWXG|S_IRWXO);
	}
#ifdef	PERM_DEBUG
	error("WHO %4.4lo\n", (long)who);
	error("getsperm(--->%s)\n", p);
#endif

nextop:
	if ((op = isop(*p)) != '\0')
		p++;
#ifdef	PERM_DEBUG
	error("op '%c'\n", op);
	error("getsperm(--->%s)\n", p);
#endif
	if (op == 0) {
		errmsgno(EX_BAD, "Missing -perm op.\n");
		return (p);
	}

	perms = 0;
	while ((pm = isperm(*p)) != 0) {
		p++;
		perms |= pm;
	}
#ifdef	PERM_DEBUG
	error("PERM %4.4lo\n", (long)perms);
	error("START PERMVAL %4.4lo\n", (long)permval);
#endif
	if ((perms == 0) && (*p == 'u' || *p == 'g' || *p == 'o')) {
		mode_t	what = 0;

		/*
		 * First select the bit triple we like to copy.
		 */
		switch (*p) {

		case 'u':
			what = permval & S_IRWXU;
			break;
		case 'g':
			what = permval & S_IRWXG;
			break;
		case 'o':
			what = permval & S_IRWXO;
			break;
		}
		/*
		 * Now copy over bit by bit without relying on shifts
		 * that would make implicit assumptions on values.
		 */
		if (what & (S_IRUSR|S_IRGRP|S_IROTH))
			perms |= (who & (S_IRUSR|S_IRGRP|S_IROTH));
		if (what & (S_IWUSR|S_IWGRP|S_IWOTH))
			perms |= (who & (S_IWUSR|S_IWGRP|S_IWOTH));
		if (what & (S_IXUSR|S_IXGRP|S_IXOTH))
			perms |= (who & (S_IXUSR|S_IXGRP|S_IXOTH));
		p++;
	}
#ifdef	PERM_DEBUG
	error("getsperm(--->%s)\n", p);
#endif
	switch (op) {

	case '=':
		permval &= ~who;
		/* FALLTHRU */
	case '+':
		permval |= (who & perms);
		break;

	case '-':
		permval &= ~(who & perms);
		break;
	}
#ifdef	PERM_DEBUG
	error("END PERMVAL %4.4lo\n", (long)permval);
#endif
	if (isop(*p))
		goto nextop;
	if (*p == ',') {
		p++;
		goto nextclause;
	}
	t->val.mode = permval;
	return (p);
}

LOCAL mode_t
iswho(c)
	int	c;
{
	switch (c) {

	case 'u':					/* user */
		return (S_ISUID|S_ISVTX|S_IRWXU);
	case 'g':					/* group */
		return (S_ISGID|S_ISVTX|S_IRWXG);
	case 'o':					/* other */
		return (S_ISVTX|S_IRWXO);
	case 'a':					/* all */
		return (S_ISUID|S_ISGID|S_ISVTX|S_IRWXU|S_IRWXG|S_IRWXO);
	default:
		return (0);
	}
}

LOCAL int
isop(c)
	int	c;
{
	switch (c) {

	case '+':
	case '-':
	case '=':
		return (c);
	default:
		return (0);
	}
}

LOCAL mode_t
isperm(c)
	int	c;
{
	switch (c) {

	case 'r':
		return (S_IRUSR|S_IRGRP|S_IROTH);
	case 'w':
		return (S_IWUSR|S_IWGRP|S_IWOTH);
	case 'x':
#ifdef	__NOT_FIND__
	case 'X':
#endif
		return (S_IXUSR|S_IXGRP|S_IXOTH);
	case 's':
		return (S_ISUID|S_ISGID);
	case 'l':
		return (S_ISGID);
	case 't':
		return (S_ISVTX);
	default:
		return (0);
	}
}


LOCAL int
walkfunc(nm, fs, type, state)
	char		*nm;
	struct stat	*fs;
	int		type;
	struct WALK	*state;
{
	if (type == WALK_NS) {
		errmsg("Cannot stat '%s'.\n", nm);
		err = 1;
		return (0);
	} else if (type == WALK_SLN && (walkflags & WALK_PHYS) == 0) {
		errmsg("Cannot follow symlink '%s'.\n", nm);
		err = 1;
		return (0);
	} else if (type == WALK_DNR) {
		if (state->flags & WALK_NOCHDIR)
			errmsg("Cannot chdir to '%s'.\n", nm);
		else
			errmsg("Cannot read '%s'.\n", nm);
		err = 1;
		return (0);
	}
	expr(nm, nm + state->base, fs, state, Tree);
	return (0);
}

LOCAL inline BOOL
expr(f, ff, fs, state, t)
	char		*f;
	char		*ff;
	struct stat	*fs;
	struct WALK	*state;
	node		*t;
{
	time_t	xtime;
	char	*p;
	char	lname[8192];

	if (!have_print) {
		have_print = TRUE;
		if (t == 0 || expr(f, ff, fs, state, t)) {
			filewrite(stdout, f, strlen(f)); putchar('\n');
		}
		have_print = FALSE;
		return (TRUE);
	}

	switch (t->op) {

	case LNAME: {
		int	lsize;

		if (!S_ISLNK(fs->st_mode))
			return (FALSE);

		lname[0] = '\0';
		lsize = readlink(state->level ? ff : f, lname, sizeof (lname));
		if (lsize < 0) {
			errmsg("Cannot read link '%s'.\n", ff);
			return (FALSE);
		}
		lname[sizeof (lname)-1] = '\0';
		if (lsize >= 0)
			lname[lsize] = '\0';
		p = lname;
		goto nmatch;
	}
	case PATH:
		p = f;
		goto nmatch;
	case NAME:
		p = ff;
	nmatch:
#if	defined(HAVE_FNMATCH_H) & defined(HAVE_FNMATCH)
		return (!fnmatch(t->this, p, 0));
#else
		goto pattern;		/* Use patmatch() as "fallback" */
#endif

	case LPAT: {
		int	lsize;

		if (!S_ISLNK(fs->st_mode))
			return (FALSE);

		lname[0] = '\0';
		lsize = readlink(state->level ? ff : f, lname, sizeof (lname));
		if (lsize < 0) {
			errmsg("Cannot read link '%s'.\n", ff);
			return (FALSE);
		}
		lname[sizeof (lname)-1] = '\0';
		if (lsize >= 0)
			lname[lsize] = '\0';
		p = lname;
		goto pattern;
	}
	case PPAT:
		p = f;
		goto pattern;
	case PAT:
		p = ff;
	pattern: {
		Uchar	*pr;		/* patmatch() return */

		pr = patmatch((Uchar *)t->this, (int *)t->right,
			(Uchar *)p, 0, strlen(p), t->val.i, pstate);
		return (*p && pr && (*pr == '\0'));
	}

	case SIZE:
		switch (*(t->left)) {
		case '+':
			if (t->this)
				return (fs->st_size    > t->val.size);
			return ((fs->st_size+511)/512  > t->val.size);
		case '-':
			if (t->this)
				return (fs->st_size   <  t->val.size);
			return ((fs->st_size+511)/512 <  t->val.size);
		default:
			if (t->this)
				return (fs->st_size   == t->val.size);
			return ((fs->st_size+511)/512 == t->val.size);
		}

	case LINKS:
		switch (*(t->left)) {
		case '+':
			return (fs->st_nlink  > t->val.nlink);
		case '-':
			return (fs->st_nlink <  t->val.nlink);
		default:
			return (fs->st_nlink == t->val.nlink);
		}

	case INUM:
		switch (*(t->left)) {
		case '+':
			return (fs->st_ino  > t->val.ino);
		case '-':
			return (fs->st_ino <  t->val.ino);
		default:
			return (fs->st_ino == t->val.ino);
		}

	case ATIME:
		xtime = fs->st_atime;
		goto times;
	case CTIME:
		xtime = fs->st_ctime;
		goto times;
	case MTIME:
	case TIME:
		xtime = fs->st_mtime;
	times:
		if (t->val2 != 0)
			goto timex;

		switch (*(t->left)) {
		case '+':
			return ((now-xtime)/DAYSECS >  t->val.time);
		case '-':
			return ((now-xtime)/DAYSECS <  t->val.time);
		default:
			return ((now-xtime)/DAYSECS == t->val.time);
		}
	timex:
		switch (*(t->left)) {
		case '+':
			return ((now-xtime) >  t->val.time);
		case '-':
			return ((now-xtime) <  t->val.time);
		default:
			return ((now-xtime) == t->val.time);
		}

	case NEWER:
		return (t->val.time < fs->st_mtime);

	case TYPE:
		switch (*(t->this)) {
		case 'b':
			return (S_ISBLK(fs->st_mode));
		case 'c':
			return (S_ISCHR(fs->st_mode));
		case 'd':
			return (S_ISDIR(fs->st_mode));
		case 'D':
			return (S_ISDOOR(fs->st_mode));
		case 'e':
			return (S_ISEVC(fs->st_mode));
		case 'f':
			return (S_ISREG(fs->st_mode));
		case 'l':
			return (S_ISLNK(fs->st_mode));
		case 'p':
			return (S_ISFIFO(fs->st_mode));
		case 's':
			return (S_ISSOCK(fs->st_mode));
		default:
			return (FALSE);
		}

	case FSTYPE:
#ifdef	HAVE_ST_FSTYPE
		return (streql(t->this, fs->st_fstype));
#else
		return (TRUE);
#endif

	case LOCL:
#ifdef	HAVE_ST_FSTYPE
		if (streql("nfs", fs->st_fstype) ||
		    streql("autofs", fs->st_fstype) ||
		    streql("cachefs", fs->st_fstype))
			return (FALSE);
#endif
		return (TRUE);

	case USER:
		switch (*(t->left)) {
		case '+':
			return (fs->st_uid  > t->val.uid);
		case '-':
			return (fs->st_uid <  t->val.uid);
		default:
			return (fs->st_uid == t->val.uid);
		}

	case GROUP:
		switch (*(t->left)) {
		case '+':
			return (fs->st_gid  > t->val.gid);
		case '-':
			return (fs->st_gid <  t->val.gid);
		default:
			return (fs->st_gid == t->val.gid);
		}

	case PERM:
		if (t->left[0] == '-')
			return ((fs->st_mode & t->val.mode) == t->val.mode);
		else
			return ((fs->st_mode & 07777) == t->val.mode);

	case MODE:
		return (TRUE);

	case XDEV:
	case MOUNT:
	case DEPTH:
	case FOLLOW:
		return (TRUE);

	case NOUSER:
/*		return (nameuid(NULL, 32, fs->st_uid));*/
		return (getpwuid(fs->st_uid) == NULL);

	case NOGRP:
/*		return (namegid(NULL, 32, fs->st_gid));*/
		return (getgrgid(fs->st_gid) == NULL);

	case PRUNE:
		state->flags |= WALK_PRUNE;
		return (TRUE);

	case OK_EXEC: {
		char qbuf[32];

		flush();
		fprintf(stderr, "< %s ... %s > ? ", ((char **)t->this)[0], f);
		fflush(stderr);
		getline(qbuf, sizeof (qbuf) - 1);

		switch (qbuf[0]) {
		case 'y':
			if (qbuf[1] == '\0' || streql(qbuf, "yes")) break;
		default:
			return (FALSE);
		}
	}
	/* FALLTHRU */

	case EXEC:
		return (doexec(f, t->val.i, (char **)t->this));

	case EXECPLUS:
		return (plusexec(f, t));

	case PRINT:
		filewrite(stdout, f, strlen(f)); putchar('\n');
		return (TRUE);

	case PRINTNNL:
		filewrite(stdout, f, strlen(f)); putchar(' ');
		return (TRUE);

	case LS:
		/*
		 * The third parameter is the file name used for readlink()
		 * (inside list()) relatively to the current working directory.
		 * For file names from the command line, we did not perform a
		 * chdir() before, so we need to use the full path name.
		 */
		list(fs, f, state->level ? ff : f);
		return (TRUE);

	case OPEN:
		return (expr(f, ff, fs, state, (node *)t->this));
	case LNOT:
		return (!expr(f, ff, fs, state, (node *)t->this));
	case AND:
		return (expr(f, ff, fs, state, (node *)t->left) ? expr(f, ff, fs, state, (node *)t->right) : 0);
	case LOR:
		return (expr(f, ff, fs, state, (node *)t->left) ? 1 : expr(f, ff, fs, state, (node *)t->right));
	}
	return (FALSE);		/* Unknown operator ??? */
}

LOCAL BOOL
doexec(f, ac, av)
	char	*f;
	int	ac;
	char	**av;
{
	pid_t	pid;
	int	retval;

	if ((pid = fork()) < 0)
		comerr("Cannot fork.\n");
	if (pid) {
		while (wait(&retval) != pid)
			/* LINTED */
			;
		return (retval == 0);
	} else {
		register int	i;
		register char	**pp = av;

#ifdef	HAVE_FCHDIR
		if (fchdir(FHome) < 0)
			comerr("Can not chdir to '.'.\n");
#ifndef	F_SETFD
		close(FHome);	/* close() on exec() is not set */
#endif
#else
		if (chdir(FHome) < 0)
			comerr("Can not chdir to '%s'.\n",
				FHome);
#endif
		if (f) {
			for (i = 0; i < ac; i++, pp++) {
				if (*pp == NULL)
					*pp = f;
			}
		}
		fexecv(av[0], stdin, stdout, stderr, ac, av);
#ifdef	PLUS_DEBUG
		error("argsize %d\n",
			(plusp->endp - (char *)&plusp->nextarg[0]) -
			(plusp->laststr - (char *)&plusp->nextarg[1]));
#endif
		comerr("Cannot execute '%s'.\n", av[0]);
		/* NOTREACHED */
		return (-1);
	}
}

LOCAL int
argsize()
{
	static int	ret = 0;
	extern char	**environ;

	if (ret == 0) {
		register int	evs = 0;
		register char	**ep;

		for (ep = environ; *ep; ep++) {
			evs += strlen(*ep) + 1 + sizeof (ep);
		}
		evs += sizeof (char **); /* The environ NULL ptr at the end */

#ifdef	_SC_ARG_MAX
		ret = sysconf(_SC_ARG_MAX);
		if (ret < 0)
			ret = _POSIX_ARG_MAX;
#else
#ifdef	ARG_MAX
		ret = ARG_MAX;
#else
#ifdef	NCARGS
		ret = NCARGS;
#endif
#endif
#endif
		if (ret <= 0)
			ret = 10000;	/* Just a guess */

		ret -= evs;
	}
	return (ret);
}

LOCAL void
pluscreate(ac, av)
	int	ac;
	char	**av;
{
	struct plusargs	*pp;
	register char	**ap = av;
	register int	i;

	for (i = ac; --i >= 0; ap++) {	/* Restore marked "{}" args */
		if (*ap == NULL)
			*ap = "{}";
	}

#ifdef	PLUS_DEBUG
	printf("Arvc %d\n", ac);
	ap = av;
	for (i = 0; i < ac; i++, ap++)
		printf("ARG %d '%s'\n", i, *ap);
#endif

	pp = __malloc(argsize() + sizeof (struct plusargs), "-exec args");
	pp->laststr = pp->endp = (char *)(&pp->av[0]) + argsize();
	pp->ac = 0;
	pp->nextarg = &pp->av[0];

#ifdef	PLUS_DEBUG
	printf("pp          %d\n", pp);
	printf("pp->laststr %d\n", pp->laststr);
#endif

	/*
	 * Copy args from command line.
	 */
	ap = av;
	for (i = 0; i < ac; i++, ap++) {
#ifdef	PLUS_DEBUG
		printf("ARG %d '%s'\n", i, *ap);
#endif
		*(pp->nextarg++) = *ap;
		pp->laststr -= strlen(*ap) + 1;
		pp->ac++;
		if (pp->laststr <= (char *)pp->nextarg)
			comerrno(EX_BAD, "No space for exec args.\n");

	}

#ifdef	PLUS_DEBUG
	ap = &pp->av[0];
	for (i = 0; i < pp->ac; i++, ap++) {
		printf("ARG %d '%s'\n", i, *ap);
	}
#endif
	pp->next = plusp;
	plusp = pp;
}

LOCAL BOOL
plusexec(f, t)
	char	*f;
	node	*t;
{
	register struct plusargs *pp = (struct plusargs *)t->this;
#ifdef	PLUS_DEBUG
	register char	**ap;
	register int	i;
#endif
	int	size;
	int	slen = strlen(f) + 1;
	char	*cp;
	int	ret = TRUE;

	size = pp->laststr - (char *)&pp->nextarg[2];

	if (slen > size) {
		pp->nextarg[0] = NULL;
		ret = doexec(NULL, pp->ac, pp->av);
		pp->laststr = pp->endp;
		pp->ac = t->val.i;
		pp->nextarg = &pp->av[t->val.i];
		size = pp->laststr - (char *)&pp->nextarg[2];
	}
	if (slen > size)
		comerrno(EX_BAD, "No space for arg '%s'.\n", f);

	cp = pp->laststr - slen;
	strcpy(cp, f);
	pp->nextarg[0] = cp;
	pp->ac++;
	pp->nextarg++;
	pp->laststr -= slen;

#ifdef	PLUS_DEBUG
	ap = &plusp->av[0];
	for (i = 0; i < plusp->ac; i++, ap++) {
		printf("ARG %d '%s'\n", i, *ap);
	}
	error("EXECPLUS '%s'\n", f);
#endif
	return (ret);
}

LOCAL void
usage(ret)
	int	ret;
{
	error("Usage:	%s [options] [path_1 ... path_n] [expression]\n", get_progname());
	error("Options:\n");
	error("	-H	follow symbolics links from command line\n");
	error("	-L	follow all symbolic links\n");
	error("*	-P	do not follow symbolic links (default)\n");
	error("*	-help	Print this help.\n");
	error("*	-version Print version number.\n");
	error("Operators in decreasing precedence:\n");
	error("	( )	group an expression\n");
	error("	! -a -o	negate a primary (unary NOT), logical AND, logical OR\n");
	error("Primaries:\n");
	error("	-atime #      TRUE if st_atime is in specified range\n");
	error("	-ctime #      TRUE if st_ctime is in specified range\n");
	error("	-depth	      evaluate dir content before dir (always TRUE)\n");
	error("	-exec program [argument ...] \\;\n");
	error("	-exec program [argument ...] {} +\n");
	error("*	-follow	      outdated: follow all symbolic links (always TRUE)\n");
	error("*	-fstype type  TRUE if st_fstype matches type\n");
	error("	-group gname/gid TRUE if st_gid matches gname/gid\n");
	error("*	-inum #	      TRUE is st_ino is in specified range\n");
	error("	-links #      TRUE is st_nlink is in specified range\n");
	error("*	-lname glob   TRUE if symlink name matches shell glob\n");
	error("*	-local	      TRUE if st_fstype does not matche remote fs types\n");
	error("*	-lpat pattern TRUE if symlink name matches pattern\n");
	error("*	-ls	      list files similar to 'ls -ilds' (always TRUE)\n");
	error("	-mtime #      TRUE if st_mtime is in specified range\n");
	error("	-name glob    TRUE if path component matches shell glob\n");
	error("	-newer file   TRUE if st_mtime newer then mtime of file\n");
	error("	-nogroup      TRUE if not in group database\n");
	error("	-nouser       TRUE if not in user database\n");
	error("	-ok program [argument ...] \\;\n");
	error("*	-pat pattern  TRUE if path component matches pattern\n");
	error("*	-path glob    TRUE if full path matches shell glob\n");
	error("	-perm mode/onum TRUE if symbolic/octal permission matches\n");
	error("*	-ppat pattern TRUE if full path matches pattern\n");
	error("	-print	      print file names line separated to stdout (always TRUE)\n");
	error("*	-printnnl     print file names space separated to stdout (always TRUE)\n");
	error("	-prune	      do not descent current directory (always TRUE)\n");
	error("	-size #	      TRUE if st_size is in specified range\n");
	error("	-type c	      TRUE if file type matches, c is from (b c d D e f l p s)\n");
	error("	-user uname/uid TRUE if st_uid matches uname/uid\n");
	error("	-xdev, -mount restrict search to current filesystem (always TRUE)\n");
	error("Primaries marked with '*' are POSIX extensions.\n");
	error("If path is omitted, '.' is used. If expression is omitted, -print is used.\n");
	exit(ret);
}

/* ARGSUSED */
LOCAL int
getflg(optstr, argp)
	char	*optstr;
	long	*argp;
{
/*	error("optstr: '%s'\n", optstr);*/

	if (optstr[1] != '\0')
		return (-1);

	switch (*optstr) {

	case 'H':
		walkflags |= WALK_ARGFOLLOW;
		return (TRUE);
	case 'L':
		walkflags |= WALK_ALLFOLLOW;
		return (TRUE);
	case 'P':
		walkflags &= ~(WALK_ARGFOLLOW | WALK_ALLFOLLOW);
		return (TRUE);

	default:
		return (-1);
	}
}

EXPORT int
main(ac, av)
	int	ac;
	char	**av;
{
	int	cac  = ac;
	char *	*cav = av;
	char *	*firstpath;
	char *	*firstprim;
	char	c;
	BOOL	help = FALSE;
	BOOL	prversion = FALSE;

	save_args(ac, av);

	cac--, cav++;
	getargs(&cac, (char * const **)&cav, "help,version,&",
			&help, &prversion,
			getflg, NULL);
	if (help) {
		usage(0);
	}
	if (prversion) {
		printf("sfind release %s (%s-%s-%s) Copyright (C) 2004 Jrg Schilling\n",
				strvers,
				HOST_CPU, HOST_VENDOR, HOST_OS);
		exit(0);
	}

	firstpath = cav;	/* Remember first file type arg */
	while (cac > 0 &&
		(c = **cav) != '-' && c != '(' && c != ')' && c != '!') {
		cav++;
		cac--;
	}
	firstprim = cav;	/* Remember first Primary type arg */
	Argv = cav;
	Argc = cac;

	if (cac) {
		nexttoken();
		Tree = parse();
		if (primtype != ENDARGS)
			comerrno(EX_BAD, "Incomplete expression.\n");
	} else {
		Tree = 0;
	}
	if (patlen > 0)
		pstate = __malloc(sizeof (int)*patlen, "space for pattern state");

#ifdef	HAVE_FCHDIR
	if ((FHome = open(".", 0)) < 0)
		comerr("Cannot get working directory\n");
#ifdef	F_SETFD
	fcntl(FHome, F_SETFD, 1);
#endif
#else
	if (getcwd(FHome, sizeof (FHome)) == NULL)
		comerr("Cannot get working directory\n");
#endif
	now	 = time(0);
	sixmonth = now - 6L*30L*24L*60L*60L;
	now	 = now + 60;

	if (firstpath == firstprim) {
		treewalk(".", walkfunc, walkflags);
	} else {
		for (cav = firstpath; cav != firstprim; cav++) {
			treewalk(*cav, walkfunc, walkflags);
			/*
			 * XXX hier break wenn treewalk() Fehler gemeldet
			 */
			if (cav != firstprim &&
#ifdef	HAVE_FCHDIR
			    fchdir(FHome) < 0) {
				comerr("Cannot change directory back to '.'.\n");
#else
			    chdir(FHome) < 0) {
				comerr("Cannot change directory back to '%s'.\n",
					FHome);
#endif
			}
		}
	}
	/*
	 * Execute all unflushed '-exec .... {} +' expressions.
	 */
	while (plusp) {
		if (plusp->laststr != plusp->endp) {
			plusp->nextarg[0] = NULL;
			doexec(NULL, plusp->ac, plusp->av);
		}
		plusp = plusp->next;
	}
	return (err);
}
