/* * This program reads various mode pages and bits of other
 * information from a scsi device and interprets the raw data for you
 * with a report written to stdout.  Usage:
 *
 * ./sginfo [options] /dev/sg2 [replace parameters]
 *
 * Options are:
 * -6    Do 6 byte mode sense + select (deafult: 10 byte)
 * -a    Display all recognised mode pages: equivalent to '-cCDefgiInpPsV'.
 * -c    Display information from Cache control page.
 * -C    Display information from Control Page.
 * -d    Display defect lists (default format: index).
 * -D    Display information from disconnect-reconnect page.
 * -e    Display information from error recovery page.
 * -f    Display information from Format Device Page.
 * -Farg Defect list format (-Flogical, -Fphysical, -Findex)
 * -g    Display information from Hard disk geometry page.
 * -G    Display only "grown" defect list (default format: index)
 * -i    Display all information from Inquiry command.
 * -I    Display information from Informational Exceptions page.
 * -l    List known scsi devices on the system
 * -L    List pages supported by this program and target device
 * -n    Display information from notch parameters page.
 * -P    Display information from Power Condition Page.
 * -s    Display all information from unit serial number page.
 * -u <n[,spn]> Display info from page number <n> and subpage <spn>.
 * -v    Show version number
 * -V    Display information from Verify Error Recovery Page.
 * -T    Trace commands (for debugging, double for more debug)
 *
 * Only one of the following three options can be specified.
 * None of these three implies the current values are returned.
 * -m    Display modifiable fields instead of current values
 * -M    Display manufacturer defaults instead of current values
 * -S    Display saved defaults instead of current values
 *
 * -X    Display output values in a list.
 * -R    Replace parameters - best used with -X
 *
 * Eric Youngdale - 11/1/93.  Version 1.0.
 *
 * Version 1.1: Ability to change parameters on cache page, support for
 *  X front end.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 *
 * Michael Weller (eowmob@exp-math.uni-essen.de) 
 *      11/23/94 massive extensions from 1.4a
 *      08/23/97 fix problems with defect lists
 *
 * Douglas Gilbert (dgilbert@interlog.com) 
 *      990628   port to sg .... (version 1.81)
 *               up 4KB limit on defect list to 32KB
 *               'sginfo -l' also shows sg devices and mapping to other
 *                    scsi devices
 *               'sginfo' commands can take either an sd, sr (scd), st
 *                    or an sg device (all non-sg devices converted to a
 *                    sg device)
 *
 *      001208   Add Kurt Garloff's "-uno" flag for displaying info
 *               from a page number. <garloff@suse.de> [version 1.90]
 *
 * Kurt Garloff <garloff@suse.de>
 *    20000715  allow displaying and modification of vendor specific pages
 *                      (unformatted - @ hexdatafield)
 *              accept vendor lengths for those pages
 *              enabled page saving
 *              cleaned parameter parsing a bit (it's still a terrible mess!)
 *              Use sr (instead of scd) and sg%d (instead of sga,b,...) in -l
 *                      and support much more devs in -l (incl. nosst)
 *              Fix segfault in defect list (len=0xffff) and adapt formatting 
 *                      to large disks. Support up to 256kB defect lists with
 *                      0xB7 (12byte) command if necessary and fallback to 0x37 
 *                      (10byte) in case of failure. Report truncation.
 *              sizeof(buffer) (which is sizeof(char*) == 4 or 32 bit archs) 
 *                      was used incorrectly all over the place. Fixed.
 *                                      [version 1.95]
 * Douglas Gilbert (dgilbert@interlog.com) 
 *    20020113  snprintf() type cleanup [version 1.96]
 *    20021211  correct sginfo MODE_SELECT, protect against block devices
 *              that answer sg's ioctls. [version 1.97]
 *    20021228  scan for some "scd<n>" as well as "sr<n>" device names [1.98]
 *    20021020  Update control page [1.99]
 */

static const char * sginfo_version_str = "Sginfo version 2.00 [20031107]";

#include <stdio.h>
#include <string.h>
#include <getopt.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "sg_include.h"

#include "sg_err.h"



static int glob_fd;
static char *device_name;

#define MAX_RESP6_SIZE 252
#define MAX_RESP10_SIZE (4*1024)
#define MAX_BUFFER_SIZE MAX_RESP10_SIZE
#define SIZEOF_BUFFER (16*1024)
#define SIZEOF_BUFFER1 (16*1024)
static unsigned char buffer[SIZEOF_BUFFER];
static unsigned char buffer1[SIZEOF_BUFFER1];

static char cache = 0;
static char control = 0;
static char default_param = 0;
static char defect = 0;
static char defectformat = 0x4;
static char disconnect = 0;
static char error = 0;
static char format = 0;
static char geometry = 0;
static char grown_defect = 0;
static char inquiry = 0;
static char informational = 0;
static char list = 0;
static char list_pages = 0;
static char modifiable = 0;
static char negate_sp_bit = 0;
static char notch = 0;
static char power = 0;
static char replace = 0;
static char saved = 0;
static char serial_number = 0;
static char verify = 0;
static char x_interface = 0;

static char save_mode = 0;  /* All mode pages output only a single line, and 
                               precede the output with the appropriate 
                               restore command */
static char mode6byte = 0;      /* defaults to 10 byte mode sense + select */
static char trace = 0;

static char *page_names[] =
{
    "Vendor (non-page format)",
    "Read-Write Error Recovery",
    "Disconnect-Reconnect",
    "Format Device",
    "Rigid Disk Geometry",
    "Flexible Disk",
     /* 0x6 */   NULL,
    "Verify Error Recovery",
    "Caching",
    "Peripheral Device",
    /* 0xa */ "Control",
    "Medium Types Supported",
    "Notch and Partition",
    "CD device parameters",
    "CD audio control",
    "Data compression",
    /* 0x10 */ "XOR control (device config[tape])",
    /* 0x11 */ "Medium partition",
        NULL,
        NULL,
        NULL,
    /* 0x15 */ "Extended",
    /* 0x16 */ "Extended, device-type specific",
        /* 0x17 */ NULL,
    /* 0x18 */ "Protocol specific lun",
    /* 0x19 */ "Protocol specific port",
    /* 0x1a */ "Power Condition",
        /* 0x1b */ NULL,
    /* 0x1c */ "Informational Exceptions",
    /* 0x1d */ "Time-out & protect",
    /* 0x1e */ NULL,
    /* 0x1f */ NULL,
    /* 0x20 */ NULL,
    /* 0x21 */ NULL,
    /* 0x22 */ NULL,
    /* 0x23 */ NULL,
    /* 0x24 */ NULL,
    /* 0x25 */ NULL,
    /* 0x26 */ NULL,
    /* 0x27 */ NULL,
    /* 0x28 */ NULL,
    /* 0x29 */ NULL,
    /* 0x2a */ "MM capabilities & mechanical status",
};

#define MAX_PAGENO (sizeof(page_names)/sizeof(char *))

#define MAXPARM 64

static int next_parameter;
static int n_replacement_values;
static unsigned long long replacement_values[MAXPARM];
static char is_hex[MAXPARM];

#define SMODE_SENSE 0x1a
#define SMODE_SENSE_10 0x5a
#define SMODE_SELECT 0x15
#define SMODE_SELECT_10 0x55

#define MP_LIST_PAGES 0x3f
#define MPHEADER6_LEN 4
#define MPHEADER10_LEN 8


/* forward declarations */
static void usage(char *);
static void dump(void *buffer, unsigned int length);

#define DXFER_NONE        0
#define DXFER_FROM_DEVICE 1
#define DXFER_TO_DEVICE   2


struct scsi_cmnd_io
{
    unsigned char * cmnd;       /* ptr to SCSI command block (cdb) */
    size_t  cmnd_len;           /* number of bytes in SCSI command */
    int dxfer_dir;              /* DXFER_NONE, DXFER_FROM_DEVICE, or
                                   DXFER_TO_DEVICE */
    unsigned char * dxferp;     /* ptr to outgoing/incoming data */
    size_t dxfer_len;           /* bytes to be transferred to/from dxferp */
};

#define SENSE_BUFF_LEN   32
#define CMD_TIMEOUT   60000 /* 60,000 milliseconds (60 seconds) */
#define EBUFF_SZ   256


/* Returns 0 -> ok, 1 -> err */
static int do_scsi_io(struct scsi_cmnd_io * sio)
{
    unsigned char sense_b[SENSE_BUFF_LEN];
    struct sg_io_hdr io_hdr;
    int res;

    memset(&io_hdr, 0, sizeof(struct sg_io_hdr));
    io_hdr.interface_id = 'S';
    io_hdr.cmd_len = sio->cmnd_len;
    io_hdr.mx_sb_len = sizeof(sense_b);
    if (DXFER_NONE == sio->dxfer_dir)
        io_hdr.dxfer_direction = SG_DXFER_NONE;
    else
        io_hdr.dxfer_direction = (DXFER_TO_DEVICE == sio->dxfer_dir) ?
                                SG_DXFER_TO_DEV : SG_DXFER_FROM_DEV;
    io_hdr.dxfer_len = sio->dxfer_len;
    io_hdr.dxferp = sio->dxferp;
    io_hdr.cmdp = sio->cmnd;
    io_hdr.sbp = sense_b;
    io_hdr.timeout = CMD_TIMEOUT;

    if (trace) {
        printf("  cbd:");
        dump(sio->cmnd, sio->cmnd_len);
    }
    if ((trace > 1) && (DXFER_TO_DEVICE == sio->dxfer_dir)) {
        printf("  additional data:\n");
        dump(sio->dxferp, sio->dxfer_len);
    }

    if (ioctl(glob_fd, SG_IO, &io_hdr) < 0) {
        perror("do_scsi_cmd: SG_IO error");
        return 1;
    }
    res = sg_err_category3(&io_hdr);
    switch (res) {
    case SG_ERR_CAT_CLEAN:
    case SG_ERR_CAT_RECOVERED:
        return 0;
    default:
        if (trace) {
            char ebuff[EBUFF_SZ];

            snprintf(ebuff, EBUFF_SZ, "do_scsi_io: opcode=0x%x", sio->cmnd[0]);
            sg_chk_n_print3(ebuff, &io_hdr);
        }
        return 1;
    }
}

static char unkn_page_str[32];

static char *get_page_name(int pageno)
{
    if (MP_LIST_PAGES == pageno)
        return "List supported page numbers";
    else if ((pageno < 0) || (pageno >= MAX_PAGENO) || 
             (! page_names[pageno])) {
        snprintf(unkn_page_str, sizeof(unkn_page_str), 
                 "page number=0x%x", pageno);
        return unkn_page_str;
    }
    return page_names[pageno];
}

static void dump(void *buffer, unsigned int length)
{
    unsigned int i;

    printf("    ");
    for (i = 0; i < length; i++) {
#if 0
        if (((unsigned char *) buffer)[i] > 0x20)
            printf(" %c ", (unsigned int) ((unsigned char *) buffer)[i]);
        else
#endif
            printf("%02x ", (unsigned int) ((unsigned char *) buffer)[i]);
        if ((i % 16 == 15) && (i < (length - 1))) {
            printf("\n    ");
        }
    }
    printf("\n");

}

static int getnbyte(unsigned char *pnt, int nbyte)
{
    unsigned int result;
    int i;
    result = 0;
    for (i = 0; i < nbyte; i++)
        result = (result << 8) | (pnt[i] & 0xff);
    return result;
}

static int putnbyte(unsigned char *pnt, unsigned int value, 
                    unsigned int nbyte)
{
    int i;

    for (i = nbyte - 1; i >= 0; i--) {
        pnt[i] = value & 0xff;
        value = value >> 8;
    }
    return 0;
}

#define REASON_SZ 128

static void check_parm_type(int i)
{
    char reason[REASON_SZ];

    if (i == 1 && is_hex[next_parameter] != 1) {
        snprintf(reason, REASON_SZ,
                 "simple number (pos %i) instead of @ hexdatafield: %llu",
                 next_parameter, replacement_values[next_parameter]);
        usage (reason);
    }
    if (i != 1 && is_hex[next_parameter]) {
        snprintf(reason, REASON_SZ,
                 "@ hexdatafield (pos %i) instead of a simple number: %llu",
                 next_parameter, replacement_values[next_parameter]);
        usage (reason);
    }
}

static void bitfield(unsigned char *pageaddr, char * text, int mask, int shift)
{
    if (x_interface && replace) {
        check_parm_type(0);
        *pageaddr = (*pageaddr & ~(mask << shift)) |
            ((replacement_values[next_parameter++] & mask) << shift);
    } else if (x_interface)
        printf("%d ", (*pageaddr >> shift) & mask);
    else
        printf("%-35s%d\n", text, (*pageaddr >> shift) & mask);
}

#if 0
static void notbitfield(unsigned char *pageaddr, char * text, int mask, 
                        int shift)
{
    if (modifiable) {
        bitfield(pageaddr, text, mask, shift);
        return;
    }
    if (x_interface && replace) {
        check_parm_type(0);
        *pageaddr = (*pageaddr & ~(mask << shift)) |
            (((!replacement_values[next_parameter++]) & mask) << shift);
    } else if (x_interface)
        printf("%d ", !((*pageaddr >> shift) & mask));
    else
        printf("%-35s%d\n", text, !((*pageaddr >> shift) & mask));
}
#endif

static void intfield(unsigned char * pageaddr, int nbytes, char * text)
{
    if (x_interface && replace) {
        check_parm_type(0);
        putnbyte(pageaddr, replacement_values[next_parameter++], nbytes);
    } else if (x_interface)
        printf("%d ", getnbyte(pageaddr, nbytes));
    else
        printf("%-35s%d\n", text, getnbyte(pageaddr, nbytes));
}

static void hexfield(unsigned char * pageaddr, int nbytes, char * text)
{
    if (x_interface && replace) {
        check_parm_type(0);
        putnbyte(pageaddr, replacement_values[next_parameter++], nbytes);
    } else if (x_interface)
        printf("%d ", getnbyte(pageaddr, nbytes));
    else
        printf("%-35s0x%x\n", text, getnbyte(pageaddr, nbytes));
}

static void hexdatafield(unsigned char * pageaddr, int nbytes, char * text)
{
    if (x_interface && replace) {
        unsigned char *ptr;
        unsigned tmp;

        /* Though in main we ensured that a @string has the right format, 
           we have to check that we are working on a @ hexdata field */

        check_parm_type(1);

        ptr = (unsigned char *) (unsigned long) 
              (replacement_values[next_parameter++]);
        ptr++;                  /* Skip @ */

        while (*ptr) {
            if (!nbytes)
                goto illegal;
            tmp = (*ptr >= 'a') ? (*ptr - 'a' + 'A') : *ptr;
            tmp -= (tmp >= 'A') ? 'A' - 10 : '0';

            *pageaddr = tmp << 4;
            ptr++;

            tmp = (*ptr >= 'a') ? (*ptr - 'a' + 'A') : *ptr;
            tmp -= (tmp >= 'A') ? 'A' - 10 : '0';

            *pageaddr++ += tmp;
            ptr++;
            nbytes--;
        }

        if (nbytes) {
          illegal:
            fputs("sginfo: incorrect number of bytes in @hexdatafield.\n",
                  stdout);
            exit(2);
        }
    } else if (x_interface) {
        putchar('@');
        while (nbytes-- > 0)
            printf("%02x", *pageaddr++);
        putchar(' ');
    } else {
        printf("%-35s0x", text);
        while (nbytes-- > 0)
            printf("%02x", *pageaddr++);
        putchar('\n');
    }
}


/* Offset into mode sense (6 or 10 byte) response that actual mode page
 * starts at (relative to resp[0]). Returns -1 if problem */
static int modePageOffset(const unsigned char * resp, int len, 
                          int modese_6, int * mdata_lenp)
{
    int bd_len;
    int resp_len = 0;
    int offset = -1;

    if (resp) {
        if (modese_6) {
            resp_len = resp[0] + 1;
            bd_len = resp[3];
            offset = bd_len + MPHEADER6_LEN;
        } else {
            resp_len = (resp[0] << 8) + resp[1] + 2;
            bd_len = (resp[6] << 8) + resp[7];
            /* LongLBA doesn't change this calculation */
            offset = bd_len + MPHEADER10_LEN; 
        }
        if ((offset + 2) > len) {
            printf("modePageOffset: raw_curr too small, offset=%d "
                   "resp_len=%d bd_len=%d\n", offset, resp_len, bd_len);
            offset = -1;
        } else if ((offset + 2) > resp_len) {
            printf("modePageOffset: response length too short, resp_len=%d"
                   " offset=%d bd_len=%d\n", resp_len, offset, bd_len);
            offset = -1;
        }
    }
    if ((offset >= 0) && mdata_lenp)
        *mdata_lenp = resp_len;
    return offset;
}

/* Reads mode (sub-)page into 'bufer' array, returns 0 if ok */
static int get_mode_page6(int page, int subpage, int page_control, int dbd,
                          unsigned char * resp)
{
    int status;
    unsigned char cmd[6];
    const int max_resp_sz = MAX_RESP6_SIZE;
    struct scsi_cmnd_io sci;

    memset(resp, 0, max_resp_sz); /* assume resp -> large enough buffer */

    cmd[0] = SMODE_SENSE;       /* MODE SENSE (6) */
    cmd[1] = 0x00 | (dbd ? 0x8 : 0); /* disable block descriptors bit */
    cmd[2] = (page_control << 6) | page;
    cmd[3] = subpage;           /* subpage code */
    cmd[4] = (unsigned char)max_resp_sz;   /* allocation length */
    cmd[5] = 0x00;              /* control */

    sci.cmnd = cmd;
    sci.cmnd_len = sizeof(cmd);
    sci.dxfer_dir = DXFER_FROM_DEVICE;
    sci.dxfer_len = max_resp_sz;
    sci.dxferp = resp;
    status = do_scsi_io(&sci);
    if (status)
        fprintf(stdout, ">>> Unable to read %s mode page (0x%x) [mode_sense_6]\n", 
                get_page_name(page), page);
    else if (trace > 1) {
        int off, len;

        off = modePageOffset(resp, max_resp_sz, 1, &len);
        if (off >= 0) {
            printf("  cdb response:\n");
            dump(resp, len);
        }
    }
    return status;
}

/* Reads mode (sub-)page into resp, returns 0 if ok */
static int get_mode_page10(int page, int subpage, int page_control, int llbaa,
                           int dbd, unsigned char * resp)
{
    int status;
    unsigned char cmd[10];
    const int max_resp_sz = MAX_RESP10_SIZE;
    struct scsi_cmnd_io sci;

    memset(resp, 0, max_resp_sz); /* assume resp -> large enough buffer */

    cmd[0] = SMODE_SENSE_10;     /* MODE SENSE (10) */
    cmd[1] = 0x00 | (llbaa ? 0x10 : 0) | (dbd ? 0x8 : 0);         
    cmd[2] = (page_control << 6) | page;
    cmd[3] = subpage;
    cmd[4] = 0x00;              /* (reserved) */
    cmd[5] = 0x00;              /* (reserved) */
    cmd[6] = 0x00;              /* (reserved) */
    cmd[7] = (max_resp_sz >> 8) & 0xff;              
    cmd[8] = (max_resp_sz & 0xff); 
    cmd[9] = 0x00;              /* control */

    sci.cmnd = cmd;
    sci.cmnd_len = sizeof(cmd);
    sci.dxfer_dir = DXFER_FROM_DEVICE;
    sci.dxfer_len = max_resp_sz;
    sci.dxferp = resp;
    status = do_scsi_io(&sci);
    if (status)
        fprintf(stdout, ">>> Unable to read %s mode page (0x%x) [mode_sense_10]\n",
                get_page_name(page), page);
    else if (trace > 1) {
        int off, len;

        off = modePageOffset(resp, max_resp_sz, 0, &len);
        if (off >= 0) {
            printf("  cdb response:\n");
            dump(resp, len);
        }
    }
    return status;
}

static int get_mode_page(int page, int subpage, int page_control,
                         int dbd, unsigned char * resp)
{
    if (mode6byte)
        return get_mode_page6(page, subpage, page_control, dbd, resp); 
    else
        return get_mode_page10(page, subpage, page_control, 0, dbd, resp); 
}

/* Contents should point to the mode parameter header that we obtained
   in a prior read operation.  This way we do not have to work out the
   format of the beast. Assume 0 or 1 block descriptors. */
static int put_mode_page6(int page, const unsigned char * msense6_resp,
                          int sp_bit)
{
    int status;
    int bdlen, resplen;
    unsigned char cmd[6];
    struct scsi_cmnd_io sci;

    bdlen = msense6_resp[3];
    resplen = msense6_resp[0] + 1;

    cmd[0] = SMODE_SELECT;
    cmd[1] = 0x10 | (sp_bit ? 1 : 0); /* always set PF bit */
    cmd[2] = 0x00;
    cmd[3] = 0x00;              /* (reserved) */
    cmd[4] = resplen;           /* parameter list length */ 
    cmd[5] = 0x00;              /* (reserved) */

    memcpy(buffer1, msense6_resp, resplen);
    buffer1[0] = 0;             /* Mask off the mode data length
                                   - reserved field */
    buffer1[2] = 0;             /* device-specific parameter is not defined
                                   and/or reserved for mode select */

#if 0   /* leave block descriptor alone */
    if (bdlen > 0) {
        memset(buffer1 + MPHEADER6_LEN, 0, 4);  /* clear 'number of blocks'
                                                   for DAD device */
        buffer1[MPHEADER6_LEN + 4] = 0; /* clear DAD density code. Why? */
        /* leave DAD block length */
    }
#endif
    buffer1[MPHEADER6_LEN + bdlen] &= 0x7f;   /* Mask PS bit */

    sci.cmnd = cmd;
    sci.cmnd_len = sizeof(cmd);
    sci.dxfer_dir = DXFER_TO_DEVICE;
    sci.dxfer_len = resplen;
    sci.dxferp = buffer1;
    status = do_scsi_io(&sci);
    if (status)
        fprintf(stdout, ">>> Unable to store %s mode page (0x%x) [msel_6]\n",
                get_page_name(page), page);
    return status;
}

/* Contents should point to the mode parameter header that we obtained
   in a prior read operation.  This way we do not have to work out the
   format of the beast. Assume 0 or 1 block descriptors. */
static int put_mode_page10(int page, const unsigned char * msense10_resp,
                           int sp_bit)
{
    int status;
    int bdlen, resplen;
    unsigned char cmd[10];
    struct scsi_cmnd_io sci;

    bdlen = (msense10_resp[6] << 8) + msense10_resp[7];
    resplen = (msense10_resp[0] << 8) + msense10_resp[1] + 2;

    cmd[0] = SMODE_SELECT_10;
    cmd[1] = 0x10 | (sp_bit ? 1 : 0); /* always set PF bit */
    cmd[2] = 0x00;              /* (reserved) */
    cmd[3] = 0x00;              /* (reserved) */
    cmd[4] = 0x00;              /* (reserved) */
    cmd[5] = 0x00;              /* (reserved) */
    cmd[6] = 0x00;              /* (reserved) */
    cmd[7] = (resplen >> 8) & 0xff;
    cmd[8] = resplen & 0xff;
    cmd[9] = 0x00;              /* (reserved) */

    memcpy(buffer1, msense10_resp, resplen);
    buffer1[0] = 0;             /* Mask off the mode data length */
    buffer1[1] = 0;             /* Mask off the mode data length */
    buffer1[3] = 0;             /* device-specific parameter is not defined
                                   and/or reserved for mode select */
#if 0   /* leave block descriptor alone */
    if (bdlen > 0) {
        memset(buffer1 + MPHEADER10_LEN, 0, 4);  /* clear 'number of blocks'
                                                    for DAD device */
        buffer1[MPHEADER10_LEN + 4] = 0; /* clear DAD density code. Why? */
        /* leave DAD block length */
    }
#endif
    buffer1[MPHEADER10_LEN + bdlen] &= 0x7f;   /* Mask PS bit */

    sci.cmnd = cmd;
    sci.cmnd_len = sizeof(cmd);
    sci.dxfer_dir = DXFER_TO_DEVICE;
    sci.dxfer_len = resplen;
    sci.dxferp = buffer1;
    status = do_scsi_io(&sci);
    if (status)
        fprintf(stdout, ">>> Unable to store %s mode page (0x%x) [msel_10]\n",
                get_page_name(page), page);
    return status;
}

static int put_mode_page(int page, const unsigned char * msense_resp)
{
    if (mode6byte)
        return put_mode_page6(page, msense_resp, ! negate_sp_bit); 
    else
        return put_mode_page10(page, msense_resp, ! negate_sp_bit); 
}

int setup_mode_page(int npage, int subpage, int nparam, int page_control, 
                    unsigned char * buff, unsigned char ** o_pagestart)
{
    int status, offset;
  
    status = get_mode_page(npage, subpage, page_control, 0, buff); 
    if (status) {
        printf("\n");
        return status;
    }
    offset = modePageOffset(buff, MAX_BUFFER_SIZE, mode6byte, NULL);
    if (offset < 0) {
        fprintf(stdout, "mode page=0x%x has bad page format\n", npage);
        return -1;
    }
    *o_pagestart = buff + offset;
    if (x_interface && replace) {
        if ((nparam && (n_replacement_values != nparam)) || 
            ((! nparam) && (n_replacement_values != (*o_pagestart)[1]))) {
            fprintf(stdout, "Wrong number of replacement values (%i instead "
                    "of %i)\n", n_replacement_values, 
                    nparam ? nparam : (*o_pagestart)[1]);
            return 1;
        }
        next_parameter = 1;
    }
    return 0;
}

static int read_geometry(int page_control)
{
    int status;
    unsigned char *pagestart;

    if (save_mode)
        printf("sginfo -gXR %s ", device_name);

    status = setup_mode_page(4, 0, 9, page_control, buffer, &pagestart);
    if (status)
        return status;

    if (!x_interface && !replace) {
        printf("Rigid Disk Drive Geometry mode page (0x4)\n");
        printf("-----------------------------------------\n");
    };
    intfield(pagestart + 2, 3, "Number of cylinders");
    intfield(pagestart + 5, 1, "Number of heads");
    intfield(pagestart + 6, 3, "Starting cyl. write precomp");
    intfield(pagestart + 9, 3, "Starting cyl. reduced current");
    intfield(pagestart + 12, 2, "Device step rate");
    intfield(pagestart + 14, 3, "Landing Zone Cylinder");
    bitfield(pagestart + 17, "RPL", 3, 0);
    intfield(pagestart + 18, 1, "Rotational Offset");
    intfield(pagestart + 20, 2, "Rotational Rate");
    if (x_interface && replace)
        return put_mode_page(4, buffer);
    else
        printf("\n");
    return 0;

}

static int read_disconnect_reconnect_data(int page_control)
{
    int status;
    unsigned char *pagestart;

    if (save_mode)
        printf("sginfo -DXR %s ", device_name);

    status = setup_mode_page(2, 0, 11, page_control, buffer, &pagestart);
    if (status)
        return status;

    if (!x_interface && !replace) {
        printf("Disconnect-Reconnect mode page (0x2)\n");
        printf("------------------------------------\n");
    };
    intfield(pagestart + 2, 1, "Buffer full ratio");
    intfield(pagestart + 3, 1, "Buffer empty ratio");
    intfield(pagestart + 4, 2, "Bus Inactivity Limit");
    intfield(pagestart + 6, 2, "Disconnect Time Limit");
    intfield(pagestart + 8, 2, "Connect Time Limit");
    intfield(pagestart + 10, 2, "Maximum Burst Size");
    bitfield(pagestart + 12, "EMDP", 1, 7);
    bitfield(pagestart + 12, "FAIR ARBITRATION", 0x7, 4);
    bitfield(pagestart + 12, "DIMM", 1, 3);
    bitfield(pagestart + 12, "DTDC", 0x7, 0);
    intfield(pagestart + 14, 2, "First Burst Size");
    if (x_interface && replace)
        return put_mode_page(2, buffer);
    else
        printf("\n");
    return 0;

}

static int read_control_page(int page_control)
{
    int status;
    unsigned char *pagestart;

    if (save_mode)
        printf("sginfo -CXR %s ", device_name);

    status = setup_mode_page(0xa, 0, 18, page_control, buffer, &pagestart);
    if (status)
        return status;

    if (!x_interface && !replace) {
        printf("Control mode page (0xa)\n");
        printf("-----------------------\n");
    }
    bitfield(pagestart + 2, "TST", 0x7, 5);
    bitfield(pagestart + 2, "D_SENSE", 1, 2);
    bitfield(pagestart + 2, "GLTSD", 1, 1);
    bitfield(pagestart + 2, "RLEC", 1, 0);
    bitfield(pagestart + 3, "Queue Algorithm Modifier", 0xf, 4);
    bitfield(pagestart + 3, "QErr", 1, 1);
    bitfield(pagestart + 3, "DQue [obsolete]", 1, 0);
    bitfield(pagestart + 4, "TAS", 1, 7);
    bitfield(pagestart + 4, "RAC", 1, 6);
    bitfield(pagestart + 4, "UA_INTLCK_CTRL", 0x3, 4);
    bitfield(pagestart + 4, "SWP", 1, 3);
    bitfield(pagestart + 4, "RAERP [obs.]", 1, 2);
    bitfield(pagestart + 4, "UAAERP [obs.]", 1, 1);
    bitfield(pagestart + 4, "EAERP [obs.]", 1, 0);
    bitfield(pagestart + 4, "AUTOLOAD MODE", 0x7, 0);
    intfield(pagestart + 6, 2, "Ready AER Holdoff Period [obs.]");
    intfield(pagestart + 8, 2, "Busy Timeout Period");
    intfield(pagestart + 10, 2, "Extended self-test completion time");
    if (x_interface && replace)
        return put_mode_page(10, buffer);
    else
        printf("\n");
    return 0;
}

static int read_informational_page(int page_control)
{
    int status;
    unsigned char *pagestart;

    if (save_mode)
        printf("sginfo -IXR %s ", device_name);

    status = setup_mode_page(0x1c, 0, 9, page_control, buffer, &pagestart);
    if (status)
        return status;

    if (!x_interface && !replace) {
        printf("Informational Exceptions mode page (0x1c)\n");
        printf("-----------------------------------------\n");
    }
    bitfield(pagestart + 2, "PERF", 1, 7);
    bitfield(pagestart + 2, "EBF", 1, 5);
    bitfield(pagestart + 2, "EWASC", 1, 4);
    bitfield(pagestart + 2, "DEXCPT", 1, 3);
    bitfield(pagestart + 2, "TEST", 1, 2);
    bitfield(pagestart + 2, "LOGERR", 1, 0);
    bitfield(pagestart + 3, "MRIE", 0xf, 0);
    intfield(pagestart + 4, 4, "Interval Timer");
    intfield(pagestart + 8, 4, "Report Count");
    if (x_interface && replace)
        return put_mode_page(0x1c, buffer);
    else
        printf("\n");
    return 0;
}

static int error_recovery_page(int page_control)
{
    int status;
    unsigned char *pagestart;

    if (save_mode)
        printf("sginfo -eXR %s ", device_name);

    status = setup_mode_page(1, 0, 14, page_control, buffer, &pagestart);
    if (status)
        return status;

    if (!x_interface && !replace) {
        printf("Read-Write Error Recovery mode page (0x1)\n");
        printf("-----------------------------------------\n");
    }
    bitfield(pagestart + 2, "AWRE", 1, 7);
    bitfield(pagestart + 2, "ARRE", 1, 6);
    bitfield(pagestart + 2, "TB", 1, 5);
    bitfield(pagestart + 2, "RC", 1, 4);
    bitfield(pagestart + 2, "EER", 1, 3);
    bitfield(pagestart + 2, "PER", 1, 2);
    bitfield(pagestart + 2, "DTE", 1, 1);
    bitfield(pagestart + 2, "DCR", 1, 0);
    intfield(pagestart + 3, 1, "Read Retry Count");
    intfield(pagestart + 4, 1, "Correction Span");
    intfield(pagestart + 5, 1, "Head Offset Count");
    intfield(pagestart + 6, 1, "Data Strobe Offset Count");
    intfield(pagestart + 8, 1, "Write Retry Count");
    intfield(pagestart + 10, 2, "Recovery Time Limit");
    if (x_interface && replace)
        return put_mode_page(1, buffer);
    else
        printf("\n");
    return 0;
}

static int notch_parameters_page(int page_control)
{
    int status;
    unsigned char *pagestart;

    if (save_mode)
        printf("sginfo -nXR %s ", device_name);

    status = setup_mode_page(0xc, 0, 6, page_control, buffer, &pagestart);
    if (status)
        return status;

    if (!x_interface && !replace) {
        printf("Notch Parameters mode page (0xc)\n");
        printf("--------------------------------\n");
    };
    bitfield(pagestart + 2, "Notched Drive", 1, 7);
    bitfield(pagestart + 2, "Logical or Physical Notch", 1, 6);
    intfield(pagestart + 4, 2, "Max # of notches");
    intfield(pagestart + 6, 2, "Active Notch");
    if (pagestart[2] & 0x40) {
        intfield(pagestart + 8, 4, "Starting Boundary");
        intfield(pagestart + 12, 4, "Ending Boundary");
    } else {           /* Hex is more meaningful for physical notches */
        hexfield(pagestart + 8, 4, "Starting Boundary");
        hexfield(pagestart + 12, 4, "Ending Boundary");
    }

    if (x_interface && !replace) {
        if (modifiable)
            printf("0");
        else
            printf("0x%8.8x%8.8x", getnbyte(pagestart + 16, 4), 
                   getnbyte(pagestart + 20, 4));
    };
    if (!x_interface)
        printf("Pages Notched                      %8.8x %8.8x\n",
               getnbyte(pagestart + 16, 4), getnbyte(pagestart + 20, 4));
    if (x_interface && replace)
        return put_mode_page(0xc, buffer);
    else
        printf("\n");
    return 0;
}

static char *formatname(int format) {
    switch(format) {
        case 0x0: return "logical blocks";
        case 0x4: return "bytes from index [Cyl:Head:Off]\n"
        "Offset -1 marks whole track as bad.\n";
        case 0x5: return "physical blocks [Cyl:Head:Sect]\n"
        "Sector -1 marks whole track as bad.\n";
    }
    return "Weird, unknown format";
}

static int read_defect_list(int page_control, int grown_only)
{
    int i, len, reallen, table, k;
    int status = 0;
    int header = 1;
    unsigned char cmd[10];
    unsigned char cmd12[12];
    unsigned char *df = NULL;
    unsigned char *bp = NULL;
    unsigned char *heapp = NULL;
    int trunc;
    struct scsi_cmnd_io sci;


    for (table = grown_only; table < 2; table++) {
        if (heapp) {
            free(heapp);
            heapp = NULL;
        }
        bp = buffer;
        memset(bp, 0, 4);
        trunc = 0;

        cmd[0] = 0x37;          /* READ DEFECT DATA (10) */
        cmd[1] = 0x00; 
        cmd[2] = (table ? 0x08 : 0x10) | defectformat;  /*  List, Format */
        cmd[3] = 0x00;          /* (reserved) */
        cmd[4] = 0x00;          /* (reserved) */
        cmd[5] = 0x00;          /* (reserved) */
        cmd[6] = 0x00;          /* (reserved) */
        cmd[7] = 0x00;          /* Alloc len */
        cmd[8] = 0x04;          /* Alloc len (size finder) */
        cmd[9] = 0x00;          /* control */

        sci.cmnd = cmd;
        sci.cmnd_len = sizeof(cmd);
        sci.dxfer_dir = DXFER_FROM_DEVICE;
        sci.dxfer_len = 4;
        sci.dxferp = bp;
        i = do_scsi_io(&sci);
        if (i) {
            fprintf(stdout, ">>> Unable to read %s defect data.\n",
                            (table ? "grown" : "manufacturer"));
            status |= i;
            continue;
        }
        if (header) {
            printf("Defect Lists\n"
                   "------------\n");
            header = 0;
        }
        len = (bp[2] << 8) + bp[3];
        reallen = len;
        if (len > 0) {
            if ((len + 8) >= sizeof(buffer)) {

                if (len >= 0xfff0)
                    len = 0x80000;      /* go large: 512 KB */
                heapp = malloc(len);
                if (NULL == heapp)
                    continue;
                bp = heapp;
                k = len;            /* length of defect list */
                cmd12[0] = 0xB7;          /* READ DEFECT DATA (12) */
                cmd12[1] = (table ? 0x08 : 0x10) | defectformat;/*  List, Format */
                cmd12[2] = 0x00;          /* (reserved) */
                cmd12[3] = 0x00;          /* (reserved) */
                cmd12[4] = 0x00;          /* (reserved) */
                cmd12[5] = 0x00;          /* (reserved) */
                cmd12[6] = 0x00;          /* Alloc len */
                cmd12[7] = (k >> 16) & 0xff;     /* Alloc len */
                cmd12[8] = (k >> 8) & 0xff;      /* Alloc len */
                cmd12[9] = (k & 0xff);    /* Alloc len */
                cmd12[10] = 0x00;         /* reserved */
                cmd12[11] = 0x00;         /* control */

                sci.cmnd = cmd12;
                sci.cmnd_len = sizeof(cmd12);
                sci.dxfer_dir = DXFER_FROM_DEVICE;
                sci.dxfer_len = k;
                sci.dxferp = bp;
                i = do_scsi_io(&sci);
                if (i) 
                    goto trytenbyte;
                reallen = (bp[4] << 24) + (bp[5] << 16) + (bp[6] << 8) + 
                          bp[7];
                len = reallen;
                if (len > k) { 
                    len = k; 
                    trunc = 1;
                }
                df = (unsigned char *) (bp + 8);
            }
            else {
trytenbyte:
                if ((len + 4) > sizeof(buffer)) { 
                    len = sizeof(buffer) - 4; 
                    trunc = 1; 
                }
                k = len;            /* length of defect list */
                cmd[0] = 0x37;          /* READ DEFECT DATA (10) */
                cmd[1] = 0x00;
                cmd[2] = (table ? 0x08 : 0x10) | defectformat;/*  List, Format */
                cmd[3] = 0x00;          /* (reserved) */
                cmd[4] = 0x00;          /* (reserved) */
                cmd[5] = 0x00;          /* (reserved) */
                cmd[6] = 0x00;          /* (reserved) */
                cmd[7] = (k >> 8);      /* Alloc len */
                cmd[8] = (k & 0xff);    /* Alloc len */
                cmd[9] = 0x00;          /* control */

                sci.cmnd = cmd;
                sci.cmnd_len = sizeof(cmd);
                sci.dxfer_dir = DXFER_FROM_DEVICE;
                sci.dxfer_len = k;
                sci.dxferp = bp;
                i = do_scsi_io(&sci);
                df = (unsigned char *) (bp + 4);
            }
        }
        if (i) {
            fprintf(stdout, ">>> Unable to read %s defect data.\n",
                            (table ? "grown" : "manufacturer"));
            status |= i;
            continue;
        }
        else {
            if (table && !status)
                printf("\n");
            printf("%d entries (%d bytes) in %s table.\n"
                   "Format (%x) is: %s\n", reallen / ((bp[1] & 7) ? 8 : 4), reallen,
                   (table ? "grown" : "manufacturer"),
                   bp[1] & 7, 
                   formatname(bp[1] & 7));
            i = 0;
            if ((bp[1] & 7) == 4) {
                while (len > 0) {
                    snprintf((char *)buffer1, 40, "%6d:%3u:%8d", getnbyte(df, 3),
                             df[3], getnbyte(df + 4, 4));
                    printf("%19s", (char *)buffer1);
                    len -= 8;
                    df += 8;
                    i++;
                    if (i >= 4) {
                        printf("\n");
                        i = 0;
                    }
                    else printf("|");
                }
            } else if ((bp[1] & 7) == 5) {
                while (len > 0) {
                    snprintf((char *)buffer1, 40, "%6d:%2u:%5d", getnbyte(df, 3),
                             df[3], getnbyte(df + 4, 4));
                    printf("%15s", (char *)buffer1);
                    len -= 8;
                    df += 8;
                    i++;
                    if (i >= 5) {
                        printf("\n");
                        i = 0;
                    }
                    else printf("|");
                }                           
            }
            else {
                while (len > 0) {
                    printf("%10d", getnbyte(df, 4));
                    len -= 4;
                    df += 4;
                    i++;
                    if (i >= 7) {
                        printf("\n");
                        i = 0;
                    }
                    else
                        printf("|");
                }
            }
            if (i)
                printf("\n");
        }
        if (trunc) 
                printf("[truncated]\n");
    }
    printf("\n");
    if (heapp) {
        free(heapp);
        heapp = NULL;
    }
    return status;
}

static int read_cache(int page_control)
{
    int status;
    unsigned char *pagestart;

    if (save_mode)
        printf("sginfo -cXR %s ", device_name);

    status = setup_mode_page(8, 0, 20, page_control, buffer, &pagestart);
    if (status)
        return status;

    if (!x_interface && !replace) {
        printf("Caching mode page (0x8)\n");
        printf("-----------------------\n");
    };
    bitfield(pagestart + 2, "Initiator Control", 1, 7);
    bitfield(pagestart + 2, "ABPF", 1, 6);
    bitfield(pagestart + 2, "CAP", 1, 5);
    bitfield(pagestart + 2, "DISC", 1, 4);
    bitfield(pagestart + 2, "SIZE", 1, 3);
    bitfield(pagestart + 2, "Write Cache Enabled", 1, 2);
    bitfield(pagestart + 2, "MF", 1, 1);
    bitfield(pagestart + 2, "Read Cache Disabled", 1, 0);
    bitfield(pagestart + 3, "Demand Read Retention Priority", 0xf, 4);
    bitfield(pagestart + 3, "Demand Write Retention Priority", 0xf, 0);
    intfield(pagestart + 4, 2, "Disable Pre-fetch Transfer Length");
    intfield(pagestart + 6, 2, "Minimum Pre-fetch");
    intfield(pagestart + 8, 2, "Maximum Pre-fetch");
    intfield(pagestart + 10, 2, "Maximum Pre-fetch Ceiling");
    bitfield(pagestart + 12, "FSW", 1, 7);
    bitfield(pagestart + 12, "LBCSS", 1, 6);
    bitfield(pagestart + 12, "DRA", 1, 5);
    intfield(pagestart + 13, 1, "Number of Cache Segments");
    intfield(pagestart + 14, 2, "Cache Segment size");
    intfield(pagestart + 17, 3, "Non-Cache Segment size");
    if (x_interface && replace)
        return put_mode_page(8, buffer);
    else
        printf("\n");
    return 0;
}

static int read_format_info(int page_control)
{
    int status;
    unsigned char *pagestart;

    if (save_mode)
        printf("sginfo -fXR %s ", device_name);

    status = setup_mode_page(3, 0, 13, page_control, buffer, &pagestart);
    if (status)
        return status;

    if (!x_interface && !replace) {
        printf("Format Device mode page (0x3)\n");
        printf("-----------------------------\n");
    };
    intfield(pagestart + 2, 2, "Tracks per Zone");
    intfield(pagestart + 4, 2, "Alternate sectors per zone");
    intfield(pagestart + 6, 2, "Alternate tracks per zone");
    intfield(pagestart + 8, 2, "Alternate tracks per lu");
    intfield(pagestart + 10, 2, "Sectors per track");
    intfield(pagestart + 12, 2, "Data bytes per physical sector");
    intfield(pagestart + 14, 2, "Interleave");
    intfield(pagestart + 16, 2, "Track skew factor");
    intfield(pagestart + 18, 2, "Cylinder skew factor");
    bitfield(pagestart + 20, "Supports Soft Sectoring", 1, 7);
    bitfield(pagestart + 20, "Supports Hard Sectoring", 1, 6);
    bitfield(pagestart + 20, "Removable Medium", 1, 5);
    bitfield(pagestart + 20, "Surface", 1, 4);
    if (x_interface && replace)
        return put_mode_page(3, buffer);
    else
        printf("\n");
    return 0;

}

static int verify_error_recovery(int page_control)
{
    int status;
    unsigned char *pagestart;

    if (save_mode)
        printf("sginfo -VXR %s ", device_name);

    status = setup_mode_page(7, 0, 7, page_control, buffer, &pagestart);
    if (status)
        return status;

    if (!x_interface && !replace) {
        printf("Verify Error Recovery mode page (0x7)\n");
        printf("-------------------------------------\n");
    }
    bitfield(pagestart + 2, "EER", 1, 3);
    bitfield(pagestart + 2, "PER", 1, 2);
    bitfield(pagestart + 2, "DTE", 1, 1);
    bitfield(pagestart + 2, "DCR", 1, 0);
    intfield(pagestart + 3, 1, "Verify Retry Count");
    intfield(pagestart + 4, 1, "Verify Correction Span (bits)");
    intfield(pagestart + 10, 2, "Verify Recovery Time Limit (ms)");

    if (x_interface && replace)
        return put_mode_page(7, buffer);
    else
        printf("\n");
    return 0;
}

#if 0
static int peripheral_device_page(int page_control)
{
    static char *idents[] =
    {
        "X3.131: Small Computer System Interface",
        "X3.91M-1987: Storage Module Interface",
        "X3.170: Enhanced Small Device Interface",
        "X3.130-1986; X3T9.3/87-002: IPI-2",
        "X3.132-1987; X3.147-1988: IPI-3"
    };
    int status;
    unsigned ident;
    unsigned char *pagestart;
    char *name;

    if (save_mode)
        printf("sginfo -pXR %s ", device_name);

    status = setup_mode_page(9, 0, 2, page_control, buffer, &pagestart);
    if (status)
        return status;

    if (!x_interface && !replace) {
        printf("Peripheral Device mode page (0x9)\n");
        printf("---------------------------------\n");
    };

#if 0
    dump(pagestart, 20);
    pagestart[1] += 2;          /*TEST */
    buffer[8] += 2;             /*TEST */
#endif

    ident = getnbyte(pagestart + 2, 2);
    if (ident < (sizeof(idents) / sizeof(char *)))
         name = idents[ident];
    else if (ident < 0x8000)
        name = "Reserved";
    else
        name = "Vendor Specific";

#ifdef DPG_CHECK_THIS_OUT
    bdlen = pagestart[1] - 6;
    if (bdlen < 0)
        bdlen = 0;
    else {
        status = setup_mode_page(9, 0, 2, page_control, buffer, &bdlen, 
                                 &pagestart);
        if (status)
            return status;
    }
        
    hexfield(pagestart + 2, 2, "Interface Identifier");
    if (!x_interface) {
        for (ident = 0; ident < 35; ident++)
            putchar(' ');
        puts(name);
    }
    hexdatafield(pagestart + 8, bdlen, "Vendor Specific Data");
#endif

    if (x_interface && replace)
        return put_mode_page(9, buffer);
    else
        printf("\n");
    if (x_interface && !save_mode)
        puts(name);
    return 0;
}
#endif

static int power_condition_page(int page_control)
{
    int status;
    unsigned char *pagestart;

    if (save_mode)
        printf("sginfo -PXR %s ", device_name);

    status = setup_mode_page(0x1a, 0, 4, page_control, buffer, &pagestart);
    if (status)
        return status;

    if (!x_interface && !replace) {
        printf("Power Condition mode page (0x1a)\n");
        printf("--------------------------------\n");
    }
    bitfield(pagestart + 3, "Idle", 1, 1);
    bitfield(pagestart + 3, "Standby", 1, 0);
    intfield(pagestart + 4, 4, "Idle Condition counter (100ms)");
    intfield(pagestart + 8, 4, "Standby Condition counter (100ms)");

    if (x_interface && replace)
        return put_mode_page(0x1a, buffer);
    else
        printf("\n");
    return 0;
}

static int do_user_page(int page_control, int page_no, int subpage_no)
{
    int status;
    int k, len, off, resp_page_no, resp_subpage_no, resp_len;
    int offset = 0;
    unsigned char *pagestart;
    char * pg_name;

    if (x_interface && replace && (63 == page_no)) {
        printf("Can't list all pages (pn=63) and replace (-R) together\n");
        return 1;
    }
    if (63 == page_no) {
        status = get_mode_page(page_no, subpage_no, page_control, 0, buffer); 
        if (status)
            printf("\n");
        else {
            offset = modePageOffset(buffer, MAX_BUFFER_SIZE, mode6byte, &resp_len);
            if (offset < 0) {
                fprintf(stdout, "mode page=0x%x has bad page format\n", page_no);
                return -1;
            }
            pagestart = buffer + offset;
        }
    } else
        status = setup_mode_page(page_no, subpage_no, 0, page_control, buffer, 
                                 &pagestart);
    if (status)
        return status;

    do {
        resp_page_no = (pagestart[0] & 0x3f);
        resp_subpage_no = (pagestart[0] & 0x40) ? pagestart[1] : 0;

        if ((63 == page_no) && x_interface) {
            if (resp_subpage_no)
                printf("sginfo -u %i,%i -XR %s ", resp_page_no, 
                       resp_subpage_no, device_name);
            else
                printf("sginfo -u %i -XR %s ", resp_page_no, device_name);
        }
        if (!x_interface && !replace) {
            pg_name = get_page_name(resp_page_no);
            if (resp_subpage_no) {
                if (pg_name && (unkn_page_str != pg_name))
                    printf("mode page: 0x%02x  subpage: 0x%02x   [%s]\n",
                           resp_page_no, resp_subpage_no, pg_name);
                else
                    printf("mode page: 0x%02x  subpage: 0x%02x\n", resp_page_no, 
                       resp_subpage_no);
                printf("------------------------------\n");
            } else {
                if (pg_name && (unkn_page_str != pg_name))
                    printf("mode page: 0x%02x   [%s]\n", resp_page_no, 
                           pg_name);
                else
                    printf("mode page: 0x%02x\n", resp_page_no);
                printf("---------------\n");
            }
        }
        if(0 == resp_page_no) { /* page==0 last and vendor (unknown) format */
            off = 0;
            len = resp_len - offset;
        } else if (resp_subpage_no) {
            off = 4;
            len = (pagestart[2] << 8) + pagestart[3] + 4;
        } else {
            off = 2;
            len = pagestart[1] + 2;
        }
        for (k = off; k < len; k++)
        {
            char nm[8]; 

            snprintf(nm, sizeof(nm), "0x%02x", k);
            hexdatafield(pagestart + k, 1, nm);
        }
        if (x_interface && replace)
            return put_mode_page(page_no, buffer);
        else
            printf("\n");
        offset += len;
        pagestart = buffer + offset;
    } while ((63 == page_no) && (offset < resp_len));
    return 0;
}

static int do_inquiry(int page_control)
{
    int status;
    const int inq_resp_len = 36;
    unsigned char cmd[6];
    unsigned char *pagestart;
    unsigned char tmp;
    struct scsi_cmnd_io sci;

    memset(buffer, 0, inq_resp_len);

    cmd[0] = 0x12;              /* INQUIRY */
    cmd[1] = 0x00;              /* evpd=0 */
    cmd[2] = 0x00;              /* page code = 0 */
    cmd[3] = 0x00;              /* (reserved) */
    cmd[4] = inq_resp_len;      /* allocation length */
    cmd[5] = 0x00;              /* control */

    sci.cmnd = cmd;
    sci.cmnd_len = sizeof(cmd);
    sci.dxfer_dir = DXFER_FROM_DEVICE;
    sci.dxfer_len = inq_resp_len;
    sci.dxferp = buffer;
    status = do_scsi_io(&sci);
    if (status) {
        printf("Error doing INQUIRY (1)");
        return status;
    }

    pagestart = buffer;

    if (!x_interface && !replace) {
        printf("INQUIRY reponse (cmd: 0x12)\n");
        printf("---------------------------\n");
    };
    bitfield(pagestart + 7, "Relative Address", 1, 7);
    bitfield(pagestart + 7, "Wide bus 32", 1, 6);
    bitfield(pagestart + 7, "Wide bus 16", 1, 5);
    bitfield(pagestart + 7, "Synchronous neg.", 1, 4);
    bitfield(pagestart + 7, "Linked Commands", 1, 3);
    bitfield(pagestart + 7, "Command Queueing", 1, 1);
    bitfield(pagestart + 7, "SftRe", 1, 0);
    bitfield(pagestart + 0, "Device Type", 0x1f, 0);
    bitfield(pagestart + 0, "Peripheral Qualifier", 0x7, 5);
    bitfield(pagestart + 1, "Removable?", 1, 7);
    bitfield(pagestart + 1, "Device Type Modifier", 0x7f, 0);
    bitfield(pagestart + 2, "ISO Version", 3, 6);
    bitfield(pagestart + 2, "ECMA Version", 7, 3);
    bitfield(pagestart + 2, "ANSI Version", 7, 0);
    bitfield(pagestart + 3, "AENC", 1, 7);
    bitfield(pagestart + 3, "TrmIOP", 1, 6);
    bitfield(pagestart + 3, "Response Data Format", 0xf, 0);
    if (x_interface)
        printf("\n");
    tmp = pagestart[16];
    pagestart[16] = 0;
    printf("%s%s\n", (!x_interface ? "Vendor:                    " : ""),
           pagestart + 8);
    pagestart[16] = tmp;

    tmp = pagestart[32];
    pagestart[32] = 0;
    printf("%s%s\n", (!x_interface ? "Product:                   " : ""),
           pagestart + 16);
    pagestart[32] = tmp;

    printf("%s%s\n", (!x_interface ? "Revision level:            " : ""),
           pagestart + 32);

    printf("\n");
    return status;

}

static int do_serial_number(int page_control)
{
    int status, i, pagelen;
    unsigned char cmd[6];
    unsigned char *pagestart;
    struct scsi_cmnd_io sci;

    cmd[0] = 0x12;              /* INQUIRY */
    cmd[1] = 0x01;              /* evpd=1 */
    cmd[2] = 0x80;              /* page code = 0x80, serial number */
    cmd[3] = 0x00;              /* (reserved) */
    cmd[4] = 0x04;              /* allocation length */
    cmd[5] = 0x00;              /* control */

    sci.cmnd = cmd;
    sci.cmnd_len = sizeof(cmd);
    sci.dxfer_dir = DXFER_FROM_DEVICE;
    sci.dxfer_len = 4;
    sci.dxferp = buffer;
    status = do_scsi_io(&sci);
    if (status) {
        printf("Error doing INQUIRY (evpd=1, serial number)\n");
        return status;
    }

    pagestart = buffer;
    
    pagelen = 4 + pagestart[3];

    cmd[0] = 0x12;              /* INQUIRY */
    cmd[1] = 0x01;              /* evpd=1 */
    cmd[2] = 0x80;              /* page code = 0x80, serial number */
    cmd[3] = 0x00;              /* (reserved) */
    cmd[4] = (unsigned char)pagelen; /* allocation length */
    cmd[5] = 0x00;              /* control */

    sci.cmnd = cmd;
    sci.cmnd_len = sizeof(cmd);
    sci.dxfer_dir = DXFER_FROM_DEVICE;
    sci.dxfer_len = pagelen;
    sci.dxferp = buffer;
    status = do_scsi_io(&sci);
    if (status) {
        printf("Error doing INQUIRY (evpd=1, serial number (page 0x80)\n");
        return status;
    }

    printf("Serial Number '");
    for (i = 0; i < pagestart[3]; i++)
        printf("%c", pagestart[4 + i]);
    printf("'\n");
    printf("\n");

    return status;
}


typedef struct sg_map {
    int bus;
    int channel;
    int target_id;
    int lun;
    char * dev_name;
} Sg_map;

typedef struct my_scsi_idlun
{
    int mux4;
    int host_unique_id;

} My_scsi_idlun;

#define MDEV_NAME_SZ 256

static void make_dev_name(char * fname, int k, int do_numeric)
{
    char buff[MDEV_NAME_SZ];
    size_t len;

    strncpy(fname, "/dev/sg", MDEV_NAME_SZ);
    fname[MDEV_NAME_SZ - 1] = '\0';
    len = strlen(fname);
    if (do_numeric)
        snprintf(fname + len, MDEV_NAME_SZ - len, "%d", k);
    else {
        if (k <= 26) {
            buff[0] = 'a' + (char)k;
            buff[1] = '\0';
            strcat(fname, buff);
        }
        else
            strcat(fname, "xxxx");
    }
}

#define MAX_SG_DEVS 48

char *devices[] =
{"/dev/sda", "/dev/sdb", "/dev/sdc", "/dev/sdd", "/dev/sde", "/dev/sdf", 
 "/dev/sdg", "/dev/sdh", "/dev/sdi", "/dev/sdj", "/dev/sdk", "/dev/sdl",
 "/dev/sdm", "/dev/sdn", "/dev/sdo", "/dev/sdp", "/dev/sdq", "/dev/sdr",
 "/dev/sds", "/dev/sdt", "/dev/sdu", "/dev/sdv", "/dev/sdw", "/dev/sdx",
 "/dev/sdy", "/dev/sdz", "/dev/sdaa", "/dev/sdab", "/dev/sdac", "/dev/sdad",
 "/dev/scd0", "/dev/scd1", "/dev/scd2", "/dev/scd3", "/dev/scd4", "/dev/scd5",
 "/dev/scd6", "/dev/scd7", "/dev/scd8", "/dev/scd9", "/dev/scd10", "/dev/scd11",
 "/dev/sr0", "/dev/sr1", "/dev/sr2", "/dev/sr3", "/dev/sr4", "/dev/sr5",
 "/dev/sr6", "/dev/sr7", "/dev/sr8", "/dev/sr9", "/dev/sr10", "/dev/sr11",
 "/dev/nst0", "/dev/nst1", "/dev/nst2", "/dev/nst3", "/dev/nst4", "/dev/nst5",
 "/dev/nosst0", "/dev/nosst1", "/dev/nosst2", "/dev/nosst3", "/dev/nosst4"
};

static Sg_map sg_map_arr[(sizeof(devices) / sizeof(char *)) + 1];

#define MAX_HOLES 4

/* Print out a list of the known devices on the system */
static void show_devices()
{
    int k, j, fd, err, bus;
    My_scsi_idlun m_idlun;
    char name[MDEV_NAME_SZ];
    char ebuff[EBUFF_SZ];
    int do_numeric = 1;
    int max_holes = MAX_HOLES;

    for (k = 0, j = 0; k < sizeof(devices) / sizeof(char *); k++) {
        fd = open(devices[k], O_RDONLY | O_NONBLOCK);
        if (fd < 0)
            continue;
        err = ioctl(fd, SCSI_IOCTL_GET_BUS_NUMBER, &(sg_map_arr[j].bus));
        if (err < 0) {
            snprintf(ebuff, EBUFF_SZ,
                     "SCSI(1) ioctl on %s failed", devices[k]);
            perror(ebuff);
            close(fd);
            continue;
        }
        err = ioctl(fd, SCSI_IOCTL_GET_IDLUN, &m_idlun);
        if (err < 0) {
            snprintf(ebuff, EBUFF_SZ, 
                     "SCSI(2) ioctl on %s failed", devices[k]);
            perror(ebuff);
            close(fd);
            continue;
        }
        sg_map_arr[j].channel = (m_idlun.mux4 >> 16) & 0xff;
        sg_map_arr[j].lun = (m_idlun.mux4 >> 8) & 0xff;
        sg_map_arr[j].target_id = m_idlun.mux4 & 0xff;
        sg_map_arr[j].dev_name = devices[k];
#if 0
        printf("[scsi%d ch=%d id=%d lun=%d %s] ", sg_map_arr[j].bus,
        sg_map_arr[j].channel, sg_map_arr[j].target_id, sg_map_arr[j].lun,
        sg_map_arr[j].dev_name);
#endif
        ++j;
        printf("%s ", devices[k]);
        close(fd);
    };
    printf("\n"); // <<<<<<<<<<<<<<<<<<<<<
    for (k = 0; k < MAX_SG_DEVS; k++) {
        make_dev_name(name, k, do_numeric);
        fd = open(name, O_RDWR | O_NONBLOCK);
        if (fd < 0) {
            if ((ENOENT == errno) && (0 == k)) {
                do_numeric = 0;
                make_dev_name(name, k, do_numeric);
                fd = open(name, O_RDWR | O_NONBLOCK);
            }
            if (fd < 0) {
                if (EBUSY == errno)
                    continue;   /* step over if O_EXCL already on it */
                else {
#if 0
                    snprintf(ebuff, EBUFF_SZ,
                             "open on %s failed (%d)", name, errno);
                    perror(ebuff);
#endif
                    if (max_holes-- > 0)
                        continue;
                    else
                        break;
                }
            }
        }
        max_holes = MAX_HOLES;
        err = ioctl(fd, SCSI_IOCTL_GET_BUS_NUMBER, &bus);
        if (err < 0) {
            snprintf(ebuff, EBUFF_SZ, "SCSI(3) ioctl on %s failed", name);
            perror(ebuff);
            close(fd);
            continue;
        }
        err = ioctl(fd, SCSI_IOCTL_GET_IDLUN, &m_idlun);
        if (err < 0) {
            snprintf(ebuff, EBUFF_SZ, "SCSI(3) ioctl on %s failed", name);
            perror(ebuff);
            close(fd);
            continue;
        }
#if 0
        printf("[scsi%d ch=%d id=%d lun=%d %s]", bus,
               (m_idlun.mux4 >> 16) & 0xff, m_idlun.mux4 & 0xff,
               (m_idlun.mux4 >> 8) & 0xff, name);
#endif
        for (j = 0; sg_map_arr[j].dev_name; ++j) {
            if ((bus == sg_map_arr[j].bus) &&
                ((m_idlun.mux4 & 0xff) == sg_map_arr[j].target_id) &&
                (((m_idlun.mux4 >> 16) & 0xff) == sg_map_arr[j].channel) &&
                (((m_idlun.mux4 >> 8) & 0xff) == sg_map_arr[j].lun)) {
                printf("%s [=%s  scsi%d ch=%d id=%d lun=%d]\n", name, 
                       sg_map_arr[j].dev_name, bus,
                       ((m_idlun.mux4 >> 16) & 0xff), m_idlun.mux4 & 0xff,
                       ((m_idlun.mux4 >> 8) & 0xff));
                break;
            }
        }
        if (NULL == sg_map_arr[j].dev_name)
            printf("%s [scsi%d ch=%d id=%d lun=%d]\n", name, bus, 
                   ((m_idlun.mux4 >> 16) & 0xff), m_idlun.mux4 & 0xff,
                   ((m_idlun.mux4 >> 8) & 0xff));
        close(fd);
    }
    printf("\n");
}

/* select the given notch w/o changing anything else, if save_mode is
   set output code to do the same in a script as well.. */
static int select_notch(int notch)
{
    int status;
    int page_control = 0;
    unsigned char *pagestart;

    if (save_mode) {
        printf("set -- `sginfo -nX %s`\n", device_name);
        printf("sginfo -nXR %s $1 $2 $3 %d $5 $6 $7\n", 
               device_name, notch);
    }
    status = setup_mode_page(0xc, 0, 0, page_control, buffer, &pagestart);
    if (status)
        return status;
    putnbyte(pagestart + 6, notch, 2);
    return put_mode_page(0xc, buffer);
}

static int show_pages(int page_control)
{
    int offset;
    int length;
    int i;
    int status = 0;
    unsigned long long pages_sup = 0;
    unsigned long long pages_mask = 0;

    status = get_mode_page(MP_LIST_PAGES, 0, (page_control | 0x10), 0, 
                           buffer);
    if (status)
        return status;
    offset = modePageOffset(buffer, sizeof(buffer), mode6byte, &length);
    if (offset < 0) {  /* Assume SCSI-1 and fake settings to report NO pages */
        offset = 10;
        length = 0;
    }

    /* Get mask of pages supported by prog: */
    for (i = 0; i < MAX_PAGENO; i++)
        if (page_names[i])
            pages_mask |= (1LL << i);

    /* Get pages listed in mode_pages */
    while (offset < length) {
        pages_sup |= (1LL << (buffer[offset] & 0x3f));
        offset += 2 + buffer[offset + 1];
    }

    /* Mask out pages unsupported by this binary */
    pages_sup &= pages_mask;

    /* Notch page supported? */
    if (pages_sup & (1LL << 12)) {
        if (get_mode_page(12, 0, 0, 0, buffer))
            return 2;
        offset = 12 + buffer[11];
    } else {                    /* Fake empty notch page */
        memset(buffer, 0, SIZEOF_BUFFER);
        offset = 0;
    }

    if (replace) {
        /* Ok we are creating a safe file.. first of all reset the
           fake replace setting. */
        replace = 0;
        save_mode = 1;
        x_interface = 1;

        if (modifiable)
            usage("do not use -LR with -m");

        /* Just a reminder: */
        puts("#!/bin/sh");

        if (pages_sup & (1 << 12)) {
            /* save away the notched pages list.. */
            pages_mask = (unsigned long) getnbyte(buffer + offset + 16, 4),
                pages_mask <<= 32;
            pages_mask = getnbyte(buffer + offset + 20, 4),

                i = getnbyte(buffer + offset + 4, 2);

            /* loop through all notches > 0 */
            while (i > 0) {
                status |= select_notch(i);
                if (pages_mask & (1 << 1))
                    status |= error_recovery_page(page_control);
                if (pages_mask & (1 << 2))
                    status |= read_disconnect_reconnect_data(page_control);
                if (pages_mask & (1 << 3))
                    status |= read_format_info(page_control);
                if (pages_mask & (1 << 4))
                    status |= read_geometry(page_control);
                if (pages_mask & (1 << 7))
                    status |= verify_error_recovery(page_control);
                if (pages_mask & (1 << 8))
                    status |= read_cache(page_control);
                if (pages_mask & (1 << 10))
                    status |= read_control_page(page_control);
                if (pages_mask & (1 << 12))
                    status |= notch_parameters_page(page_control);
                if (pages_mask & (1 << 0x1a))
                    status |= power_condition_page(page_control);
                if (pages_mask & (1 << 0x1c))
                    status |= read_informational_page(page_control);
                i--;
            }

            /* Back to notch 0 and safe the notch 0 page itself */
            status |= select_notch(0);
            status |= notch_parameters_page(page_control);
        }
        if (pages_sup & (1 << 1))
            status |= error_recovery_page(page_control);
        if (pages_sup & (1 << 2))
            status |= read_disconnect_reconnect_data(page_control);
        if (pages_sup & (1 << 3))
            status |= read_format_info(page_control);
        if (pages_sup & (1 << 4))
            status |= read_geometry(page_control);
        if (pages_sup & (1 << 7))
            status |= verify_error_recovery(page_control);
        if (pages_sup & (1 << 8))
            status |= read_cache(page_control);
        if (pages_sup & (1 << 10))
            status |= read_control_page(page_control);
        return status;
    }
    if (x_interface) {
        printf("0x%08lx%08lx 0x%08x%08x %d\n",
               (unsigned long) (pages_sup >> 32),
               (unsigned long) pages_sup,
               getnbyte(buffer + offset + 16, 4),
               getnbyte(buffer + offset + 20, 4),
               getnbyte(buffer + offset + 6, 2));
    } else {
        pages_mask = getnbyte(buffer + offset + 16, 4);
        pages_mask <<= 32;
        pages_mask += getnbyte(buffer + offset + 20, 4);

        puts("Mode Pages supported by this binary and target:");
        puts("-----------------------------------------------");
        for (i = 0; i < MAX_PAGENO; i++)
            if (pages_sup & (1LL << i))
                printf("0x%02x: %s Page%s\n", i, get_page_name(i),
                       (pages_mask & (1LL << i)) ? " (notched)" : "");
        if (pages_sup & (1LL << 12)) {
            printf("\nCurrent notch is %d.\n", 
                   getnbyte(buffer + offset + 6, 2));
        }
        if (!pages_sup)
            puts("No mode pages supported (SCSI-1?).");
    }

    return 0;
}

#define DEVNAME_SZ 256

static int open_sg_dev(char * devname)
{
    int fd, err, bus, bbus, k;
    My_scsi_idlun m_idlun, mm_idlun;
    int do_numeric = 1;
    char name[DEVNAME_SZ];
    struct stat a_st;
    int block_dev = 0;

    strncpy(name, devname, DEVNAME_SZ);
    name[DEVNAME_SZ - 1] = '\0';
    fd = open(name, O_RDONLY);
    if (fd < 0)
        return fd;
    if (fstat(fd, &a_st) < 0) {
        fprintf(stderr, "could do fstat() on fd ??\n");
        close(fd);
        return -9999;
    }
    if (S_ISBLK(a_st.st_mode))
        block_dev = 1;
    if (block_dev || (ioctl(fd, SG_GET_TIMEOUT, 0) < 0)) {
        err = ioctl(fd, SCSI_IOCTL_GET_BUS_NUMBER, &bus);
        if (err < 0) {
            perror("A SCSI device name is required\n");
            close(fd);
            return -9999;
        }
        err = ioctl(fd, SCSI_IOCTL_GET_IDLUN, &m_idlun);
        if (err < 0) {
            perror("A SCSI device name is required\n");
            close(fd);
            return -9999;
        }
        close(fd);
    
        for (k = 0; k < MAX_SG_DEVS; k++) {
            make_dev_name(name, k, do_numeric);
            fd = open(name, O_RDWR | O_NONBLOCK);
            if (fd < 0) {
                if ((ENOENT == errno) && (0 == k)) {
                    do_numeric = 0;
                    make_dev_name(name, k, do_numeric);
                    fd = open(name, O_RDWR | O_NONBLOCK);
                }
                if (fd < 0) {
                    if (EBUSY == errno)
                        continue;   /* step over if O_EXCL already on it */
                    else
                        break;
                }
            }
            err = ioctl(fd, SCSI_IOCTL_GET_BUS_NUMBER, &bbus);
            if (err < 0) {
                perror("sg ioctl failed");
                close(fd);
                fd = -9999;
            }
            err = ioctl(fd, SCSI_IOCTL_GET_IDLUN, &mm_idlun);
            if (err < 0) {
                perror("sg ioctl failed");
                close(fd);
                fd = -9999;
            }
            if ((bus == bbus) && 
                ((m_idlun.mux4 & 0xff) == (mm_idlun.mux4 & 0xff)) &&
                (((m_idlun.mux4 >> 8) & 0xff) == 
                                        ((mm_idlun.mux4 >> 8) & 0xff)) &&
                (((m_idlun.mux4 >> 16) & 0xff) == 
                                        ((mm_idlun.mux4 >> 16) & 0xff)))
                break;
            else {
                close(fd);
                fd = -9999;
            }
        }
    }
    if (fd >= 0) {
#ifdef SG_GET_RESERVED_SIZE
        int size;

        if (ioctl(fd, SG_GET_RESERVED_SIZE, &size) < 0) {
            fprintf(stderr, "Compiled with new driver, running on old!!\n");
            close(fd);
            return -9999;
        }
#endif
        close(fd);
        return open(name, O_RDWR);
    }
    else
        return fd;
}

static void usage(char *errtext)
{
    fprintf(stderr, "Error: sginfo - %s\n", errtext);
    fprintf(stderr, "Usage: sginfo [-options] [device] [replacement_values]\n");
    fputs("\tAllowed options are:\n"
          "\t-6    Do 6 byte mode sense and select commands (def: 10 bytes).\n"
          "\t-a    Display mode pages: equivalent to '-cCDefgiInpPsV'.\n"
          "\t-c    Display information from Caching Page.\n"
          "\t-C    Display information from Control Mode Page.\n"
          "\t-d    Display defect lists (default format: index).\n"
          "\t-D    Display information from Disconnect-Reconnect Page.\n"
          "\t-e    Display information from Error Recovery page.\n"
          "\t-f    Display information from Format Device Page.\n"
          "\t-Farg Format of the defect list:\n"
          "\t\t-Flogical  - logical blocks\n"
          "\t\t-Fphysical - physical blocks\n"
          "\t\t-Findex    - defect bytes from index\n", stdout);
    fputs("\t-g    Display information from Rigid Disk Drive Geometry Page.\n"
          "\t-G    Display 'grown' defect list (default format: index).\n"
          "\t-i    Display information from Inquiry command.\n"
          "\t-I    Display Informational Exception page.\n"
          "\t-l    List known scsi devices on the system\n"
          "\t-L    List pages supported by this program and target device\n"
          "\t-n    Display information from Notch and Partition Page.\n"
          "\t-N    Negate (stop) storing to saved page (active with -R).\n"
          "\t-P    Display information from Power Condition Page.\n"
          "\t-s    Display information from unit serial number page.\n"
          "\t-T    Trace commands (for debugging, double for more)\n"
          "\t-u<no> Display information from page number <no>.\n"
          "\t-v    Show version number\n"
          "\t-V    Display information from Verify Error Recovery Page.\n"
          "\n", stdout);
    fputs("\tOnly one of the following three options can be specified.\n"
   "\tNone of these three implies the current values are returned.\n", stdout);
    fputs("\t-m    Display modifiable fields instead of current values\n"
      "\t-M    Display manufacturer defaults instead of current values\n"
          "\t-S    Display saved defaults instead of current values\n\n"
          "\t-X    Display output values in a list.\n"
    "\t-R    Replace parameters - best used with -X (expert use only)\n"
    "\t      [replacement parameters placed after device on command line]\n\n"
    "\t      See man page.\n", stdout);
    exit(2);
}

int main(int argc, char *argv[])
{
    int k;
    int user_page = -1;
    int user_subpage = 0;
    char c;
    int page_control;
    int status = 0;
    char all = 0;
    int i;
    long tmp;

    if (argc < 2)
        usage("too few arguments");
    while ((k = getopt(argc, argv, "6acCdDefgGiIlLmMnNPRsSTvVXu:F:")) != EOF) {
        c = (char)k;
        switch (c) {
        case '6':
            mode6byte = 1;
            break;
        case 'a':
            all = 1;
            /* equivalent to '-cCDefgiInPsV' */
            control = 1;
            cache = 1;
            disconnect = 1;
            error = 1;
            format = 1;
            geometry = 1;
            inquiry = 1;
            informational = 1;
            notch = 1;
            power = 1;
            serial_number = 1;
            verify = 1;
            break;
        case 'c':
            cache = 1;
            break;
        case 'C':
            control = 1;
            break;
        case 'd':
            defect = 1;
            break;
        case 'D':
            disconnect = 1;
            break;
        case 'e':
            error = 1;
            break;
        case 'f':
            format = 1;
            break;
        case 'F':
            if (!strcasecmp(optarg, "logical"))
                defectformat = 0x0;
            else if (!strcasecmp(optarg, "physical"))
                defectformat = 0x5;
            else if (!strcasecmp(optarg, "index"))
                defectformat = 0x4;
            else usage(
        "Illegal -F parameter, must be one of logical, physical, or index.");
            break;
        case 'g':
            geometry = 1;
            break;
        case 'G':
            grown_defect = 1;
            break;
        case 'i':
            inquiry = 1;
            break;
        case 'I':
            informational = 1;
            break;
        case 'l':
            list = 1;
            break;
        case 'L':
            list_pages = 1;
            break;
        case 'm':
            modifiable = 1;
            break;
        case 'M':
            default_param = 1;
            break;
        case 'n':
            notch = 1;
            break;
        case 'N':
            negate_sp_bit = 1;
            break;
        case 'P':
            power = 1;
            break;
        case 'R':
            replace = 1;
            break;
        case 's':
            serial_number = 1;
            break;
        case 'S':
            saved = 1;
            break;
        case 'T':
            trace++;
            break;
        case 'u':
            //x_interface = 1;
            i = sscanf(optarg, "%d,%d", &user_page, &user_subpage);
            if (1 == i)
                user_subpage = 0;
            else if (i < 1)
                usage("argument following '-u' should be of form "
                      "<pg>[,<subpg>]");
            if ((user_page < 0) || (user_page > 63) || (user_subpage < 0) ||
                (user_subpage > 255))
                usage("mode pages range from 0 .. 63, subpages from "
                      "0 .. 255");
            break;
        case 'v':
            fprintf(stdout, " %s\n", sginfo_version_str);
            return 0;
        case 'V':
            verify = 1;
            break;
        case 'X':
            x_interface = 1;
            break;
        case '?':
            usage("Unknown option");
            break;
        default:
            fprintf(stdout, "Unknown option '-%c' (ascii 0x%02x)\n", c, c);
            usage("bad option");
        };
    };

    if (saved + modifiable + default_param > 1)
        usage("only one of -m, -M, or -S allowed");
    if (x_interface && (inquiry + geometry + cache + format +
                 error + control + disconnect + defect + list_pages) > 1)
        usage("-X can be used only with exactly one display page option.");
    if (replace && !x_interface)
        usage("-R requires -X");
    if (replace && (modifiable || default_param || saved) && !list_pages)
        usage("-R not allowed for -m, -M or -S (except when -L is also used)");

    if (replace && !saved) {
        memset (is_hex, 0, 32);
        for (i = 1; i < argc - optind; i++) {
            if (strncmp(argv[optind + i], "0x", 2) == 0) {
                char *pnt = argv[optind + i] + 2;
                replacement_values[i] = 0;
        /* This is a kluge, but we can handle 64 bit quantities this way. */
                while (*pnt) {
                    if (*pnt >= 'a' && *pnt <= 'f')
                        *pnt -= 32;
                    replacement_values[i] = (replacement_values[i] << 4) |
                        (*pnt > '9' ? (*pnt - 'A' + 10) : (*pnt - '0'));
                    pnt++;
                }
                continue;
            }
            if (argv[optind + i][0] == '@') {
        /*Ensure that this string contains an even number of hex-digits */
                int len = strlen(argv[optind + i] + 1);

                if ((len & 1) || (len != strspn(argv[optind + i] + 1, 
                                                "0123456789ABCDEFabcdef")))
                            usage("Odd number of chars or non-hex digit in @hexdatafield");

                replacement_values[i] = (unsigned long) argv[optind + i];
                is_hex[i] = 1;
                continue;
            }
            /* Using a tmp here is silly but the most clean approach */
            sscanf(argv[optind + i], "%ld", &tmp);
            replacement_values[i] = tmp;
        };
        n_replacement_values = argc - optind - 1;
    };
    if (list) {
        show_devices();
        exit(0);
    }
    if (optind >= argc)
        usage("no device name given");
    glob_fd = open_sg_dev(device_name = argv[optind]);
    if (glob_fd < 0) {
        if (-9999 == glob_fd)
            fprintf(stderr, "Couldn't find sg device corresponding to %s\n",
                    device_name);
        else {
            perror("sginfo(open)");
            fprintf(stderr, "file=%s, or no corresponding sg device found\n",
                    device_name);
            fprintf(stderr, "Is sg driver loaded?\n");
        }
        exit(1);
    }

    page_control = 0;
    if (modifiable)
        page_control = 1;
    if (default_param)
        page_control = 2;
    if (saved)
        page_control = 3;

    if (!x_interface)
        printf("\n");

    if (inquiry)
        status |= do_inquiry(page_control);
    if (serial_number)
        status |= do_serial_number(page_control);
    if (error)
        status |= error_recovery_page(page_control);
    if (disconnect)
        status |= read_disconnect_reconnect_data(page_control);
    if (format)
        status |= read_format_info(page_control);
    if (geometry)
        status |= read_geometry(page_control);
    if (verify)
        status |= verify_error_recovery(page_control);
    if (cache)
        status |= read_cache(page_control);
    if (control)
        status |= read_control_page(page_control);
    if (notch)
        status |= notch_parameters_page(page_control);
    if (power)
        status |= power_condition_page(page_control);
    if (informational)
        status |= read_informational_page(page_control);
    if (list_pages)
        status |= show_pages(page_control);
    if (user_page != -1)
        status |= do_user_page(page_control, user_page, user_subpage);
    if (defect)
        status |= read_defect_list(page_control, 0);
    if (grown_defect)
        status |= read_defect_list(page_control, 1);

    if (all)
        return 0;
    return status ? 1 : 0;
}
