
/***************************************************************************
                          imageviewer.cpp  -  description
                             -------------------
    begin                : Sat Dec 1 2001
    copyright            : (C) 2001 by Richard Groult
    email                : rgroult@jalix.org 
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "imageviewer.h"

#include "imagelistview.h"
#include "mainwindow.h"
#include "fileiconitem.h"
#include "directoryview.h"

#include <stdlib.h>
#include <math.h>

#include <kmessagebox.h>
#include <kiconloader.h>
#include <kimageeffect.h>
#include <krun.h>
#include <kaction.h>
#include <kpixmapio.h>
#include <kaction.h>
#include <klocale.h>
#include <kwin.h>
#include <kpopupmenu.h>
#include <kstandarddirs.h>
#include <kapplication.h>
#include <kfiledialog.h>


#include <qcanvas.h>
#include <qimage.h>
#include <qpainter.h>
#include <qmovie.h>
#include <qmenubar.h>
#include <qfiledialog.h>
#include <qmessagebox.h>
#include <qpopupmenu.h>
#include <qlabel.h>
#include <qpainter.h>
#include <qkeycode.h>
#include <qdatetime.h>  
#define ZOOM_MAX 150.0

int min(int a, int b)
{
	return a<b?a:b;
}
int max(int a, int b)
{
	return a>b?a:b;
}

static int nbrMU;

ImageViewer::ImageViewer (QWidget * parent,
			ImageListView * il,
			MainWindow *mw,
			QString name, int wFlags):
QWidget (parent, name.ascii(), wFlags),
filename (0)
{
	this->il = il;
	this->mw = mw;
	e_grayscale=e_normalize=e_equalize=e_intensity=e_emboss=e_swirl=e_spread=e_implode=e_charcoal=e_invert=false;
	total = 0;
	ss = true;	
	fit = false;
	shrink=false;
	enlarge=false;
	isFitWidth=false;
	isFitHeight=false;

	preloadedImage = NULL;
	preloadedScaledImage = NULL;
	image = NULL;
	imageScaled=NULL;

	sp=ep=NULL;
	movie = 0L;
	hasimage = false;
	nbImg=0;
	//
	dragStartPosX = -1;
	dragStartPosY = -1;
	difTopPosX = -1;
	difTopPosX = -1;
	realPosX=-1,
	realPosY=-1;
	button=NoButton;
	scale=1;
	setToGrayscale(-1);
	imageIndex=-1;
	//
	pIO=new KPixmapIO();
	
	setFocusPolicy (StrongFocus);
	setFocusPolicy (WheelFocus);
}


void
ImageViewer::initMenu(KActionCollection *actionCollection)
{
	m_popup = new KPopupMenu();
	actionCollection->action("FullScreen")->plug(m_popup);

	m_popup->insertSeparator();
	KPopupMenu *popupsort = new KPopupMenu ();
	actionCollection->action("Zoom in")->plug(popupsort);
	actionCollection->action("Zoom out")->plug(popupsort);
	actionCollection->action("Fit to Screen")->plug(popupsort);
	actionCollection->action("Fit the width")->plug(popupsort);
	actionCollection->action("Fit the height")->plug(popupsort);
	actionCollection->action("Originale size")->plug(popupsort);
	actionCollection->action("ZoomLock")->plug(popupsort);
	popupsort->insertSeparator();
	actionCollection->action("Enlarge")->plug(popupsort);
	actionCollection->action("Shrink")->plug(popupsort);
	m_popup->insertItem(i18n("Zoom"), popupsort);

	actionCollection->action("view_Orientation")->plug(m_popup);
	
	actionCollection->action("view_effects")->plug(m_popup);

	KPopupMenu *popupmove = new KPopupMenu ();
	actionCollection->action("First Image")->plug(popupmove);
	actionCollection->action("Previous Image")->plug(popupmove);
	actionCollection->action("Next Image")->plug(popupmove);
	actionCollection->action("Last Image")->plug(popupmove);
	actionCollection->action("Slideshow")->plug(popupmove);
	m_popup->insertItem(i18n("Go to..."), popupmove);

	m_popup->insertSeparator();
	//actionCollection->action("editcopypixmap")->plug(m_popup);
	m_popup->insertSeparator();
	actionCollection->action("filesave")->plug(m_popup);
	actionCollection->action("editcopy")->plug(m_popup);

	m_popup->insertSeparator();
	actionCollection->action("edittrash")->plug(m_popup);
	actionCollection->action("editdelete")->plug(m_popup);

	m_popup->insertSeparator();
	actionCollection->action("Image Info")->plug(m_popup);
	actionCollection->action("Properties")->plug(m_popup);
}



QPixmap
ImageViewer::getPixmap()
{
	return pIO->convertToPixmap(*image);
}


QString
ImageViewer::getFilename()
{
	return imageName;
}


void
ImageViewer::setToGrayscale(int togray)
{
	grayscale=togray;
}


int
ImageViewer::toGrayscale()
{
	return grayscale;
}

bool
ImageViewer::preloadImage (QString fileName)
{
//	kdDebug() << __FILE__ << __LINE__ << " begin ImageViewer::preloadImage (QString fileName=" << fileName <<  endl;
	if(!mw->dirView->isImage(fileName)) fileName=QString();
	
	QFile::Offset big = 0x200000;// 2MB
	QFile qfile( fileName );
	if( qfile.size() > big ) fileName=QString();
	
	
	
	preimageName=fileName;
	delete(preloadedImage); preloadedImage=new QImage();
	if(!preloadedImage->load(fileName))
	{
		delete(preloadedImage); preloadedImage=NULL;
		delete(preloadedScaledImage);preloadedScaledImage=NULL;
		return false;
	}
	scalePreload();
	return true;
//	kdDebug() << __FILE__ << __LINE__ << " end ImageViewer::preloadImage (QString fileName=" << fileName <<  endl;	return true;
}


void 
ImageViewer::reload()
{
	loadImage(filename);
}


bool
ImageViewer::loadImage (QString fileName, int index)
{
//	kdDebug() << __FILE__ << __LINE__ << " begin ImageViewer::loadImage (QString fileName, int index)" << fileName << " " << index << endl;
	if(!mw->dirView->isImage(fileName)) fileName=QString();
	bool ok = false;
	if (!fileName.isEmpty())
	{
		imageIndex=index;
		FileIconItem *item=il->findItem(QFileInfo(fileName).filePath(), true);
		if(item)
		{
			if(!item->isSelected())
			{
				delete (preloadedImage);preloadedImage=NULL;
				delete (imageScaled);imageScaled=NULL;
				return false;
			}
		}
		else
		{
			delete (preloadedImage);preloadedImage=NULL;
			delete (imageScaled);imageScaled=NULL;
			return false;
		}

		mw->setMessage(i18n("Loading image..."));

		filename = fileName;
		nbImg++;
		if (!fileName.compare(preimageName))
		{
			delete (image); image = preloadedImage;
			preloadedImage=NULL;
			if(preloadedScaledImage)
			{
				delete(imageScaled);imageScaled=preloadedScaledImage;
				preloadedScaledImage=NULL;
			}

			ok=reconvertImage();
		}
		if(!ok)
		{
			delete(image); image = new QImage ();
			ok=image->load(filename);
			delete (preloadedImage);preloadedImage=NULL;
			delete (imageScaled);imageScaled=NULL;

			ok=reconvertImage();
		}
	}
	
	if(ok)
	{
		if(movie)
		{
			movie->disconnectUpdate(this);
			movie->disconnectStatus(this);
			movie->pause();
		}
		//autoRotate(false);
		applyFilter();
		//kdDebug() << __FILE__ << __LINE__ << "  \tImageViewer::loadImage (QString fileName, int index)" << fileName << " " << index << endl;
		doScale(false);
		//kdDebug() << __FILE__ << __LINE__ << "  \tImageViewer::loadImage (QString fileName, int index)" << fileName << " " << index << endl;

		imageName=fileName;
		mw->setZoom((int)(scale*100));
		//
		imageType=QImageIO::imageFormat(filename);
		if( (imageType == "MNG")  || (imageType == "GIF")   )
		{
			repaint();
			startMovie();
		}
		else
		{
			movie=NULL;
			//repaint();
		}
	}
	else
	{
		filename=strdup("(none)");
		delete(movie);movie=NULL;
		delete(image);image=NULL;
		delete (imageScaled);imageScaled=NULL;
		delete (preloadedImage);preloadedImage=NULL;
		delete (preloadedScaledImage);preloadedScaledImage=NULL;
	}
	updateStatus ();
	hasimage = image!=NULL;

	mw->setMessage(i18n("Ready"));
	//kdDebug() << __FILE__ << __LINE__ << "  \tImageViewer::loadImage (QString fileName, int index)" << fileName << " " << index << endl;
	if(!movie)repaint();
//	kdDebug() << __FILE__ << __LINE__ << " end ImageViewer::loadImage (QString fileName, int index)" << fileName << " " << index << endl;
	return ok;

}


bool
ImageViewer::hasImage()
{
	return hasimage;
}


bool
ImageViewer::reconvertImage()
{
	if (!image)
		return FALSE;

	if(image->hasAlphaBuffer())
	{
		QPixmap pix(image->width(), image->height(),QPixmap::defaultDepth () );
		QPainter p;
			p.begin(&pix);
			p.drawTiledPixmap (0, 0, image->width(), image->height(), QPixmap(locate("appdata", "pics/bgxpm.png")));
			p.drawImage(0,0,*image);
			p.end();
		*image=pix.convertToImage();
	}
	return true;
}


bool
ImageViewer::smooth () const
{
	return ss;
}


void
ImageViewer::scalePreload ()
{
	//kdDebug() << __FILE__ << __LINE__ << " ImageViewer::scalePreload ()" << endl;
	if(!( (e_grayscale+e_normalize+e_equalize+e_intensity+e_invert+e_emboss+e_swirl+e_spread+e_implode+e_charcoal==0)
		&& preloadedImage
		&& (! preloadedImage->hasAlphaBuffer()) ))
	{
		delete(preloadedScaledImage); preloadedScaledImage=NULL;
		return;
	}
	
	
	float s;
	if ((((double)height()) / preloadedImage->height ()) < (((double) width ()) / preloadedImage->width ()))
		s = ((double)height()) / preloadedImage->height ();
	else
		s = ((double) width ()) / preloadedImage->width ();

	if(!lock)
	{
		if( (s>1 && enlarge) ||  (s<1 && shrink) )
		{
		}
		else
			s=1;
	}
	else
		s=scale;

	QPoint rtp(0,0);
	QPoint rbp((int)ceil(width()/s),(int)ceil(height()/s));
	QRect redraw(rtp,rbp);
	int
		cx=0,
		cy=0,
		cw=min(preloadedImage->width(),  redraw.width()),
		ch=min(preloadedImage->height(), redraw.height());

	int options=ColorOnly||ThresholdDither||ThresholdAlphaDither||AvoidDither;
	delete(preloadedScaledImage);preloadedScaledImage=new QImage();
	*preloadedScaledImage=preloadedImage->copy(cx, cy, cw, ch, options).smoothScale((int)ceil(cw*s), (int)ceil(ch*s));
}


void
ImageViewer::startMovie()
{
	delete(movie); movie=NULL;
	if (!filename.isEmpty()) // Start a new movie - have it delete when closed.
		initMovie();
}


void
ImageViewer::initMovie()
{
	movie = new QMovie(filename);
	nbrMU=-1;
	QPixmap pix(image->width(), image->height()); pix.fill(bgBrush.color()); *image=pIO->convertToImage(pix);//repaint();
	movie->setBackgroundColor(bgBrush.color());
	movie->connectUpdate(this, SLOT(movieUpdated(const QRect&)));
	movie->connectStatus(this, SLOT(movieStatus(int)));
}


void
ImageViewer::movieUpdated(const QRect& )
{
	if(! movie)	
		return;
	nbrMU++;
	if(nbrMU>movie->frameNumber()) /* the gif is not animated */
	{
		movie->disconnectUpdate(this);
		movie->disconnectStatus(this);
		movie->pause();
		movie=0L;
		delete(movie); movie=NULL;
		delete(image); image=new QImage(filename);
		delete (imageScaled);imageScaled=NULL;
		reconvertImage();
		applyFilter();
		doScale(false);
		//repaint();
		return;
	}
	else
	{
		*image = pIO->convertToImage(movie->framePixmap());
		if(nbrMU!=0)repaint();
	}
}


void
ImageViewer::movieStatus(int status)
{
	if (movie && status<0)
		KMessageBox::error(this, i18n("Could not play movie \"%1\"").arg(filename));
	if(status==3) nbrMU=-1;
}


void
ImageViewer::movieResized(const QSize& )
{
}


void
ImageViewer::mousePressEvent (QMouseEvent * e)
{
	QWidget::mousePressEvent(e);

	button= e->button ();
	if (button==RightButton)
	{
		m_popup->removeItemAt(7);
		m_popup_openWith=il->popupOpenWith();
		m_popup->insertItem(i18n("Open with"), m_popup_openWith, -1, 7);
		il->setSelected(il->currentItem(), true, false);
		m_popup->exec(e->globalPos());
	}
	else
	if (button==LeftButton)
	{
		if(!image) return;
		KApplication::setOverrideCursor (sizeAllCursor);
		dragStartPosX=e->pos().x();
		dragStartPosY=e->pos().y();	
		difTopPosX = getVirtualPosX()-dragStartPosX;
		difTopPosY = getVirtualPosY()-dragStartPosY;
	}
	else
	{
		delete(sp); sp = new QPoint(e->pos()) ;
		ep = new QPoint(*sp);
	}
}


void
ImageViewer::mouseReleaseEvent (QMouseEvent * e)
{
	if (e->button()==LeftButton)
	{
		if(!image) return;
		KApplication::restoreOverrideCursor ();
		QWidget::mouseReleaseEvent(e);
		dragStartPosX=-1;
		dragStartPosY=-1;	
		repaint();
	}
	else
	if (e->button()==RightButton)
	{
	}
	else
	if(e->button()!=MidButton)
	{
	}
	else
	if(image)
	{
		delete(ep); ep = new QPoint(e->pos()) ;
		if(*sp==*ep)
		{
			ep=NULL;
			KApplication::setOverrideCursor (waitCursor);	// this might take time
			doScale(false);placeImage(false);repaint();
			KApplication::restoreOverrideCursor ();	// restore original cursor
			return;
		}
		else
		{
			QRect rectDraged= QRect(QPoint(min(sp->x(), ep->x()), min(sp->y(), ep->y())),
					 QPoint(max(sp->x(), ep->x()), max(sp->y(), ep->y())));
			rectDraged.moveBy(-getVirtualPosX(), -getVirtualPosY());
			QPoint oldCenter=QPoint( (int)(((max(sp->x(), ep->x())+min(sp->x(), ep->x()))/2-getVirtualPosX())/scale),
						 (int)(((max(sp->y(), ep->y())+min(sp->y(), ep->y()))/2-getVirtualPosY())/scale));			
			float
				sx=width()/rectDraged.width(),
				sy=height()/rectDraged.height();
			if(scale*((float)sx+(float)sy)/2 > ZOOM_MAX)	
				scale=ZOOM_MAX;
			else
				scale*=((float)sx+(float)sy)/2;
			
			mw->setZoom((int)(scale*100));
			oldCenter*=scale;			
			ep=NULL;
			
			centerImage(oldCenter.x(), oldCenter.y(), true);
		}
		ep=NULL;
		KApplication::restoreOverrideCursor ();
	}
	ep=NULL;
	button=NoButton;
}


void
ImageViewer::mouseMoveEvent(QMouseEvent * e)
{
	if (button==LeftButton)
	{
		QWidget::mouseMoveEvent(e);
		if ( (dragStartPosX+dragStartPosY!=-2.0))
		{
			double
				diffX=e->pos().x()-dragStartPosX,
				diffY=e->pos().y()-dragStartPosY,
				panX=0, panY=0;
				
			if (virtualPictureWitdth()>width())
			{
				if(fabs(diffX)>=scale)
				{
					panX=(int)diffX;
					dragStartPosX+=panX;
					if(!posXForTopXIsOK(difTopPosX+dragStartPosX))
					{	
						dragStartPosX-=panX;
						panX=0;
					}
				}
			}
			if (virtualPictureHeight()>height())
			{
				if(fabs(diffY)>=scale)
				{
					panY=diffY;
					dragStartPosY+=panY;
					if(!posYForTopYIsOK(difTopPosY+dragStartPosY))
					{
						dragStartPosY-=panY;
						panY=0;
					}
				}
			}
			if(panX!=0 || panY!=0)
				scroll((int)panX, (int)panY);
		}
		return;
	}
	else
	if(movie || !ep)
		return;

	QPainter p(this); p.setPen(QColor("black"));
	lp = new QPoint(*ep);
	ep = new QPoint(e->pos()) ;
	int
		lx = lp->x(),
		ly = lp->y(),
		ex = ep->x(),
		ey = ep->y(),
		sx = sp->x(),
		sy = sp->y();
		
	int tlx,tly,brx,bry;

		tlx=(sx<ex?sx:ex);
		tly=(ly<ey?ly:ey);
		brx=(sx>ex?sx:ex);
		bry=(ly>ey?ly:ey);
		repaint(QRect(QPoint(tlx ,tly), QPoint(brx ,bry)));
		
		tlx=(lx<ex?lx:ex);
		tly=(sy<ey?sy:ey);
		brx=(lx>ex?lx:ex);
		bry=(sy>ey?sy:ey);
		repaint(QRect(QPoint(tlx ,tly), QPoint(brx ,bry)));

		tlx=(lx<ex?lx:ex);
		tly=(ly<ey?ly:ey);
		brx=(lx>ex?lx:ex);
		bry=(ly>ey?ly:ey);
		repaint(QRect(QPoint(tlx ,tly), QPoint(brx ,bry)));

		tlx=(lx<sx?lx:sx);
		tly=(sy<ly?sy:ly);
		brx=(lx>sx?lx:sx);
		bry=(sy>ly?sy:ly);
		repaint(QRect(QPoint(tlx ,tly), QPoint(brx ,bry)));

	p.drawRect(QRect(*sp,*ep));
}

void
ImageViewer::mouseDoubleClickEvent ( QMouseEvent * e )
{
	if(e->button () == LeftButton)
	{
		mw->slotFullScreen();
	}
}

void
ImageViewer::wheelEvent (QWheelEvent * e)
{
	//kdDebug() << __FILE__ << __LINE__ << " begin ImageViewer::wheelEvent (QWheelEvent * e)" << endl;
	e->accept();
	if(e->state() == Qt::ShiftButton)
	{
		if (e->delta () > 0)
			zoomOut(1.5);
		else
			zoomIn(1.5);
		
	}
	else
	if (e->delta () < 0)
		il->next ();
	else
		il->previous ();
	//kdDebug() << __FILE__ << __LINE__ << " end ImageViewer::wheelEvent (QWheelEvent * e)" << endl;
}

#define CENTER 1
#define MOSAIC 2
#define CENTER_MOSAIC 3
#define CENTER_MAX 4
#define ADAPT 5
#define LOGO 6
			
void
ImageViewer::wallpaperC(){wallpaper(CENTER);}
void
ImageViewer::wallpaperM(){wallpaper(MOSAIC);}
void
ImageViewer::wallpaperCM(){wallpaper(CENTER_MOSAIC);}
void
ImageViewer::wallpaperCMa(){wallpaper(CENTER_MAX);}
void
ImageViewer::wallpaperA(){wallpaper(ADAPT);}
void
ImageViewer::wallpaperL(){wallpaper(LOGO);}


void
ImageViewer::wallpaper(int mode)
{
	/*
	1:center
	2:mosaic
	3:center mosaic
	4:center max
	5:adapt
	6:logo
	*/
	if(mode>LOGO)
		return
	
	mw->setMessage(i18n("Set as Wallpaper"));	
	QString com=QString("dcop kdesktop KBackgroundIface setWallpaper '%1' %2 >/dev/null 2>/dev/null")
				.arg(filename)
				.arg(mode);
	KRun::runCommand(com);
	mw->setMessage(i18n("Ready"));
}



void
ImageViewer::setFit(bool fit, bool keep)
{
	//kdDebug() << __FILE__ << __LINE__ << " ImageViewer::setFit(bool fit=, bool keep=)" << fit << keep<< endl;

	if(keep)
		this->fit=fit;
	delete(imageScaled); imageScaled=NULL;
	delete(preloadedImage); preloadedImage=NULL;
	imageScaled=NULL;if(fit)
		fitSize();
	else
		doScale(true);
}

void
ImageViewer::setSmooth(bool s)
{
	ss=s;
	doScale(true);
}




bool ImageViewer::autoRotate(bool r)
{
    KFileMetaInfo metadatas((QString)filename );
    if ( !metadatas.isValid() )
        return false;

    KFileMetaInfoItem metaitem = metadatas.item("Orientation");
    if ( !metaitem.isValid()
#if QT_VERSION >= 0x030100
        || metaitem.value().isNull()
#endif
        )
        return false;

    switch ( metaitem.value().toInt() )
    {
        //  Orientation:
        //  1:      normal
        //  2:      flipped horizontally
        //  3:      ROT 180
        //  4:      flipped vertically
        //  5:      ROT 90 -> flip horizontally
        //  6:      ROT 90
        //  7:      ROT 90 -> flip vertically
        //  8:      ROT 270

        case 1:
        default:
		break;
        case 2:
		mirror(true, false, r);
		break;
        case 3:
		mirror(true, true, r);
		break;
        case 4:
		mirror(false, true, r);
		break;
        case 5:
		rotateLeft(r);
		mirror(true, false, r);
		break;
        case 6:
		rotateRight(r);
		break;
        case 7:
		rotateRight(r);
		mirror(false, true, r);
		break;
        case 8:
		rotateLeft(r);
            break;
    }
    return true;
}





void
ImageViewer::scroll( int dx, int dy )
{
	QWidget::scroll(dx,dy);	
}

void
ImageViewer::mirror (bool horizontal, bool vertical, bool r)
{
	if(!image) return;
	KApplication::setOverrideCursor (waitCursor);	// this might take time

	QPixmap pix(image->width(), image->height());
	QPainter p;
	p.begin(&pix);
	if(vertical)
	{
		p.scale(1,-1);
		p.drawImage(0, -image->height(), *image);
	}
	else
	if(horizontal)
	{
		p.scale(-1,1);	
		p.drawImage(-image->width(), 0, *image);
	}
	p.end();
	*image=pIO->convertToImage(pix);
	
	delete(imageScaled); imageScaled=NULL;
	
	centerImage(false);
	placeImage(r);

	KApplication::restoreOverrideCursor ();	// restore original cursor
}


void
ImageViewer::rotateLeft(bool r)
{
	if(!image) return;
	KApplication::setOverrideCursor (waitCursor);	// this might take time

	QPixmap pix(image->height(), image->width());
	QPainter p;
		p.begin(&pix);
		p.rotate(-90);
		p.drawImage(-image->width(), 0, *image);
	p.end();
	*image=pIO->convertToImage(pix);
	
	delete(imageScaled); imageScaled=NULL;
	
	centerImage(false);
	placeImage(r);

	KApplication::restoreOverrideCursor ();	// restore original cursor
}


void
ImageViewer::rotateRight(bool r)
{
	if(!image) return;
	KApplication::setOverrideCursor (waitCursor);	// this might take time

	QPixmap pix(image->height(), image->width());
	QPainter p;
	p.begin(&pix);
		p.rotate(90);
		p.drawImage( 0, -image->height(),*image);
	p.end();
	*image=pIO->convertToImage(pix);
	
	delete(imageScaled); imageScaled=NULL;
	
	centerImage(false);
	placeImage(r);

	KApplication::restoreOverrideCursor ();	// restore original cursor
}


void
ImageViewer::zoomIn(float rate)
{
	if(scale<ZOOM_MAX)
	{	
		KApplication::setOverrideCursor (waitCursor);	// this might take time
		
		QPoint center(width()/2, height()/2);
		center/=scale;
		center+=QPoint(getPosX(), getPosY());		
		if(scale*rate>ZOOM_MAX)
			scale=ZOOM_MAX;
		else
			scale*=rate;
		centerImage((int)(center.x()*scale), (int)(center.y()*scale), true);
		
		KApplication::restoreOverrideCursor ();	// restore original cursor
		mw->setZoom((int)(scale*100));	
		
		delete(imageScaled); imageScaled=NULL;
	}
}


void
ImageViewer::zoomOut(float rate)
{
	if(scale>1.0/ZOOM_MAX)
	{	
		KApplication::setOverrideCursor (waitCursor);	// this might take time
		QPoint center(width()/2, height()/2);
		center/=scale;
		center+=QPoint(getPosX(), getPosY());		
		if(scale/rate<=1.0/ZOOM_MAX)
			scale=1.0/ZOOM_MAX;
		else
			scale/=rate;
		centerImage((int)(center.x()*scale), (int)(center.y()*scale), true);
		KApplication::restoreOverrideCursor ();	// restore original cursor
		mw->setZoom((int)(scale*100));
		
		delete(imageScaled); imageScaled=NULL;
	}
}


void
ImageViewer::updateStatus ()
{
	if (!image || (image && image->size() == QSize(0,0)))
	{
		if (filename!=QString("(none)"))
		{
			mw->setNbrItems(total);
			mw->setZoom((int)(scale*100));			
			QString *fileName=new QString(filename);
			int pos = fileName->findRev ("/");
			mw->setImagename(fileName->right (fileName->length () - pos-1));
			pos = fileName->findRev(".");
			mw->setImagetype(imageType);
			mw->setDim(new QSize(0,0));
			QFileInfo fi(filename);
			mw->setSize(fi.size());
			mw->setDate(new QDateTime(fi.lastModified()));
			mw->setImageIndex(imageIndex);
		}
		else
		{
			mw->setZoom((int)(scale*100));
			mw->setImagename(NULL);
			mw->setImagetype(NULL);
			mw->setDim(NULL);
			mw->setSize(-1);
			mw->setDate(NULL);
			mw->setImageIndex(-1);
		}
	}
	else
	{		
		if (filename!=QString("(none)"))
		{
			mw->setZoom((int)(scale*100));
			//
			QString *fileName=new QString(filename);
			int pos = fileName->findRev ("/");
			mw->setImagename(fileName->right (fileName->length () - pos-1));
			//
			pos = fileName->findRev(".");
			mw->setImagetype(imageType);
			//
			mw->setDim(new QSize(image->size()));
			//
			QFileInfo fi(filename);
			mw->setSize(fi.size());
			//
			mw->setDate(new QDateTime(fi.lastModified()));
			mw->setImageIndex(imageIndex);
		}
		else
		{
			mw->setZoom((int)(scale*100));
			mw->setImagename("(none)");
			mw->setImagetype("");
			mw->setDim(new QSize(0,0));
			mw->setSize(0);		
			mw->setDate(NULL);
		}
	}	
}



int ImageViewer::virtualScreenWitdth(){return (int)floor(width()/scale);}
int ImageViewer::virtualScreenHeight(){return (int)floor(height()/scale);}

int ImageViewer::virtualPictureWitdth(){ return (image!=NULL)?(int)floor(image->width()*scale):0;}
int ImageViewer::virtualPictureHeight(){return (image!=NULL)?(int)floor(image->height()*scale):0;}

void ImageViewer::setVirtualPosX(double posX){topPosX=posX; realPosX=-(int)floor(posX/scale);}
void ImageViewer::setVirtualPosY(double posY){topPosY=posY; realPosY=-(int)floor(posY/scale);}
int ImageViewer::getVirtualPosX(){return (int)topPosX;}
int ImageViewer::getVirtualPosY(){return (int)topPosY;}
QPoint ImageViewer::getVirtualPos(){return QPoint(getVirtualPosX(), getVirtualPosY());}
void ImageViewer::setVirtualPos(QPoint pos){setVirtualPosX(pos.x());setVirtualPosY(pos.y());}

void ImageViewer::setPosX(double posX){realPosX= (int)((posX/fabs(posX)) * floor(fabs(posX)));}
void ImageViewer::setPosY(double posY){realPosY= (int)((posY/fabs(posY)) * floor(fabs(posY)));}
void ImageViewer::setPos(QPoint pos){setPosX(pos.x()); setPosY(pos.y());}
int ImageViewer::getPosX(){return (int)floor(realPosX);}
int ImageViewer::getPosY(){return (int)floor(realPosY);}
QPoint ImageViewer::getPos(){return QPoint(getPosX(), getPosY());}

void
ImageViewer::placeImage(bool redraw)
{
	//kdDebug() << __FILE__ << __LINE__ << " begin ImageViewer::placeImage(bool redraw=" << redraw << endl;

/*
	int
		oldVPX=getVirtualPosX(),
		oldPY=getVirtualPosY();
*/
	//	
	if(!posXForTopXIsOK(getVirtualPosX()))
		setVirtualPosX(0);
	if(virtualPictureWitdth()<=width())
		centerXImage();
	//	
	if(!posYForTopYIsOK(getVirtualPosY()))
		setVirtualPosY(0);		
	if(virtualPictureHeight()<=height())
		centerYImage();
	//
	if(
	redraw 
	/*
	&&
		((oldVPX!=getVirtualPosX()) ||
		 (oldPY!=getVirtualPosY()))
	*/
	 )
	
		repaint();
	//kdDebug() << __FILE__ << __LINE__ << " end ImageViewer::placeImage(bool redraw=" << redraw << endl;
}


void
ImageViewer::centerImage(int posX, int posY, bool redraw)
{
	//kdDebug() << __FILE__ << __LINE__ << " begin ImageViewer::centerImage(int posX=, int posY=, bool redraw=)" <<posX  <<posY<< redraw << endl;
	int
		oldVPX=getVirtualPosX(),
		oldVPY=getVirtualPosY();
	
	if (virtualPictureWitdth()>width())
	{
		int possibleX = width()/2 - posX;
		if(posXForTopXIsOK(possibleX))
		{
			setVirtualPosX(possibleX);
		}
		else
		{
			posX-=getVirtualPosX();
			if ( (posX>width()/2) &&
				(virtualPictureWitdth()+getVirtualPosX()-width()<abs(width()-posX)))
			{
				posX=width()-virtualPictureWitdth();
				setVirtualPosX(posX);
			}
			else
			{
				posX=0;
				setVirtualPosX(posX);
			}
		}
	}
	else
		centerXImage();
	//
	if(virtualPictureHeight()>height())
	{
		int possibleY = height()/2 - posY;
		
		if(posYForTopYIsOK(possibleY))
		{
			setVirtualPosY(possibleY);
		}
		else
		{
			posY-=getVirtualPosY();	
			if ((posY>height()/2) &&
				(virtualPictureHeight()+getVirtualPosY()-height()<abs(height()-posY)))
			{
				posY=height()-virtualPictureHeight();
				setVirtualPosY(posY);
			}
			else
			{
				posY=0;
				setVirtualPosY(posY);
			}
		}
	}
	else
		centerYImage();	
	
	if(redraw &&
		((oldVPX!=getVirtualPosX()) ||
		 (oldVPY!=getVirtualPosY())))
	{
		repaint();
	}
	//kdDebug() << __FILE__ << __LINE__ << " end ImageViewer::centerImage(int posX=, int posY=, bool redraw=)" <<posX  <<posY<< redraw << endl;
}

void
ImageViewer::centerImage(bool redraw)
{
	//kdDebug() << __FILE__ << __LINE__ << " ImageViewer::centerImage(bool redraw=" <<redraw  << endl;

	centerXImage();
	centerYImage();
	if(redraw)
	{
		repaint();
	}
}

void
ImageViewer::centerXImage(bool redraw)
{
	int oldVPX=getVirtualPosX();
	setVirtualPosX((width () - virtualPictureWitdth()) / 2);
	
	if(redraw &&
		oldVPX!=getVirtualPosX())
	{
		repaint();
	}
}

void
ImageViewer::centerYImage(bool redraw)
{
	int oldVPY=getVirtualPosX();
	setVirtualPosY((height() - virtualPictureHeight()) / 2);
	
	if(redraw &&
		oldVPY!=getVirtualPosY())
	{
		repaint();
	}
}


void
ImageViewer::originalSize()
{
	scale=1;
	placeImage(false);
	mw->setZoom((int)(scale*100));
	
	delete(imageScaled); imageScaled=NULL;
	delete(preloadedImage); preloadedImage=NULL;
	repaint();
}

void
ImageViewer::enlargeSize()
{
}

/**
	reduce size if larger than the viewer
*/
void
ImageViewer::reduceSize()
{
	if (!image)
		return;
	else
	if (image->isNull ())
		return;
	else
	{
		float s;
		if ((((double)height()) / image->height ()) <  (((double)width ()) / image->width ()))
			s = ((double)height()) / image->height ();
		else
			s = ((double)width ()) / image->width ();

		if ( (s < 1) )
			scale=s;
		else
			scale=1;	
	}
	placeImage(false);
	mw->setZoom((int)(scale*100));
}

void
ImageViewer::fitSize(bool redraw)
{
//	kdDebug() << __FILE__ << __LINE__ << " debut ImageViewer::fitSize(bool redraw=" << redraw  << endl;
	if (!image)
		return;
	if (image->isNull ())
		return;

	float s;
	if ((((double)height()) / image->height ()) < (((double) width ()) / image->width ()))
		s = ((double)height()) / image->height ();
	else
		s = ((double) width ()) / image->width ();

	scale=s;
	placeImage(false);
	mw->setZoom((int)(scale*100));
	
	//delete(imageScaled); imageScaled=NULL;
	if(redraw)
	{
		repaint();
	}
//	kdDebug() << __FILE__ << __LINE__ << " fin ImageViewer::fitSize()"   << endl;
}

void
ImageViewer::fitWidth(bool setFitWidth, bool redraw)
{
	//kdDebug() << __FILE__ << __LINE__ << " begin ImageViewer::fitWidth(bool setFitWidth=" << setFitWidth << endl;

	isFitWidth=setFitWidth;
	isFitHeight=false;
	if(!isFitWidth)return;
	
	if (!image)
		return;
	if (image->isNull ())
		return;
	scale=((double)width()) / image->width ();
	placeImage(false);
	mw->setZoom((int)(scale*100));
	
	delete(imageScaled); imageScaled=NULL;
	delete(preloadedImage); preloadedImage=NULL;
	if(redraw)
	{
		repaint();
	}
	//kdDebug() << __FILE__ << __LINE__ << " end ImageViewer::fitWidth(bool setFitWidth=" << setFitWidth << endl;
}

void
ImageViewer::fitHeight(bool setFitHeight, bool redraw)
{
	//kdDebug() << __FILE__ << __LINE__ << " begin ImageViewer::fitHeight(bool setFitHeight=" << setFitHeight << endl;

	isFitHeight=setFitHeight;
	isFitWidth=false;
	if(!isFitHeight)return;
	
	if (!image)
		return;
	if (image->isNull ())
		return;
	scale=((double)height()) / image->height();
	placeImage(false);
	mw->setZoom((int)(scale*100));
	
	delete(imageScaled); imageScaled=NULL;
	delete(preloadedImage); preloadedImage=NULL;
	if(redraw)
	{
		repaint();
	}
	//kdDebug() << __FILE__ << __LINE__ << " end ImageViewer::fitHeight(bool setFitHeight=" << setFitHeight << endl;
}

void
ImageViewer::doScale(bool repaint)
{
	//kdDebug() << __FILE__ << __LINE__ << " begin ImageViewer::doScale(bool repaint=" << repaint << endl;

	if(!image)return;
	if (image->size () == QSize (0, 0))
		return;
	
	float s;
	if ((((double) height ()) / image->height ()) < (((double) width ()) / image->width ()))
		s = ((double) height ()) / image->height ();
	else
		s = ((double) width ()) / image->width ();
	
	if(isFitWidth) 
		fitWidth(true, false);
	else 
	if(isFitHeight) 
		fitHeight(true, false);
	else
	if(!lock)
	{
		if(s>1 && enlarge)
			scaleFit();
		else
		if(s<1 && shrink)
			scaleFit();
		else
			scale=1;
	}
	centerImage(repaint);
	//kdDebug() << __FILE__ << __LINE__ << " end ImageViewer::doScale(bool repaint=" << repaint << endl;
}


void
ImageViewer::scaleFit()
{
	fitSize(false);
}

bool
ImageViewer::posXForTopXIsOK(double posX)
{				
	int maxX=width();
	return !(
			(
				(posX+virtualPictureWitdth()>maxX) &&
				(posX>=0)
			)
		||
			(
				(posX<0) &&
				(posX+virtualPictureWitdth()<maxX)
			)
		);		
}

bool
ImageViewer::posYForTopYIsOK(double posY)
{
	int maxY=height();
	return !(
			(
				(posY+virtualPictureHeight()>maxY) &&
				(posY>=0)
			)
		||
			(
				(posY<0) &&
				(posY+virtualPictureHeight()<maxY)
			)
		);		
}



void
ImageViewer::resizeEvent (QResizeEvent *e)
{
	//kdDebug() << __FILE__ << __LINE__ << " ImageViewer::resizeEvent (QResizeEvent *e) " << endl;
	delete(imageScaled); imageScaled=NULL;
	delete(preloadedImage); preloadedImage=NULL;
	QWidget::resizeEvent(e);
	doScale(true);	
}


void
ImageViewer::setBackgroundColor(QColor col)
{
	bgBrush = QBrush(col);
	QWidget::setBackgroundColor(col);
	setBackgroundMode(NoBackground);
	repaint();
}

void
ImageViewer::paintEvent (QPaintEvent * e)
{
	//kdDebug() << __FILE__ << __LINE__ << " begin ImageViewer::paintEvent (QPaintEvent * e)"  << endl;
	if(!mw->isUpdatesEnabled()) return;
	//QWidget::paintEvent(e);
	if(image)
	{
		int options=ColorOnly||ThresholdDither||ThresholdAlphaDither||AvoidDither;
		if(dragStartPosX + dragStartPosY != -2)
		{
			setVirtualPosX(difTopPosX+dragStartPosX);
			setVirtualPosY(difTopPosY+dragStartPosY);
		}

		//////////////////
		QPoint rtp((int)floor(e->rect().topLeft().x()/scale),
				(int)floor(e->rect().topLeft().y()/scale));
		QPoint rbp((int)ceil(e->rect().bottomRight().x()/scale),
				(int)ceil(e->rect().bottomRight().y()/scale));
		QRect redraw(rtp,rbp);
		redraw.moveBy(getPosX(), getPosY());

		int
			cx=max(0,redraw.topLeft().x()),
			cy=max(0,redraw.topLeft().y()),
			cw=min(image->width(),  redraw.width() +min(0,redraw.topLeft().x())+1),
			ch=min(image->height(), redraw.height()+min(0,redraw.topLeft().y())+1);
		if(image->hasAlphaBuffer()){cw++; ch++;}

		int
			x= e->rect().topLeft().x()-min(0, (int)(ceil(redraw.topLeft().x()*scale))),
			y= e->rect().topLeft().y()-min(0, (int)(ceil(redraw.topLeft().y()*scale)));

		////////////////
		QPainter painter;
		painter.begin(this);

		if(cw>0 && ch>0)
		{
			//kdDebug() << __FILE__ << __LINE__ << " \t ImageViewer::paintEvent (QPaintEvent * e)"  << endl;
			
			if(cx==0 && cy==0 && imageScaled)
			{
				//kdDebug() << __FILE__ << __LINE__ << " \t ImageViewer::paintEvent (QPaintEvent * e)"  << endl;
				painter.drawImage(x,y,*imageScaled);
				//delete(imageScaled); imageScaled=NULL;
			}
			else
			
			if( (!smooth()) ||
				(scale==1.0)  ||
				(dragStartPosX + dragStartPosY != -2) ||
				(ep!=NULL))
			{
				//kdDebug() << __FILE__ << __LINE__ << " \t ImageViewer::paintEvent (QPaintEvent * e)"  << endl;
				QImage copiedImage=image->copy(cx, cy, cw, ch, options);
				QPixmap scaleCopy((int)ceil(cw*scale), (int)ceil(ch*scale));
				QPainter pC(&scaleCopy);
					pC.scale(scale, scale);
					pC.drawImage(0, 0, copiedImage);
					pC.end();
				painter.drawPixmap(x, y, scaleCopy);
			}			
			else
			{
				//kdDebug() << __FILE__ << __LINE__ << " \t ImageViewer::paintEvent (QPaintEvent * e)"  << endl;
				painter.drawImage (x,y, image->copy(cx, cy, cw, ch, options).smoothScale((int)ceil(cw*scale), (int)ceil(ch*scale)));
			}
		}
		
		if(getVirtualPosX()>0)
		{
			painter.fillRect(0,0,
					getVirtualPosX(), height(),
					bgBrush);
			painter.fillRect(getVirtualPosX()+virtualPictureWitdth(), 0,
					width()-(getVirtualPosX()+virtualPictureWitdth()), height(),
					bgBrush);
			painter.flush();
		}
		if(getVirtualPosY()>0)
		{
			painter.fillRect(0,0,
					width(), getVirtualPosY(),
					bgBrush);
			painter.fillRect(0,getVirtualPosY()+virtualPictureHeight(),
					width(), height()-(getVirtualPosY()+virtualPictureHeight()),
					bgBrush);
			painter.flush();
		}
		painter.flush();
		painter.end();
	}
	else
	{
		QPainter painter;
		painter.begin(this);
		painter.fillRect(0,0,width(), height(), bgBrush);
		painter.end();
	}
	//kapp->processEvents();

	//kdDebug() << __FILE__ << __LINE__ << " end ImageViewer::paintEvent (QPaintEvent * e)"  << endl;
}

void ImageViewer::setShrink(bool shrink)
{
	//kdDebug() << __FILE__ << __LINE__ << " ImageViewer::setShrink(bool shrink=" << shrink << endl;

	this->shrink=shrink;
	delete(imageScaled); imageScaled=NULL;
	delete(preloadedImage); preloadedImage=NULL;
	if(shrink)
	{
		doScale(true);		
	}
}

/** Read property of bool enlarge. */
const bool&
ImageViewer::getEnlarge()
{
	return enlarge;
}

/** Write property of bool enlarge. */
void
ImageViewer::setEnlarge(bool enlarge)
{
	//kdDebug() << __FILE__ << __LINE__ << " ImageViewer::setEnlarge(bool enlarge=" << enlarge << endl;

	this->enlarge = enlarge;
	delete(imageScaled); imageScaled=NULL;
	delete(preloadedImage); preloadedImage=NULL;
	if(enlarge)
	{
		doScale(true);
	}
}

void
ImageViewer::setZoomLock(bool lock)
{
	this->lock=lock;
}

void
ImageViewer::applyFilter(int filter, bool activate)
{
	switch(filter)
	{
		case EFFECT_GRAYSCALE : e_grayscale=activate; break;
	
		case EFFECT_NORMALIZE : e_normalize=activate; break;
		case EFFECT_EQUALIZE : e_equalize=activate; break;
		case EFFECT_INTENSITY : e_intensity=activate; break;
		case EFFECT_INVERT : e_invert=activate; break;

		case EFFECT_EMBOSS : e_emboss=activate; break;
		case EFFECT_SWIRL : e_swirl=activate; break;
		case EFFECT_SPREAD : e_spread=activate; break;
		case EFFECT_IMPLODE : e_implode=activate; break;
		case EFFECT_CHARCOAL : e_charcoal=activate; break;
	}
}

void
ImageViewer::applyFilter()
{
	if(image->size() == QSize(0,0)) return;

	if(e_normalize)
		KImageEffect::normalize(*image);
	if(e_equalize)
		KImageEffect::equalize(*image);
	if(e_intensity)
		*image=KImageEffect::intensity(*image, 0.5);
	if(e_invert)
		image->invertPixels(false);
	if(e_emboss)
		*image=KImageEffect::emboss(*image);
	if(e_swirl)
		*image=KImageEffect::swirl(*image);
	if(e_spread)
		*image=KImageEffect::spread(*image);
	if(e_implode)
		*image=KImageEffect::implode(*image);
	if(e_charcoal)
		*image=KImageEffect::charcoal(*image);	
	if(e_grayscale)
		*image=KImageEffect::desaturate(*image, toGrayscale()/100.0);


}

void
ImageViewer::applyFilterPreloaded()
{
}

void
ImageViewer::scrolldxR()
{
	if (virtualPictureWitdth()<width())
		return;
	dragStartPosX=-floor(10*scale);
	dragStartPosY=0;
	difTopPosX = getVirtualPosX();
	difTopPosY = getVirtualPosY();
	if(!posXForTopXIsOK(difTopPosX+dragStartPosX))
	{
		dragStartPosX=-1;
		dragStartPosY=-1;	
		return;
	}
	scroll((int)(dragStartPosX), (int)(dragStartPosY));
	dragStartPosX=-1;
	dragStartPosY=-1;	
}

void
ImageViewer::scrolldyB()
{
	if (virtualPictureHeight()<height())
		return;
	dragStartPosX=0;
	dragStartPosY=-floor(10*scale);
	difTopPosX = getVirtualPosX();
	difTopPosY = getVirtualPosY();
	if(!posYForTopYIsOK(difTopPosY+dragStartPosY))
	{
		dragStartPosX=-1;
		dragStartPosY=-1;	
		return;
	}
	scroll((int)(dragStartPosX), (int)(dragStartPosY));
	dragStartPosX=-1;
	dragStartPosY=-1;	
}
 void
ImageViewer::scrolldxL()
{
	if (virtualPictureWitdth()<width())
		return;
	dragStartPosX=floor(10*scale);
	dragStartPosY=0;
	difTopPosX = getVirtualPosX();
	difTopPosY = getVirtualPosY();
	if(!posXForTopXIsOK(difTopPosX+dragStartPosX))
	{
		dragStartPosX=-1;
		dragStartPosY=-1;	
		return;
	}
	scroll((int)(dragStartPosX), (int)(dragStartPosY));
	dragStartPosX=-1;
	dragStartPosY=-1;	
}

void
ImageViewer::scrolldyT()
{
	if (virtualPictureHeight()<height())
		return;
	dragStartPosX=0;
	dragStartPosY=floor(10*scale);
	difTopPosX = getVirtualPosX();
	difTopPosY = getVirtualPosY();
	if(!posYForTopYIsOK(difTopPosY+dragStartPosY))
	{
		dragStartPosX=-1;
		dragStartPosY=-1;	
		return;
	}
	scroll((int)(dragStartPosX), (int)(dragStartPosY));
	dragStartPosX=-1;
	dragStartPosY=-1;
}

QString
ImageViewer::slotSave(QString path)
{
	QString destDir=KFileDialog::getSaveFileName(path,
							"*.png *.jpg *.gif *.bmp",
							this,
							i18n("Save file as..."));
	if(destDir.isEmpty()) return destDir;
	mw->setMessage(i18n("Saving image...")); kapp->processEvents();
	KApplication::setOverrideCursor (waitCursor);
	QString ext=QFileInfo(destDir).extension().upper();
	if(ext.isEmpty())
	{
		destDir+=".png";
		ext="PNG";
	}
	else
	if(ext=="JPG")
	{
		ext="JPEG";
	}
	if(!image->save(destDir, ext.local8Bit(), 100))
	{
		KApplication::restoreOverrideCursor ();
		KMessageBox::error(this, i18n("Error saving image."));
	}
	else
		KApplication::restoreOverrideCursor ();
	mw->setMessage(i18n("Ready"));
	return destDir;

}

#include "imageviewer.moc"
