/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/
#include <inttypes.h>
#include <config.h>
#include <dvd.h>
#include <glib/gi18n.h>


#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#include "dvd.h"

/* --- other ASCII characters --- */
#define	DVD_ASCII_NUL		0x00 /* '\0' */
#define	DVD_ASCII_H_TAB		0x09 /* tab */
#define	DVD_ASCII_V_TAB		0x0b /* vertical tab */
#define	DVD_ASCII_NEW_LINE	0x0a /* new line or line feed */
#define	DVD_ASCII_NEW_PAGE	0x0c /* new page or form feed */
#define	DVD_ASCII_C_RETURN	0x0d /* carrage return */
#define	DVD_ASCII_DBL_QUOTE	0x22 /* '"' */
#define	DVD_ASCII_HASH		0x23 /* '#' */
#define	DVD_ASCII_L_PAREN	0x28 /* '(' */
#define DVD_ASCII_R_PAREN	0x29 /* ')' */
#define	DVD_ASCII_COMMA		0x2c /* ',' */
#define	DVD_ASCII_MINUS		0x2d /* '-' */
#define	DVD_ASCII_PLUS		0x2b /* '+' */
#define	DVD_ASCII_SPACE		0x20 /* ' ' */
#define	DVD_ASCII_SEMICOLAN	0x3a /* ':' */
#define	DVD_ASCII_COLAN		0x3b /* ';' */
#define	DVD_ASCII_EQUALS	0x3d /* '=' */

#define DVD_FILE_IO_PRIV(file) (DVD_FILE_IO (file)->priv)
#define DVD_FILE_IO_READ_SIZE 512

/* file error messages */
const gchar *FILE_IO_ERRORS[G_FILE_ERROR_FAILED + 1] = {
	N_("No such file or directory; The file name does not exist."),
	N_("File is a directory; You cannot open a directory for writing, or create or remove hard links to it."),
	N_("Permission denied; The file permissions do not allow the attempted operation."),
	N_("File name too long; The file name exceeds the maximum file name length on your system."),
	N_("File does not exist; A file that should be available does not exist"),
	N_("Directory is just a regular file; A file that isn't a directory was specified when a directory is required."),
	N_("No such device or address; The system tried to use the device represented by a file you specified, and it couldn't find the device. This can mean that the device file was installed incorrectly, or that the physical device is missing or not correctly attached to the computer."),
	N_("Unable to map file; This file is of a type that doesn't support mapping."),
	N_("Read only file system; The directory containing the new link can't be modified because it's on a read-only file system."),
	N_("Text file busy; Text file is busy at the moment, you may try again later"),
	N_("Bad pointer; An internal error occured - please close the program immediately and report this bug"),
	N_("Possible symbolic link loop; Too many levels of symbolic links were encountered in looking up a file name. This often indicates a cycle of symbolic links."),
	N_("Disk is full; Write operation on a file failed because the disk is full."),
	N_("No memory available; The system cannot allocate more virtual memory because its capacity is full."),
	N_("Can't open file; Program has too many files open"),
	N_("Can't open file; There are too many distinct file openings in the entire system."),
	N_("Bad file descriptor; I/O on a descriptor that has been closed or reading from a descriptor open only for writing (or vice versa)."),
	N_("Invalid argument; Program passed a wrong argument to a library function."),
	N_("Broken pipe; An internal error occured - please close the program immediately and report this bug"),
	N_("Resource temporarily unavailable; The resource may be available later."),
	N_("Interupted function call; You should try again"),
	N_("Input/output error; The disk or other physical device hardware is returning errors."),
	N_("Operation not permitted; Only the owner of the file (or other resource) or processes with special privileges can perform the operation."),
	N_("Unknown error; An unknown error occurred")
};

struct	_DvdFileIOPriv {
	gchar		*path;
	DvdFileIORW	 rw;
	gint		 fd;		/* file descriptor */
	gsize		 size;		/* file size in bytes */
	gsize		 position;	/* file descriptor position */
	guint8		 percent;	/* file progress */
};

enum {
	FILE_IO_PROGRESS,
	LAST_SIGNAL
};

static guint dvd_file_io_signals[LAST_SIGNAL];
static GObjectClass *dvd_file_io_parent_class = NULL;

static void	dvd_file_io_class_init	(DvdFileIOClass *class);
static void	dvd_file_io_instance_init	(GTypeInstance	*instance,
					 gpointer	g_class);
static void	dvd_file_io_dispose	(GObject	*object);


static void
dvd_file_io_class_init		(DvdFileIOClass *class)
{
	GObjectClass *object_class = (GObjectClass *) class;
	
	dvd_file_io_signals[FILE_IO_PROGRESS] =
		g_signal_new("file_progress",
			     G_TYPE_FROM_CLASS(class),
			     G_SIGNAL_RUN_LAST,
			     G_STRUCT_OFFSET (DvdFileIOClass, file_progress),
			     NULL, NULL,
			     g_cclosure_marshal_VOID__UINT,
			     G_TYPE_NONE, 1, G_TYPE_UINT);
	dvd_file_io_parent_class = g_type_class_ref (G_TYPE_OBJECT);
	object_class->dispose = dvd_file_io_dispose;
}

static void
dvd_file_io_instance_init	(GTypeInstance	*instance,
				 gpointer	g_class)
{
	DvdFileIO *file;
	DvdFileIOPriv *P;
	
	file = DVD_FILE_IO (instance);
	P = g_malloc0 (sizeof (DvdFileIOPriv));
	DVD_FILE_IO_PRIV (file) = P;
	P->fd = -1;
}


static void
dvd_file_io_dispose		(GObject	*object)
{
	DvdFileIO *file;
	DvdFileIOPriv *P;
	GError *error = NULL;
	
	file = DVD_FILE_IO (object);
	P = DVD_FILE_IO_PRIV (file);
	
	if (dvd_file_io_close (file, &error) == FALSE) {
		g_warning (error->message);
		if (P->path != NULL) g_free (P->path);
		g_error_free (error);
	}
	g_free (P);
	G_OBJECT_CLASS (dvd_file_io_parent_class)->dispose (G_OBJECT (file));
}


/**
 * dvd_file_io_get_type:
 *
 * Returns the GType for the DvdFileIO class.
 */

GType
dvd_file_io_get_type		(void)
{
	static GType file_type = 0;

	if (!file_type) {
		GTypeInfo file_info = {
			sizeof (DvdFileIOClass),
			NULL,
			NULL,
			(GClassInitFunc) dvd_file_io_class_init,
			NULL,
			NULL, /* class_data */
			sizeof (DvdFileIO),
			0, /* n_preallocs */
			(GInstanceInitFunc) dvd_file_io_instance_init
			};
		file_type = g_type_register_static (G_TYPE_OBJECT,
						    "DvdFileIO",
						    &file_info, 0);
	}
	return file_type;
}

DvdFileIO
*dvd_file_io_new		(void)
{
	DvdFileIO *file;
	
	file = g_object_new (DVD_FILE_IO_TYPE, NULL);
	return file;
}

gboolean
dvd_file_io_open		(DvdFileIO	 *file,
			 const gchar	 *file_path,
			 DvdFileIORW	  rw,
			 gboolean	  allow_trunc,
			 GError		**error)
{
	DvdFileIOPriv *P;
	gint length = 0;
	gint flags = 0;
	gint mode = 0;
	gboolean use_mode = FALSE;
	
	g_assert (file != NULL);
	P = DVD_FILE_IO_PRIV (file);
	g_assert (P->fd == -1);
	
	if (rw == DVD_FILE_IO_READ) {
		struct stat fstat;
		
		flags = O_RDONLY;
		
		if (g_file_test (file_path, G_FILE_TEST_EXISTS) == FALSE) {
			g_set_error (error,
				     G_FILE_ERROR,
				     G_FILE_ERROR_NOENT,
				     FILE_IO_ERRORS[G_FILE_ERROR_NOENT]);
			return FALSE;
		}
		if ((stat (file_path, &fstat) == -1) ||
		    (access (file_path, R_OK) == -1)) {
			GFileError file_error;
			
			file_error = g_file_error_from_errno (errno);
			g_set_error (error,
				     G_FILE_ERROR,
				     file_error,
				     FILE_IO_ERRORS[file_error]);
			return FALSE;
		}
		/* check for directory because */
		/* open will not fail unless a write flag is on */
		if (S_ISREG (fstat.st_mode) == 0) {
			g_set_error (error,
				     G_FILE_ERROR,
				     G_FILE_ERROR_ISDIR,
				     FILE_IO_ERRORS[G_FILE_ERROR_ISDIR]);
			return FALSE;
		}
		length = fstat.st_size;
	} else {
		use_mode = TRUE;
		mode = S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH;
		if (allow_trunc == TRUE) {
			flags = O_CREAT | O_WRONLY | O_TRUNC;
		} else {
			if (g_file_test (file_path, G_FILE_TEST_EXISTS) == TRUE) {
				g_set_error (error,
					     G_FILE_ERROR,
					     G_FILE_ERROR_FAILED,
					     _("File operation failed;"
					       "Refusing to overwrite existing file"));
				return FALSE;
			}
			flags = O_CREAT | O_WRONLY;
		}
	}
	
	if (use_mode == TRUE) {
		P->fd = open (file_path, flags, mode);
	} else {
		P->fd = open (file_path, flags);
	}
	if (P->fd < 1) {
		GFileError file_error;
		
		file_error = g_file_error_from_errno (errno);
		g_set_error (error,
			     G_FILE_ERROR,
			     file_error,
			     FILE_IO_ERRORS[file_error]);
		return FALSE;
	} else {
		P->path = g_strdup (file_path);
		P->rw = rw;
		P->size = (gsize) length;
	}
	return TRUE;
}

gboolean
dvd_file_io_open_temp		(DvdFileIO	 *file,
				 gchar		 *template,
				 GError		**error)
{
	DvdFileIOPriv *P;
	
	g_assert (file != NULL);
	P = DVD_FILE_IO_PRIV (file);
	g_assert (P->fd == -1);
	
	P->fd = g_mkstemp (template);
	if (P->fd == -1) {
		GFileError file_error;
			
		file_error = g_file_error_from_errno (errno);
		g_set_error (error,
			     G_FILE_ERROR,
			     file_error,
			     FILE_IO_ERRORS[file_error]);
		return FALSE;
	}
	P->rw = DVD_FILE_IO_WRITE;
	P->size = 0;
	P->path = g_strdup (template);
	return TRUE;
}

/**
 * dvd_file_io_size:
 * @file: Opened #DvdFileIO
 *
 * Get, in bytes, the size of an opened file
 *
 * Returns: size of file if open, 0 if not
 */

gsize
dvd_file_io_size		(DvdFileIO	*file)
{
	DvdFileIOPriv *P;
	gsize size;
	
	g_assert (file != NULL);
	P = DVD_FILE_IO_PRIV (file);
	
	size = P->size;
	return size;
}

/**
 * dvd_file_io_get_path:
 * @file: Opened #DvdFileIO
 *
 * Get the file path of an opened file
 *
 * Returns: file path if open, NULL if not
 */
G_CONST_RETURN gchar
*dvd_file_io_get_path	(DvdFileIO	*file)
{
	DvdFileIOPriv *P;
	const gchar *path;
	
	g_assert (file != NULL);
	P = DVD_FILE_IO_PRIV (file);
	
	path = P->path;
	return path;
}

/**
 * dvd_file_io_get_fd:
 * @file: Opened #DvdFileIO
 *
 * Get the file descriptor of an opened file
 * While making use of the descriptor, you should
 * ensure the position is current with dvd_file_io_set_position()
 *
 * Returns: file descriptor of file if open, -1 if not
 */
gint
dvd_file_io_get_fd		(DvdFileIO	*file)
{
	DvdFileIOPriv *P;
	gint fd;
	
	g_assert (file != NULL);
	P = DVD_FILE_IO_PRIV (file);
	
	fd = P->fd;
	return fd;
}

/**
 * dvd_file_io_get_position:
 * @file: Opened #DvdFileIO
 *
 * Get, in bytes, the read/write position of an opened file
 *
 * Returns: position of file if open, 0 if not
 */

gsize
dvd_file_io_get_position	(DvdFileIO	*file)
{
	DvdFileIOPriv *P;
	gsize pos;
	
	g_assert (file != NULL);
	P = DVD_FILE_IO_PRIV (file);
	
	pos = P->position;
	return pos;
}

/**
 * dvd_file_io_set_position:
 * @file: Opened #DvdFileIO
 *
 * Set, in bytes, the read/write position of an opened #DvdFileIO
 * The validity of position is not checked but must be the true
 * position of the file file descriptor.
 * If the progress value [0 to 100] has changed the file_progress signal
 * is emitted.
 *
 * Returns:
 */

void
dvd_file_io_set_position	(DvdFileIO	*file,
				 gsize		 position)
{
	DvdFileIOPriv *P;
	guint8 percent;
	
	g_assert (file != NULL);
	P = DVD_FILE_IO_PRIV (file);
	
	if (P->position == position) return;
	P->position = position;
	if (P->position == P->size) {
		percent = 100;
	} else {
		percent = (guint8) ((gdouble) P->position /
			     (gdouble) P->size * 100);
	}
	if (P->percent != percent) {
		P->percent = percent;
		g_signal_emit (G_OBJECT (file),
			       dvd_file_io_signals[FILE_IO_PROGRESS],
		 	       0, P->percent);
	}
}

/**
 * dvd_file_io_close:
 * @file: Opened #DvdFileIO to close
 * @error: #GError
 *
 * Close a #DvdFileIO if open, removing current file
 * path and position. The #DvdFileIO @file can then
 * be reused or unreferenced.
 *
 * Possible @error values are
 * domain #G_FILE_ERROR with code #GFileError
 *
 * Returns: TRUE if closed, FALSE on close error and sets @error
 */

gboolean
dvd_file_io_close		(DvdFileIO	 *file,
				 GError		**error)
{
	DvdFileIOPriv *P;
	gboolean ok = TRUE;
	
	g_assert (file != NULL);
	P = DVD_FILE_IO_PRIV (file);
	
	if (P->fd == -1) {
		return TRUE;
	}
	g_free (P->path);
	P->path = NULL;
	P->size = 0;
	P->position = 0;
	P->percent = 0;
	P->fd = close (P->fd);
	if (P->fd == -1) {
		GFileError file_error;
			
		file_error = g_file_error_from_errno (errno);
		g_set_error (error,
			     G_FILE_ERROR,
			     file_error,
			     FILE_IO_ERRORS[file_error]);
		ok = FALSE;
	}
	P->fd = -1;
	return ok;
}


/**
 * dvd_file_io_get_contents:
 * @file: Opened #DvdFileIO to read from
 * @data: Pointer to the returned data
 * @length: Length of data read
 * @error: #GError
 *
 * Read all of #DvdFileIO @file after the current position,
 * determined by dvd_file_io_get_position().
 * Returned @data must be freed after read with g_free ()
 * if length > 0.
 *
 * Possible @error values are
 * domain #G_FILE_ERROR with code #GFileError
 *
 * Returns: TRUE if read ok, FALSE on read error or 0 length and sets @error
 */

gboolean
dvd_file_io_get_contents	(DvdFileIO	 *file,
				 guint8		**data,
				 gsize		 *length,
				 GError		**error)
{
	DvdFileIOPriv *P;
	guint8 *ptr;
	gsize len;

	g_assert (file != NULL);
	P = DVD_FILE_IO_PRIV (file);
	
	len = P->size - P->position;
	if (len < 1) {
		*length = 0;
		*data = NULL;
		g_set_error (error,
			     G_FILE_ERROR,
			     G_FILE_ERROR_FAILED,
			     FILE_IO_ERRORS[G_FILE_ERROR_FAILED]);
		return FALSE;
	}
	*data = g_malloc (len);
	ptr = *data;
	while (1) {
		gssize rlen;
		
		rlen = read (P->fd, ptr, DVD_FILE_IO_READ_SIZE);
		if (rlen < 0) {
			GFileError file_error;
			
			file_error = g_file_error_from_errno (errno);
			g_set_error (error,
				     G_FILE_ERROR,
				     file_error,
				     FILE_IO_ERRORS[file_error]);
			g_free (*data);
			*length = 0;
			*data = NULL;
			return FALSE;
		} else if (rlen == 0) {
			g_set_error (error,
				     G_FILE_ERROR,
				     G_FILE_ERROR_FAILED,
			 	     _("File operation failed;"
				       "Unable to read the rest of the file because"
			  	       "the current position is not the real file position"));
			g_free (*data);
			*length = 0;
			*data = NULL;
			return FALSE;
		}
		ptr += rlen;
		dvd_file_io_set_position (file, P->position + rlen);
		if (P->position == P->size) {
			*length = len;
			break;
		}
	}
	return TRUE;
}


/**
 * dvd_file_io_read_line:
 * @file: Opened #DvdFileIO to read from
 * @line: Pointer to the returned null terminated line
 * @ignore_comments: Ignore comment lines
 * @error: #GError
 *
 * Read untill new line or end of #DvdFileIO @file, starting at the current position.
 * Note that if @skip_comments is TRUE, lines beginning with "#" will not be read,
 * instead, the next non comment line will be returned (if any).
 * If after execution @line is NULL you have reached the end of the file.
 *
 * The returned @line is null terminated and must be freed with g_free ()
 *
 * Possible @error values are
 * domain #G_FILE_ERROR with code #GFileError
 *
 * Returns: TRUE if read ok, FALSE on read error and sets @error.
 */

gboolean
dvd_file_io_read_line		(DvdFileIO	 *file,
				 guint8		**line,
				 gboolean	  ignore_comments,
				 GError		**error)
{
	DvdFileIOPriv *P;
	GByteArray *barray;
	guint8 byte;
	gboolean first;
	gboolean ignore;
	gsize left;

	g_assert (file != NULL);
	P = DVD_FILE_IO_PRIV (file);
	
	left = P->size - P->position;
	barray = g_byte_array_new ();
	first = ignore_comments;
	ignore = FALSE;
	while (left > 0) {
		gssize rlen;
		
		rlen = read (P->fd, &byte, 1);
		if (rlen < 0) {
			GFileError file_error;
			
			file_error = g_file_error_from_errno (errno);
			g_set_error (error,
				     G_FILE_ERROR,
				     file_error,
				     FILE_IO_ERRORS[file_error]);
			g_byte_array_free (barray, TRUE);
			dvd_file_io_set_position (file, P->size - left);
			*line = NULL;
			return FALSE;
		} else if (rlen == 0) {
			g_set_error (error,
				     G_FILE_ERROR,
				     G_FILE_ERROR_FAILED,
			 	     _("File operation failed;"
				       "Unable to read the rest of the file because"
			  	       "the current position is not the real file position"));
			g_byte_array_free (barray, TRUE);
			dvd_file_io_set_position (file, P->size - left);
			*line = NULL;
			return FALSE;
		} else {
			left--;
		}
		if (byte == DVD_ASCII_C_RETURN) {
			/* ignore */
			continue;
		} else if (byte == DVD_ASCII_NEW_LINE) {
			if (ignore == TRUE) {
				ignore = FALSE;
				first = TRUE;
				continue;
			} else {
				/* done */
				break;
			}
		}
		if (first == TRUE) {
			first = FALSE;
			if (byte == DVD_ASCII_HASH) {
				ignore = TRUE;
				continue;
			}
		}
		if (ignore == FALSE) {
			g_byte_array_append (barray, &byte, 1);
		}
	}
	if (barray->len > 0) {
		byte = DVD_ASCII_NUL;
		g_byte_array_append (barray, &byte, 1);
		*line = g_byte_array_free (barray, FALSE);
	} else {
		/* all comments or End Of File */
		g_byte_array_free (barray, TRUE);
		*line = NULL;
	}
	dvd_file_io_set_position (file, P->size - left);
	return TRUE;
}

/**
 * dvd_file_io_read:
 * @file: Opened #DvdFileIO to read from
 * @data: Pointer to a data buffer
 * @length: Length of @data
 * @bytes_read: Returned number of bytes read into @data
 * @error: #GError
 *
 * Read up to @length from #DvdFileIO @file into buffer @data.
 * A returned @bytes_read of 0 indicates end of file.
 *
 * Possible @error values are
 * domain #G_FILE_ERROR with code #GFileError
 *
 * Returns: TRUE if read or EOF, FALSE on read error and sets @error
 */

gboolean
dvd_file_io_read		(DvdFileIO	 *file,
				 guint8		 *data,
				 gsize		  length,
				 gsize		 *bytes_read,
				 GError		**error)
{
	DvdFileIOPriv *P;
	gssize rlen;
	
	g_assert (file != NULL);
	g_assert (data != NULL);
	g_assert (bytes_read != NULL);
	P = DVD_FILE_IO_PRIV (file);
	g_assert (P->size > 0);
	
	rlen = read (P->fd, data, length);
	if (rlen == -1) {
		GFileError file_error;
			
		file_error = g_file_error_from_errno (errno);
		g_set_error (error,
			     G_FILE_ERROR,
			     file_error,
			     FILE_IO_ERRORS[file_error]);
		return FALSE;
	} else if (rlen == 0) {
		/* EOF */
		*bytes_read = 0;
	} else {
		*bytes_read = (gsize) rlen;
		dvd_file_io_set_position (file, P->position + rlen);
	}
	return TRUE;
}

gssize
dvd_file_io_read_byte		(DvdFileIO	 *file,
				 guint8		 *byte,
				 GError		**error)
{
	DvdFileIOPriv *P;
	gssize rlen;
	
	g_assert (file != NULL);
	P = DVD_FILE_IO_PRIV (file);
	g_assert (P->size > 0);
	
	rlen = read (P->fd, byte, 1);
	if (rlen == -1) {
		GFileError file_error;
			
		file_error = g_file_error_from_errno (errno);
		g_set_error (error,
			     G_FILE_ERROR,
			     file_error,
			     FILE_IO_ERRORS[file_error]);
	} else if (rlen == 1) {
		dvd_file_io_set_position (file, P->position + 1);
	}
	return rlen;
}

gboolean
dvd_file_io_seek		(DvdFileIO	 *file,
				 gsize		  position,
				 GError		**error)
{
	DvdFileIOPriv *P;
	off_t rlen;
	
	g_assert (file != NULL);
	P = DVD_FILE_IO_PRIV (file);
	g_assert (P->size > 0);
	
	if (position >= P->size) {
		g_set_error (error,
			     G_FILE_ERROR,
			     G_FILE_ERROR_FAILED,
			     _("File operation failed;"
			       "Unable to seek to requested position because"
			       "it is greater than the file's length"));
		return FALSE;
	}
	rlen = lseek (P->fd, position, SEEK_SET);
	if (rlen == -1) {
		GFileError file_error;
			
		file_error = g_file_error_from_errno (errno);
		g_set_error (error,
			     G_FILE_ERROR,
			     file_error,
			     FILE_IO_ERRORS[file_error]);
		return FALSE;
	} else {
		dvd_file_io_set_position (file, position);
		return TRUE;
	}
}

/**
 * dvd_file_io_write:
 * @file: Opened #DvdFileIO to write to
 * @data: The data to write to file
 * @length: Length of data to write
 * @error: #GError
 *
 * Write @length @data to #DvdFileIO @file.
 *
 * Possible @error values are
 * domain #G_FILE_ERROR with code #GFileError
 *
 * Returns: TRUE if written, FALSE on write error and sets @error
 */

gboolean
dvd_file_io_write		(DvdFileIO	 *file,
				 const guint8	 *data,
				 gsize		  length,
				 GError		**error)
{
	DvdFileIOPriv *P;
	gssize written = 0;
	const guint8 *end;
	
	end = data + length;
	g_assert (file != NULL);
	g_assert (data != NULL);
	P = DVD_FILE_IO_PRIV (file);
	
	while (data < end) {
		gssize wlen;
		
		wlen = write (P->fd, data, length - written);
		if (wlen == -1) {
			GFileError file_error;
			
			file_error = g_file_error_from_errno (errno);
			g_set_error (error,
				     G_FILE_ERROR,
				     file_error,
				     FILE_IO_ERRORS[file_error]);
			return FALSE;
		} else {
			written += wlen;
			data += wlen;
			dvd_file_io_set_position (file, P->position + wlen);
		}
	}
	return TRUE;
}


/**
 * dvd_file_io_move:
 * @file: Opened #DvdFileIO to move
 * @file_path: The file path of moved file
 * @error: #GError
 *
 * Moves a file to a new location.
 * The file will be closed on completion and
 * must be opened again before use.
 *
 * Possible @error values are
 * domain #G_FILE_ERROR with code #GFileError
 *
 * Returns: TRUE if moved, FALSE on error and sets @error
 */

gboolean
dvd_file_io_move		(DvdFileIO	 *file,
				 const gchar	 *file_path,
				 GError		**error)
{
	DvdFileIOPriv *P;
	gboolean ok;
	gchar *src, *dir;
	struct stat fdstat, dstat;
	
	g_assert (file != NULL);
	P = DVD_FILE_IO_PRIV (file);
	
	if (fstat (P->fd, &fdstat) != -1) {
		GFileError file_error;
			
		file_error = g_file_error_from_errno (errno);
		g_set_error (error,
			     G_FILE_ERROR,
			     file_error,
			     FILE_IO_ERRORS[file_error]);
		return FALSE;
	}
	dir = g_path_get_dirname (file_path);
	if (stat (dir, &dstat) != -1) {
		GFileError file_error;
			
		file_error = g_file_error_from_errno (errno);
		g_set_error (error,
			     G_FILE_ERROR,
			     file_error,
			     FILE_IO_ERRORS[file_error]);
		g_free (dir);
		return FALSE;
	}
	g_free (dir);
	
	src = g_strdup (P->path);
	
	/* if on the same device, use rename() */
	if (fdstat.st_dev == dstat.st_dev) {
		if (dvd_file_io_close (file, error) == FALSE) {
			g_free (src);
			return FALSE;
		}
		if (rename (src, file_path) == -1) {
			GFileError file_error;
			
			file_error = g_file_error_from_errno (errno);
			g_set_error (error,
				     G_FILE_ERROR,
				     file_error,
				     FILE_IO_ERRORS[file_error]);
			g_free (src);
			return FALSE;
		}
		g_free (src);
		return TRUE;
	}
	
	ok = dvd_file_io_copy (file,
			    file_path,
			    FALSE,
			    error);
	if (ok == TRUE) {
		if (unlink (src) == -1) {
			GFileError file_error;
			
			file_error = g_file_error_from_errno (errno);
			g_set_error (error,
				     G_FILE_ERROR,
				     file_error,
				     FILE_IO_ERRORS[file_error]);
			ok = FALSE;
		}
	}
	g_free (src);
	return ok;
}

/**
 * dvd_file_io_copy:
 * @src_file: Opened #DvdFileIO to copy
 * @file_path: The file path of copied file
 * @error: #GError
 *
 * Copies a file to a new location.
 * The file will be closed on completion and
 * must be opened again before use.
 *
 * Possible @error values are
 * domain #G_FILE_ERROR with code #GFileError
 *
 * Returns: TRUE if copied, FALSE on error and sets @error
 */

gboolean
dvd_file_io_copy		(DvdFileIO	 *src_file,
				 const gchar	 *file_path,
				 gboolean	  allow_trunc,
				 GError		**error)
{
	DvdFileIOPriv *P;
	DvdFileIO *tofile;
	gboolean ok;
	
	g_assert (src_file != NULL);
	P = DVD_FILE_IO_PRIV (src_file);
	g_assert (P->fd != -1);

	tofile = dvd_file_io_new ();
 	ok = dvd_file_io_open (tofile,
 			    file_path,
 			    DVD_FILE_IO_WRITE,
 			    allow_trunc,
 			    error);
	if (ok == FALSE) {
		g_object_unref (G_OBJECT (tofile));
		return FALSE;
	}
	while (ok == TRUE) {
		guint8 data[DVD_FILE_IO_READ_SIZE];
		gsize length;
		
		ok = dvd_file_io_read (src_file,
				    data,
				    DVD_FILE_IO_READ_SIZE,
				    &length,
				    error);
		if (ok == TRUE) {
			if (length == 0) {
				/* EOF */
				break;
			}
			ok = dvd_file_io_write (tofile, data, length, error);
		}
	}
	if (ok == FALSE) {
		dvd_file_io_close (tofile, NULL);
		dvd_file_io_close (src_file, NULL);
		if (unlink (file_path) == -1) {
			g_warning ("file copy failed and cannot delete dest file %s",
				   file_path);
		}
	} else {
		ok = dvd_file_io_close (tofile, error);
		if (ok == FALSE) {
			dvd_file_io_close (src_file, NULL);
			if (unlink (file_path) == -1) {
				g_warning ("file copy failed and cannot delete dest file %s",
					   file_path);
			}
		} else {
			ok = dvd_file_io_close (src_file, error);
		}
	}
	g_object_unref (G_OBJECT (tofile));
	return ok;
}

gboolean
dvd_file_io_truncate		(DvdFileIO	 *file,
				 GError		**error)
{
	DvdFileIOPriv *P;
	gsize original_size;
	
	g_assert (file != NULL);
	P = DVD_FILE_IO_PRIV (file);
	g_assert (P->fd != -1);
	
	if (dvd_file_io_seek (file,
			   0,
			   error) == FALSE) {
		return FALSE;
	}
	original_size = P->size;
	if (original_size > 0x2800) {
		gint64 fsize;
		
		P->percent = 0;
		for (fsize = P->size - DVD_FILE_IO_READ_SIZE;
		     fsize > 0;
		     fsize -= DVD_FILE_IO_READ_SIZE) {
			guint8 percent;
			gsize trunk_size;
			
			if (fsize < 0) {
				trunk_size = P->size;
			} else {
				trunk_size = P->size - fsize;
			}
			if (ftruncate (P->fd, trunk_size) == -1) {
				GFileError file_error;
			
				file_error = g_file_error_from_errno (errno);
				g_set_error (error,
					     G_FILE_ERROR,
					     file_error,
					     FILE_IO_ERRORS[file_error]);
				return FALSE;
			}
			P->size = P->size - trunk_size;
			percent = (guint8) ((gdouble) P->size /
				  (gdouble) original_size * 100);
			if (P->percent != percent) {
				P->percent = percent;
				g_signal_emit (G_OBJECT (file),
					       dvd_file_io_signals[FILE_IO_PROGRESS],
		 			       0, P->percent);
			}
		}
	} else {
		if (ftruncate (P->fd, P->size) == -1) {
			GFileError file_error;
		
			file_error = g_file_error_from_errno (errno);
			g_set_error (error,
				     G_FILE_ERROR,
				     file_error,
				     FILE_IO_ERRORS[file_error]);
			return FALSE;
		}
		P->size = 0;
		g_signal_emit (G_OBJECT (file),
			       dvd_file_io_signals[FILE_IO_PROGRESS],
			       0, 100);
	}
	return TRUE;
}

