/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <inttypes.h>
#include <string.h>
#include <ctype.h>

#include "dvd.h"
#include "dvd-time.h"
#include <dvdread/ifo_types.h>

guint32
dvd_read_time_to_time_stamp		(gpointer	dvd_read_time)
{
	guint16 hours, minutes, seconds, frames;
	guint32 ts = 0;
	dvd_time_t *dt = (dvd_time_t *) dvd_read_time;
	
	/* First 4 bits are 1/10th of value (0-5) */
	/* Last 4 represent remainder of value (0-9) */
	hours = (((dt->hour >> 4) * 10) + (dt->hour & 0x0f));
	minutes = (((dt->minute >> 4) * 10) + (dt->minute & 0x0f));
	seconds = (((dt->second >> 4) * 10) + (dt->second & 0x0f));
	
	/* First bit is fps 0=pal, 1=ntsc */
	/* Second bit indicates value is in use */
	/* Next 2 bits represent frames / 10 (0-2) */
	/* The last 4 bits represent the remaining number of frames (0-9) */
	if (dt->frame_u & 0x40) {
		frames = (((dt->frame_u & 0x30) >> 4) * 10) + (dt->frame_u & 0x0f);
		if (dt->frame_u & 0x80) {
			ts = frames * NTSC_CLOCK_CYCLES_PER_FRAME;
		} else {
			ts = frames * PAL_CLOCK_CYCLES_PER_FRAME;
		}
	}
	
	ts += hours   * 324000000;
	ts += minutes *   5400000;
	ts += seconds *     90000;
	/*g_message ("ts = %d, h = %d, m = %d, s = %d", ts, hours, minutes, seconds);*/
	return ts;
}

DvdTime
*dvd_time_new_from_time_stamp	(guint32	 ts)
{
	DvdTime *time;
	
	time = g_malloc (sizeof (DvdTime));
	
	return dvd_time_set_from_time_stamp (time, ts);
}

DvdTime
*dvd_time_new_from_dvd_read_time (gpointer	dvd_read_time)
{
	guint32 ts;
	dvd_time_t *dt = (dvd_time_t *) dvd_read_time;
	g_assert (dt != NULL);
	
	ts = dvd_read_time_to_time_stamp (dt);
	return dvd_time_new_from_time_stamp (ts);
}

DvdTime
*dvd_time_set_from_time_stamp	(DvdTime	*time,
				 guint32	 ts)
{
	gint seconds;
	guint32 whole_ms;
	
	g_assert (time != NULL);
	
	time->ts = ts;
	whole_ms = ts / 90;
	
	time->bk_milliseconds = whole_ms % 1000;
	seconds = (whole_ms - time->bk_milliseconds) / 1000;
	time->bk_hours = seconds / 3600;
	seconds = seconds - time->bk_hours * 3600;
	time->bk_minutes = seconds / 60;
	time->bk_seconds = seconds - time->bk_minutes * 60;
	
	return time;
}

DvdTime
*dvd_time_set_from_dvd_read_time (DvdTime	*time,
				  gpointer	dvd_read_time)
{
	guint32 ts;
	dvd_time_t *dt = (dvd_time_t *) dvd_read_time;
	
	ts = dvd_read_time_to_time_stamp (dt);
	
	return dvd_time_set_from_time_stamp (time, ts);
}
