/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/
#include <inttypes.h>
#include <config.h>
#include <glib/gi18n.h>
#include <glib-object.h>
#include <string.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "dvd.h"
#include "dvd-time.h"
#include "dvd-marshal.h"



static GObjectClass *dvd_title_parent_class = NULL;

static void     dvd_title_class_init	(DvdTitleClass	*class);
static void     dvd_title_instance_init	(GTypeInstance	*instance,
					 gpointer	 g_class);
static void     dvd_title_dispose	(GObject	*object);

static void
dvd_title_class_init	(DvdTitleClass	*class)
{
	GObjectClass *object_class = (GObjectClass *) class;
	
	dvd_title_parent_class = g_type_class_ref (G_TYPE_OBJECT);
	object_class->dispose = dvd_title_dispose;
}

static void
dvd_title_instance_init	(GTypeInstance	*instance,
			 gpointer	 g_class)
{
	DvdTitle *title;
	
	title = DVD_TITLE (instance);
}

static void
dvd_title_dispose	(GObject	*object)
{
	DvdTitle *title = DVD_TITLE (object);
	GSList *iter;
	gint track;
	
	for (track = 0;
	     track < MAX_DVD_AUDIO_TRACKS;
	     track++) {
		if (title->audio[track] != NULL) {
			g_object_unref (G_OBJECT (title->audio[track]));
		}
	}
	if (title->video != NULL) {
		g_object_unref (G_OBJECT (title->video));
	}
	for (iter = title->chapters;
	     iter != NULL;
	     iter = g_slist_next (iter)) {
		DvdChapter *chapter;
		
		chapter = (DvdChapter *) iter->data;
		g_object_unref (G_OBJECT (chapter));
	}
	g_slist_free (title->chapters);

	g_free (title->time);
	G_OBJECT_CLASS (dvd_title_parent_class)->dispose (G_OBJECT (title));
}

/**
 * Gets the GType for the dvd title object class
 * @returns GType.
 */
GType
dvd_title_get_type	(void)
{
	static GType title_type = 0;

	if (title_type == 0) {
		GTypeInfo title_info = {
			sizeof (DvdTitleClass),
			NULL,
			NULL,
			(GClassInitFunc) dvd_title_class_init,
			NULL,
			NULL, /* class_data */
			sizeof (DvdTitle),
			0, /* n_preallocs */
			(GInstanceInitFunc) dvd_title_instance_init
	    	};
		title_type = g_type_register_static (G_TYPE_OBJECT,
						    "DvdTitle",
						    &title_info, 0);
	}
	return title_type;
}

/**
 * Create a new dvd title object with #DvdSource source.
 * You must call g_object_unref() when you are done with the returned object.
 * @returns A #DvdTitle object.
 */
DvdTitle
*dvd_title_new		(DvdSource	 source)
{
	DvdTitle *title;
	
	title = g_object_new (dvd_title_get_type (), NULL);
	title->source = source;
	
	return title;
}

void
dvd_title_append_chapter(DvdTitle	 *title,
			 DvdChapter	 *chapter)
{
	DvdChapter *prev_chapter;
	
	g_assert (title != NULL);
	g_assert (chapter != NULL);
	g_assert (chapter->source == title->source);
	
	dvd_chapter_set_id (DVD_CHAPTER (chapter), title->chapter_count);
	if (title->chapter_count > 0) {
		prev_chapter = dvd_title_get_chapter (title, title->chapter_count);
		g_assert (prev_chapter != NULL);
		prev_chapter->next_chapter = g_object_ref (G_OBJECT (chapter));
	}
	title->chapters = g_slist_append (title->chapters, g_object_ref (G_OBJECT (chapter)));
	++title->chapter_count;
}

/* Information */
/**
 * Gets the title ID. Title ID's start from 0
 * @param title A #DvdTitle.
 * @returns The title ID.
 */
guint8
dvd_title_get_id	(DvdTitle	*title)
{
	g_assert (title != NULL);
	return title->title_id;
}

/**
 * Gets the number of chapters in the given #DvdTitle title
 * @param title A #DvdTitle.
 * @returns The number of chapters.
 */
guint8	 
dvd_title_get_chapters	(DvdTitle	*title)
{
	g_assert (title != NULL);
	return title->chapter_count;
}

/**
 * Gets the #DvdChapter according to chapter_id.
 * @param title A #DvdTitle.
 * @param chapter_id A valid chapter ID.
 * You must call g_object_unref() when you are done with the returned object.
 * @returns #DvdChapter.
 */
DvdChapter
*dvd_title_get_chapter	(DvdTitle	*title,
			 guint8		 chapter_id)
{
	DvdChapter *chapter;
	
	g_assert (title != NULL);
	g_assert (chapter_id <= title->chapter_count);
	
	chapter = g_slist_nth_data (title->chapters, chapter_id - 1);
	g_object_ref (chapter);
	return chapter;
}

/**
 * Gets the total title time.
 * @param title A #DvdTitle.
 * @returns #DvdTime.
 */
G_CONST_RETURN
DvdTime
*dvd_title_get_time	(DvdTitle	*title)
{
	g_assert (title != NULL);
	return title->time;
}

/**
 * Gets the estimated (vob) kilobytes of the title.
 * @param title A #DvdTitle.
 * @returns Number of kilobytes.
 */
guint32
dvd_title_get_kbs	(DvdTitle	*title)
{
	g_assert (title != NULL);
	
	return title->kbs;
}

/**
 * Gets the estimated (vob) size of a title.
 * @param title A #DvdTitle.
 * @param gbytes Returned number of gigabytes.
 * @param mbytes Returned number of megabytes.
 * @param kbytes Returned number of kilobytes.
 * @note This is only an estimate.
 * @note No field may be null.
 * @returns Number of kilobytes.
 */
void
dvd_title_get_size	(DvdTitle	*title,
			 guint8		*gbytes,
			 guint16	*mbytes,
			 guint16	*kbytes)
{
	guint32 kbs;
	
	g_assert (title != NULL);
	g_assert (gbytes != NULL);
	g_assert (mbytes != NULL);
	g_assert (kbytes != NULL);
	
	kbs = title->kbs;
	*gbytes = kbs / 0x100000;
	kbs = kbs - *gbytes * 0x100000;
	*mbytes = kbs / 0x400;
	*kbytes = kbs - *mbytes * 0x400;
}

/**
 * Gets the video properties of the title.
 * @param title A #DvdTitle.
 * @returns #DvdVideo that must be freed with g_free().
 */
DvdVideo
*dvd_title_get_video	(DvdTitle	*title)
{
	
	g_assert (title != NULL);
	g_assert (title->video != NULL);
	
	return g_object_ref (G_OBJECT (title->video));
}

/**
 * Gets the number of audio tracks in the title.
 * @param title A #DvdTitle.
 * @returns Number of audio tracks.
 */
guint8
dvd_title_get_audio_tracks	(DvdTitle	*title)
{
	g_assert (title != NULL);
	return title->audio_tracks;
}

/**
 * Gets the audio properties of the title according to the track_number.
 * @param title A #DvdTitle.
 * @param track_number Valid audio track number to return.
 * @returns #DvdAudio object.
 */
DvdAudio
*dvd_title_get_audio	(DvdTitle	*title,
			 guint8		 track_number)
{
	DvdAudio *audio;
	
	g_assert (title != NULL);
	g_assert (track_number < title->audio_tracks);
	audio = title->audio[track_number];
	g_assert (audio != NULL);
	
	return g_object_ref (G_OBJECT (audio));
}

