/* qprogram-starter, a program to start programs or commands, with
   the option to log output and errors and to shutdown the system.
 * Copyright (C) 2009 2010  Christian Metscher <hakaishi@web.de>

 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef GUI_CPP
#define GUI_CPP

#include "gui.h"

Gui::Gui(QWidget *parent) : QWidget(parent){

     setupUi(this);

     dateTimeEdit->setDateTime(QDateTime::currentDateTime()); //For displaying the current time
     timer = new QTimer(this);
     process = new QProcess(this);
     process2 = new QProcess(this);

     hintMsgBox = new QPlainTextEdit;
     hintMsgBox->setReadOnly(true);
     hintMsgBox->resize(500,450);
     hintMsgBox->setWindowTitle("Info");
     hintMsgBox->setWindowModality(Qt::NonModal);
     hintMsgBox->setPlainText(tr("The command in the second textbox (if there is any) will be executed after the first one. The message boxes will close themselves after 10 seconds.\n\nTo start a program just type i.e. \"firefox\" or \"firefox www.google.com\" and then click on OK.\n\nIf the process is \"finished\" while it is still running, then try the --nofork option (i.e. kopete --nofork). Note that this will also occure for some programs like gedit, firefox or gnome-terminal if they are already running.\n\nWhen you want to start a program or command with sudo, please use i.e. gksudo.\n\nmake examples:\n make -C /path/to/project\n make clean -C /path/to/project\n\nIf the shutdown won't work, it means that \"sudo shutdown -P now\" is used. This needs root permissions, so one needs to do the following:\n\nPost the following in a terminal:\n\"EDITOR:nano sudo -E visudo\" and add this line:\n\"* ALL = NOPASSWD:/sbin/shutdown\" whereas * replaces the username or %groupname.\n\nThe configuration- and logging-files can be found at ~/.qprogram-starter/"));

     connect(buttons, SIGNAL(accepted()), this, SLOT(run()));
     connect(timer, SIGNAL(timeout()), this, SLOT(check()));
     connect(process, SIGNAL(readyReadStandardOutput()), this, SLOT(output()));
     connect(process, SIGNAL(readyReadStandardError()), this, SLOT(errorOutput()));
     connect(process, SIGNAL(finished(int)), this, SLOT(checkForProcess2()));
     connect(process, SIGNAL(error(QProcess::ProcessError)), this, SLOT(message()));
     connect(process2, SIGNAL(readyReadStandardOutput()), this, SLOT(output()));
     connect(process2, SIGNAL(readyReadStandardError()), this, SLOT(errorOutput()));
     connect(process2, SIGNAL(finished(int)), this, SLOT(shutdown_or_message()));
     connect(process2, SIGNAL(error(QProcess::ProcessError)), this, SLOT(message()));
     connect(this, SIGNAL(finished()), this, SLOT(shutdown_or_message()));
     connect(hint, SIGNAL(clicked(bool)), this, SLOT(info_hint()));
}

Gui::~Gui(){}

void Gui::info_hint(){ hintMsgBox->show(); }

void Gui::check(){ //To check if start time is reached
     timeInTheFuture = dateTimeEdit->dateTime();
     secondsToTimeInTheFuture = QDateTime::currentDateTime().secsTo(timeInTheFuture);
     textEdit->setDisabled(true);
     atDateCheckBox->setDisabled(true);
     dateTimeEdit->setDisabled(true);
     if(secondsToTimeInTheFuture == 0){
       process->start(textEdit->toPlainText());
       timer->stop();
     }
}

void Gui::run(){ //To start either the timer or start the process
     if(atDateCheckBox->isChecked() && !textEdit->toPlainText().isEmpty())
       timer->start(1000);
     else if(!textEdit->toPlainText().isEmpty()){
       process->start(textEdit->toPlainText());
       textEdit->setDisabled(true);
       atDateCheckBox->setDisabled(true);
       dateTimeEdit->setDisabled(true);
     }
}

void Gui::checkForProcess2(){ //check if there is a second command
     if((process->exitCode()==0) && (!textEdit2->toPlainText().isEmpty())){
       process2->start(textEdit2->toPlainText());
       textEdit2->setDisabled(true);
     }
     else
       finished();
}

void Gui::output(){ //write output into a file if loggingCheckBox is checked
     if(loggingCheckBox->isChecked()){
       QFile outputLog(QDir::homePath() + "/.qprogram-starter/outputLog.txt");
       if(!outputLog.open(QIODevice::Append))
         return;
       outputLog.write(process->readAllStandardOutput());
       outputLog.write(process2->readAllStandardOutput());
       outputLog.close();
     }
}

void Gui::errorOutput(){ //write error output into a file if loggingCheckBox is checked
     if(loggingCheckBox->isChecked()){
     QFile errorLog(QDir::homePath() + "/.qprogram-starter/errorLog.txt");
       if(!errorLog.open(QIODevice::Append))
         return;
       errorLog.write(process->readAllStandardError());
       errorLog.write(process2->readAllStandardError());
       errorLog.close();
     }
}

void Gui::shutdown_or_message(){
     if(shutdownCheckBox->isChecked()){
       bool shutdown_works = false;
       bool gnome_power1 = false;
       bool gnome_power2 = false;
       bool hal_works = false;
       QDBusMessage response;

       QDBusInterface gnomeSessionManager("org.gnome.SessionManager", "/org/gnome/SessionManager",
         "org.gnome.SessionManager", QDBusConnection::sessionBus());
       response = gnomeSessionManager.call("RequestShutdown");
       if(response.type() == QDBusMessage::ErrorMessage){
         gnome_power1 = QProcess::startDetached("gnome-power-cmd.sh shutdown");
         gnome_power2 = QProcess::startDetached("gnome-power-cmd shutdown");
       }
       else
         shutdown_works = true;

       QDBusInterface kdeSessionManager("org.kde.ksmserver", "/KSMServer",
         "org.kde.KSMServerInterface", QDBusConnection::sessionBus());
       response = kdeSessionManager.call("logout", 0, 2, 2);
       if(response.type() == QDBusMessage::ErrorMessage){ /*nothing to do*/ ; }
       else
         shutdown_works = true;

       if(!shutdown_works && !gnome_power1 && !gnome_power2){
         QDBusInterface powermanagement("org.freedesktop.Hal", "/org/freedesktop/Hal/devices/computer",
           "org.freedesktop.Hal.Device.SystemPowerManagement", QDBusConnection::systemBus());
         response = powermanagement.call("Shutdown");
         if(response.type() == QDBusMessage::ErrorMessage){ /*nothing to do*/ ; }
         else
           hal_works = true;
       }

       if(!hal_works && !shutdown_works && !gnome_power1 && !gnome_power2){
         QDBusInterface powermanagement("org.freedesktop.ConsoleKit", "/org/freedesktop/ConsoleKit/Manager",
           "org.freedesktop.ConsoleKit.Manager", QDBusConnection::systemBus());
         response = powermanagement.call("Stop");
         if(response.type() == QDBusMessage::ErrorMessage)
           QProcess::startDetached("sudo shutdown -P now");
       }
     }
     else
       message();
}

void Gui::message(){
       QMessageBox msgBox;
       if(((process->exitCode()==254)||(process->error()==1)||
          (process->error()==0)||(process->exitCode()==1))||
          ((process2->exitCode()==254)||(process2->error()==1)||
          (process2->error()==0)||(process2->exitCode()==1))){
         msgBox.setWindowTitle("Error");
         msgBox.setIcon(QMessageBox::Critical);
         if((process->exitCode()==254)||(process2->exitCode()==254))
           msgBox.setInformativeText("Failed to start!\n"
            "Wrong arguments or options.");
         if((process->error()==0)||(process2->error()==0))
           msgBox.setInformativeText("Failed to start!\n"
            "No such program or command.");
         if(((process->error()==1)||(process->exitCode()==1))||
            ((process2->error()==1)||(process2->exitCode()==1)))
           msgBox.setInformativeText("Process crashed");  
       }
       else{
         msgBox.setWindowTitle("Information");
         msgBox.setIcon(QMessageBox::Information);
         if(((process->exitCode()==0) && (!shutdownCheckBox->isChecked()))||
            ((process2->exitCode()==0) && (!shutdownCheckBox->isChecked())))
           msgBox.setInformativeText("Process finished");
       }
       msgBox.setWindowModality(Qt::NonModal);
       msgBox.setWindowFlags(Qt::WindowStaysOnTopHint);
       QTimer::singleShot(10000, &msgBox, SLOT(close()));
       msgBox.exec();

       textEdit->setEnabled(true);
       atDateCheckBox->setEnabled(true);
       if(atDateCheckBox->isChecked())
         dateTimeEdit->setEnabled(true);
       textEdit2->setEnabled(true);

       if(quitCheckBox->isChecked())
         close();
}

void Gui::loadSettings(){
     QString file(QDir::homePath() + "/.qprogram-starter/qprogram-starter.conf");
     QSettings settings(file, QSettings::NativeFormat);
     if(!QFile::exists(file)){ //if path or file doesn't exist
       settings.setValue("CheckBoxes/atDate", false);
       settings.setValue("CheckBoxes/logging", false);
       settings.setValue("CheckBoxes/shutdown", false);
       settings.setValue("CheckBoxes/quitWithLasProcess", false);
       settings.setValue("Text/text", QString());
     }

     atDateCheckBox->setChecked(settings.value("CheckBoxes/atDate").toBool());
     loggingCheckBox->setChecked(settings.value("CheckBoxes/logging").toBool());
     shutdownCheckBox->setChecked(settings.value("CheckBoxes/shutdown").toBool());
     quitCheckBox->setChecked(settings.value("CheckBoxes/quitWithLasProcess").toBool());
     textEdit->setText(settings.value("Text/text").toString());
}


#endif
