/* $Id: libc.c,v 1.10 2003/05/30 00:42:39 cgd Exp $ */

/*
 * Copyright 2001, 2003
 * Broadcom Corporation. All rights reserved.
 *
 * This software is furnished under license and may be used and copied only
 * in accordance with the following terms and conditions.  Subject to these
 * conditions, you may download, copy, install, use, modify and distribute
 * modified or unmodified copies of this software in source and/or binary
 * form. No title or ownership is transferred hereby.
 *
 * 1) Any source code used, modified or distributed must reproduce and
 *    retain this copyright notice and list of conditions as they appear in
 *    the source file.
 *
 * 2) No right is granted to use any trade name, trademark, or logo of
 *    Broadcom Corporation.  The "Broadcom Corporation" name may not be
 *    used to endorse or promote products derived from this software
 *    without the prior written permission of Broadcom Corporation.
 *
 * 3) THIS SOFTWARE IS PROVIDED "AS-IS" AND ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING BUT NOT LIMITED TO, ANY IMPLIED WARRANTIES OF
 *    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 *    NON-INFRINGEMENT ARE DISCLAIMED. IN NO EVENT SHALL BROADCOM BE LIABLE
 *    FOR ANY DAMAGES WHATSOEVER, AND IN PARTICULAR, BROADCOM SHALL NOT BE
 *    LIABLE FOR DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 *    OR OTHERWISE), EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <limits.h>
#include "port.h"
#include "libc.h"

/* Bits for the allocation bitfield */
#define HEAP_ALLOCATED 0x1



/* These are defined by the backend port code */
void lib_port_init(void *);
void lib_port_uninit(void);
void lib_port_exit(void);
extern unsigned char *lib_heap;
extern unsigned int lib_heap_size;


void lib_init(void *backend_data)
{
	lib_port_init(backend_data);
	*(uint32_t *)lib_heap = lib_heap_size - 8;
	*(((uint32_t *)lib_heap) + 1) = 0;
}

void lib_uninit()
{
	lib_port_uninit();
}

#define SIZE_T_MAX ULONG_MAX
#define is_digit(c) ((c) >= '0' && (c) <= '9')


enum flag {
    flag_none = 0,
    flag_minus = 1,
    flag_plus = 2,
    flag_space = 4,
    flag_hash = 8,
    flag_zero = 16,
    flag_short = 32,
    flag_long = 64
};



static void put_string(char *s, int len, int width, int prec, enum flag f)
{
	int i;
	if (len < 0) { 
		len = lib_strnlen(s, prec >= 0 ? prec : SIZE_T_MAX); 
	} else if (prec >= 0 && prec < len) { 
		len = prec; 
	}
	if (width > len && !(f & flag_minus)) {
		for (i = len; i < width; ++i) { 
			lib_putc(' '); 
		}
	}
	for (i = 0; i < len; ++i) { 
		lib_putc(s[i]); 
	}
	if (width > len && f & flag_minus) {
		for (i = len; i < width; ++i) { 
			lib_putc(' '); 
		}
	}
}

static void put_ulong(unsigned long int value, int base, int upper,
		      char *prefix, int width, int prec, enum flag f)
{
	int len, zeros, i;
	unsigned long int x, x2;
	for (len = 1, x = 1; x < ULONG_MAX / base; ++len, x = x2) {
		x2 = x * base;
		if (x2 > value) { 
			break; 
		}
	}
	zeros = (prec > len) ? prec - len : 0;
	width -= zeros + len;
	if (prefix != 0) { 
		width -= lib_strnlen(prefix, 0xffffffff); 
	}
	if (!(f & flag_minus)) {
		for (i = 0; i < width; ++i) { 
			lib_putc(' '); 
		}
	}
	if (prefix != 0) { 
		put_string(prefix, -1, 0, -1, flag_none); 
	}
	for (i = 0; i < zeros; ++i) { 
		lib_putc('0'); 
	}
	for ( ; x > 0; x /= base) {
		int digit = (value / x) % base;
		lib_putc((digit < 10 ? '0' : (upper ? 'A' : 'a') - 10) + digit);
	}
	if (f & flag_minus) {
		for (i = 0; i < width; ++i) { 
			lib_putc(' '); 
		}
	}
}

void lib_vprintf(char *format, va_list arg)
{
	char c;
	while ((c = *format++) != '\0') {
		if (c != '%' || (c = *format++) == '%') {
			lib_putc(c);
		} else {
			enum flag f = flag_none;
			int width = 0;
			int prec = -1;
			for ( ; c != '\0'; c = *format++) {
				if (c == '-') { 
					f |= flag_minus; 
				} else if (c == '+') { 
					f |= flag_plus; 
				} else if (c == ' ') { 
					f |= flag_space; 
				} else if (c == '#') { 
					f |= flag_hash; 
				} else if (c == '0') { 
					f |= flag_zero; 
				} else {
					break;
				}
			}
			if (c == '*') {
				width = va_arg(arg, int);
				if (width < 0) {
					f |= flag_minus;
					width = -width;
				}
				c = *format++;
			} else {
				for ( ; is_digit(c); c = *format++) {
					width = width * 10 + (c - '0');
				}
			}
			if (c == '.') {
				f &= ~flag_zero;
				c = *format++;
				if (c == '*') {
					prec = va_arg(arg, int);
					c = *format++;
				} else {
					for (prec = 0; is_digit(c); c = *format++) {
						prec = prec * 10 + (c - '0');
					}
				}
			}
			if (c == 'h' || c == 'l') {
				f |= (c == 'h' ? flag_short : flag_long);
				c = *format++;
			}
			if (c == 'd' || c == 'i') {
				long int value;
				if (f & flag_short) {
					value = (short int) va_arg(arg, int);
				} else if (f & flag_long) {
					value = va_arg(arg, long int);
				} else {
					value = va_arg(arg, int);
				}
				if (value >= 0) {
					if (f & flag_plus) {
						put_ulong(value, 10, 0, "+", width, prec, f);
					} else if (f & flag_space) {
						put_ulong(value, 10, 0, " ", width, prec, f);
					} else {
						put_ulong(value, 10, 0, 0, width, prec, f);
					}
				} else {
					put_ulong(-value, 10, 0, "-", width, prec, f);
				}
			} else if (c == 'o' || c == 'u' || c == 'x' || c == 'X') {
				unsigned long int value;
				if (f & flag_short) {
					value = (unsigned short int) va_arg(arg, unsigned int);
				} else if (f & flag_long) {
					value = va_arg(arg, unsigned long int);
				} else {
					value = va_arg(arg, unsigned int);
				}
				if (c == 'o') {
					if (f & flag_hash && value != 0) {
						put_ulong(value, 8, 0, "0", width, prec, f);
					} else {
						put_ulong(value, 8, 0, 0, width, prec, f);
					}
				} else if (c == 'u') {
					put_ulong(value, 10, 0, 0, width, prec, f);
				} else {
					if (f & flag_hash && value != 0) {
						if (c == 'x') {
							put_ulong(value, 16, 0, "0x", width, prec, f);
						} else {
							put_ulong(value, 16, 1, "0X", width, prec, f);
						}
					} else {
						put_ulong(value, 16, 0, 0, width, prec, f);
					}
				}
			} else if (c == 'c') {
				char value = va_arg(arg, int);
				lib_putc(value);
			} else if (c == 's') {
				char *value = va_arg(arg, char *);
				if (value == 0) {
					put_string("(null)", -1, width, prec, f);
				} else if (f & flag_hash) {
					put_string(value + 1, (unsigned char) *value, width, prec, f);
				} else {
					put_string(value, -1, width, prec, f);
				}
			} else if (c == 'p') {
				void *value = va_arg(arg, void *);
				put_ulong((unsigned long int) value, 16, 0, "0x", width, prec, f);
			} else {
				lib_putc(c);
			}
		}
	}
}

void lib_printf(char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	lib_vprintf(fmt, ap);
	va_end(ap);
}




void *lib_malloc(unsigned int amount)
{
	unsigned int cur_block_size;
	unsigned int cur_block_flags;
	unsigned char *heap_ptr = lib_heap;
	void *retval = 0;
	amount = (amount + 7) & ~0x7;

	while (heap_ptr < (lib_heap + lib_heap_size)) {
		cur_block_size = *((uint32_t *)heap_ptr);
		cur_block_flags = *(((uint32_t *)heap_ptr) + 1);
		if ((cur_block_flags & HEAP_ALLOCATED) 
		    || (cur_block_size < amount)) {
			heap_ptr += (cur_block_size + 8);
			continue;
		}
		/* Found a chunk...
		   Set up the header for the next boundary, if one doesn't
		   already exist.  */
		if (cur_block_size > amount) {
			*((uint32_t *)(heap_ptr + amount + 8)) = cur_block_size - (amount + 8);
			*(((uint32_t *)(heap_ptr + amount + 8)) + 1) = 0;
		}

		/* Modify this chunk header */
		*((uint32_t *)heap_ptr) = amount;
		*(((uint32_t *)heap_ptr) + 1) = HEAP_ALLOCATED;
		retval = heap_ptr + 8;
		break;
	}
	
	/*
	 * Be mean to all those that expect the return value to be 0
	 */
	if (retval) {
		lib_memset(retval, 7, amount);
	}
	return retval;
}

void lib_free(void *ptr)
{
	unsigned int cur_block_size, next_block_size;
	unsigned int flags;
	unsigned char *heap_ptr, *next_ptr;
	/* Check extreme bogosity.  This will also catch the required free(NULL) handling */
	if (((unsigned char *)ptr < (lib_heap-8)) || ((unsigned char *)ptr >= (lib_heap + lib_heap_size - 8))) {
		return;
	}
	
	/* First, just do the dealloc */
	*(((uint32_t *)ptr) - 1) = 0;

	/* Scan the list, looking for consolidations */
	for (heap_ptr = lib_heap; heap_ptr < (lib_heap + lib_heap_size); heap_ptr += (cur_block_size + 8)) {
		cur_block_size = *((uint32_t *)heap_ptr);
		flags = *(((uint32_t *)heap_ptr) + 1);
		if (flags & HEAP_ALLOCATED) {
			continue;
		} else {
			next_ptr = heap_ptr + cur_block_size + 8;
			while ((next_ptr < (lib_heap + lib_heap_size))
			       && !(*(((uint32_t *)next_ptr) + 1) & HEAP_ALLOCATED)) {
				next_block_size = *((uint32_t *)next_ptr);
				*((uint32_t *)heap_ptr) += 8 + next_block_size;
				next_ptr += 8 + next_block_size;
			}
			
		}
	}
}

void *lib_memset(void *s, int c, unsigned int n)
{
	unsigned char *tmp = (unsigned char *)s;
	while (n--) {
		*tmp = c;
		tmp++;
	}
	return s;
}

int lib_memcmp(void *s1, void *s2, unsigned int n)
{
	unsigned char *tmp1 = s1, *tmp2 = s2;
	while (n--) {
		if (*tmp1 < *tmp2) {
			return -1;
		} else if (*tmp2 < *tmp1) {
			return 1;
		}
		tmp1++;
		tmp2++;
	}
	return 0;
}

void lib_bzero(void *s, unsigned int n)
{
	lib_memset(s, 0, n);
}

void *lib_memcpy(void *dest, void *src, unsigned int n)
{
	unsigned char *s1 = (unsigned char *)dest; 
	unsigned char *s2 = (unsigned char *)src;
	while (n--) {
		*s1 = *s2;
		s1++;
		s2++;
	}
	return dest;
}

char *lib_strncpy(char *dest, char *src, unsigned int amount)
{
	char *retval = dest;
	while (amount && *src) {
		*dest = *src;
		dest++;
		src++;
		amount--;
	}
	if (amount) {
		*dest = *src;
	}
	return retval;
}

int lib_strlen(char *s) 
{
	int retval = 0;
	while (*s++) retval++;
	return retval;
}

char *lib_strcpy(char *dest, char *src)
{
	char *tmp = dest;
	while (*src) {
		*tmp = *src;
		tmp++;
		src++;
	}
	*tmp = *src;
	return dest;
}

int   lib_strnlen(char *str, unsigned int maxlen)
{
	unsigned int retval = 0;
	while ((retval < maxlen) 
	       && *str++) {
		retval++;
	}
	return retval;
	       
}

void lib_usleep(unsigned long usec)
{
	uint64_t targ_time = lib_get_time() + usec;
	while (lib_get_time() < targ_time) {
		/* Do nothing */
	}
		
}

char *lib_strdup(char *src)
{
	char *retval = lib_malloc(lib_strlen(src)+1);
	if (!retval) {
		return 0;
	}
	lib_strcpy(retval, src);
	return retval;
}

char *lib_strchr(char *str,char c)
{
    while (*str && (*str != c)) str++;
    if (*str) return str;
    return 0;
}

int   lib_isspace(char c)
{
	return ((c == ' ') 
		|| (c == '\t') 
		|| (c == '\n') 
		|| (c == '\r'));
}

int   lib_isdigit(char c)
{
	return ((c >= '0') && (c <= '9'));
}

char lib_toupper(char foo)
{
	if ((foo >= 'a')
	    && (foo <= 'z')) {
		return (foo-'a') + 'A';
	}
	return foo;
}

char lib_tolower(char foo)
{
	if ((foo >= 'A')
	    && (foo <= 'Z')) {
		return (foo-'A') + 'a';
	}
	return foo;
}

int lib_isalpha(char c)
{
	return ((c >= 'A') && (c <= 'Z')) 
		|| (c >= 'a' && (c <= 'z'));
}

int lib_strncmp(char *s1, char *s2, unsigned long n)
{
	while (n && *s1 && (*s1 == *s2)) {
		n--;
		if (n) {
			s1++;
			s2++;
		}
	}
	if (*s1 < *s2) {
		return -1;
	} else if (*s1 > *s2) {
		return 1;
	} 
	return 0;
}

int lib_strcmp(char *s1, char *s2)
{
//	if (!(s1 && s2)) {
//	lib_printf("strcmp\n");
//		}
	return lib_strncmp(s1, s2, UINT32_MAX);
}

char *lib_strcat(char *s1, char *s2)
{
	char *tmp = s1 + lib_strlen(s1);
	lib_strcpy(tmp, s2);
	return s1;
}


void lib_exit(void) 
{
	lib_uninit();
	lib_port_exit();
}

/* Returns number of digits printed, not including null */
int  lib_ultostr(char *ptr, unsigned long num)
{
	int digits = 0;
	char *ptr2;
	if (num) {
		unsigned long tmp = num;
		while (tmp) {
			digits++;
			tmp >>= 4;
		}
	} else {
		digits = 1;
	}
	
	ptr2 = ptr + digits;
	*ptr2 = '\0';
	ptr2--;
	while (num) {
		int digit;
		digit = num & 0xf;
		if (digit < 0xa) {
			*ptr2 = '0' + digit;
		} else {
			*ptr2 = 'A' + (digit - 10);
		}
		ptr2--;
		num >>= 4;
	}
	return digits;
}

uint64_t lib_strtomacaddr(char *str)
{
	uint64_t result = 0;
	int tmp;
	while (*str) {
		if (lib_isdigit(*str)) {
			tmp = *str - '0';
		} else if (lib_isalpha(*str)) {
			tmp = 10 + (lib_toupper(*str) - 'A');
		} else if ((*str == ':') || (*str == '-')) {
			str++;
			continue;
		} else {
			break;
		}
		result <<= 4;
		result += tmp;
		str++;
	}
	return result;
}

uint64_t lib_strtou64(char *str, char **endptr, int base)
{
	uint64_t result = 0;
	int tmp;
	if (!base) {
		if (str[0] == '0') {
			str++;
			if (str[0] == 'x' || str[0] == 'X') {
				base = 16;
				str++;
			} else {
				base = 8;
			}
		} else {
			base = 10;
		}
	}
	if (base == 16) {
		if ((str[0] == '0') &&
		    ((str[1] == 'x') ||
		     (str[1] == 'X'))) {
			str += 2;
		}
	}
	while (*str) {
		if (lib_isdigit(*str)) {
			tmp = *str - '0';
		} else if (lib_isalpha(*str)) {
			tmp = 10 + (lib_toupper(*str) - 'A');
		} else {
			break;
		}
		if (tmp >= base) {
			break;
		}
		result *= base;
		result += tmp;
		str++;
	}
	*endptr = str;
	return result;
}

static int char_is_delim(char c, char *delims) 
{
	for (; *delims; delims++) {
		if (c == *delims) {
			return 1;
		}
	}
	return 0;
}

char *lib_strtok(char *str, char *delims)
{
	static char *saved;
	char *retval;
	if (!str) {
		retval = saved;
	} else {
		retval = str;
	}
	while (*retval && char_is_delim(*retval, delims)) {
		retval++;
	}
	if (*retval) { 
		saved = retval;
		while (*saved && !char_is_delim(*saved, delims)) {
			saved++;
		}
		if (*saved) {
			*saved = 0;
			saved++;
		}
	} else {
		retval = 0;
	}
	return retval;
}

uint32_t lib_parse_ip(char *str)
{
	uint32_t retval = 0;
	int tmp = 0;
	while (*str) {
		if (lib_isdigit(*str)) {
			tmp = (tmp * 10) + (*str - '0');
		} else if (*str == '.') {
			retval = (retval << 8) | tmp;
			tmp = 0;
		}
		str++;
	}
	retval = (retval << 8) | tmp;
	return retval;
}

void lib_die(char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	lib_vprintf(fmt, ap);
	va_end(ap);
	lib_printf("\n");
	lib_exit();
}

void lib_print_ip(uint32_t ip)
{
	lib_printf("%i.%i.%i.%i", ip>>24, (ip>>16) & 0xff, (ip>>8) & 0xff, ip & 0xff);
}

void  lib_print_mac_addr(uint64_t addr)
{
	lib_printf("%.02X:%.02X:%.02X:%.02X:%.02X:%.02X",
		   (int)((addr >> 40) & 0xff),
		   (int)((addr >> 32) & 0xff),
		   (int)((addr >> 24) & 0xff),
		   (int)((addr >> 16) & 0xff),
		   (int)((addr >>  8) & 0xff),
		   (int)(addr & 0xff));
}

int lib_atoi(char *str) 
{
	int retval = 0;
	while ((*str >= '0') && ((*str <= '9'))) {
		retval = (retval * 10) + (*str - '0');
		str++;
	}
	return retval;
}
