/*
 * Copyright 2008 Klaus Triendl
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free 
 * Software Foundation, 51 Franklin Street, Fifth Floor, 
 * Boston, MA 02110-1301, USA.
*/

#include <gtkmm.h>
#include <sigx/sigx.h>


// fwd decl
class TheGUI;

class TheThread: public sigx::glib_threadable
{
public:
	TheThread(TheGUI* thegui);


protected:
	// virtuals from sigx::threadable
	virtual void on_startup();


	void on_first_idle();
	void on_button_clicked();


private:
	TheGUI* m_thegui;
};


class TheGUI: public Gtk::Window, public sigx::glib_auto_dispatchable
{
public:
	TheGUI();


public:
	// expose Gtk::Button::signal_clicked of m_btn
	sigx::signal_f<Glib::SignalProxy0<void> > signal_button_clicked;


private:
	void on_first_idle();
	void on_thethread_ready();


private:	
	TheThread m_thethread;
	Gtk::Button* m_btn;
};


#include <iostream>


TheThread::TheThread(TheGUI* thegui): 
	m_thegui(thegui)
{}

//virtual 
void TheThread::on_startup()
{
	// one-shot idle handler
	maincontext()->signal_idle().connect(
		sigc::bind_return(
			sigc::mem_fun(this, &TheThread::on_first_idle), 
			false
		)
	);
}

void TheThread::on_first_idle()
{
	std::cout << "TheTread connects to the button_clicked signal." << std::endl;
	// connect to TheGUI::signal_button_clicked
	m_thegui->signal_button_clicked().connect_notify(
		sigc::mem_fun(this, &TheThread::on_button_clicked)
	);
}

void TheThread::on_button_clicked()
{
	std::cout << "TheThread got notified that the button was clicked" << std::endl;
}


TheGUI::TheGUI(): 
	Gtk::Window(), 
	sigx::glib_auto_dispatchable(), 
	// set up signal functor with late object instance binding (because m_btn is still invalid
	// but will be constructed later)
	signal_button_clicked(*this, sigc::ref(m_btn), &Gtk::Button::signal_clicked), 
	m_thethread(this), 
	m_btn()
{
	m_btn = Gtk::manage(new Gtk::Button("notify thread"));
	m_btn->set_sensitive(false);

	add(*m_btn);

	// end the thread if close button is clicked
	signal_delete_event().connect(
		// hide parameter GdkEvent*
		sigc::hide(sigc::bind_return(sigc::mem_fun(m_thethread, &TheThread::finish), false))
	);
	// one-shot idle handler to start the thread
	Glib::signal_idle().connect(
		sigc::bind_return(sigc::mem_fun(this, &TheGUI::on_first_idle), false)
	);
	
	show_all_children();
}


void TheGUI::on_first_idle()
{
	std::cout << "starting TheThread." << std::endl;
	// start the thread
	m_thethread.run();
	// it's not yet correct to activate the "notify thread" button because we don't
	// know exactly when the thread will have connected to our signal unless we in turn 
	// let us notify but this is ok for this test example
	m_btn->set_sensitive(true);
}


int main(int arg_count, char** args)
{
	Glib::thread_init();
	
	Gtk::Main main(arg_count, args);
	TheGUI thegui;
	main.run(thegui);
}
