/*
    Silky - A GTK+ client for SILC.
    Copyright (C) 2003,2004 Toni Willberg

    - Various auxiliary functions

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    http://silky.sourceforge.net/

*/

#include "includes.h"

GHashTable *mimetable; /* parsed mime.types */

extern GladeXML *xmlmain;
extern silkyStruct *silky;

extern gchar *glade_silky_query;
extern gchar *glade_silky_query_label;
extern gchar *homedir;
extern SilkyChannel *channel_active;
extern SilkyQuery *query_active;

/* Returns HH:MM */
gchar * utf_timestamp (void) {
  time_t curtime;
  struct tm *loctime;
  gchar *utf_time;
  curtime = time(0);
  loctime = localtime (&curtime);
  utf_time = g_strdup_printf("%02d:%02d", loctime->tm_hour, loctime->tm_min);

  return utf_time;
}




/* Shows a MODAL error dialog */
void errordialog(const char *errormessage) {
  GtkWindow *mainwin;
  GtkMessageDialog *dialog;

  mainwin = GTK_WINDOW(glade_xml_get_widget (xmlmain, "window_main"));
  dialog = GTK_MESSAGE_DIALOG(
			      gtk_message_dialog_new (mainwin,
						      GTK_DIALOG_DESTROY_WITH_PARENT,
						      GTK_MESSAGE_ERROR,
						      GTK_BUTTONS_CLOSE,
						      errormessage)
			      );
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (GTK_WIDGET(dialog));

  return;
}


/* Returns GtkTreeIter for GtkListStore pointing to row that contains
   data of given SilcClientID
*/
GtkTreeIter get_nicklist_iter_by_clientid (GtkListStore *list_store, SilcClientID *client_id) {
  GtkTreeIter iter;
  gboolean valid;
  SilcClientID *stored_clientid;

  debug("get_nicklist_iter_by_clientid()");

  if (!client_id) {
    debug("Did not get client_id");
    return iter;
  }
  if (!list_store) {
    debug("Did not get list_store");
    return iter;
  }

  /* Get the first iter in the list */
  valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(list_store), &iter);

  debug ("before valid...");
  while (valid) {

    /* get clientid from the list */
    gtk_tree_model_get (GTK_TREE_MODEL(list_store), &iter, 3, &stored_clientid, -1);

    if (!stored_clientid) {
      debug("Did not get stored_clientid for this client. This should not happen.");
    }

    /* try to find matching client id */
    if(SILC_ID_CLIENT_COMPARE(client_id, stored_clientid)) {
      debug("\tmatching IDs found");
      break;
    }

    /* next */
    valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(list_store), &iter);
  }
  debug ("\tlist iterated");

  return iter;
}



/* this prints given message to the CONSOLE tab */
int printconsole (const char *str) {
  GtkTextView *textview = NULL;
  gchar *timestamp;
  gchar *outstring;
  GtkTextBuffer *buffer;
  GtkTextIter iter; /* pointer to current location */
  GtkNotebook *storednotebook;
  GtkWidget *active_page;
  gint apagenr;
  gchar *channel_name;
  GladeXML *stored_query_xml;

  /* make sure it was utf8 */
  if (!g_utf8_validate(str, -1, NULL)) {
    debug("Found non-utf8 string: '%s'", str);
    str = g_strdup("Non-utf8 string received. Can not display.");
  }


  timestamp = g_strdup_printf("[%s] ", utf_timestamp() );
  outstring = g_strdup_printf("%s", str);

  /* see if there's active channel or query tab first */
  storednotebook = GTK_NOTEBOOK(glade_xml_get_widget (xmlmain, "tabs"));
  apagenr = gtk_notebook_get_current_page(GTK_NOTEBOOK(storednotebook));

  /*
     console or other type of tab? 0 is always console
  */
  if (apagenr == 0) { /* console tab */
    debug("active: CONSOLE");
    textview = GTK_TEXT_VIEW(glade_xml_get_widget (xmlmain, "textviewwidget"));
  }
  else { /* other tab */
    debug("active: other tab... looking for type...");
    active_page = gtk_notebook_get_nth_page(storednotebook, apagenr); /* active pagenr from GTK */
    if (!active_page) {
      debug("no page was active, this should not happen!");
      return FALSE;
    }

    /* test if it was a channel tab */
    channel_name = g_object_get_data(G_OBJECT(active_page), "channel_name");
    if (channel_name) {
      debug("active CHANNEL");
      textview = GTK_TEXT_VIEW(get_stored_channel_widget(channel_name, "textviewwidget"));
    }
    else {

      /* this way we can get the textview widget without knowing the clientid */
      stored_query_xml = glade_get_widget_tree(active_page);
      textview = GTK_TEXT_VIEW(glade_xml_get_widget (stored_query_xml, "textviewwidget"));
    }

  }

  if (!textview) {
    debug("active NONE. no textview found!");
    return FALSE;
  }


  buffer = gtk_text_view_get_buffer(textview);

  /* append \n */
  outstring = g_strdup_printf("%s\n", outstring);

  /*  debug("`%s'", outstring); */
  /* Get the end of the buffer */
  gtk_text_buffer_get_end_iter(buffer, &iter);

  /* Print the text */

  /* timestamp */
  gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, timestamp, -1, "timestamp", NULL);

  /* server message */
  gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, outstring, -1, "server_messages", NULL);

  gtk_text_buffer_get_end_iter(buffer, &iter);
  gtk_text_buffer_create_mark(buffer, "end_mark", &iter, FALSE);
  gtk_text_view_scroll_to_mark(textview, gtk_text_buffer_get_mark(buffer, "end_mark"), 0.1, FALSE, 0,1);

  g_free(timestamp);
  g_free(outstring);

  return TRUE;
}



/*
   Get Channel Name of the active channel, if any
*/
char * get_active_channel_name (void) {
  GtkNotebook *storednotebook;
  GtkWidget *active_page;
  gchar *channel_name;


  /* where are the names stored... */
  storednotebook = GTK_NOTEBOOK(glade_xml_get_widget (xmlmain, "tabs"));
  if (!storednotebook) {
    debug("can not find storednotebook, returning NULL");
    return NULL;
  }

  /* get active page entry */

  active_page = gtk_notebook_get_nth_page(storednotebook,
					 gtk_notebook_get_current_page(GTK_NOTEBOOK(storednotebook))
					 );
  if (active_page == 0) /* if on console tab */ {
    debug("no console tab, returning NULL");
    return NULL;
  }

  /* get name of the channel from active_page object */
  channel_name = g_object_get_data(G_OBJECT(active_page), "channel_name");

  if (!channel_name) {
    debug("didn't get channel_name, returning NULL");
    return NULL;
  }

  debug("found active channel: '%s'", channel_name);
  return channel_name;
}


/* Get Channel Name of the active channel, if any
   requires [X] button as parameter, the button is supposed to be
   the one located inside hbox in the channel's tab */
char * get_active_channel_name_by_x (GtkWidget *button) {
  GtkWidget *active_hbox;
  gchar *channel_name;

  debug("get_active_channel_name_by_x ()");

  active_hbox = gtk_widget_get_parent(button);

  /* get name of the channel from active_page object */
  channel_name = g_object_get_data(G_OBJECT(active_hbox), "channel_name");

  debug("found channel '%s'", channel_name);

  return channel_name;
}

/* Get Channel ID of the active channel, if any
   requires [X] button as parameter, the button is supposed to be
   the one located inside hbox in the channel's tab */
SilcChannelID * get_active_channel_id_by_x (GtkWidget *button) {
  GtkWidget *active_hbox;
  SilcChannelID *channel_id;

  debug("get_active_channel_id_by_x ()");

  active_hbox = gtk_widget_get_parent(button);

  /* get name of the channel from active_page object */
  channel_id = g_object_get_data(G_OBJECT(active_hbox), "channel_id");

  debug("found channel id");

  return channel_id;
}


/* Get Channel ID of the active channel, if any */
SilcChannelID * get_active_channel_id (void) {
  GtkNotebook *storednotebook;
  GtkWidget *active_page;
  SilcChannelID *channel_id;

  debug("get_active_channel_id ()");

  /* where are the names stored... */
  storednotebook = GTK_NOTEBOOK(glade_xml_get_widget (xmlmain, "tabs"));
  if (!storednotebook) {
    debug("Did not get stored notebook");
    return NULL;
  }

  /* get active page entry */
  active_page = gtk_notebook_get_nth_page(storednotebook,
					  gtk_notebook_get_current_page(GTK_NOTEBOOK(storednotebook))
					  );

  if(!active_page) {
    debug("Did not get active page...");
    return NULL;
  }

  /* get name of the channel from active_page object */
  channel_id = g_object_get_data(G_OBJECT(active_page), "channel_id");
  if (!channel_id) {
    debug("Did not get channel_id from active page");
    return NULL;
  }

  debug ("\treturning channel_id");
  return channel_id;
}







/* creates new query window or returns an old one if such exists */
void * get_query_window (SilcClientID *clientid) {
  gchar *str_clientid;
  GtkNotebook *notebook;
  GtkWidget *widget_query;
  SilcClientEntry cliententry;
  GladeXML *xml_query;
  GtkWindow *window_query;
  GtkLabel *label_nickname;
  GtkLabel *label_realname;
  GtkWidget *widget_tabhbox;
  GtkWidget *wid_tab_label = NULL;
  GtkWidget *stored_widget;
  SilkyQuery *query = NULL;

  debug("get_query_window()");

  if (!clientid) {
    debug("Did not get clientid at get_query_window() ");
    return NULL;
  }

  str_clientid =  silc_id_id2str(clientid, SILC_ID_CLIENT);
  if (!str_clientid) {
    debug("did not find str_clientid at get_query_window() ");
    return NULL;
  }

  cliententry = silc_client_get_client_by_id(silky->client, silky->conn, clientid);
  if (!cliententry) {
    debug("Did not get cliententry at get_query_window(). Returning. ");
    return NULL;
  }

  /* get notebook widget */
  notebook = GTK_NOTEBOOK(glade_xml_get_widget (xmlmain, "tabs"));

  /* fetch query of this user, if exists */
  query = query_find_by_cliententry(cliententry);

  //  if (!widget_query) {
  if (!query) {
    debug("No existing query widget found for '%s', creating new...", cliententry->nickname);


    query = query_add( cliententry ); /* allocate new SilkyQuery */



    /* get xml for query */
    xml_query = glade_xml_new (glade_silky_query, NULL, NULL);
    if (!xml_query) {
      g_error("Can not find the silky-query.glade file.");
    }

    query->glade_xml = xml_query;

    /* connect signals */
    glade_xml_signal_autoconnect (xml_query);

    /* get parent widget */
    window_query = GTK_WINDOW(glade_xml_get_widget (xml_query, "window_query"));
    /* get channel widget */
    widget_query = glade_xml_get_widget (xml_query, "query_hbox");



    /* set nickname and realname of the queried user */
    label_nickname =  GTK_LABEL(glade_xml_get_widget (xml_query, "query_nickname"));
    label_realname =  GTK_LABEL(glade_xml_get_widget (xml_query, "query_realname"));

    gtk_label_set_text(label_nickname, cliententry->nickname);
    if (g_utf8_validate(cliententry->realname, -1, NULL)) {
      gtk_label_set_text(label_realname, cliententry->realname);
    }

    /* DON'T try to use stored data before this line! :) */

    /* store pointer to the widget_query by str_client to the notebook object */
    g_object_set_data(G_OBJECT(notebook), str_clientid, widget_query);

    /* store the clientid into the widget */
    g_object_set_data(G_OBJECT(widget_query), "clientid", clientid);

    g_object_set_data(G_OBJECT(widget_query), "cliententry", cliententry);

    /* store tab type */
    g_object_set_data(G_OBJECT(widget_query), "tab_type", (gpointer *)TAB_TYPE_QUERY);

    /* destroy unnecessary parent */
    g_object_ref(widget_query);
    gtk_container_remove(GTK_CONTAINER(window_query), widget_query);


    /* query text widget, temporary */
    stored_widget = glade_xml_get_widget (xml_query, "textviewwidget");

    /* set all color and font tags */
    set_textview_colors(GTK_TEXT_VIEW(stored_widget) );


    /* generate tab label stuff and pack it */
    widget_tabhbox = gtk_hbox_new(FALSE, 2);
    query->label_text = (gpointer)gtk_label_new(cliententry->nickname);
    query->label_image = (gpointer)gtk_image_new_from_stock( GTK_STOCK_JUSTIFY_FILL, GTK_ICON_SIZE_BUTTON );
    gtk_box_pack_start(GTK_BOX(widget_tabhbox), (gpointer)query->label_image, TRUE, TRUE, 0);
    gtk_box_pack_end(GTK_BOX(widget_tabhbox), (gpointer)query->label_text, TRUE, TRUE, 0);


    /* create new page/tab to the notebook */
    gtk_notebook_append_page(notebook, widget_query, widget_tabhbox); /* create new tab */
    query->pagenr = gtk_notebook_get_n_pages(notebook)-1;

    gtk_widget_show_all(widget_tabhbox);

    /* Store widget of tab label for later user, FIXME: this can(?) be removed after query struct is done */
    g_object_set_data(G_OBJECT(widget_query), "tablabel", wid_tab_label);

    g_object_unref(widget_query);

  }
  else {
    debug("Found existing query widget, using it...");
  }

  /* ACTIVATE current page */
  //  gtk_notebook_set_current_page(notebook, gtk_notebook_page_num(notebook, widget_query));
//  gtk_notebook_set_current_page(notebook, query->pagenr);
  //  stored_query_xml = glade_get_widget_tree(widget_query);
  // stored_widget = glade_xml_get_widget (stored_query_xml, "textviewwidget");
  stored_widget = glade_xml_get_widget (query->glade_xml, "textviewwidget");

  debug("\treturning the widget");

  return stored_widget;

}



/* get stored channel widget from notebook by channel name */
GtkWidget * get_stored_channel_widget (const char *channel_name, const char *widget_name) {
  GtkNotebook *storednotebook;
  GtkWidget *stored_channel_widget;
  GladeXML *stored_channel_xml;
  GtkWidget *stored_widget;

  debug("get_stored_channel_widget('%s', '%s')", channel_name, widget_name);

  /* get notebook widget */
  storednotebook = GTK_NOTEBOOK(glade_xml_get_widget (xmlmain, "tabs"));

  /* get channel widget by channel_name */
  stored_channel_widget = g_object_get_data(G_OBJECT(storednotebook), channel_name);

  if (!stored_channel_widget) {
    debug("no stored channel widget found, not on active channel");
    printconsole(_("You are not on any channel."));
    return NULL;
  }

  stored_channel_xml = glade_get_widget_tree(stored_channel_widget);
  stored_widget = glade_xml_get_widget (stored_channel_xml, widget_name);

  return stored_widget;
}


/* get stored channel "window" widget from notebook by channel name */
GtkWidget * get_stored_channel_window (const char *channel_name) {
  GtkNotebook *notebook;
  GtkWidget *page;

  notebook = GTK_NOTEBOOK(glade_xml_get_widget (xmlmain, "tabs"));
  page = g_object_get_data(G_OBJECT(notebook), channel_name);

  return page;
}

/* marks active the channel tab of given channel ch,
   called usually when some text was printed there and
   user needs to be notified about activity in given channel struct
*/
void hilight_channel_window_by_ch(SilkyChannel *ch, gint type) {
  if (ch == channel_active) {
    debug("channel is active, doing nothing");
    return;
  }


  switch (type) {
  case HILIGHT_MOTION: /* wtf? */
      debug("HILIGHT_MOTION");
      gtk_label_set_markup( (gpointer)ch->label_text, g_strconcat("<b>", CHANNEL_NAME(ch), "</b>", NULL));
      gtk_image_set_from_stock( (gpointer)ch->label_image, GTK_STOCK_REFRESH, GTK_ICON_SIZE_BUTTON );

      break;

  case HILIGHT_SPEECH: /* any public message */
      debug("HILIGHT_SPEECH");
      gtk_label_set_markup( (gpointer)ch->label_text, g_strconcat("<b>", CHANNEL_NAME(ch), "</b>", NULL));
      gtk_image_set_from_stock( (gpointer)ch->label_image, GTK_STOCK_DIALOG_INFO, GTK_ICON_SIZE_BUTTON );

      break;

  case HILIGHT_WORD: /* wordlist match */
      debug("HILIGHT_WORD");
      gtk_label_set_markup( (gpointer)ch->label_text, g_strconcat("<b>", CHANNEL_NAME(ch), "</b>", NULL));
      gtk_image_set_from_stock( (gpointer)ch->label_image, GTK_STOCK_DIALOG_WARNING, GTK_ICON_SIZE_BUTTON );

      break;

  case HILIGHT_NICK: /* my nick match */
      debug("HILIGHT_NICK");
      gtk_label_set_markup( (gpointer)ch->label_text, g_strconcat("<b>", CHANNEL_NAME(ch), "</b>", NULL));
      gtk_image_set_from_stock( (gpointer)ch->label_image, GTK_STOCK_DIALOG_QUESTION, GTK_ICON_SIZE_BUTTON );

      break;

  default:
      debug("default hilight, this should not happen?");
      break;
  }
}



/* marks active the query tab of given channel ch,
   called usually when some text was printed there and
   user needs to be notified about activity in given channel struct
*/
void hilight_query_window_by_q(SilkyQuery *q, gint type) {
  if (q == query_active) {
    debug("query is active, doing nothing");
    return;
  }


  switch (type) {
  case HILIGHT_MOTION: /* wtf? */
      debug("HILIGHT_MOTION");
      gtk_label_set_markup( (gpointer)q->label_text, g_strconcat("<b>", q->client_entry->nickname, "</b>", NULL));
      gtk_image_set_from_stock( (gpointer)q->label_image, GTK_STOCK_REFRESH, GTK_ICON_SIZE_BUTTON );

      break;

  case HILIGHT_SPEECH: /* any public message */
      debug("HILIGHT_SPEECH");
      gtk_label_set_markup( (gpointer)q->label_text, g_strconcat("<b>", q->client_entry->nickname, "</b>", NULL));
      gtk_image_set_from_stock( (gpointer)q->label_image, GTK_STOCK_DIALOG_INFO, GTK_ICON_SIZE_BUTTON );

      break;

  case HILIGHT_WORD: /* wordlist match */
      debug("HILIGHT_WORD");
      gtk_label_set_markup( (gpointer)q->label_text, g_strconcat("<b>", q->client_entry->nickname, "</b>", NULL));
      gtk_image_set_from_stock( (gpointer)q->label_image, GTK_STOCK_DIALOG_WARNING, GTK_ICON_SIZE_BUTTON );

      break;

  case HILIGHT_NICK: /* my nick match */
      debug("HILIGHT_NICK");
      gtk_label_set_markup( (gpointer)q->label_text, g_strconcat("<b>", q->client_entry->nickname, "</b>", NULL));
      gtk_image_set_from_stock( (gpointer)q->label_image, GTK_STOCK_DIALOG_QUESTION, GTK_ICON_SIZE_BUTTON );

      break;

  default:
      debug("default hilight, this should not happen?");
      break;
  }
}



/* prints stuff to channel, FIXME: rewrite this */
int print_mime_to_channel (SilcClientEntry client_entry, const char *channel_name, const gchar *content_type, ...) {
  gchar *nick_name;
  GtkTextView *textview;
  GtkTextBuffer *buffer = NULL;
  GtkTextIter iter;
  GtkScrolledWindow *swindow;
  GtkAdjustment *vadjustment;
  
  SilkyChannel *ch; /* fixme; should get ch as input, not channel_name */
  unsigned char *message;
  int flags;
  unsigned char *mime_data_buffer;
  int mime_data_len;

  va_list va;
  gchar *timestamp;
  gchar *outstring;
  GdkPixbufLoader *loader;
  GError *err;
  GdkPixbuf *pixbuf;
  /* strdup because modified here */
  gchar *hilightwords = g_strdup( prefs_get("highlight_words") );
  
  debug("print_mime_to_channel()");
  
  
  /* get ch */
  ch = channel_find_by_name(g_strdup(channel_name));
  
  if (!content_type) {
    debug("content-type was null, aborting");
    return FALSE;
  }
  
  nick_name = client_entry->nickname;
  textview =  GTK_TEXT_VIEW(get_stored_channel_widget(channel_name, "textviewwidget"));
  if (!textview) {
    debug("didn't find textview for channel '%s'", channel_name);
    return FALSE;
  }
  
  /* get text buffer for output */
  buffer = gtk_text_view_get_buffer(textview);
  
  if (!buffer) {
    debug("Can not find text buffer for channel '%s'", channel_name);
    return FALSE;
  }
  
  swindow = GTK_SCROLLED_WINDOW(gtk_widget_get_parent(GTK_WIDGET(textview)));
  vadjustment = gtk_scrolled_window_get_vadjustment( swindow );

  /* get end position for printing */
  gtk_text_buffer_get_end_iter(buffer, &iter);
  
  va_start(va, content_type); /* last named argument before ... */

  /* handle TYPE */
  if (!strcmp(content_type, "text/plain")) {
    debug("content-type: '%s'", content_type);

    flags = va_arg(va, int);
    message = va_arg(va, unsigned char *);

    /* make sure it was utf8 */
    if (!g_utf8_validate(message, -1, NULL)) {
      debug("Found non-utf8 string, can not display: '%s'", message);
      message = g_strdup(_("Non-unicode string received. Cannot display."));
    }

    /* timestamp */
    timestamp = utf_timestamp();
    outstring = g_strdup_printf("[%s] ", timestamp);

    gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, outstring, -1, "timestamp", NULL);


    /* set action flags to tabs */
    if (g_pattern_match_simple(g_strconcat("*", silky->conn->local_entry->nickname, "*", NULL), message)) {
      debug("* my nick match");
      hilight_channel_window_by_ch(ch, HILIGHT_NICK);
    }
    else if (g_pattern_match_simple(g_strconcat("*", hilightwords, "*", NULL), message)) {
      debug("* hilight wordlist match");
      hilight_channel_window_by_ch(ch, HILIGHT_WORD);
    }
    else {
      hilight_channel_window_by_ch(ch, HILIGHT_SPEECH);
    }


    
    /* <mynickname> */
    if ( client_entry->id == silky->conn->local_entry->id) { /* myself */
      /* signed [S] */
      if (flags & SILC_MESSAGE_FLAG_SIGNED) {
	gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, "[S]", -1, "nickname-own", NULL);
      }
      
      if (flags & SILC_MESSAGE_FLAG_ACTION) { /* action.. uuggggly.. */
        gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, "* ", -1, "nickname-own", NULL);
        gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, nick_name, -1, "nickname-own", "context-nick", NULL);
        gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, " ", -1, "nickname-own", NULL);
        gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, message, -1, "message-own", NULL);
      }
      else {
        gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, "<", -1, "nickname-own", NULL);
        gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, nick_name, -1, "nickname-own", "context-nick", NULL);
        gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, "> ", -1, "nickname-own", NULL);
        gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, message, -1, "message-own", NULL);
      }
    }
    else { /* someone else */
      /* signed [S] */
      if (flags & SILC_MESSAGE_FLAG_SIGNED) {
	gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, "[?]", -1, "nickname", NULL);
      }

      if ((flags & SILC_MESSAGE_FLAG_ACTION)) { /* Print actions differently */
        gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, "* ", -1, "nickname", NULL);
        gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, nick_name, -1, "nickname", "context-nick", NULL);
        gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, " ", -1, "nickname", NULL);
        gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, message, -1, "message", NULL);
      }
      else {
        gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, "<", -1, "nickname", NULL);
        gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, nick_name, -1, "nickname", "context-nick", NULL);
        gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, "> ", -1, "nickname", NULL);
        gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, message, -1, "message", NULL);
      }
    }

    gtk_text_buffer_insert (buffer, &iter, "\n", -1);

  }

  if (!strcmp(content_type, "image/gif") || !strcmp(content_type, "image/png") || !strcmp(content_type, "image/jpeg")) {
    debug("content-type: '%s'", content_type);

    mime_data_buffer = va_arg(va, unsigned char *);
    mime_data_len = va_arg(va, int);
    debug("datalen: '%d'", mime_data_len);

    loader = gdk_pixbuf_loader_new ();
    err = NULL;

    /* this gives a warning but works?! */
    if (!gdk_pixbuf_loader_write (loader, mime_data_buffer, mime_data_len, &err)) {
      if (err) {
	debug ("Unable to load pixbuf data, error: '%s'", err->message);
      } else {
	debug("Unable to load pixbuf data, no error!");
      }
    }
    
    if (!gdk_pixbuf_loader_close (loader, &err)) {
      if (err) {
	debug ("Unable to close pixbuf, error: '%s'", err->message);
      } else {
	debug("Unable to close pixbuf, no error!");
      }
    }
    
    pixbuf = gdk_pixbuf_loader_get_pixbuf (loader);
    g_object_ref (pixbuf);
    g_object_unref (loader);

    /* insert nickname */
    gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, "<", -1, "nickname", NULL);
    gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, nick_name, -1, "nickname", NULL);
    gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, ">", -1, "nickname", NULL);

    /* insert the image and a newline*/
    gtk_text_buffer_insert_pixbuf (buffer, &iter, pixbuf);
    gtk_text_buffer_insert (buffer, &iter, "", -1);
    
  }
  

  va_end(va);
  
  
  gtk_text_buffer_get_end_iter(buffer, &iter);
  gtk_text_buffer_create_mark(buffer, "end_mark", &iter, FALSE);
  
  /* scroll to end only if we were already at the end before printing,
     this avoids scrolling when user is reading backlog */
  if (vadjustment->upper - vadjustment->page_size == vadjustment->value) {
    gtk_text_view_scroll_to_mark(textview, gtk_text_buffer_get_mark(buffer, "end_mark"), 0.1, FALSE, 0, 1);
  }
  
  return 1;
}



/* this prints given message to the CHANNEL tab, non-mime message */
int printchannel (const char *channel_name, const char *str) {
GtkTextView *textview;

  GtkTextBuffer *buffer;
  gchar *timestamp;
  gchar *outstring;
  GtkTextIter iter; /* pointer to current location */

  /* make sure it was utf8 */
  if (!g_utf8_validate(str, -1, NULL)) {
    debug("Found non-utf8 string: '%s'", str);
    str = g_strdup(_("Non-unicode string received. Cannot display."));
  }

  textview =  GTK_TEXT_VIEW(get_stored_channel_widget(channel_name, "textviewwidget"));

  buffer = gtk_text_view_get_buffer(textview);

  if (!buffer) {
    debug("Can not find text buffer!");
    return FALSE;
  }

  timestamp = utf_timestamp();
  outstring = g_strdup_printf("[%s] %s\n", timestamp, str);

  gtk_text_buffer_get_end_iter(buffer, &iter);
  gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, outstring, -1, "message", NULL);
  gtk_text_buffer_get_end_iter(buffer, &iter);
  gtk_text_buffer_create_mark(buffer, "end_mark", &iter, FALSE);
  gtk_text_view_scroll_to_mark(textview, gtk_text_buffer_get_mark(buffer, "end_mark"), 0.1, FALSE, 0,1);

  g_free(timestamp);
  g_free(outstring);

  return TRUE;
}


/*
   This prints given message to the QUERY tab.
   args: target clientid tab, printed nickname of target/sender, flags and message string
*/
int printquery (SilcClientID *clientid, gchar *nickname, SilcMessageFlags flag, const char *str) {
  GtkTextView *textview;
  GtkTextBuffer *buffer;
  GtkTextIter iter; /* pointer to current location */
  gchar *timestamp;
  gchar *outstring;
  gchar *nickstring;
  gchar *flagstring;
  SilcClientEntry cliententry;

  debug("printquery()");

  textview = GTK_TEXT_VIEW(get_query_window(clientid));

  if (!textview) {
    debug("Can not find text view!");
    return FALSE;
  }

  buffer = GTK_TEXT_BUFFER(gtk_text_view_get_buffer(textview));

  if (!buffer) {
    debug("Can not find text buffer!");
    return FALSE;
  }

  /* Get the end of the buffer */
  gtk_text_buffer_get_end_iter(buffer, &iter);

  timestamp = g_strdup_printf("[%s] ", utf_timestamp() );
  outstring = g_strdup_printf("%s\n", str); /* append \n */

  /* was it an action? */
  if ((flag & SILC_MESSAGE_FLAG_ACTION)) {
    debug("flag: SILC_MESSAGE_FLAG_ACTION");
    nickstring = g_strdup_printf("* %s ", nickname); /* *me actions */
  }
  else {
    nickstring = g_strdup_printf("<%s> ", nickname); /* me: says */
  }

  /* was it signed message? */
  if (flag & SILC_MESSAGE_FLAG_SIGNED) {
    debug("flag: SILC_MESSAGE_FLAG_SIGNED");
    flagstring = g_strdup("[S]");
  }
  else {
    flagstring = g_strdup("");
  }


  /* timestamp */
  gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, timestamp, -1, "timestamp", NULL);

  /* flagstring */
  gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, flagstring, -1, "nickname", NULL);

  /* nickname */
  gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, nickstring, -1, "nickname", NULL);

  /* Print the text */
  gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, outstring, -1, "message", NULL);

  gtk_text_buffer_get_end_iter(buffer, &iter);
  gtk_text_buffer_create_mark(buffer, "end_mark", &iter, FALSE);
  gtk_text_view_scroll_to_mark(textview, gtk_text_buffer_get_mark(buffer, "end_mark"), 0.1, FALSE, 0,1);

  g_free(timestamp);
  g_free(outstring);
  g_free(flagstring);
  g_free(nickstring);

  cliententry = silc_client_get_client_by_id(silky->client, silky->conn, clientid);

  hilight_query_window_by_q(query_find_by_cliententry(cliententry), HILIGHT_SPEECH);

  return TRUE;
}


void add_user_to_nicklist (GtkListStore *liststore, SilcChannelEntry channel, SilcChannelUser user) {
  GtkTreeIter iter, i;
  gchar *nick = NULL;
  int x = 0, pos = 0;

  if (!liststore) {
    debug("Missing liststore");
    return;
  }
  if (!channel) {
    debug("Missing channel");
    return;
  }
  if (!user) {
    debug("Missing user");
    return;
  }

  /* new line, sort alphabetically */
  if( gtk_tree_model_get_iter_first(GTK_TREE_MODEL(liststore), &i) ) {
    do {
      gtk_tree_model_get(GTK_TREE_MODEL(liststore), &i, 2, &nick, -1);
      if( g_utf8_collate( g_utf8_casefold(user->client->nickname, 256), g_utf8_casefold(nick, 256) ) <= 0) {
	g_free(nick);
	gtk_list_store_insert_before(liststore, &iter, &i);
	x++;
      } else pos++;
    } while ( gtk_tree_model_iter_next(GTK_TREE_MODEL(liststore), &i) && !x );
  }

  if (!x) gtk_list_store_append(liststore, &iter);

  debug("Added new line to liststore");
  debug("Adding nick '%s'", user->client->nickname);

  channel_member_add(channel_find_by_entry(channel), user);


  /* founder column*/
  if (user->mode & SILC_CHANNEL_UMODE_CHANFO) {
    gtk_list_store_set (liststore, &iter,
			0,  "*", -1);
  }

  /* muted? using same column as founder can not be muted :) */
  if (user->mode & SILC_CHANNEL_UMODE_QUIET) {
    gtk_list_store_set (liststore, &iter,
			0,  "&", -1);
  }

  /* chanop column */
  if (user->mode & SILC_CHANNEL_UMODE_CHANOP) {
    gtk_list_store_set (liststore, &iter,
			1,  "@", -1);
  }

  /* nickname column */
  gtk_list_store_set (liststore, &iter,
		      2,  user->client->nickname, -1);

  /* clientid column */
  gtk_list_store_set (liststore, &iter,
		      3 , user->client->id, -1);

  debug("--------------");
}







gchar * set_cmode_window_widgets (gchar *channel_name, SilcUInt32 mode) {
  GtkToggleButton *toggle_button;
  gchar *str_mode;

  debug("cmode for '%s'", channel_name);

  /* set all modes cleared */
  str_mode = g_strdup(""); /* clear dynamic string */
  if (mode == SILC_CHANNEL_MODE_NONE) {
    str_mode = g_strdup("none");
  }


  /*
     Check mode and construct str_mode string accordingly.
     Also TOGGLE togglebutton widgets in the channel
  */
  toggle_button = GTK_TOGGLE_BUTTON(get_stored_channel_widget(channel_name, "cmode_private")); /* get toggle button */
  if (mode & SILC_CHANNEL_MODE_PRIVATE) {
    str_mode = g_strconcat(str_mode, "p", NULL);
    gtk_toggle_button_set_active(toggle_button, TRUE); /* button down */
  }
  else {
    gtk_toggle_button_set_active(toggle_button, FALSE); /* button up */
  }

  /* CMODE TOPIC */
  debug("BEFORE");
  toggle_button = GTK_TOGGLE_BUTTON(get_stored_channel_widget(channel_name, "cmode_topic")); /* get toggle button */
  if (mode & SILC_CHANNEL_MODE_TOPIC) {
    str_mode = g_strconcat(str_mode, "t", NULL);
    gtk_toggle_button_set_active(toggle_button, TRUE);
  }
  else {
    gtk_toggle_button_set_active(toggle_button, FALSE);
  }
  debug("AFTER");

  /* CMODE SECRET */
  toggle_button = GTK_TOGGLE_BUTTON(get_stored_channel_widget(channel_name, "cmode_secret")); /* get toggle button */
  if (mode & SILC_CHANNEL_MODE_SECRET) {
    str_mode = g_strconcat(str_mode, "s", NULL);
    gtk_toggle_button_set_active(toggle_button, TRUE);
  }
  else {
    gtk_toggle_button_set_active(toggle_button, FALSE);
  }

  /* CMODE PRIVKEY */
  /*
  toggle_button = GTK_TOGGLE_BUTTON(get_stored_channel_widget(channel_name, "cmode_privkey"));
  if (mode & SILC_CHANNEL_MODE_PRIVKEY) {
    str_mode = g_strconcat(str_mode, "k", NULL);
    gtk_toggle_button_set_active(toggle_button, TRUE);
  }
  else {
    gtk_toggle_button_set_active(toggle_button, FALSE);
  }
  */

  /* CMODE INVITE */

  toggle_button = GTK_TOGGLE_BUTTON(get_stored_channel_widget(channel_name, "cmode_invite")); /* get toggle button */
  if (mode & SILC_CHANNEL_MODE_INVITE) {
    str_mode = g_strconcat(str_mode, "i", NULL);
      gtk_toggle_button_set_active(toggle_button, TRUE);
  }
  else {
    gtk_toggle_button_set_active(toggle_button, FALSE);
  }


  /* CMODE ULIMIT */
  /*
  toggle_button = GTK_TOGGLE_BUTTON(get_stored_channel_widget(channel_name, "cmode_ulimit"));
  if (mode & SILC_CHANNEL_MODE_ULIMIT) {
    str_mode = g_strconcat(str_mode, "l", NULL);
      gtk_toggle_button_set_active(toggle_button, TRUE);
  }
  else {
    gtk_toggle_button_set_active(toggle_button, FALSE);
  }
  */

  /* CMODE PASSPHRASE */
  /*
  toggle_button = GTK_TOGGLE_BUTTON(get_stored_channel_widget(channel_name, "cmode_passphrase"));
  if (mode & SILC_CHANNEL_MODE_PASSPHRASE) {
    str_mode = g_strconcat(str_mode, "a", NULL);
    gtk_toggle_button_set_active(toggle_button, TRUE);
  }
  else {
    gtk_toggle_button_set_active(toggle_button, FALSE);
  }
  */

  /* CMODE SILENCE_USERS */
  toggle_button = GTK_TOGGLE_BUTTON(get_stored_channel_widget(channel_name, "cmode_umute")); /* get toggle button */
  if (mode & SILC_CHANNEL_MODE_SILENCE_USERS) {
    str_mode = g_strconcat(str_mode, "m", NULL);
    gtk_toggle_button_set_active(toggle_button, TRUE);
  }
  else {
    gtk_toggle_button_set_active(toggle_button, FALSE);
  }


  /* CMODE SILENCE_OPERS */
  toggle_button = GTK_TOGGLE_BUTTON(get_stored_channel_widget(channel_name, "cmode_omute")); /* get toggle button */
  if (mode & SILC_CHANNEL_MODE_SILENCE_OPERS) {
    str_mode = g_strconcat(str_mode, "M", NULL);
    gtk_toggle_button_set_active(toggle_button, TRUE);
  }
  else {
    gtk_toggle_button_set_active(toggle_button, FALSE);
  }

  /* CMODE FOUNDER_AUTH */
  /*
  toggle_button = GTK_TOGGLE_BUTTON(get_stored_channel_widget(channel_name, "cmode_founder"));
  if (mode & SILC_CHANNEL_MODE_FOUNDER_AUTH) {
    str_mode = g_strconcat(str_mode, "f", NULL);
    gtk_toggle_button_set_active(toggle_button, TRUE);
  }
  else {
    gtk_toggle_button_set_active(toggle_button, FALSE);
  }
  */


  return str_mode;
}


void error_no_channel_oper_priv (gchar *channel_name) {
  printchannel(channel_name,
	       g_strdup_printf(_("Permission denied. You are not channel operator of channel '%s'."), channel_name)
	       );

}

void error_no_channel_founder_priv (gchar *channel_name) {
  printchannel(channel_name,
	       g_strdup_printf(_("Permission denied. You are not channel founder of channel '%s'."), channel_name)
	       );

}

void error_user_not_on_channel (gchar *channel_name) {
  printchannel(channel_name,
	       g_strdup_printf(_("User is not on channel '%s'."), channel_name)
	       );
}


/* delete the list, call this just before exiting Silky */
void mime_types_list_destroy (void) {
  debug("mime_types_list_destroy()");
  if (!mimetable) {
    return;
  }

  g_hash_table_destroy(mimetable);

}

gchar *mime_types_get_type(gchar *filename) {
  gchar *extension;

  debug ("mime_types_get_type()");
  if (!mimetable || !filename) {
    debug ("No mimetable or filename specified");
    return NULL;
  }

  extension = strrchr(filename, '.');
  if (!extension || *(extension+1) == 0) {
    debug("unknown extension");
    return NULL;
  }

  extension++;
  return g_hash_table_lookup(mimetable, extension);

}



/* read given mime.types file and initialize the hash */
GHashTable *mime_init(gchar *file) {
  FILE *f1;
  gchar str[4096];
  GHashTable *mimetable;

  debug("mime_init()");

  f1 = fopen(file, "r");
  if (!f1) {
    debug("file open failed for '%s'", file);
    return NULL;
  }

  /* init the table */
  mimetable = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);

  if (!mimetable) {
    debug("was not able to initialize mimetable. out of mem?");
    return NULL;
  }

  /* parse the mime.types file */
  while (fgets(str, sizeof(str), f1)) {
    gchar *key, *value;
    gchar *tmp;
    //    gchar buf[4096];

    // ignore lines starting with '#'
    if (str[0] == '#') {
      continue;
    }

    // ignore empty lines
    if (strlen(str) < 3) {
      continue;
    }

    str[strlen(str) - 1] = 0; /* nullify the end of the string */

    tmp = strrchr(str, '\t'); /* ?? */
    if (!tmp) {
      continue;
    }

    *tmp = 0;
    tmp++;
    if (*tmp == 0) {
      continue;
    }

    key = tmp;

    tmp = strchr(str, '\t'); /* ?? */
    if (!tmp) {
      continue;
    }

    *tmp = 0;

    value = str;

    tmp = strtok(key, " ");
    if (!tmp) {
      g_hash_table_insert(mimetable, strdup(key), strdup(value));
    }  else {
      g_hash_table_insert(mimetable, strdup(tmp), strdup(value));

      while ( ( tmp = strtok(NULL, " ") )) {
        g_hash_table_insert(mimetable, strdup(tmp), strdup(value));
      }

    }
  }

  return mimetable;
}

/* this function closes all tabs in UI except the first (console) tab */
void close_all_tabs( void ) {
  GtkWidget *notebook;
  int i, n;

  if( !(notebook = glade_xml_get_widget(xmlmain, "tabs")) ) {
    debug("could not retrieve notebook, bailing out");
    return;
  }

  if( (n = gtk_notebook_get_n_pages(GTK_NOTEBOOK(notebook))) > 1 ) {
    debug("removing tabs left over from previous connection");
    gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), 0);
    for( i = n; i > 0; i-- )
      gtk_notebook_remove_page(GTK_NOTEBOOK(notebook), i);
  }
}

/* called when an event was triggered for a <nickname> in the text view */
gboolean context_nick_event(GtkTextTag *texttag, GObject *widget, GdkEvent *event, GtkTextIter *iter, gpointer user_data) {
  GtkMenu *menu;
  GtkWidget *menuitem;
  SilcClientID *clientid;
  gchar *target_nick = NULL;
  GtkTextIter *start_iter, *end_iter;
  GtkTextBuffer *text_buffer;

  if (event->type == GDK_BUTTON_PRESS || event->type == GDK_2BUTTON_PRESS) {

    /* Get the nickname from the text */
    start_iter = gtk_text_iter_copy (iter);
    end_iter = gtk_text_iter_copy (iter);
    while (!gtk_text_iter_begins_tag (start_iter, texttag)) {
      gtk_text_iter_backward_char (start_iter);
    }
    while (!gtk_text_iter_ends_tag (end_iter, texttag)) {
      gtk_text_iter_forward_char (end_iter);
    }

    text_buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (widget));
    target_nick = gtk_text_buffer_get_text (text_buffer, start_iter, end_iter, FALSE);

    if (!target_nick) {
      debug("no tagtext found?!");
    }

    gtk_text_iter_free (start_iter);
    gtk_text_iter_free (end_iter);

  }


  /* single right-click */
  if (target_nick && event->type == GDK_BUTTON_PRESS && event->button.button == 3 ) {
    debug ("right-clicked on '%s'", target_nick );

    // temp, should get one from the tags
    clientid = silky->conn->local_entry->id;

    /* create the menu */
    menu = GTK_MENU(gtk_menu_new());

    /* ADD MENU items and signal handlers*/

    /* WHOIS */
    menuitem = gtk_menu_item_new_with_label(_("Who is"));
    g_signal_connect(menuitem, "activate", G_CALLBACK(handle_command_callback), g_strdup_printf("WHOIS %s", target_nick));
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
    gtk_widget_show (menuitem);

    /* send a message */
    menuitem = gtk_menu_item_new_with_label(_("Send message"));
    g_signal_connect(menuitem, "activate", G_CALLBACK(handle_command_callback), g_strdup_printf("QUERY %s", target_nick));
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
    gtk_widget_show (menuitem);

    /* a separator here */
    menuitem = gtk_separator_menu_item_new();
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
    gtk_widget_show(menuitem);

    /* cumode +o user */
    menuitem = gtk_menu_item_new_with_label(_("Op"));
    g_signal_connect(menuitem, "activate", G_CALLBACK(handle_command_callback), g_strdup_printf("OP %s", target_nick));
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
    gtk_widget_show(menuitem);

    /* cumode -o user */
    menuitem = gtk_menu_item_new_with_label(_("Deop"));
    g_signal_connect(menuitem, "activate", G_CALLBACK(handle_command_callback), g_strdup_printf("DEOP %s", target_nick));
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
    gtk_widget_show(menuitem);

    /* cumode +q user */
    menuitem = gtk_menu_item_new_with_label(_("Mute"));
    g_signal_connect(menuitem, "activate", G_CALLBACK(handle_command_callback), g_strdup_printf("MUTE %s", target_nick));
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
    gtk_widget_show(menuitem);

    /* cumode -q user */
    menuitem = gtk_menu_item_new_with_label(_("Unmute"));
    g_signal_connect(menuitem, "activate", G_CALLBACK(handle_command_callback), g_strdup_printf("UNMUTE %s", target_nick));
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
    gtk_widget_show(menuitem);

    /* a separator here */
    menuitem = gtk_separator_menu_item_new();
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
    gtk_widget_show(menuitem);

    /* kick user */
    menuitem = gtk_menu_item_new_with_label(_("Kick"));
    g_signal_connect(menuitem, "activate", G_CALLBACK(handle_command_callback), g_strdup_printf("KICK %s", target_nick));
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
    gtk_widget_show(menuitem);


    /* kill user */
    /* FIXME, the on_command_kill needs some work
       menuitem = gtk_menu_item_new_with_label(_("Kill"));
       g_signal_connect(menuitem, "activate", G_CALLBACK(on_command_kill), clientid);
       gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
       gtk_widget_show(menuitem);
    */

    /* show the menu */
    gtk_menu_popup (menu, NULL, NULL, NULL, NULL,
		    event->button.button, event->button.time);

  }

  /* right button released, should hide a context sensitive menu if it was visible */
  if (event->type == GDK_BUTTON_RELEASE && event->button.button == 3 ) {
    debug ("Released right-click");
  }

  /* double left-click */
  if (target_nick && event->type == GDK_2BUTTON_PRESS && event->button.button == 1 ) {
    debug ("Double left-clicked on '%s'", target_nick);
    handle_command(g_strdup_printf("QUERY %s", target_nick));
  }

  g_free(target_nick);
  return TRUE;
}

SilcChannelUser get_channel_user_by_entry(SilcChannelEntry channel, SilcClientEntry client) {
  SilcChannelUser chu;
  SilcHashTableList htl;

  if( !channel ) {
    debug("channel == NULL");
    return NULL;
  }

  if (!channel->user_list) {
    debug("no ulist, returning");
    return NULL;
  }

  debug("calling silc_hash_table_list()");
  silc_hash_table_list(channel->user_list, &htl);
  debug("looping");

  while(silc_hash_table_get(&htl, NULL, (void *)&chu)) {
    if( client == chu->client ) return chu;
  }
  return NULL;
}



/*
  Tries to initialize mimetable.
*/
gboolean silky_read_mimetable() {

  gint i;

  /*
    First NULL is reserved for later use..
  */
  gchar *mimefiles[] = {
    NULL,
    "/etc/mime.types",		/* default	*/
    "/usr/pkg/etc/mime.types",	/* pkgsrc	*/
    "/usr/lib/mime.types",		/* IRIX 6.5	*/
    "/usr/local/etc/mime.types",	/* default?	*/
    "mime.types",			/* Current dir	*/
    NULL
  };

  if (!homedir) {
    g_error("No homedir\n");
  }

#ifdef SILKY_WIN32
  mimefiles[0] = g_strconcat(homedir, "\\mime.types", NULL);
#else
  mimefiles[0] = g_strconcat(homedir, "/.mime.types", NULL);
#endif

  /* Check all paths from mimefiles to find working "mime.types" file, stops on first one */
  for (i=0; mimefiles[i]; i++) {
    debug("trying %d: '%s'", i, mimefiles[i]);
    mimetable = mime_init(mimefiles[i]);
    if (mimetable) {
      break;
    }
  }

  /* We didn't find any mime.table: give error and quit */
  if (!mimetable) {
    debug("mime.types not found at all");
    return FALSE;
  }
  else {
      debug("found mime.types. Hurray!");
    return TRUE;
  }

}

gchar *silky_get_homedir() {
  gchar *homedir;

  homedir = g_strdup(g_get_home_dir());
  if (!homedir) {
    debug("homedir not found, can not continue");
    return NULL;
  }
  else {
    return homedir;
  }

}

/*
  returns full path to user's silkydir
*/

gchar *silky_get_silkydir(gchar *homedir) {

#ifdef SILKY_WIN32
  return g_strconcat(homedir, "\\silky", NULL);
#else
  return g_strconcat(homedir, "/.silky/", NULL);
#endif

}


/*
  Create silkydir.
*/
gboolean silky_create_silkydir(gchar *silkydir) {
  gint ret;

  if( g_file_test(silkydir, G_FILE_TEST_EXISTS) ) {
    if( !g_file_test(silkydir, G_FILE_TEST_IS_DIR) ) {
      debug("Error: '%s' exists, but is NOT a directory. Aborting", silkydir);
      return FALSE;
    }
    debug("'%s' exists and is a directory, continuing", silkydir);
    return TRUE;
  }

  debug("Directory '%s' does not exist, trying to create it", silkydir);

#ifdef SILKY_WIN32
  ret = mkdir(silkydir);
#else
  ret = mkdir(silkydir, 0700);
#endif

   if (ret == -1) {
    debug("error creating '%s' (error %d: %s)", silkydir, errno, strerror(errno));

    if (errno == EEXIST) {
      debug("'%s' exists, using it", silkydir);
    }
    else {
      debug("Can not create '%s'", silkydir);
      return FALSE;
    }
  }
  else {
    debug ("'%s' created", silkydir);
  }

   return TRUE;

}


/*
  creates serverkeydir
*/
gboolean silky_create_serverkeydir(gchar *serverkeydir) {
  gint ret;

  debug("trying to create '%s'", serverkeydir);
  if (!serverkeydir) return FALSE;

#ifdef SILKY_WIN32
  ret = mkdir(serverkeydir);
#else
  ret = mkdir(serverkeydir, 0700);
#endif
  if (ret == -1) {
    debug("error creating '%s' (error %d: %s)", serverkeydir, errno, strerror(errno));

    if (errno == EEXIST) {
      debug("'%s' exists, using it", serverkeydir);
    }
    else {
      g_error("Can not create '%s'\n", serverkeydir);
      return FALSE;
    }
  }
  else {
    debug ("'%s' created", serverkeydir);
  }

  return TRUE;

}

/* fixme: is this, and why, needed? */
void silky_nickname_parse(const char *nickname, char **ret_nickname) {
  //  strncpy (&ret_nickname, nickname, sizeof(&ret_nickname)-1);
  debug("got '%s'", nickname);
  *ret_nickname = strdup(nickname);
}

/* Returns a random positive number between 'min' and 'max-1' */
unsigned int get_random_number(const unsigned int min,
    const unsigned int max) {
  unsigned int range, ret;
  SilcUInt32 random;

  /* Largish random number */
  random = silc_rng_get_rn32(silky->rng);

  /* Random number in range of given values */
  range = max - min;
  ret = random % range;

  return ret + min;
}


/* inputs null, returns active textview widget */
void * silky_get_active_textview () {
  GtkTextView *textview = NULL;
  GtkNotebook *storednotebook = NULL;
  GladeXML *stored_xml;
  GtkWidget *active_page = NULL;
  
  storednotebook = GTK_NOTEBOOK(glade_xml_get_widget (xmlmain, "tabs"));
  active_page = gtk_notebook_get_nth_page(storednotebook, gtk_notebook_get_current_page(GTK_NOTEBOOK(storednotebook) ) ); /* active page from stored g_object*/
  stored_xml = glade_get_widget_tree(active_page);
  textview = GTK_TEXT_VIEW(glade_xml_get_widget (stored_xml, "textviewwidget"));
  return textview;
}
