/**
 * \file interpolater.hxx
 * Routines to handle linear interpolation from a table of x,y The
 * table must be sorted by "x" in ascending order
 */

// Written by Curtis Olson, started April 1998.
//
// Copyright (C) 1998  Curtis L. Olson  - curt@me.umn.edu
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA  02111-1307, USA.
//
// $Id: interpolater.hxx,v 1.1.1.1 2002/09/07 02:58:19 curt Exp $


#ifndef _INTERPOLATER_H
#define _INTERPOLATER_H


#ifndef __cplusplus                                                          
# error This library requires C++
#endif                                   

#include <simgear/compiler.h>

#include <vector>
SG_USING_STD(vector);

#include STL_STRING
SG_USING_STD(string);


/**
 * A class that provids a simple linear 2d interpolation lookup table.
 * The actual table is expected to be loaded from a file.  The
 * independant variable must be strictly ascending.  The dependent
 * variable can be anything.
 */
class SGInterpTable {

    struct Entry
    {
      Entry ()
	: ind(0.0L), dep(0.0L) {}
      Entry (double independent, double dependent)
	: ind(independent), dep(dependent) {}
      double ind;
      double dep;
    };

    int size;
    vector<Entry> table;

public:

    /**
     * Constructor. Creates a new, empty table.
     */
    SGInterpTable();

    /**
     * Constructor. Loads the interpolation table from the specified file.
     * @param file name of interpolation file
     */
    SGInterpTable( const string& file );


    /**
     * Add an entry to the table, extending the table's length.
     *
     * @param ind The independent variable.
     * @param dep The dependent variable.
     */
    void addEntry (double ind, double dep);
    

    /**
     * Given an x value, linearly interpolate the y value from the table.
     * @param x independent variable
     * @return interpolated dependent variable
     */
    double interpolate(double x) const;

    /** Destructor */
    ~SGInterpTable();
};


#endif // _INTERPOLATER_H


