/*
** Copyright (C) 10 Feb 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
//Originally part of gcombust
#include <gtk/gtk.h>
#include <glib.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>

//#include "vector_commands.h"
//#include "contractions.h"
#include "clist_common.h"
//#include "globals.h"

static char *last_cwd = NULL;

/* The following functions for unescaping uris are from
 * libgnomevfs/gnome-vfs-utils.c, cvs-lxr version (May 19 2001) */

static int
hex_to_int (gchar c)
{
	return  c >= '0' && c <= '9' ? c - '0'
		: c >= 'A' && c <= 'F' ? c - 'A' + 10
		: c >= 'a' && c <= 'f' ? c - 'a' + 10
		: -1;
}

static int
unescape_character (const char *scanner)
{
	int first_digit;
	int second_digit;

	first_digit = hex_to_int (*scanner++);
	if (first_digit < 0) {
		return -1;
	}

	second_digit = hex_to_int (*scanner++);
	if (second_digit < 0) {
		return -1;
	}

	return (first_digit << 4) | second_digit;
}

/*  Decode %xx escaped characters
 **  -----------------------------
 **
 ** This function takes a pointer to a string in which some
 ** characters may have been encoded in %xy form, where xy is
 ** the ASCII hex code for character 16x+y.
 */
#define HEX_ESCAPE '%'

static gchar *
gnome_vfs_unescape_string (const gchar *escaped, const gchar *illegal_characters)
{
	const gchar *in;
	gchar *out;
	gint character;
   gulong temp;
	if (escaped == NULL) {
		return NULL;
	}
   temp=strlen(escaped) + 1;
	gchar *result =(gchar *)g_malloc(temp);

	out = result;
	for (in = escaped; *in != '\0'; in++) {
		character = *in;
		if (*in == HEX_ESCAPE) {
			character = unescape_character (in + 1);

			/* Check for an illegal character. We consider '\0' illegal here. */
			if (character <= 0
					|| (illegal_characters != NULL
						&& strchr (illegal_characters, (char)character) != NULL)) {
				g_free (result);
				return NULL;
			}
			in += 2;
		}
		*out++ = (char)character;
	}

	*out = '\0';
	g_assert (out - result <= (signed long) strlen (escaped));
	return result;

}

/* These are from gnome-libs-1.0.9/libgnome
 * If gcombust ever gets gnome support they can be replaced with the
 * corresponding gnome-libs functions, but for now I'm too lazy... ;)
 * 
 * Copyright 1997 Paolo Molaro
 * Copyright 1998 Miguel de Icaza.
 *
 */

/**
 * gnome_uri_list_extract_uris:
 * @uri_list: an uri-list in the standard format.
 *
 * Returns a GList containing strings allocated with g_malloc
 * that have been splitted from @uri-list.
 */
static GList*        
gnome_uri_list_extract_uris (const gchar* uri_list, const gint type)
{
	const gchar *p, *q;
	gchar *retval;
	GList *result = NULL;
	
	g_return_val_if_fail (uri_list != NULL, NULL);

	p = uri_list;
	
	/* We don't actually try to validate the URI according to RFC
	 * 2396, or even check for allowed characters - we just ignore
	 * comments and trim whitespace off the ends.  We also
	 * allow LF delimination as well as the specified CRLF.
	 */
	while (p) {
		if (*p != '#') {
			while (isspace(*p))
				p++;
			
			q = p;
			while (*q && (*q != '\n') && (*q != '\r'))
				q++;
		  
			if (q > p) {
			        q--;
				while (q > p && isspace(*q))
					q--;
				
				retval = (gchar *)g_malloc (q - p + 2);
				strncpy (retval, p, q - p + 1);
				retval[q - p + 1] = '\0';

				if (0 == type)
					result = g_list_prepend (result, retval);
				else {
					gchar *unescaped = gnome_vfs_unescape_string(retval, "");
					g_free(retval);
					result = g_list_prepend (result, unescaped);
				}
			}
		}
		p = strchr (p, '\n');
		if (p)
			p++;
	}
	
	return g_list_reverse (result);
}

/**
 * gnome_uri_list_extract_filenames:
 * @uri_list: an uri-list in the standard format
 *
 * Returns a GList containing strings allocated with g_malloc
 * that contain the filenames in the uri-list.
 *
 * Note that unlike gnome_uri_list_extract_uris() function, this
 * will discard any non-file uri from the result value.
 */
GList*        
gnome_uri_list_extract_filenames (const gchar* uri_list, const gint type)
{
	GList *tmp_list, *node, *result;
	
	g_return_val_if_fail (uri_list != NULL, NULL);

	result = gnome_uri_list_extract_uris (uri_list, type);

	tmp_list = result;
	while (tmp_list) {
		gchar *s = (gchar *)tmp_list->data;
		
		node = tmp_list;
		tmp_list = tmp_list->next;

		if (!strncmp (s, "file:", 5)) {
			node->data = g_strdup (s+5);
		} else {
			result = g_list_remove_link(result, node);
			g_list_free_1 (node);
		}
		g_free (s);
	}
	return result;
}

/**
 * gnome_uri_list_free_strings:
 * @list: A GList returned by gnome_uri_list_extract_uris() or gnome_uri_list_extract_filenames()
 *
 * Releases all of the resources allocated by @list.
 */
void          
gnome_uri_list_free_strings      (GList *list)
{
	g_list_foreach (list, (GFunc)g_free, NULL);
	g_list_free (list);
}

