/*
 SimpleCDR Copyright (C) 2001 John Tobin

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 If you would like to negotiate alternate licensing terms, you may do
 so by contacting the author: John Tobin <ogre@sirinet.net>

wave.cpp

	Copyright (c) 1996, 1988, 2000 by Timothy J. Weber.
*/

#ifndef _MSC_VER
#include <stdlib.h>
#else
// Microsoft doesn't include min, though it's part of the standard library!
template<class T>
T min(T a, T b) { return a < b? a: b; }
#endif

#include "wave.h"

using namespace std;

/***************************************************************************
	macros and constants
***************************************************************************/

// constants for the canonical WAVE format
const int fmtChunkLength = 16;  // length of fmt contents
const int waveHeaderLength = 4 + 8 + fmtChunkLength + 8;  // from "WAVE" to sample data

/***************************************************************************
	typedefs and class definitions
***************************************************************************/

/***************************************************************************
	prototypes for static functions
***************************************************************************/

/***************************************************************************
	static variables
***************************************************************************/

/***************************************************************************
	public member functions for WaveFile
***************************************************************************/

WaveFile::WaveFile():
	readFile(0),
	formatType(0),
	numChannels(0),
	sampleRate(0),
	bytesPerSecond(0),
	bytesPerSample(0),
	bitsPerChannel(0),
	dataLength(0),
	error(0),
	changed(true)
{
}

WaveFile::~WaveFile()
{
	Close();
}

bool WaveFile::OpenRead(const char* name)
{
	if (readFile || writeFile)
		Close();

	try {
		// open the RIFF file
		readFile = new RiffFile(name);
		if (!readFile->filep())
			throw error = "Couldn't open file";

		// read the header information
		if (strcmp(readFile->chunkName(), "RIFF")
			|| strcmp(readFile->subType(), "WAVE")
			|| !readFile->push("fmt "))
			throw error = "Couldn't find RIFF, WAVE, or fmt";

		size_t dwFmtSize = size_t(readFile->chunkSize());
		char* fmtChunk = new char[dwFmtSize];
		try {
			if (fread(fmtChunk, dwFmtSize, 1, readFile->filep()) != 1)
				throw error = "Error reading format chunk";
			readFile->pop();

			// set the format attribute members
			formatType = *((short*) fmtChunk);
			numChannels = *((short*) (fmtChunk + 2));
			sampleRate = *((long*) (fmtChunk + 4));
			bytesPerSecond = *((long*) (fmtChunk + 8));
			bytesPerSample = *((short*) (fmtChunk + 12));
			bitsPerChannel = *((short*) (fmtChunk + 14));

			// position at the data chunk
			if (!readFile->push("data"))
				throw error = "Couldn't find data chunk";

			// get the size of the data chunk
			dataLength = readFile->chunkSize();

			delete[] fmtChunk;
		} catch (...) {
			delete[] fmtChunk;
			throw error;
		}
	} catch (...) {
		Close();
		return false;
	}
	return true;
}



bool WaveFile::ResetToStart()
{
	if (readFile) {
		// pop out of the data chunk
		if (!readFile->rewind()
			|| !readFile->push("data"))
		{
			error = "Couldn't find data chunk on reset";
			return false;
		} else
			return true;
	} else if (writeFile) {
		return fseek(writeFile, waveHeaderLength, SEEK_SET) == 0;
	} else
		return false;
}

bool WaveFile::Close()
{
	bool retval = true;

	if (readFile) {
		delete readFile;  // closes the file before it's destroyed
		readFile = 0;
	}

	return retval;
}

bool WaveFile::FormatMatches(const WaveFile& other)
{
	return formatType == other.formatType
		&& numChannels == other.numChannels
		&& sampleRate == other.sampleRate
		&& bytesPerSecond == other.bytesPerSecond
		&& bytesPerSample == other.bytesPerSample
		&& bitsPerChannel == other.bitsPerChannel;
}

void WaveFile::SetupFormat(int sampleRate, short bitsPerChannel, short channels)
{
	SetFormatType(1);
	SetNumChannels(channels);
	SetSampleRate(sampleRate);
	SetBytesPerSample((unsigned short)((bitsPerChannel >> 3) / channels));
	SetBytesPerSecond(sampleRate * channels * GetBytesPerSample());
	SetBitsPerChannel(bitsPerChannel);
	SetNumSamples(0);
}

bool WaveFile::GetFirstExtraItem(string& type, string& value)
{
	if (readFile)
		return readFile->rewind() && readFile->getNextExtraItem(type, value);
	else
		return false;
}

bool WaveFile::GetNextExtraItem(string& type, string& value)
{
	if (readFile)
		return readFile->getNextExtraItem(type, value);
	else
		return false;
}

bool WaveFile::CopyFrom(WaveFile& other)
{
	const size_t transferBufSize = 4096;

	if (!writeFile) {
		error = "Copy to an unopened file";
		return false;
	} else if (!other.readFile) {
		error = "Copy from an unopened file";
		return false;
	}

	try {
		// allocate the transfer buffer
		char* transferBuffer = new char[transferBufSize];
		unsigned long bytesRead = 0;

		try {
			if (!other.ResetToStart())
				throw error = "Couldn't reset input file to start";

			while (bytesRead < other.dataLength) {
				// calculate the size of the next buffer
				size_t bytesToRead = (size_t) min(transferBufSize,
					size_t(other.dataLength - bytesRead));

				// read the buffer
				if (fread(transferBuffer, 1, bytesToRead, other.readFile->filep())
					!= bytesToRead)
					throw error = "Error reading samples from input file";
				bytesRead += bytesToRead;

				// write the buffer
				if (fwrite(transferBuffer, 1, bytesToRead, writeFile) != bytesToRead)
					throw error = "Error writing samples to output file";
				dataLength += bytesToRead;
				changed = true;
			}

			// delete the transfer buffer
			delete[] transferBuffer;
		} catch (...) {
			delete[] transferBuffer;
			throw error;
		}
	} catch (...) {
		return false;
	}

	return true;
}



/***************************************************************************
	private member functions for WaveFile
***************************************************************************/

/***************************************************************************
	main()
***************************************************************************/

#ifdef TEST_WAVE

#include <iostream>

static void reportProblem()
{
	cout << "  *** ERROR: Result incorrect." << endl;
}

static void checkResult(bool got, bool expected)
{
	if (got)
		cout << "success." << endl;
	else 
		cout << "fail." << endl;

	if (got != expected)
		reportProblem();
}

static void pause()
{
	cout << "Press Enter to continue." << endl;
	cin.get();
}

static void ShowErrors(WaveFile& from, WaveFile& to)
{
	bool any = from.GetError() || to.GetError();

	if (from.GetError())
		cout << "Error on input: " << from.GetError() << "." << endl;

	if (to.GetError())
		cout << "Error on output: " << to.GetError() << "." << endl;

	if (!any)
		cout << "Success." << endl;
}

static void ShowFormat(WaveFile& wave)
{
	cout
		<< "Format:           " << wave.GetFormatType()
		<< (wave.IsCompressed()? " (compressed)" : " (PCM)") << endl
		<< "Channels:         " << wave.GetNumChannels() << endl
		<< "Sample rate:      " << wave.GetSampleRate() << endl
		<< "Bytes per second: " << wave.GetBytesPerSecond() << endl
		<< "Bytes per sample: " << wave.GetBytesPerSample() << endl
		<< "Bits per channel: " << wave.GetBitsPerChannel() << endl
		<< "Bytes:            " << wave.GetDataLength() << endl
		<< "Samples:          " << wave.GetNumSamples() << endl
		<< "Seconds:          " << wave.GetNumSeconds() << endl
		<< "File pointer:     " << (wave.GetFile()? "good" : "null") << endl;

	string type, value;
	if (wave.GetFirstExtraItem(type, value)) {
		cout << "Extra data:" << endl;
		do {
			cout << "  " << type << ": " << value << endl;
		} while (wave.GetNextExtraItem(type, value));

		wave.ResetToStart();
	}

	pause();
}

int main(int argc, const char* argv[])
{
	if (argc < 3)
		cout << "Copies one WAVE file to another, in canonical form." << endl;
	else {
		WaveFile From, To;

		cout << "Opening input..." << endl;
		From.OpenRead(argv[1]);
		ShowErrors(From, To);
		ShowFormat(From);

		cout << "Setting formats..." << endl;
		To.CopyFormatFrom(From);
		ShowFormat(To);

		cout << "Opening output..." << endl;
		To.OpenWrite(argv[2]);
		ShowErrors(From, To);

		cout << "Copying..." << endl;
		To.CopyFrom(From);
		ShowErrors(From, To);
		cout << "Resulting format: " << endl;
		ShowFormat(To);
		cout << "Source format: " << endl;
		ShowFormat(From);

		cout << "Closing..." << endl;
		To.Close();
		From.Close();
		ShowErrors(From, To);
	}

	return 0;
}

#endif
