/*
 * $Id: ports.c,v 1.15 2003/12/01 09:10:15 troth Exp $
 *
 ****************************************************************************
 *
 * simulavr - A simulator for the Atmel AVR family of microcontrollers.
 * Copyright (C) 2001, 2002, 2003  Theodore A. Roth
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 ****************************************************************************
 */

/**
 * \file ports.c
 * \brief Module for accessing simulated I/O ports.
 *
 * Defines an abstract Port class as well as subclasses for each individual
 * port.
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>

#include "avrerror.h"
#include "avrmalloc.h"
#include "avrclass.h"
#include "utils.h"
#include "callback.h"
#include "op_names.h"

#include "storage.h"
#include "flash.h"

#include "vdevs.h"
#include "memory.h"
#include "stack.h"
#include "register.h"
#include "sram.h"
#include "eeprom.h"
#include "timers.h"
#include "ports.h"

#include "avrcore.h"

/****************************************************************************\
 *
 * global variables
 *
\****************************************************************************/

/** \brief FIXME: This should be static. */
char *name_PIN[] = { "PINA", "PINB", "PINC", "PIND", "PINE", "PINF" };

/** \brief FIXME: This should be static. */
char *name_DDR[] = { "DDRA", "DDRB", "DDRC", "DDRD", "DDRE", "DDRF" };

/** \brief FIXME: This should be static. */
char *name_PORT[] = { "PORTA", "PORTB", "PORTC", "PORTD", "PORTE", "PORTF" };

/****************************************************************************\
 *
 * Local static prototypes.
 *
\****************************************************************************/

static uint8_t port_reg_read (VDevice *dev, int addr);
static void port_reg_write (VDevice *dev, int addr, uint8_t val);
static void port_reset (VDevice *dev);
static char *port_reg_name (VDevice *dev, int addr);

static uint8_t port_read_pin (Port *p, int addr);

static uint8_t port_read_port (Port *p, int addr);
static void port_write_port (Port *p, int addr, uint8_t val);

static uint8_t port_read_ddr (Port *p, int addr);
static void port_write_ddr (Port *p, int addr, uint8_t val);

/****************************************************************************\
 *
 * Port(VDevice) : I/O Port registers
 *
\****************************************************************************/

/**
 * \brief Allocates a new Port object.
 *
 * This is a virtual method for higher level port implementations and as such
 * should not be used directly.
 */
Port *
port_new (char *name, int base, int pins, PortFP_AltRd alt_rd,
          PortFP_AltWr alt_wr)
{
    Port *p;

    p = avr_new (Port, 1);
    port_construct (p, name, base, pins, alt_rd, alt_wr);
    class_overload_destroy ((AvrClass *)p, port_destroy);

    return p;
}

/**
 * \brief Constructor for the Port object.
 *
 * This is a virtual method for higher level port implementations and as such
 * should not be used directly.
 */
void
port_construct (Port *p, char *name, int base, int pins, PortFP_AltRd alt_rd,
                PortFP_AltWr alt_wr)
{
    if (p == NULL)
        avr_error ("passed null ptr");

    vdev_construct ((VDevice *)p, name, base, PORT_SIZE, port_reg_read,
                    port_reg_write, port_reset, port_reg_name);

    p->mask = (uint8_t) (((uint16_t) 1 << pins) - 1);

    p->alt_rd = alt_rd;
    p->alt_wr = alt_wr;

    p->ext_rd = NULL;
    p->ext_wr = NULL;

    port_reset ((VDevice *)p);
}

static void
port_reset (VDevice *dev)
{
    Port *p = (Port *)dev;

    p->port = 0;
    p->ddr = 0;

    p->ext_enable = 1;
}

/**
 * \brief Destructor for the Port object
 *
 * This is a virtual method for higher level port implementations and as such
 * should not be used directly.
 */
void
port_destroy (void *p)
{
    if (p == NULL)
        return;

    vdev_destroy (p);
}

/** \brief Disable external port functionality.
 *
 * This is only used when dumping memory to core file. See mem_io_fetch().
 */
void
port_ext_disable (Port *p)
{
    p->ext_enable = 0;
}

/** \brief Enable external port functionality.
 *
 * This is only used when dumping memory to core file. See mem_io_fetch().
 */
void
port_ext_enable (Port *p)
{
    p->ext_enable = 1;
}

/**
 * \brief Attaches read and write functions to a particular port
 *
 * I think I may have this backwards. Having the virtual hardware supply
 * functions for the core to call on every io read/write, might cause missed
 * events (like edge triggered). I'm really not too sure how to handle this.
 *
 * In the future, it might be better to have the core supply a function for
 * the virtual hardware to call when data is written to the device. The device
 * supplied function could then check if an interrupt should be generated or
 * just simply write to the port data register.
 *
 * For now, leave it as is since it's easier to test if you can block when the
 * device is reading from the virtual hardware.
 */
void
port_add_ext_rd_wr (Port *p, PortFP_ExtRd ext_rd, PortFP_ExtWr ext_wr)
{
    p->ext_rd = ext_rd;
    p->ext_wr = ext_wr;
}

static uint8_t
port_read_pin (Port *p, int addr)
{
    uint8_t data;

    /* get the data from the external virtual hardware if connected */
    if (p->ext_rd && p->ext_enable)
        data = p->ext_rd (addr) & p->mask;
    else
        data = 0;

    /*
     * For a pin n to be enabled as input, DDRn == 0,
     * otherwise it will always read 0.
     */
    data &= ~(p->ddr);

    /*
     * Pass data to alternate read so as to check alternate functions of
     * pins for that port.
     */
/*      if (p->alt_rd) */
/*          data = p->alt_rd(p, addr, data); */

    return data;
}

static uint8_t
port_read_port (Port *p, int addr)
{
    return (p->port & p->mask);
}

static void
port_write_port (Port *p, int addr, uint8_t val)
{
    /* update port register */
    p->port = (val & p->mask);

    /*
     * Since changing p->port might change what the virtual hardware
     * sees, we need to call ext_wr() to pass change allong.
     */
    if (p->ext_wr && p->ext_enable)
        p->ext_wr (addr, (p->port & p->ddr));
}

static uint8_t
port_read_ddr (Port *p, int addr)
{
    return (p->ddr & p->mask);
}

static void
port_write_ddr (Port *p, int addr, uint8_t val)
{
    /* update ddr register */
    p->ddr = (val & p->mask);

    /*
     * Since changing p->ddr might change what the virtual hardware
     * sees, we need to call ext_wr() to pass change allong.
     */
    if (p->ext_wr && p->ext_enable)
        p->ext_wr (addr, (p->port & p->ddr));
}

static uint8_t
port_reg_read (VDevice *dev, int addr)
{
    Port *p = (Port *)dev;
    uint8_t val = 0;

    switch (addr - vdev_get_base (dev))
    {
        case PORT_DDR:
            val = port_read_ddr ((Port *)p, addr);
            break;
        case PORT_PIN:
            val = port_read_pin ((Port *)p, addr);
            break;
        case PORT_PORT:
            val = port_read_port ((Port *)p, addr);
            break;

        default:
            avr_error ("Invalid Port Address: 0x%02x", addr);
    }

    return val;
}

static void
port_reg_write (VDevice *dev, int addr, uint8_t val)
{
    Port *p = (Port *)dev;

    switch (addr - vdev_get_base (dev))
    {
        case PORT_PIN:
            avr_error ("Attempt to write to readonly PINx register");

        case PORT_DDR:
            port_write_ddr ((Port *)p, addr, val);
            return;

        case PORT_PORT:
            port_write_port ((Port *)p, addr, val);
            return;
    }
    avr_error ("Invalid Port Address: 0x%02x", addr);
}

static char *
port_reg_name (VDevice *dev, int addr)
{
    int port = vdev_get_name (dev)[4] - 'A';

    switch (addr - vdev_get_base (dev))
    {
        case PORT_PIN:
            return name_PIN[port];
        case PORT_DDR:
            return name_DDR[port];
        case PORT_PORT:
            return name_PORT[port];
    }
    avr_error ("Invalid Port %c Address: 0x%02x", addr, 'A' + port);
    return NULL;
}

/****************************************************************************\
 *
 * PortA(Port) : Port A Definition.
 *
\****************************************************************************/

/** \brief Allocate a new PortA object. */

PortA *
porta_new (int pins)
{
    PortA *p;

    p = avr_new (PortA, 1);
    porta_construct (p, pins);
    class_overload_destroy ((AvrClass *)p, porta_destroy);

    return p;
}

/**
 * \brief Constructor for the PortA object.
 *
 * Port A also functions as low byte of address into external SRAM if enabled.
 *   - PAn <--> ADn
 */
void
porta_construct (PortA *p, int pins)
{
    if (p == NULL)
        avr_error ("passed null ptr");

    port_construct ((Port *)p, name_PORT[PORT_A], PORT_A_BASE, pins, NULL,
                    NULL);
}

/** \brief Destructor for the PortA object. */

void
porta_destroy (void *p)
{
    if (p == NULL)
        return;

    port_destroy (p);
}

/****************************************************************************\
 *
 * PortB(Port) : Port B Definition.
 *
\****************************************************************************/

/** \brief Allocate a new PortB object. */

PortB *
portb_new (int pins)
{
    PortB *p;

    p = avr_new (PortB, 1);
    portb_construct (p, pins);
    class_overload_destroy ((AvrClass *)p, portb_destroy);

    return p;
}

/** \brief Constructor for the PortB object. */

void
portb_construct (PortB *p, int pins)
{
    if (p == NULL)
        avr_error ("passed null ptr");

    port_construct ((Port *)p, name_PORT[PORT_B], PORT_B_BASE, pins, NULL,
                    NULL);
}

/** \brief Destructor for the PortB object. */

void
portb_destroy (void *p)
{
    if (p == NULL)
        return;

    port_destroy (p);
}

/****************************************************************************\
 *
 * PortC(Port) : Port C Definition.
 *
\****************************************************************************/

/** \brief Allocate a new PortC object. */

PortC *
portc_new (int pins)
{
    PortC *p;

    p = avr_new (PortC, 1);
    portc_construct (p, pins);
    class_overload_destroy ((AvrClass *)p, portc_destroy);

    return p;
}

/** \brief Constructor for the PortC object. */

void
portc_construct (PortC *p, int pins)
{
    if (p == NULL)
        avr_error ("passed null ptr");

    port_construct ((Port *)p, name_PORT[PORT_C], PORT_C_BASE, pins, NULL,
                    NULL);
}

/** \brief Destructor for the PortC object. */

void
portc_destroy (void *p)
{
    if (p == NULL)
        return;

    port_destroy (p);
}

/****************************************************************************\
 *
 * PortD(Port) : Port D Definition.
 *
\****************************************************************************/

/** \brief Allocate a new PortD object. */

PortD *
portd_new (int pins)
{
    PortD *p;

    p = avr_new (PortD, 1);
    portd_construct (p, pins);
    class_overload_destroy ((AvrClass *)p, portd_destroy);

    return p;
}

/** \brief Constructor for the PortD object. */

void
portd_construct (PortD *p, int pins)
{
    if (p == NULL)
        avr_error ("passed null ptr");

    port_construct ((Port *)p, name_PORT[PORT_D], PORT_D_BASE, pins, NULL,
                    NULL);
}

/** \brief Destructor for the PortD object. */

void
portd_destroy (void *p)
{
    if (p == NULL)
        return;

    port_destroy (p);
}

/****************************************************************************\
 *
 * PortE(Port) : Port E Definition.
 *
\****************************************************************************/

/** \brief Allocate a new PortE object. */

PortE *
porte_new (int pins)
{
    PortE *p;

    p = avr_new (PortE, 1);
    porte_construct (p, pins);
    class_overload_destroy ((AvrClass *)p, porte_destroy);

    return p;
}

/** \brief Constructor for the PortE object. */

void
porte_construct (PortE *p, int pins)
{
    if (p == NULL)
        avr_error ("passed null ptr");

    port_construct ((Port *)p, name_PORT[PORT_E], PORT_E_BASE, pins, NULL,
                    NULL);
}

/** \brief Destructor for the PortE object. */

void
porte_destroy (void *p)
{
    if (p == NULL)
        return;

    port_destroy (p);
}

/****************************************************************************\
 *
 * PortF(Port) : Port F Definition.
 *
\****************************************************************************/

/** \brief Allocate a new PortF object. */

PortF *
portf_new (int pins)
{
    PortF *p;

    p = avr_new (PortF, 1);
    portf_construct (p, pins);
    class_overload_destroy ((AvrClass *)p, portf_destroy);

    return p;
}

/** \brief Constructor for the PortF object. */

void
portf_construct (PortF *p, int pins)
{
    if (p == NULL)
        avr_error ("passed null ptr");

    port_construct ((Port *)p, name_PORT[PORT_F], PORT_F_BASE, pins, NULL,
                    NULL);
}

/** \brief Destructor for the PortF object. */

void
portf_destroy (void *p)
{
    if (p == NULL)
        return;

    port_destroy (p);
}
