/*
 * This file is part of Siril, an astronomy image processor.
 * Copyright (C) 2005-2011 Francois Meyer (dulle at free.fr)
 * Copyright (C) 2012-2014 team free-astro (see more in AUTHORS file)
 * Reference site is http://free-astro.vinvin.tf/index.php/Siril
 *
 * Siril is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Siril is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Siril. If not, see <http://www.gnu.org/licenses/>.
*/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include "ser.h"

void ser_display_info(struct ser_struct *ser_file) {
	fprintf(stdout, "=========== SER file info ==============\n");
	fprintf(stdout, "file id: %s\n", ser_file->file_id);
	fprintf(stdout, "lu id: %d\n", ser_file->lu_id);
	fprintf(stdout, "color: %s\n",
			ser_file->color_id == SER_MONO ? "mono" : "colored" );
	fprintf(stdout, "image size: %d x %d (%d bits)\n",
			ser_file->image_width, ser_file->image_height,
			ser_file->pixel_depth == SER_PIXEL_DEPTH_8 ? 8 : 16);
	fprintf(stdout, "frame count: %d\n", ser_file->frame_count);
	fprintf(stdout, "observer: %s\n", ser_file->observer);
	fprintf(stdout, "instrument: %s\n", ser_file->instrument);
	fprintf(stdout, "telescope: %s\n", ser_file->telescope);
	fprintf(stdout, "date: unsupported\n");
	fprintf(stdout, "========================================\n");
}

int _ser_read_header(struct ser_struct *ser_file) {
	char header[178];
	int pdepth, endianness;
	if (!ser_file || ser_file->fd <= 0)
		return -1;
	if (178 != read(ser_file->fd, header, 178)) {
		perror("read");
		return -1;
	}
	memcpy(&ser_file->lu_id, header+14, 4);
	memcpy(&endianness, header+22, 4);
	memcpy(&ser_file->image_width, header+26, 4);
	memcpy(&ser_file->image_height, header+30, 4);
	memcpy(&pdepth, header+34, 4);
	memcpy(&ser_file->frame_count, header+38, 4);

	if (endianness)
		ser_file->endianness = SER_LITTLE_ENDIAN;
	else 	ser_file->endianness = SER_BIG_ENDIAN;
	if (pdepth <= 8)
		ser_file->pixel_depth = SER_PIXEL_DEPTH_8;
	else 	ser_file->pixel_depth = SER_PIXEL_DEPTH_16;
	ser_file->file_id = strndup(header, 14);
	ser_file->observer = strndup(header+42, 40);
	ser_file->instrument = strndup(header+82, 40);
	ser_file->telescope = strndup(header+122, 40);
	return 0;
}

int ser_open_file(char *filename, struct ser_struct *ser_file) {
	if (ser_file->fd > 0) {
		fprintf(stderr, "SER: file already opened, or badly closed\n");
		return -1;
	}
	ser_file->fd = open(filename, O_RDONLY);
	if (ser_file->fd == -1) {
		perror("SER file open");
		return -1;
	}
	if (_ser_read_header(ser_file)) {
		fprintf(stderr, "SER: reading header failed, closing file %s\n",
				filename);
		ser_close_file(ser_file);
		return -1;
	}
	ser_file->filename = strdup(filename);
	return 0;
}

int ser_close_file(struct ser_struct *ser_file) {
	int retval = 0;
	if (!ser_file) return retval;
	if (ser_file->fd > 0) {
		retval = close (ser_file->fd);
		ser_file->fd = -1;
	}
	if (ser_file->file_id) free(ser_file->file_id);
	if (ser_file->observer) free(ser_file->observer);
	if (ser_file->instrument) free(ser_file->instrument);
	if (ser_file->telescope) free(ser_file->telescope);
	if (ser_file->filename) free(ser_file->filename);
	ser_init_struct(ser_file);
	return retval;
}

void ser_init_struct(struct ser_struct *ser_file) {
	//ser_file->fd = -1;
	//ser_file->filename = NULL;
	memset(ser_file, 0, sizeof(struct ser_struct));
}

/* frame number starts at 0 */
int ser_read_frame(struct ser_struct *ser_file, int frame_no, fits *fit) {
	int frame_size, retval, i;
	off_t offset;
	BYTE pixel;
	WORD *olddata;
	if (!ser_file || ser_file->fd <= 0 || !fit ||
			frame_no < 0 || frame_no >= ser_file->frame_count)
		return -1;
	frame_size = ser_file->image_width * ser_file->image_height;
	olddata = fit->data;
	if((fit->data = realloc(fit->data, frame_size * sizeof(WORD)))==NULL){
		fprintf(stderr,"ser_read: error realloc %s %d\n",
				ser_file->filename, frame_size);
		if (olddata) free(olddata);
		return -1;
	}

	offset = 178+frame_size*ser_file->pixel_depth*frame_no;
	/*fprintf(stdout, "offset is %lu (frame %d, %d pixels, %d-byte)\n", offset,
			frame_no, frame_size, ser_file->pixel_depth);*/
	if ((off_t)-1 == lseek(ser_file->fd, offset, SEEK_SET))
		return -1;
	retval = read(ser_file->fd, fit->data, frame_size * ser_file->pixel_depth);
	if (retval != frame_size * ser_file->pixel_depth) return -1;
	
	if (ser_file->pixel_depth == SER_PIXEL_DEPTH_8) {
		// inline conversion to 16 bit
		for (i = frame_size-1; i >= 0; i--)
			fit->data[i] = (WORD)(((BYTE*)fit->data)[i]);
	} else if (ser_file->endianness == SER_BIG_ENDIAN) {
		// inline conversion to little endian
		for (i = frame_size-1; i >= 0; i--) {
			pixel = fit->data[i];
			pixel = (pixel >> 8) | (pixel << 8);
			fit->data[i] = pixel;
		}
	}

	fit->bitpix = USHORT_IMG;
	fit->naxis = 2;
	fit->naxes[0] = fit->rx = ser_file->image_width;
	fit->naxes[1] = fit->ry = ser_file->image_height;
	fit->naxes[2] = 1;
	fit->pdata[RLAYER]=fit->data;
	fit->pdata[GLAYER]=fit->data;
	fit->pdata[BLAYER]=fit->data;
	return 0;
}

