/*
 * This file is part of Siril, an astronomy image processor.
 * Copyright (C) 2005-2011 Francois Meyer (dulle at free.fr)
 * Copyright (C) 2012-2014 team free-astro (see more in AUTHORS file)
 * Reference site is http://free-astro.vinvin.tf/index.php/Siril
 *
 * Siril is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Siril is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Siril. If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <assert.h>
#include <stdlib.h>
#include <math.h>
#include <locale.h>
#include <gsl/gsl_matrix.h>

#include "siril.h"
#include "PSF.h"
#include "PSF_list.h"
#include "callbacks.h"
#include "star_finder.h"
#include "proto.h"

WORD Compute_threshold(fits *fit, double starfinder, int layer){
	WORD threshold;
	double bgnoise[3];
	
	assert(layer <=3);
	
	backgroundnoise(fit, bgnoise);
	threshold=(WORD)(starfinder*bgnoise[layer]);
	
	return threshold;
}

/*
This is an implementation of a simple peak detector algorithm which 
identifies any pixel that is greater than any of its eight neighbors.

Original algorithm come from:
Copyleft (L) 1998 Kenneth J. Mighell (Kitt Peak National Observatory)
*/

/* returns a NULL-ended array of FWHM info */
Param_GAUSS **peaker(fits *fit, int layer){
	int nx = fit->rx;
	int ny = fit->ry;
	int x, y, xx, yy, i, j, radius, nbstars = 0;
	WORD pixel, neighbor, threshold;
	WORD **wave_image, **real_image;
	fits *wave_fit = calloc(1, sizeof(fits));
	double bg, starfinder;
	gboolean bingo;
	Param_GAUSS *cur_star;
	Param_GAUSS **results = malloc((MAX_STARS+1) * sizeof(Param_GAUSS*));
	gsl_matrix *z;
	static GtkSpinButton *spin_radius = NULL, *spin_threshold = NULL;
	
	assert(nx > 0 && ny > 0);
	
	struct timeval t_start, t_end;

	siril_log_color_message("Findstar: processing...\n", "red");	
	gettimeofday (&t_start, NULL);
		
	results[0]=NULL;
	
	/* Get radius and starfinder */
	if (spin_radius == NULL) {
		spin_radius = GTK_SPIN_BUTTON(gtk_builder_get_object(builder, "spinstarfinder_radius"));
		spin_threshold = GTK_SPIN_BUTTON(gtk_builder_get_object(builder, "spinstarfinder_threshold"));
	}
	radius = (int)gtk_spin_button_get_value(spin_radius);
	starfinder = gtk_spin_button_get_value(spin_threshold);
	
	threshold = Compute_threshold(fit, starfinder, layer);
	
	bg = background(fit, layer, NULL);
	copyfits(fit, wave_fit, CP_ALLOC|CP_FORMAT|CP_COPYA, 0);
	get_wavelet_layers(wave_fit, 4, 2, 2);
	
	/* FILL wavelet image upside-down */
	wave_image = malloc(ny*sizeof(WORD *));
	for (i=0; i<ny; i++)
		wave_image[ny-i-1] = wave_fit->pdata[layer] + i*nx;
		
	/* FILL real image upside-down */
	real_image = malloc(ny * sizeof(WORD *));
	for (i=0; i<ny; i++)
		real_image[ny-i-1] = fit->pdata[layer] + i*nx;

	for (y=radius; y<ny-radius; y++){
		for (x=radius; x<nx-radius; x++){
			pixel = wave_image[y][x];
			if (pixel >= threshold && pixel <= fit->maxi){
				bingo = TRUE;
				for (yy=y-1; yy<=y+1; yy++){
					for (xx=x-1; xx<=x+1; xx++){
						if (xx == x && yy == y) continue;
						neighbor = wave_image[yy][xx];
						if (neighbor > pixel) {
							bingo = FALSE;
							break;
						} else if (neighbor == pixel) {
							if ((xx <= x && yy <= y) || (xx > x && yy < y)) {
								bingo = FALSE;
								break;
							}
						}
					}
				}
				if (bingo && nbstars < MAX_STARS){
					int ii, jj;
					//~ fprintf(stdout, "Found a probable star at position (%d, %d) with a value of %hu\n", x, y, pixel);
					z = gsl_matrix_alloc (radius*2, radius*2);
					/* FILL z */
					for (jj=0, j=y-radius; j<y+radius; j++, jj++){
						for (ii=0, i=x-radius; i<x+radius; i++, ii++){
							gsl_matrix_set (z, ii, jj, (double)real_image[j][i]);
						}
					}
					/* ****** */
					/* In this case the angle is not fitted because it
					 *  slows down the algorithm too much 
					 * To fit the angle, set the 3rd parameter to TRUE */
					cur_star = Global_minimisation(z, bg, layer, FALSE);
					if (cur_star) {
						update_units(fit, &cur_star);
						if (is_star(cur_star, radius)){
							cur_star->xpos = x+cur_star->x0-radius;
							cur_star->ypos = y+cur_star->y0-radius;
							results[nbstars] = cur_star;
							results[nbstars+1] = NULL;
							nbstars++;
						}
					}
					gsl_matrix_free(z);
				}
			}
		}
	}
	if (nbstars==0) {
		free(results);
		results=NULL;
	}
	siril_log_message("Found %d stars in image, channel #%d\n", nbstars, layer);
	free(wave_image);
	free(real_image);
	clearfits(wave_fit);

	gettimeofday (&t_end, NULL);
	
	show_time(t_start, t_end);
	return results;
}

gboolean is_star(Param_GAUSS *result, int radius){
	if (isnan(result->FWHMX) || isnan(result->FWHMY)) 		return FALSE;
	if (isnan(result->x0) || isnan(result->y0)) 		return FALSE;
	if ((result->x0<0.) || (result->y0<0.))				return FALSE;
	if (result->A<=result->B)							return FALSE;
	if (result->B<0.)									return FALSE;
	if (result->FWHMX <= 0. || result->FWHMY <= 0.) 			return FALSE;
	if (result->FWHMX > 2*radius || result->FWHMY > 2*radius)	return FALSE;
	if (result->FWHMX > result->FWHMY)
		if ((result->FWHMY / result->FWHMX) < 0.8) 			return FALSE;
	if (result->FWHMY > result->FWHMX)
		if ((result->FWHMX / result->FWHMY) < 0.8) 			return FALSE;
	return TRUE;
}

/* Function to add star one by one, from the selection rectangle, the
 * minimisation is run and the star is detected and added to the list of stars.
 *
 * IF A STAR IS FOUND and not already present in com.stars, the return value is
 * the new star and index is set to the index of the new star in com.stars.
 * IF NO NEW STAR WAS FOUND, either because it was already in the list, or a
 * star failed to be detected in the selection, or any other error, the return
 * value is NULL and index is set to -1.
 */
Param_GAUSS *add_star(fits *fit, int layer, int *index){
	Param_GAUSS *result=NULL;
	int i=0;
	gboolean already_found = FALSE;
	char *msg;

	*index = -1;
	result = Get_Minimisation(&gfit, layer, &com.selection);
	if (!result) return NULL;
	/* We do not check if it's matching with the "is_star()" criteria.
	 * Indeed, in this case the user can add manually stars missed by star_finder */

	if (com.stars){
		// check if the star was already detected/peaked
		while (com.stars[i]) {
			if (fabs(result->x0 + com.selection.x - com.stars[i]->xpos) < 0.9  && 
					fabs(com.selection.y + com.selection.h - result->y0 - com.stars[i]->ypos) < 0.9)
				already_found = TRUE;
			i++;
		}
	} else {
		com.stars = malloc((MAX_STARS + 1) * sizeof(Param_GAUSS*));
		if (com.stars == NULL) return NULL;
		com.star_is_seqdata = FALSE;
	}

	if (already_found) {
		free(result);
		result = NULL;
		msg = siril_log_message("This star has already been peaked !\n");
		show_dialog(msg, "Peaker", "gtk-dialog-info");
	} else {
		if (i < MAX_STARS) {
			result->xpos = result->x0 + com.selection.x;
			result->ypos = com.selection.y + com.selection.h - result->y0;
			com.stars[i] = result;
			com.stars[i+1] = NULL;
			*index = i;
		} else {
			free(result);
			result = NULL;
		}
	}
	return result;
}

/* Remove a star from com.stars, at index index. The star is freed. */
void remove_star(int index){
	int i=0;
	if (index < 0) return;

	if (com.stars){
		/* TODO: it would be much faster to swap with the last position,
		 * but is the order of the list important? */
		while (com.stars[i]) {
			if (i == index)
				free(com.stars[i]);
			if (i >= index)
				com.stars[i] = com.stars[i+1];
			i++;
		}
		redraw(com.cvport, REMAP_NONE);
	}
}
