/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
* This file is part of Skrooge and defines classes SKGObjectModelBase.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgobjectmodelbase.h"
#include "skgtraces.h"
#include "skgdocument.h"
#include "skgnodeobject.h"
#include "skgtransactionmng.h"
#include "skgmainpanel.h"

#include <kstandarddirs.h>
#include <klocalizedstring.h>
#include <klocale.h>
#include <kicon.h>

#include <QApplication>
#include <QMimeData>
#include <qcolor.h>
#include <assert.h>

SKGObjectModelBase::SKGObjectModelBase(SKGDocument* iDocument,
                                       const QString& iTable,
                                       const QString& iWhereClause,
                                       QObject *parent,
                                       const QString& iWhereClauseChild,
                                       bool iResetOnCreation)
                : QAbstractItemModel(parent), isResetRealyNeeded(iResetOnCreation), document(iDocument),
                whereClause(iWhereClause),
                whereClauseChild(iWhereClauseChild)
{
        SKGTRACEIN(1, "SKGObjectModelBase::SKGObjectModelBase");

        listSupportedDefault 	<< "t_name"
        << "d_date"
        << "t_savestep"
        ;
        setTable(iTable);

        //Build schemas
        SKGModelTemplate def;
        def.id="default";
        def.name=tr("Default");
        def.icon="edit-undo";
        def.schema="";
        listSchema.push_back(def);

        connect((const QObject*) document, SIGNAL(tableModified(QString, int)), this, SLOT(dataModified(QString, int)));
        refresh();
}

SKGObjectModelBase::~SKGObjectModelBase()
{
        SKGTRACEIN(1, "SKGObjectModelBase::~SKGObjectModelBase");
        clear();
}

void SKGObjectModelBase::clear()
{
        SKGTRACEIN(1, "SKGObjectModelBase::clear");
        QHashIterator<int, SKGObjectBase*> i(objectsHashTable);
        while (i.hasNext()) {
                i.next();
                SKGObjectBase* val=i.value();
                if (val) {
                        delete val;
                        val=NULL;
                }
        }

        QHashIterator<int, SKGObjectBase::SKGListSKGObjectBase*> i2(parentChildRelations);
        while (i2.hasNext()) {
                i2.next();
                SKGObjectBase::SKGListSKGObjectBase* val=i2.value();
                if (val) {
                        delete val;
                        val=NULL;
                }
        }

        parentChildRelations.clear();
        childParentRelations.clear();
        objectsHashTable.clear();
        objectsHashTableRows.clear();
}

int SKGObjectModelBase::getID(const SKGObjectBase* obj, int row) const
{
        return 100*(obj->getID())+row;
}

QList<SKGObjectModelBase::SKGModelTemplate> SKGObjectModelBase::getSchemas() const
{
        return listSchema;
}

void SKGObjectModelBase::setSupportedAttributes ( const QStringList & iListAttribute )
{
        listSupported.clear();
        listVisibility.clear();
        listSize.clear();

        if (iListAttribute.count()==0) {
                listSupported=listSupportedDefault;
                int nb=listSupported.count();
                for (int i=0; i<nb; ++i) {
                        listVisibility.push_back("Y");
                        listSize.push_back(-1);
                }
        } else {
                int nb=iListAttribute.count();
                for (int i=0; i<nb; ++i) {
                        QStringList values = iListAttribute.at(i).split("|");
                        int nbValues=values.count();

                        if (nbValues>0) {
                                listSupported.push_back(values.at(0));
                                if (nbValues>1) listVisibility.push_back(values.at(1)=="Y");
                                else listVisibility.push_back(true);
                                if (nbValues>2) listSize.push_back(SKGServices::stringToInt(values.at(2)));
                                else listSize.push_back(-1);
                        }
                }
        }
        isResetRealyNeeded=true;
}

void SKGObjectModelBase::setFilter ( const QString & iWhereClause )
{
        if (iWhereClause!=whereClause) isResetRealyNeeded=true;
        whereClause=iWhereClause;
}

void SKGObjectModelBase::setTable ( const QString & iTable )
{
        if (iTable!=table && !table.isEmpty()) isResetRealyNeeded=true;
        table=iTable;
        realTable=SKGServices::getRealTable(table);
}

QString SKGObjectModelBase::getTable() const
{
        return table;
}

QString SKGObjectModelBase::getRealTable() const
{
        return realTable;
}

QString SKGObjectModelBase::getWhereClause() const
{
        return whereClause;
}

SKGDocument* SKGObjectModelBase::getDocument() const
{
        return document;
}

void SKGObjectModelBase::buidCache()
{
        SKGTRACEIN(1, "SKGObjectModelBase::buidCache");
        doctransactionTable=(getRealTable()=="doctransaction");
        nodeTable=(getRealTable()=="node");
}

void SKGObjectModelBase::refresh()
{
        if (!isResetRealyNeeded) return;
        SKGTRACEIN(1, "SKGObjectModelBase::refresh");
        {
                _SKGTRACEIN(10, "QAbstractItemModel::refresh-remove");
                clear();
                listAttibutes.clear();
                listAttibuteTypes.clear();
                /*beginRemoveRows(QModelIndex(), 0, rowCount(QModelIndex())-1);
                endRemoveRows();*/
        }

        SKGObjectBase::SKGListSKGObjectBase* listObjects=NULL;
        {
                _SKGTRACEIN(10, "QAbstractItemModel::refresh-fill");
                QStringList listAttibutesTmp;
                if (SKGServices::getAttributesList(document, table, listAttibutesTmp).isSucceeded()) {
                        isResetRealyNeeded=false;
                        if (listAttibutesTmp.count()) {
                                if (listSupported.count()==0) listSupported=listSupportedDefault;

                                //Filter attributes
                                int nb=listSupported.count();
                                for (int i=0 ; i<nb ; ++i) {
                                        QString att=listSupported[i];
                                        if (listAttibutesTmp.contains(att)) {
                                                listAttibutes.push_back(att);
                                                if (att.startsWith("t_")) listAttibuteTypes.push_back(SKGObjectModelBase::STRING);
                                                else if (att.startsWith("f_")) listAttibuteTypes.push_back(SKGObjectModelBase::DOUBLE);
                                                else if (att.startsWith("i_")) listAttibuteTypes.push_back(SKGObjectModelBase::INTEGER);
                                                else if (att.startsWith("d_")) listAttibuteTypes.push_back(SKGObjectModelBase::DATE);
                                                else listAttibuteTypes.push_back(SKGObjectModelBase::OTHER);
                                        }
                                }

                                //Remove double
                                nb=listAttibutes.count();
                                for (int i=nb-1 ; i>=0 ; --i) {
                                        QString att=listAttibutes[i];
                                        if (att.contains("_REAL")) {
                                                att.replace("_REAL", "_");
                                                int p=listAttibutes.indexOf(att);
                                                if (p==-1) {
                                                        att=att.toLower();
                                                        p=listAttibutes.indexOf(att);
                                                }
                                                if (p!=-1) {
                                                        ;
                                                        listAttibutes.removeAt(p);
                                                        listAttibuteTypes.removeAt(p);
                                                        if (p<i) --i;
                                                }
                                        }
                                }
                        }

                        //Get objects
                        listObjects=new SKGObjectBase::SKGListSKGObjectBase();
                        if (listObjects) {
                                SKGObjectBase::getObjects(document, table, whereClause, *listObjects);
                                parentChildRelations.insert(0, listObjects);

                                QList<SKGObjectBase*> listObjectToTreat;
                                QList<int> listObjectToTreatId;

                                //Initialize object to treat
                                int nb=listObjects->count();
                                SKGTRACEL(1) << nb << " objects found" << endl;
                                for (int t=0; t<nb; ++t) {
                                        SKGObjectBase* c=new SKGObjectBase(listObjects->at(t));
                                        int id=getID(c, t);

                                        listObjectToTreat.push_back(c);
                                        listObjectToTreatId.push_back(id);

                                        childParentRelations.insert(id, 0);
                                        objectsHashTableRows.insert(id, t);
                                        objectsHashTable.insert(id, c);
                                }

                                //Build tree if needed
                                if (!whereClauseChild.isEmpty()) {
                                        //For each object, we have to create hash tables to find children from parent and parent for a child
                                        for (int i=0; i<listObjectToTreat.count(); ++i) { //WARNING: the list is modified during treatement
                                                //Get parent
                                                SKGObjectBase* parent=listObjectToTreat.at(i);
                                                int parentid=listObjectToTreatId.at(i);

                                                //Get children
                                                SKGObjectBase::SKGListSKGObjectBase children;
                                                QString wc=whereClauseChild;
                                                wc.replace("#ID#", SKGServices::intToString(parent->getID()));
                                                SKGObjectBase::SKGListSKGObjectBase* listObjectsChildren=new SKGObjectBase::SKGListSKGObjectBase();
                                                if (listObjectsChildren) {
                                                        SKGObjectBase::getObjects(document, table, wc, *listObjectsChildren);

                                                        //Addition relationship in hash table
                                                        parentChildRelations.insert(parentid, listObjectsChildren);
                                                        int nbc=listObjectsChildren->count();
                                                        for (int k=0; k<nbc; ++k) {
                                                                //Addition children in list to treat
                                                                SKGObjectBase* c=new SKGObjectBase(listObjectsChildren->at(k));
                                                                int childId=getID(c, k);

                                                                listObjectToTreat.push_back(c);
                                                                listObjectToTreatId.push_back(childId);

                                                                childParentRelations.insert(childId, parentid);
                                                                objectsHashTableRows.insert(childId, k);
                                                                objectsHashTable.insert(childId, c);
                                                        }
                                                }
                                        }
                                }
                        }
                }

                //Build cache
                buidCache();

                {
                        SKGTRACEIN(1, "SKGObjectModelBase::refresh-reset");
                        QAbstractItemModel::reset();
                }
        }
}

int SKGObjectModelBase::rowCount(const QModelIndex &parent ) const
{
        if (parent.column()>0) return 0;
        _SKGTRACEIN(10, "SKGObjectModelBase::rowCount");

        int idParent=0;
        if (parent.isValid()) idParent = parent.internalId();

        SKGObjectBase::SKGListSKGObjectBase* children=parentChildRelations.value(idParent);
        //SKGTRACE << table << "-rowCount(" << idParent << " )=" << (children ? children->count() : 0) << endl;
        return (children ? children->count() : 0);

}

int SKGObjectModelBase::columnCount(const QModelIndex &parent ) const
{
        Q_UNUSED(parent);
        return listAttibutes.count();
}

QModelIndex SKGObjectModelBase::index ( int row, int column, const QModelIndex & parent) const
{
        if (!hasIndex(row, column, parent)) return QModelIndex();
        _SKGTRACEIN(10, "SKGObjectModelBase::index");

        int idParent=0;
        if (parent.isValid())  idParent = parent.internalId();

        SKGObjectBase* child=NULL;
        SKGObjectBase::SKGListSKGObjectBase* children=parentChildRelations.value(idParent);
        if (children)  child=(SKGObjectBase*) &(children->at(row));

        int idChild=getID(child, row);
        //SKGTRACE << table << "-" << idParent << "(" << row << ") ==> " << idChild << endl;
        return createIndex(row, column, idChild);
}

QModelIndex SKGObjectModelBase::parent ( const QModelIndex & index ) const
{
        if (!index.isValid() || index.column()!=0) return QModelIndex();
        _SKGTRACEIN(10, "SKGObjectModelBase::parent");

        int idChild=0;
        if (index.isValid()) idChild=index.internalId();

        int idParent=childParentRelations.value(idChild);
        int row=objectsHashTableRows.value(idParent);
        //SKGTRACE << table << "-" << idChild << "(" << row << ") <== " << idParent << endl;
        return idParent ? createIndex(row, 0, idParent) : QModelIndex();
}

int SKGObjectModelBase::getIndexAttribute (const QString& iAttributeName ) const
{
        int output=listAttibutes.indexOf(iAttributeName);
        if (output==-1) {
                SKGTRACE << "[" << iAttributeName << "] not found in [" << getRealTable() << "]" << endl;
        }
        return output;
}

QString SKGObjectModelBase::getAttribute (int iIndex ) const
{
        return listAttibutes.at(iIndex);
}

SKGObjectModelBase::AttributeType SKGObjectModelBase::getAttributeType (int iIndex ) const
{
        return listAttibuteTypes.at(iIndex);
}

QVariant SKGObjectModelBase::headerData ( int section, Qt::Orientation orientation, int role ) const
{
        _SKGTRACEIN(10, "SKGObjectModelBase::headerData");

        if (orientation==Qt::Horizontal) {
                if (role == Qt::DisplayRole) {
                        QString att;
                        if (section>=0 && section< listAttibutes.count()) att=listAttibutes[section];
                        else att=SKGServices::intToString(section);

                        return getDocument()->getDisplay(getTable()+'.'+att);
                } else if (role == Qt::UserRole ) {
                        QString att;
                        if (section>=0 && section< listAttibutes.count()) att=listAttibutes[section];
                        else att=SKGServices::intToString(section);

                        att=getDocument()->getDisplay(getTable()+'.'+att);

                        if (section>=0 && section< listVisibility.count()) {
                                att+=QString("|")+(listVisibility[section] ? "Y" : "N");
                                if (section>=0 && section< listSize.count()) att+="|"+SKGServices::intToString(listSize.at(section));
                        }
                        return att;
                }
        }
        return QVariant();
}

SKGObjectBase SKGObjectModelBase::getObject(const QModelIndex &index) const
{
        SKGObjectBase* obj = getObjectPointer(index);
        SKGObjectBase output;
        if (obj!=NULL) output=*obj;
        return output;
}

SKGObjectBase* SKGObjectModelBase::getObjectPointer(const QModelIndex &index) const
{
        _SKGTRACEIN(10, "SKGObjectModelBase::getObject");
        //SKGTRACE << "getObject:" << index.internalId() << endl;
        return objectsHashTable.value(index.internalId());
}

QVariant SKGObjectModelBase::data(const QModelIndex &index, int role) const
{
        if (!index.isValid()) return QVariant();
        _SKGTRACEIN(10, "SKGObjectModelBase::data");

        switch ( role ) {
        case Qt::DisplayRole:
        case Qt::EditRole:
        case Qt::UserRole: {
                SKGObjectBase* obj = getObjectPointer(index);
                QString att=listAttibutes[index.column()];
                QString val=obj->getAttribute(att);

                switch ( getAttributeType(index.column()) ) {
                case SKGObjectModelBase::DOUBLE: {
                        double dval=SKGServices::stringToDouble(val);
                        return dval;
                }
                case SKGObjectModelBase::INTEGER: {
                        return SKGServices::stringToInt(val);
                }
                case SKGObjectModelBase::DATE: {
                        QDate dval=SKGServices::stringToTime(val).date();
                        if (role == Qt::DisplayRole) {
                                return KGlobal::locale()->formatDate (dval, KLocale::FancyShortDate);
                        } else {
                                return dval;
                        }
                }
                default: {}
                }

                if (doctransactionTable && att=="t_savestep") return "";
                return val;
        }

        case Qt::TextColorRole: {
                //Text color
                if (getAttributeType(index.column())==SKGObjectModelBase::DOUBLE) {
                        QVariant value_displayed = SKGObjectModelBase::data(index, Qt::UserRole);
                        bool ok=false;
                        double value_double=value_displayed.toDouble(&ok);
                        if (ok && value_double<0) return qVariantFromValue(QColor(Qt::red));
                }
                break;
        }
        case Qt::TextAlignmentRole: {
                //Text alignment
                SKGObjectModelBase::AttributeType attType=getAttributeType(index.column());
                return (int)(Qt::AlignVCenter|(attType==SKGObjectModelBase::DOUBLE || attType==SKGObjectModelBase::INTEGER ? Qt::AlignRight : Qt::AlignLeft));
        }
        case Qt::DecorationRole: {
                //Decoration
                SKGObjectBase* obj = getObjectPointer(index);
                if (index.column() == 0 && nodeTable) {
                        SKGNodeObject node =*obj;
                        QStringList data=SKGServices::splitCSVLine(node.getData());
                        QIcon icon=KIcon("folder");
                        if (data.count()>2)  icon=(QIcon) KIcon(data.at(2));

                        if (node.isAutoStart()) {
                                //Modify icon
                                QStringList overlay;
                                overlay.push_back("user-online");
                                if (data.count()>2)  icon=(QIcon) KIcon(data.at(2), NULL, overlay);
                                else icon=(QIcon) KIcon("folder", NULL, overlay);
                        }
                        return qVariantFromValue(icon);
                } else if (index.column() == 0 && doctransactionTable) {
                        return qVariantFromValue((QIcon) KIcon(obj->getAttribute("t_mode")=="U" ? "edit-undo" :"edit-redo"));
                } else if (doctransactionTable && listAttibutes[index.column()]=="t_savestep") {
                        if (obj->getAttribute("t_savestep")=="Y") return qVariantFromValue((QIcon) KIcon("document-save"));
                }
                break;
        }
        case Qt::ToolTipRole: {
                //Tooltip
                if (doctransactionTable) {
                        SKGObjectBase* obj = getObjectPointer(index);
                        QStringList msg;
                        document->getMessages(obj->getID(), msg);
                        int nbMessages=msg.count();
                        if (nbMessages) {
                                QString message;
                                for (int i=0; i<nbMessages; ++i) {
                                        if (i!=0) message+='\n';
                                        message+=msg.at(i);
                                }

                                return message;
                        }
                } else if (getAttributeType(index.column())==SKGObjectModelBase::DATE) {
                        SKGObjectBase* obj = getObjectPointer(index);
                        QString att=listAttibutes[index.column()];
                        QString val=obj->getAttribute(att);

                        QDate dval=SKGServices::stringToTime(val).date();
                        QString fancyDate=KGlobal::locale()->formatDate (dval, KLocale::FancyShortDate);
                        QString shortDate=KGlobal::locale()->formatDate (dval, KLocale::ShortDate);


                        if (shortDate!=fancyDate) return shortDate;
                }
                break;
        }
        default : {}
        }

        return QVariant();
}

Qt::ItemFlags SKGObjectModelBase::flags(const QModelIndex &index) const
{
        _SKGTRACEIN(10, "SKGObjectModelBase::flags");

        Qt::ItemFlags flags=QAbstractItemModel::flags(index);

        if (nodeTable) {
         //       flags|=Qt::ItemIsEditable;

                if (index.isValid()) flags|=Qt::ItemIsDragEnabled | Qt::ItemIsDropEnabled;
                else flags|=Qt::ItemIsDropEnabled;
        }
        if (index.column() == 0) flags|=Qt::ItemIsUserCheckable;
        return flags;

}

Qt::DropActions SKGObjectModelBase::supportedDragActions() const
{
        return (nodeTable ? Qt::MoveAction : Qt::IgnoreAction);
}

Qt::DropActions SKGObjectModelBase::supportedDropActions() const
{
        return (nodeTable ? Qt::MoveAction : Qt::IgnoreAction);
}

QStringList SKGObjectModelBase::mimeTypes() const
{
        QStringList types;
        types << "application/skrooge."+getRealTable()+".ids";
        return types;
}

QMimeData* SKGObjectModelBase::mimeData(const QModelIndexList &indexes) const
{
        QMimeData *mimeData = new QMimeData();
        QByteArray encodedData;

        QDataStream stream(&encodedData, QIODevice::WriteOnly);

        QString t=getTable();
        foreach (const QModelIndex& index, indexes) {
                if (index.isValid()) {
                        SKGObjectBase obj=getObject(index);
                        t=obj.getRealTable();
                        stream << t;
                        stream << obj.getID();
                }
        }

        mimeData->setData("application/skrooge."+t+".ids", encodedData);
        return mimeData;
}

bool SKGObjectModelBase::dropMimeData(const QMimeData *data,
                                      Qt::DropAction action,
                                      int row, int column,
                                      const QModelIndex &parent)
{
        Q_UNUSED(row);
        if (action == Qt::IgnoreAction) return true;
        if (!data->hasFormat("application/skrooge.node.ids")) return false;
        if (column > 0) return false;

        QByteArray encodedData = data->data("application/skrooge.node.ids");
        QDataStream stream(&encodedData, QIODevice::ReadOnly);
        QStringList newItems;

        SKGError err;
        SKGNodeObject parentNode;
        if (parent.isValid()) parentNode=getObject(parent);
        {
                SKGBEGINTRANSACTION(*getDocument(), i18n("Move bookmark"), err);

                double min=0;
                double max=0;
                if (row>=1) {
                        QModelIndex previousIndex=SKGObjectModelBase::index (row-1, 0, parent);
                        SKGNodeObject previousObject = getObject(previousIndex);
                        min=previousObject.getOrder();
                }

                if (row>=rowCount(parent)) max=min+1;
                else {
                        QModelIndex nextIndex=SKGObjectModelBase::index (row, 0, parent);
                        SKGNodeObject nextObject = getObject(nextIndex);
                        max=nextObject.getOrder();
                }
                if (max<=min) max=min+1;

                while (!stream.atEnd() && err.isSucceeded()) {
                        int o_id;
                        QString o_table;
                        stream >> o_table;
                        stream >> o_id;

                        //Set parent
                        SKGNodeObject child(getDocument(), o_id);
                        err=child.load();
                        if (err.isSucceeded()) {
                                if (parent.isValid()) err=child.setParentNode(parentNode);
                                else err=child.removeParentNode();
                        }

                        //Set order
                        if (err.isSucceeded())  err=child.setOrder((min+max)/2.0);

                        //Save
                        if (err.isSucceeded()) err=child.save();
                }
        }

        SKGMainPanel::displayErrorMessage(err);
        return err.isSucceeded();
}

bool SKGObjectModelBase::canFetchMore ( const QModelIndex & parent ) const
{
        Q_UNUSED(parent);
        _SKGTRACEIN(10, "SKGObjectModelBase::canFetchMore");
        return false;
}

void SKGObjectModelBase::dataModified(const QString& iTableName, int iIdTransaction)
{
        if (getRealTable()==iTableName || iTableName.isEmpty()) {
                SKGTRACEIN(1, "SKGObjectModelBase::dataModified");
                SKGTRACEL(1) << "getTable=" << getRealTable() << endl;
                SKGTRACEL(1) << "Parameters=" << iTableName << " , " << iIdTransaction << endl;

                //Get modifications
                /*     int nb=0;
                     SKGObjectModificationList modifications;
                     if (iIdTransaction!=0) {
                             document->getModifications(iIdTransaction, modifications);
                             nb=modifications.count();
                     }

                     if (1==1 || iTableName=="node" || iTableName=="doctransaction" || getTable()=="v_operation_consolidated" || iIdTransaction==0 || nb>50) { //TODO: optimization*/
                //Full refresh
                isResetRealyNeeded=true;

                //Refresh model
                refresh();
                /* } else {
                         //Treat
                         emit layoutAboutToBeChanged();

                         for (int i=0; i<nb; ++i) {
                                 SKGObjectModification mod=modifications[i];
                                 if (getRealTable()==mod.table) {
                                         if (mod.type==D) {
                                                 //Delete
                                                 int pos=-1;
                                                 int nbo=parentChildRelations.value("")->count();
                                                 for (int j=0; pos==-1 && j<nbo; ++j) {
                                                         if (mod.uuid==parentChildRelations.value("")->at(j).getUniqueID()) pos=j;
                                                 }

                                                 if (pos!=-1) {
                                                         beginRemoveRows(QModelIndex(), pos, pos);
                                                         parentChildRelations.value("")->removeAt(pos);
                                                         endRemoveRows();
                                                 }
                                         } else if (mod.type==I) {
                                                 //Insert
                                                 int pos=rowCount();
                                                 beginInsertRows(QModelIndex(), pos, pos);
                                                 SKGObjectBase tmp(document, getTable(), mod.id);
                                                 parentChildRelations.value("")->push_back(tmp);
                                                 endInsertRows();

                                                 QModelIndex topLeft=index(pos, 0);
                                                 QModelIndex bottomRight=index(pos, columnCount());
                                                 emit dataChanged (topLeft, bottomRight);
                                         } else if (mod.type==U) {
                                                 //Update
                                                 int nbo=parentChildRelations.value("")->count();
                                                 for (int j=0; j<nbo; ++j) {
                                                         if (mod.uuid==parentChildRelations.value("")->at(j).getUniqueID()) {
                                                                 QModelIndex idx=index(j, 0, QModelIndex());
                                                                 SKGObjectBase* obj = (SKGObjectBase*) idx.internalPointer();
                                                                 if (obj) {
                                                                         obj->load();

                                                                         QModelIndex bottomRight=index(j, columnCount());
                                                                         emit dataChanged (idx, bottomRight);
                                                                 }
                                                         }
                                                 }
                                         }
                                 }
                         }
                         emit  layoutChanged();
                 }*/
        }
}


#include "skgobjectmodelbase.moc"

