/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is part of Skrooge and implements classes SKGDocument.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgdocument.h"
#include "skgtraces.h"
#include "skgerror.h"
#include "skgservices.h"
#include "skgobjectbase.h"
#include "skgnamedobject.h"

#include <QSqlDatabase>
#include <QSqlQuery>
#include <QSqlError>
#include <QHash>
#include <QRegExp>
#include <QUuid>
#include <QVariant>
#include <QFile>
#include <QDir>

SKGError SKGDocument::lastCallbackError;
int SKGDocument::databaseUniqueIdentifier=0;

SKGDocument::SKGDocument(DatabaseMode iMode)
                :QObject(), lastSavedTransaction(0), progressFunction(NULL), progressData(NULL),
                currentFileName(""), currentDatabase(NULL),  inundoRedoTransaction(0), currentTransaction(0),
                mode(iMode),inProgress(false)
{
        SKGTRACEIN(10, "SKGDocument::SKGDocument");

        //Initialisation of undoable tables
        SKGListNotUndoable.push_back("T.doctransaction");
        SKGListNotUndoable.push_back("T.doctransactionitem");
        SKGListNotUndoable.push_back("T.doctransactionmsg");

        //Database unique identifier
        ++databaseUniqueIdentifier;
        databaseIdentifier="SKGDATABASE_"+SKGServices::intToString(databaseUniqueIdentifier);
}


SKGDocument::~SKGDocument()
{
        SKGTRACEIN(10, "SKGDocument::~SKGDocument");
        close();
        progressFunction=NULL;
        progressData=NULL;
}

SKGDocument::DatabaseMode SKGDocument::getDatabaseMode()
{
        return mode;
}

SKGError SKGDocument::setProgressCallback(int (*iProgressFunction)(int, void*), void* iData)
{
        SKGError err;
        progressFunction=(void*) iProgressFunction;
        progressData=iData;

        return err;
}

SKGError SKGDocument::stepForward(int iPosition)
{
        SKGError err;

        //Increase the step for the last transaction
        if (getDepthTransaction()) {
                posStepForTransaction.pop_back();
                posStepForTransaction.push_back(iPosition);
        }

        //Check if a callback function exists
        if (progressFunction) {
                //YES ==> compute
                double min=0;
                double max=100;

                bool emitevent=true;
                SKGIntListIterator nbIt=nbStepForTransaction.begin();
                SKGIntListIterator posIt=posStepForTransaction.begin();
                for (; emitevent && nbIt!=nbStepForTransaction.end(); ++nbIt) {
                        int p=*posIt;
                        int n=*nbIt;
                        if (p<0 || p>n) p=n;

                        if (n!=0) {
                                double pmin=min;
                                double pmax=max;
                                min=pmin+(pmax-pmin)*(static_cast<double>(p) / static_cast<double>(n));
                                max=pmin+(pmax-pmin)*(static_cast<double>(p+1) / static_cast<double>(n));
                                if (max>100) max=100;
                        } else emitevent=false;

                        ++posIt;
                }

                int posPourcent=(int) min;

                //Call the call back
                if (emitevent) {
                        inProgress=true;
                        if ((*(int (*)(int, void*)) progressFunction)(posPourcent, progressData)!=0) {
                                err.setReturnCode(ERR_ABORT);
                                err.setMessage(tr("The current operation has been interrupted"));

                                //Remove all untransactionnal message
                                unTransactionnalMessages.clear();
                        }
                        inProgress=false;
                }
        }
        return err;
}

SKGError SKGDocument::beginTransaction(const QString & iName, int iNbStep, const QDateTime& iDate)
{
        SKGError err;
        SKGTRACEINRC(5, "SKGDocument::beginTransaction", err);
        SKGTRACEL(10) << "Input parameter [name]=[" << iName << ']' << endl;
        if (nbStepForTransaction.size() == 0) {
                //Open SQLtransaction
                err = SKGServices::executeSqliteOrder(this, "BEGIN;");
                if (err.isSucceeded()) {
                        //Create undo redo transaction
                        err = SKGServices::executeSqliteOrder(this, "insert into doctransaction "
                                                              "(d_date, t_name, i_parent) values "
                                                              "('"+SKGServices::timeToString(iDate)+
                                                              "','"+SKGServices::stringToSqlString(iName)+
                                                              "', "+SKGServices::intToString(getTransactionToTreat(SKGDocument::UNDO))+");");
                        currentTransaction=getTransactionToTreat(SKGDocument::UNDO);
                        if (err.isSucceeded() && iName != "#INTERNAL#") {
                                //Create undo redo temporary triggers
                                err = createUndoRedoTemporaryTriggers();
                        }
                }
        } else {
                //A transaction already exists
                //Check if the child transaction is a opened in the progress callback
                if (inProgress) {
                        err.setReturnCode(ERR_FAIL);
                        err.setMessage(tr("A transaction can not be started during execution of an other one"));
                }
        }
        if (err.isSucceeded()) {
                nbStepForTransaction.push_back(iNbStep);
                posStepForTransaction.push_back(iNbStep);

                if (iNbStep) err=stepForward(0);
        }
        return err;
}

SKGError SKGDocument::checkExistingTransaction() const
{
        SKGError err;
        SKGTRACEINRC(10, "SKGDocument::checkExistingTransaction", err);
        if (getDepthTransaction() <= 0) {
                err.setReturnCode(ERR_ABORT);
                err.setMessage(tr("A transaction must be opened to do this action"));
        }
        return err;
}

SKGError SKGDocument::endTransaction(bool succeedded)
{
        SKGError err;
        SKGTRACEINRC(5, "SKGDocument::endTransaction", err);
        if (nbStepForTransaction.size()==0) {
                //Set error message
                err.setReturnCode(ERR_ABORT);
                err.setMessage(tr("SKGDocument::endTransaction failed because too many transactions ended"));
        } else {
                stepForward(100);
                nbStepForTransaction.pop_back();
                posStepForTransaction.pop_back();

                if (nbStepForTransaction.size()==0) {
                        if (succeedded) {
                                //Optimization of the current transaction
                                if (err.isSucceeded()) {
                                        err = SKGServices::executeSqliteOrder(this, "DELETE FROM doctransaction WHERE id="+SKGServices::intToString(getCurrentTransaction())+" AND ((SELECT count(1) FROM doctransactionitem a where a.rd_doctransaction_id=doctransaction.id)=0);");
                                }

                                //Optimisation 2: remove duplicate orders
                                if (err.isSucceeded()) {
                                        err = SKGServices::executeSqliteOrder(this,
                                                                              "DELETE FROM doctransactionitem WHERE id IN "
                                                                              "(SELECT a.id FROM doctransactionitem a, doctransactionitem b "
                                                                              "WHERE a.rd_doctransaction_id="+SKGServices::intToString(getCurrentTransaction())+
                                                                              " AND b.rd_doctransaction_id=a.rd_doctransaction_id AND a.i_object_id=b.i_object_id "
                                                                              "AND a.t_object_table=b.t_object_table AND "
                                                                              "b.t_action=a.t_action AND b.t_sqlorder=a.t_sqlorder AND a.id>b.id );");
                                }
                                //Remove oldest transaction
                                if (err.isSucceeded()) {
                                        QString maxdepthstring = getParameter("SKG_UNDO_MAX_DEPTH");
                                        if (maxdepthstring.length() == 0) maxdepthstring = "-1";
                                        int maxdepth = SKGServices::stringToInt(maxdepthstring);
                                        if (maxdepth >= 0) {
                                                err = SKGServices::executeSqliteOrder(this, "delete from doctransaction where id in (select id from doctransaction limit max(0,((select count(1) from doctransaction)-(" + maxdepthstring + "))))");
                                        }
                                }

                                //Remove SKGDocument::REDO transactions if we are not in a undo / redo transaction
                                if (!inundoRedoTransaction) {
                                        int i = 0;
                                        while ((i = getTransactionToTreat(SKGDocument::REDO)) && err.isSucceeded()) {
                                                err = SKGServices::executeSqliteOrder(this, "delete from doctransaction where id=" + SKGServices::intToString(i));
                                        }
                                }

                                //Commit the transaction
                                if (err.isSucceeded()) {
                                        err = SKGServices::executeSqliteOrder(this, "COMMIT;");
                                }

                        }

                        if (!succeedded || err.isFailed()) {
                                //Rollback the transaction
                                SKGError err2 = SKGServices::executeSqliteOrder(this, "ROLLBACK;");
                                if (err2.isSucceeded()) {
                                        //delete the transaction
                                        err2 = SKGServices::executeSqliteOrder(this, "delete from doctransaction where id=" + SKGServices::intToString(getCurrentTransaction()));
                                }

                                if (err2.isFailed()) err.addError(err2.getReturnCode(), err2.getMessage());
                        } else {
                                //For better performance, events are submitted only for the first recursive undo
                                if (inundoRedoTransaction<=1) {
                                        //Emit modification events
                                        QStringList listModifiedTables;
                                        err=SKGServices::getDistinctValues(this, "doctransactionitem",
                                                                           "t_object_table",
                                                                           "rd_doctransaction_id="+SKGServices::intToString(getCurrentTransaction()),
                                                                           listModifiedTables);
                                        if (err.isSucceeded()) {
                                                QStringList tablesRefreshed;
                                                foreach (const QString& table, listModifiedTables) {
                                                        emit tableModified(table, getCurrentTransaction());
                                                        tablesRefreshed.push_back(table);
                                                }
                                                emit tableModified("doctransaction", getCurrentTransaction());
                                                emit tableModified("doctransactionitem", getCurrentTransaction());

                                                //WARNING: list is modified during treatement
                                                for (int i=0; i<listModifiedTables.count(); ++i) {
                                                        QString table=listModifiedTables[i];
                                                        if (!tablesRefreshed.contains(table)) {
                                                                emit tableModified(table, 0);
                                                                tablesRefreshed.push_back(table);
                                                        }

                                                        QStringList toAdd=getImpactedTable(table);
                                                        int nbToAdd=toAdd.count();
                                                        for (int j=0; j<nbToAdd; ++j) {
                                                                if (!listModifiedTables.contains(toAdd.at(j)))
                                                                        listModifiedTables.push_back(toAdd.at(j));
                                                        }
                                                }

                                                emit transactionSuccessfullyEnded(getCurrentTransaction());

                                                //Remove temporary transaction if needed
                                                err = SKGServices::executeSqliteOrder(this, "delete from doctransaction where id="+SKGServices::intToString(getCurrentTransaction())+" and t_name='#INTERNAL#';");
                                        }
                                }
                        }

                        currentTransaction=0;

                        //clean cache
                        cache.clear();
                }
        }
        return err;
}

SKGError SKGDocument::sendMessage(const QString& iMessage, bool iPopup)
{
        SKGError err;
        SKGTRACEINRC(10, "SKGDocument::sendMessage", err);
        //Associate message with transaction
        if (checkExistingTransaction().isSucceeded()) {
                SKGObjectBase msg(this, "doctransactionmsg");
                err = msg.setAttribute("rd_doctransaction_id", SKGServices::intToString(getCurrentTransaction()));
                if (err.isSucceeded()) err = msg.setAttribute("t_message", iMessage);
                if (err.isSucceeded()) err = msg.setAttribute("t_popup", iPopup ? "Y" : "N");
                if (err.isSucceeded()) err = msg.save();
        }

        //Addition message in global variable in case of
        if (iPopup) unTransactionnalMessages.push_back(iMessage);
        return err;
}

SKGError SKGDocument::getMessages(int iIdTransaction, QStringList& oMessages, bool iAll)
{
        SKGError err;
        SKGTRACEINRC(10, "SKGDocument::getMessages", err);
        oMessages=unTransactionnalMessages;
        unTransactionnalMessages.clear();

        SKGStringListList listTmp;
        err=SKGServices::executeSelectSqliteOrder(this,
                        QString("SELECT t_message, t_popup FROM doctransactionmsg WHERE ")+
                        (iAll ? "t_popup IS NOT NULL" : "t_popup='Y'")+
                        " AND rd_doctransaction_id=" +
                        SKGServices::intToString(iIdTransaction) +
                        " ORDER BY id ASC",
                        listTmp);
        int nb=listTmp.count();
        for (int i=1; err.isSucceeded() && i<nb ;++i) {
                QString msg=listTmp.at(i).at(0);
                if (!oMessages.contains(msg)) oMessages.push_back(msg);
        }
        return err;
}

SKGError SKGDocument::getModifications(int iIdTransaction, SKGObjectModificationList& oModifications) const
{
        SKGError err;
        SKGTRACEINRC(10, "SKGDocument::getModifications", err);
        oModifications.clear();

        SKGStringListList listTmp;
        err=SKGServices::executeSelectSqliteOrder(this,
                        "SELECT i_object_id,t_object_table,t_action FROM doctransactionitem WHERE rd_doctransaction_id=" +
                        SKGServices::intToString(iIdTransaction) +
                        " ORDER BY id ASC",
                        listTmp);
        int nb=listTmp.count();
        for (int i=1; err.isSucceeded() && i<nb ;++i) {
                SKGObjectModification mod;
                mod.id=SKGServices::stringToInt(listTmp.at(i).at(0));
                mod.table=listTmp.at(i).at(1);
                QString type=listTmp.at(i).at(2);
                mod.type=(type=="D" ? I : (type=="I" ? D : U )); //Normal because in database we have to sql order to go back.
                mod.uuid=listTmp.at(i).at(0)+ '-' + mod.table;;

                oModifications.push_back(mod);
        }
        return err;
}

QStringList SKGDocument::getImpactedTable(const QString& /*iTable*/) const
{
        SKGTRACEIN(10, "SKGDocument::getImpactedTable");
        QStringList output;
        return output;
}

SKGError SKGDocument::groupTransactions(int iFrom, int iTo)
{
        SKGError err;
        SKGTRACEINRC(5, "SKGDocument::groupTransactions", err);

        ++inundoRedoTransaction; //It's a kind of undo redo

        //Check if a transaction is still opened
        err = checkExistingTransaction();
        if (err.isSucceeded()) {
                //A transaction is still opened
                err.setReturnCode(ERR_ABORT);
                err.setMessage(tr("Creation of a group of transactions is forbidden inside a transaction"));

        } else {
                int iidMaster=qMax(iFrom, iTo);
                QString smin=SKGServices::intToString(qMin(iFrom, iTo));
                QString smax=SKGServices::intToString(iidMaster);

                //Get transaction
                SKGStringListList transactions;
                err=SKGServices::executeSelectSqliteOrder(this,
                                QString("SELECT id, t_name, t_mode, i_parent FROM doctransaction WHERE id BETWEEN ") +
                                smin +" AND " +
                                smax+ " ORDER BY id ASC",
                                transactions);

                //Check and get main parameter for the group
                int nb=transactions.count();
                QString transactionMode;
                QString communParent;
                QString name;
                for (int i=1; err.isSucceeded() && i<nb; ++i) { //We forget header
                        QStringList transaction=transactions.at(i);
                        QString mode=transaction.at(2);
                        if (!name.isEmpty()) name+=',';
                        name+=transaction.at(1);

                        if (!transactionMode.isEmpty() && mode!=transactionMode)  err=SKGError(ERR_INVALIDARG, "Undo and Redo transactions can not be grouped");
                        else  transactionMode=mode;

                        if (i==1)  communParent=transaction.at(3);
                }


                if (err.isSucceeded()) {
                        //Group
                        err = SKGDocument::beginTransaction("#INTERNAL#");
                        if (err.isSucceeded()) {
                                //Group items
                                if (err.isSucceeded())
                                        err = SKGServices::executeSqliteOrder(this,
                                                                              QString("UPDATE doctransactionitem set rd_doctransaction_id=") +
                                                                              smax +
                                                                              " where rd_doctransaction_id BETWEEN " +
                                                                              smin+ " AND " + smax);
                                if (err.isSucceeded())
                                        err = SKGServices::executeSqliteOrder(this,
                                                                              QString("UPDATE doctransaction set i_parent=") +
                                                                              communParent +
                                                                              ", t_name='"+SKGServices::stringToSqlString(name)+
                                                                              "' where id=" +smax);

                                if (err.isSucceeded()) err = SKGServices::executeSqliteOrder(this,
                                                                     QString("DELETE FROM doctransaction WHERE id BETWEEN ") +
                                                                     smin+ " AND " + SKGServices::intToString(qMax(iFrom, iTo)-1));
                        }

                        if (err.isSucceeded()) err=endTransaction(true);
                        else  endTransaction(false);
                }
        }

        --inundoRedoTransaction;
        return err;
}

SKGError SKGDocument::undoRedoTransaction(const UndoRedoMode& iMode)
{
        SKGError err;
        SKGTRACEINRC(5, "SKGDocument::undoRedoTransaction", err);
        //Check if a transaction is still opened
        err = checkExistingTransaction();
        if (err.isSucceeded()) {
                //A transaction is still opened
                err.setReturnCode(ERR_ABORT);
                err.setMessage(tr("Undo / Redo is forbidden inside a transaction"));

        } else {
                if (iMode==SKGDocument::UNDOLASTSAVE) {
                        //Create group
                        SKGStringListList transactions;
                        err=SKGServices::executeSelectSqliteOrder(this,
                                        "SELECT id, t_savestep FROM doctransaction WHERE t_mode='U' ORDER BY id DESC",
                                        transactions);
                        int nb=transactions.count();
                        int min=0;
                        int max=0;
                        for (int i=1; err.isSucceeded() && i<nb; ++i) {
                                QStringList transaction=transactions.at(i);
                                if (i==1) {
                                        max=SKGServices::stringToInt(transaction.at(0));
                                }
                                if (i!=1 && transaction.at(1)=="Y") {
                                        break;
                                }
                                min=SKGServices::stringToInt(transaction.at(0));
                        }
                        if (min==0) min=max;
                        if (err.isSucceeded() && min!=max && min!=0) err=groupTransactions(min, max);
                } else {
                        err=SKGError(); //To ignore error generated by checkExistingTransaction.
                }

                //Get ID of the transaction to undo
                if (err.isSucceeded()) {
                        QString name;
                        bool saveStep=false;
                        QDateTime date;
                        int id = getTransactionToTreat(iMode, &name, &saveStep, &date);
                        if (id == 0) {
                                //No transaction found ==> generate an error
                                err = SKGError(ERR_INVALIDARG, "No transaction found. Undo / Redo impossible.");
                        } else {

                                //Undo transaction
                                SKGTRACEL(5) << "Undoing transaction [" << id << "]- [" << name << "]..." << endl;
                                SKGStringListList listSqlOrder;
                                err=SKGServices::executeSelectSqliteOrder(this,
                                                "SELECT t_sqlorder FROM doctransactionitem WHERE rd_doctransaction_id=" +
                                                SKGServices::intToString(id) +
                                                " ORDER BY id DESC",
                                                listSqlOrder);
                                if (err.isSucceeded()) {
                                        int nb=listSqlOrder.count();
                                        err = SKGDocument::beginTransaction(name, nb+3, date);
                                        if (err.isSucceeded()) {
                                                ++inundoRedoTransaction; //Because we will be in a undo/redo transaction
                                                //Normal the first element is ignored because it is the header
                                                for (int i=1; err.isSucceeded() && i<nb ;++i) {
                                                        err=SKGServices::executeSqliteOrder(this, listSqlOrder.at(i).at(0));

                                                        if (err.isSucceeded()) err=stepForward(i); //Undo / redo are not cancelable
                                                }

                                                if (err.isSucceeded()) {
                                                        //Set the NEW transaction in redo mode
                                                        int lastredo = getTransactionToTreat((iMode == SKGDocument::UNDO || iMode == SKGDocument::UNDOLASTSAVE  ? SKGDocument::REDO : SKGDocument::UNDO));
                                                        int newredo = getTransactionToTreat(iMode);
                                                        if (err.isSucceeded())
                                                                err = SKGServices::executeSqliteOrder(this,
                                                                                                      QString("UPDATE doctransaction set t_mode=") +
                                                                                                      (iMode == SKGDocument::UNDO || iMode == SKGDocument::UNDOLASTSAVE ? "'R'" : "'U'") +
                                                                                                      ", i_parent=" +
                                                                                                      SKGServices::intToString(lastredo) +
                                                                                                      " where id=" + SKGServices::intToString(newredo));
                                                        if (err.isSucceeded()) err=stepForward(nb);

                                                        //Move messages from previous transaction to new one
                                                        if (err.isSucceeded())
                                                                err = SKGServices::executeSqliteOrder(this,
                                                                                                      "UPDATE doctransactionmsg set rd_doctransaction_id="+
                                                                                                      SKGServices::intToString(getCurrentTransaction()) +
                                                                                                      " where rd_doctransaction_id=" +
                                                                                                      SKGServices::intToString(id));
                                                        if (err.isSucceeded()) err=stepForward(nb+1);

                                                        //delete treated transaction
                                                        if (err.isSucceeded())
                                                                err = SKGServices::executeSqliteOrder(this,
                                                                                                      "DELETE from doctransaction where id="
                                                                                                      + SKGServices::intToString(id));
                                                        if (err.isSucceeded()) err=stepForward(nb+2);

                                                        //Check that new transaction has exactly the same number of item
                                                        /* if (err.isSucceeded()) {
                                                                 SKGStringListList listSqlOrder;
                                                                 err=SKGServices::executeSelectSqliteOrder(this,
                                                                                 "SELECT count(1) FROM doctransactionitem WHERE rd_doctransaction_id=" +
                                                                                 SKGServices::intToString(getCurrentTransaction()),
                                                                                 listSqlOrder);
                                                                 if (err.isSucceeded() && SKGServices::stringToInt(listSqlOrder.at(1).at(0))!=nb-1) {
                                                                         err=SKGError(ERR_ABORT, tr("Invalid number of item after undo/redo. Expected (%1) != Result (%2)").arg(nb-1).arg(listSqlOrder.at(1).at(0)));
                                                                 }
                                                         }*/

                                                        if (err.isSucceeded()) err=stepForward(nb+3);
                                                }

                                                if (err.isSucceeded()) err=endTransaction(true);
                                                else  endTransaction(false);
                                                --inundoRedoTransaction; //We left the undo / redo transaction
                                        }
                                }
                        }
                }
        }

        return err;
}

int SKGDocument::getDepthTransaction() const
{
        return nbStepForTransaction.size();
}

int SKGDocument::getNbTransaction(const UndoRedoMode& iMode) const
{
        SKGTRACEIN(10, "SKGDocument::getNbTransaction");
        int output = 0;
        if (getDatabase()) {
                QString sqlorder = "select count(1) from doctransaction where t_mode='";
                sqlorder+=(iMode == SKGDocument::UNDO || iMode == SKGDocument::UNDOLASTSAVE ? "U" : "R");
                sqlorder+='\'';
                QSqlQuery query=getDatabase()->exec(sqlorder);
                if (query.next()) {
                        output=query.value(0).toInt();
                }
        }
        return output;
}

int SKGDocument::getTransactionToTreat(const UndoRedoMode& iMode, QString* oName, bool* oSaveStep, QDateTime* oDate) const
{
        SKGTRACEIN(10, "SKGDocument::getTransactionToTreat");
        //initialisation
        int output = 0;
        if (oName) *oName = "";
        if (getDatabase()) {
                QString sqlorder = "select A.id , A.t_name, A.t_savestep, A.d_date from doctransaction A where "
                                   "((select count(1) from doctransaction B where B.i_parent=A.id)=0) "
                                   "and A.t_mode='";
                sqlorder+=(iMode == SKGDocument::UNDO || iMode == SKGDocument::UNDOLASTSAVE ? "U" : "R");
                sqlorder+='\'';
                QSqlQuery query=getDatabase()->exec(sqlorder);
                if (query.next()) {
                        output=query.value(0).toInt();
                        if (oName != NULL) *oName = query.value(1).toString();
                        if (oSaveStep != NULL) *oSaveStep = (query.value(2).toString()=="Y");
                        if (oDate != NULL) *oDate = SKGServices::stringToTime(query.value(3).toString());
                }
        }
        return output;
}

int SKGDocument::getCurrentTransaction() const
{
        SKGTRACEIN(10, "SKGDocument::getCurrentTransaction");
        return currentTransaction;
}

SKGError SKGDocument::changePassword(const QString & newPassword)
{
        SKGError err;
        SKGTRACEINRC(10, "SKGDocument::changePassword", err);
        err = setParameter("SKG_PASSWORD", newPassword);
        if (err.isSucceeded()) {
                if (newPassword.isEmpty()) {
                        err=sendMessage(tr("The document password has been removed."));
                } else {
                        err=sendMessage(tr("The document password is now [%1].").arg(newPassword));
                }
        }
        return err;
}

SKGError SKGDocument::initialize()
{
        SKGError err;
        SKGTRACEINRC(5, "SKGDocument::initialize", err);
        err=load("", "");
        return err;
}

SKGError SKGDocument::load(const QString & name, const QString & password)
{
        //close previous document
        SKGError err;
        SKGTRACEINRC(5, "SKGDocument::load", err);
        SKGTRACEL(10) << "Input parameter [name]=[" << name << ']' << endl;

        lastSavedTransaction=-1; //To avoid double event emission
        err=close();
        if (err.isSucceeded()) {
                //Check if file exists
                QFile inputFile(name);
                if (name.length() && !inputFile.exists()) {
                        //Set error message
                        err.setReturnCode(ERR_INVALIDARG);
                        err.setMessage(tr("File [%1] not found").arg(name));
                } else {
                        // open your memory database
                        if (getDatabaseMode()==CopiedInMemory) {
                                //Create memory database
                                currentDatabase=new QSqlDatabase(QSqlDatabase::addDatabase("QSQLITE", databaseIdentifier));
                                currentDatabase->setDatabaseName(":memory:");
                                if (!currentDatabase->open()) {
                                        //Set error message
                                        QSqlError sqlErr=currentDatabase->lastError();
                                        err = SKGError(SQLLITEERROR + sqlErr.number(), sqlErr.text());

                                        //close database
                                        close();
                                }
                        }

                        if (err.isSucceeded()) {
                                if (inputFile.exists()) {
                                        //File exist
                                        if (getDatabaseMode()==DirectAccess) {
                                                //Direct load of existing file
                                                currentDatabase=new QSqlDatabase(QSqlDatabase::addDatabase("QSQLITE", databaseIdentifier));
                                                currentDatabase->setDatabaseName(name);
                                                if (!currentDatabase->open()) {
                                                        //Set error message
                                                        QSqlError sqlErr=currentDatabase->lastError();
                                                        err = SKGError(SQLLITEERROR + sqlErr.number(), sqlErr.text());
                                                }
                                        } else if (getDatabaseMode()==CopiedInFile) {
                                                //Copy in temporary file
                                                temporaryFile=QDir::tempPath()+"/skrooge_"+QUuid::createUuid ().toString()+".skg";
                                                QFile(temporaryFile).remove(); //Must must remove it to be able to copy
                                                err=SKGServices::cryptFile(name, temporaryFile, password, false);
                                                if (err.isSucceeded()) {
                                                        currentDatabase=new QSqlDatabase(QSqlDatabase::addDatabase("QSQLITE", databaseIdentifier));
                                                        currentDatabase->setDatabaseName(temporaryFile);
                                                        if (!currentDatabase->open()) {
                                                                //Set error message
                                                                QSqlError sqlErr=currentDatabase->lastError();
                                                                err = SKGError(SQLLITEERROR + sqlErr.number(), sqlErr.text());
                                                        }
                                                }
                                        } else {
                                                //Copy in memory
                                                {//This scope is needed to avoid a warning at remove
                                                        QSqlDatabase dbfile = QSqlDatabase::addDatabase("QSQLITE", "TEMPORARY");
                                                        dbfile.setDatabaseName(name);
                                                        if (!dbfile.open()) {
                                                                //Set error message
                                                                QSqlError sqlErr=dbfile.lastError();
                                                                err = SKGError(SQLLITEERROR + sqlErr.number(), sqlErr.text());
                                                        } else {
                                                                //Copy db from file to memory
                                                                err = SKGServices::copySqliteDatabase(name, &dbfile, getDatabase(), true);

                                                                //Close
                                                                dbfile.close();
                                                        }
                                                }
                                                //Remove database
                                                QSqlDatabase::removeDatabase("TEMPORARY");
                                        }

                                        currentFileName=name;
                                } else {
                                        if (getDatabaseMode()==DirectAccess || getDatabaseMode()==CopiedInFile) {
                                                //Copy in temporary file
                                                temporaryFile=QDir::tempPath()+"/skrooge_"+QUuid::createUuid ().toString()+".skg";
                                                currentDatabase=new QSqlDatabase(QSqlDatabase::addDatabase("QSQLITE", databaseIdentifier));
                                                currentDatabase->setDatabaseName(temporaryFile);
                                                if (!currentDatabase->open()) {
                                                        //Set error message
                                                        QSqlError sqlErr=currentDatabase->lastError();
                                                        err = SKGError(SQLLITEERROR + sqlErr.number(), sqlErr.text());
                                                }
                                        }

                                        //Create parameter and undo redo table
                                        /**
                                         * This constant is used to initialized the data model (table creation)
                                         */
                                        const QString InitialDataModel
                                        =
                                                // ==================================================================
                                                //Table parameters
                                                "DROP TABLE IF EXISTS parameters;;"
                                                "CREATE TABLE parameters "
                                                "(id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
                                                "t_uuid_parent TEXT NOT NULL DEFAULT '',"
                                                "t_name TEXT NOT NULL,"
                                                "t_value TEXT NOT NULL DEFAULT '',"
                                                "d_lastmodifdate DATE NOT NULL DEFAULT CURRENT_TIMESTAMP);;"

                                                // ==================================================================
                                                //Table node
                                                "DROP TABLE IF EXISTS node;;"
                                                "CREATE TABLE node ("
                                                "id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
                                                "t_name TEXT NOT NULL DEFAULT '' CHECK (t_name NOT LIKE '%"OBJECTSEPARATOR"%'),"
                                                "t_fullname TEXT,"
                                                "f_sortorder FLOAT,"
                                                "t_autostart VARCHAR(1) DEFAULT 'N' CHECK (t_autostart IN ('Y', 'N')),"
                                                "t_data TEXT,"
                                                "r_node_id INT CONSTRAINT fk_id REFERENCES node(id) ON DELETE CASCADE);;"

                                                // ==================================================================
                                                //Table doctransaction
                                                "DROP TABLE IF EXISTS doctransaction;;"
                                                "CREATE TABLE doctransaction ("
                                                "id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,"
                                                "t_name TEXT NOT NULL,"
                                                "t_mode VARCHAR(1) DEFAULT 'U' CHECK (t_mode IN ('U', 'R')),"
                                                "d_date DATE NOT NULL,"
                                                "t_savestep VARCHAR(1) DEFAULT 'N' CHECK (t_savestep IN ('Y', 'N')),"
                                                "i_parent INTEGER);;"

                                                // ==================================================================
                                                //Table doctransactionitem
                                                "DROP TABLE IF EXISTS doctransactionitem;;"
                                                "CREATE TABLE doctransactionitem ("
                                                "id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT, "
                                                "rd_doctransaction_id INTEGER NOT NULL,"
                                                "i_object_id INTEGER NOT NULL,"
                                                "t_object_table TEXT NOT NULL,"
                                                "t_action VARCHAR(1) DEFAULT 'I' CHECK (t_action IN ('I', 'U', 'D')),"
                                                "t_sqlorder TEXT NOT NULL DEFAULT '');;"

                                                "DROP TABLE IF EXISTS doctransactionmsg;;"
                                                "CREATE TABLE doctransactionmsg ("
                                                "id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT, "
                                                "rd_doctransaction_id INTEGER NOT NULL,"
                                                "t_message TEXT NOT NULL DEFAULT '',"
                                                "t_popup VARCHAR(1) DEFAULT 'Y' CHECK (t_popup IN ('Y', 'N'))"
                                                ");;";
                                        ;

                                        if (err.isSucceeded()) err = SKGServices::executeSqliteOrder(this, InitialDataModel);
                                        if (err.isSucceeded()) err = SKGDocument::refreshViewsIndexesAndTriggers();
                                }
                        }
                }
        }

        //migrate
        if (err.isSucceeded()) {
                bool mig=false;
                err = migrate(mig);
                if (err.isSucceeded() && mig && !name.isEmpty()) err=sendMessage(tr("The document has be migrated"));
        }

        //Optimization
        if (err.isSucceeded()) {
                lastSavedTransaction=getTransactionToTreat(SKGDocument::UNDO);
                err = SKGServices::executeSqliteOrder(this, "ANALYZE");
        }

        if (err.isFailed() && !name.isEmpty()) {
                err.addError(ERR_FAIL, tr("SKGDocument::load(%1,***) failed").arg(name));
                initialize();
        } else {
                //Send event
                emit tableModified("", 0);
        }

        return err;
}

bool SKGDocument::isFileModified() const
{
        //Get last executed transaction
        int last=getTransactionToTreat(SKGDocument::UNDO);
        //  if (nbStepForTransaction.size()) --last;
        return (lastSavedTransaction!=last);
}

QString SKGDocument::getCurrentFileName() const
{
        return currentFileName;
}

SKGError SKGDocument::save()
{
        SKGError err;
        SKGTRACEINRC(5, "SKGDocument::save", err);
        if (currentFileName.length() == 0) {
                err = SKGError(ERR_INVALIDARG, tr("SKGDocument::save() not authorized. Please use SKGDocument::saveAs()"));
        } else {
                //save
                err = saveAs(currentFileName, true);
        }
        return err;
}

SKGError SKGDocument::saveAs(const QString & name, bool overwrite)
{
        SKGError err;
        SKGTRACEINRC(5, "SKGDocument::saveAs", err);
        SKGTRACEL(10) << "Input parameter [name]=[" << name << ']' << endl;

        //Check if a transaction is still opened
        err = checkExistingTransaction();
        if (err.isSucceeded()) {
                //A transaction is still opened
                err.setReturnCode(ERR_ABORT);
                err.setMessage(tr("save is forbidden if a transaction is still opened"));

        } else {
                //No transaction opened ==> it's ok
                //We mark the last transaction as a save point
                err = SKGServices::executeSqliteOrder(this,"update doctransaction set t_savestep='Y' where id in (select A.id from doctransaction A where "
                                                      "((select count(1) from doctransaction B where B.i_parent=A.id)=0) "
                                                      "and A.t_mode='U')");

                //Optimization
                if (err.isSucceeded()) {
                        err = SKGServices::executeSqliteOrder(this, "VACUUM;");
                        if (err.isSucceeded()) {
                                //Check if file already exist
                                if (!overwrite && QFile(name).exists()) {
                                        //Set error message
                                        err.setReturnCode(ERR_INVALIDARG);
                                        err.setMessage(tr("File [%1] already exist").arg(name));
                                } else {
                                        //Open database
                                        if (getDatabaseMode()==CopiedInMemory) {
                                                //Remove previous file
                                                QFile(name).remove();

                                                {//This scope is needed to avoid a warning at remove
                                                        //Copy memory in file
                                                        QSqlDatabase dbfile = QSqlDatabase::addDatabase("QSQLITE", "TEMPORARY");
                                                        dbfile.setDatabaseName(name);
                                                        if (!dbfile.open()) {
                                                                //Set error message
                                                                QSqlError sqlErr=dbfile.lastError();
                                                                err = SKGError(SQLLITEERROR + sqlErr.number(), sqlErr.text());
                                                        } else {
                                                                //Copy db from memory to file
                                                                err = SKGServices::copySqliteDatabase(name, &dbfile, getDatabase(), false);

                                                                //Close db
                                                                dbfile.close();
                                                        }
                                                }

                                                //Remove database
                                                QSqlDatabase::removeDatabase("TEMPORARY");
                                        } else if (getDatabaseMode()==CopiedInFile) {
                                                //Rename previous file
                                                QFile(name).rename(name+".old");

                                                //Copy file to file
                                                QString password=getParameter("SKG_PASSWORD");
                                                err=SKGServices::cryptFile(temporaryFile, name, password, true);

                                                //Remove previous file
                                                if (err.isSucceeded()) QFile(name+".old").remove();
                                                else QFile(name+".old").rename(name);
                                        } else if (getDatabaseMode()==DirectAccess && !temporaryFile.isEmpty()) {
                                                //Rename previous file
                                                QFile(name).rename(name+".old");

                                                //Copy file to file
                                                if (!QFile::copy(temporaryFile, name)) {
                                                        err = SKGError(ERR_FAIL, tr("copy of %1 to %2 failed").arg(temporaryFile).arg(name));
                                                } else {
                                                        err=close();
                                                        if (err.isSucceeded()) err=load(name);
                                                }

                                                //Remove previous file
                                                if (err.isSucceeded()) QFile(name+".old").remove();
                                                else QFile(name+".old").rename(name);
                                        }

                                        if (err.isFailed()) err.addError(ERR_FAIL, tr("SKGDocument::saveAs(%1) failed").arg(name));
                                        else {
                                                //The document is not modified
                                                currentFileName = name;
                                                lastSavedTransaction=getTransactionToTreat(SKGDocument::UNDO);
                                        }
                                }
                        }
                }
        }
        return err;
}

SKGError SKGDocument::close()
{
        SKGError err;
        SKGTRACEINRC(5, "SKGDocument::close", err);
        if (getDatabase() != NULL) {
                getDatabase()->close();
                delete currentDatabase; //Delete handler to avoid warning and memory leak
                QSqlDatabase::removeDatabase(databaseIdentifier);
        }

        if (!temporaryFile.isEmpty()) {
                QFile(temporaryFile).remove();
                temporaryFile="";
        }

        //Emit events ?
        bool emitEvent=(lastSavedTransaction!=-1);

        //Init fields
        currentDatabase = NULL;;
        currentFileName = "";
        lastSavedTransaction=0;
        nbStepForTransaction=SKGIntList();
        posStepForTransaction=SKGIntList();

        //Send event
        if (emitEvent) {
                emit tableModified("", 0);
                emit transactionSuccessfullyEnded(0);
        }

        return err;
}

SKGError SKGDocument::refreshViewsIndexesAndTriggers() const
{
        SKGError err;
        SKGTRACEINRC(5, "SKGDocument::refreshViewsIndexesAndTriggers", err);
        /**
        * This constant is used to initialized the data model (trigger creation)
        */
        const QString InitialDataModelTrigger
        =
                DELETECASCADEPARAMETER("parameters")+
                DELETECASCADEPARAMETER("node")+

                //Compute fullname
                "DROP TRIGGER IF EXISTS cpt_node_fullname1;;"
                "CREATE TRIGGER cpt_node_fullname1 " //This trigger must be the first
                "AFTER UPDATE OF t_fullname ON node BEGIN "
                "UPDATE node SET t_name=t_name WHERE r_node_id=new.id;"
                "END;;"

                "DROP TRIGGER IF EXISTS cpt_node_fullname2;;"
                "CREATE TRIGGER cpt_node_fullname2 "
                "AFTER INSERT ON node BEGIN "
                "UPDATE node SET t_fullname="
                "CASE WHEN r_node_id IS NULL THEN new.t_name ELSE (SELECT c.t_fullname from node c where c.id=new.r_node_id)||'"OBJECTSEPARATOR"'||new.t_name END "
                "WHERE id=new.id;"
                "END;;"

                "DROP TRIGGER IF EXISTS cpt_node_fullname3;;"
                "CREATE TRIGGER cpt_node_fullname3 "
                "AFTER UPDATE OF t_name ON node BEGIN "
                "UPDATE node SET t_fullname="
                "CASE WHEN r_node_id IS NULL OR r_node_id='' THEN new.t_name ELSE (SELECT c.t_fullname from node c where c.id=new.r_node_id)||'"OBJECTSEPARATOR"'||new.t_name END "
                "WHERE id=new.id;"
                "END;;"

                //-- Foreign Key Preventing insert + Skrooge modification
                /*   "
                CREATE TRIGGER fki_node_parent_id_node_id "
                "BEFORE INSERT ON [node] "
                "FOR EACH ROW BEGIN "
                "  SELECT RAISE(ABORT, 'insert on table node violates foreign key constraint fki_node_parent_id_node_id') "
                "  WHERE NEW.r_node_id IS NOT NULL AND NEW.r_node_id!='' AND (SELECT id FROM node WHERE id = NEW.r_node_id) IS NULL; "
                "
                END;; "

                //-- Foreign key preventing update + Skrooge modification
                "
                CREATE TRIGGER fku_node_parent_id_node_id "
                "BEFORE UPDATE ON [node] "
                "FOR EACH ROW BEGIN "
                "    SELECT RAISE(ABORT, 'update on table node violates foreign key constraint fku_node_parent_id_node_id') "
                "      WHERE NEW.r_node_id IS NOT NULL AND NEW.r_node_id!='' AND (SELECT id FROM node WHERE id = NEW.r_node_id) IS NULL; "
                "
                END;; "
                */ //TODO


                //-- Cascading delete - WARNING rewriten for skrooge to support recursive mode
                "DROP TRIGGER IF EXISTS fkdc_node_parent_id_node_id;;"
                "CREATE TRIGGER fkdc_node_parent_id_node_id "
                "BEFORE DELETE ON node "
                "FOR EACH ROW BEGIN "
                "    DELETE FROM node WHERE node.t_fullname LIKE OLD.t_fullname||'"OBJECTSEPARATOR"%'; "
                "END;; "

                /*     "
                CREATE TRIGGER fkdc_node_parent_id_node_id "
                "BEFORE DELETE ON node "
                "FOR EACH ROW BEGIN "
                "    DELETE FROM node WHERE node.r_node_id=OLD.id; "
                "
                END;; "*/
                ;

        /**
        * This constant is used to initialized the data model (index creation)
        */
        const QString InitialDataModelIndex
        =
                "DROP INDEX IF EXISTS uidx_parameters_uuid_parent_name;;"
                "CREATE UNIQUE INDEX uidx_parameters_uuid_parent_name ON parameters (t_uuid_parent, t_name);;"

                "DROP INDEX IF EXISTS uidx_node_parent_id_name;;"
                "CREATE UNIQUE INDEX uidx_node_parent_id_name ON node(t_name,r_node_id);;"

                "DROP INDEX IF EXISTS uidx_node_fullname;;"
                "CREATE UNIQUE INDEX uidx_node_fullname ON node(t_fullname);;"

                "DROP INDEX IF EXISTS idx_doctransaction_parent;;"
                "CREATE INDEX idx_doctransaction_parent ON doctransaction (i_parent);;"

                "DROP INDEX IF EXISTS idx_doctransactionitem_i_object_id;;"
                "CREATE INDEX idx_doctransactionitem_i_object_id ON doctransactionitem (i_object_id);;"

                "DROP INDEX IF EXISTS idx_doctransactionitem_t_object_table;;"
                "CREATE INDEX idx_doctransactionitem_t_object_table ON doctransactionitem (t_object_table);;"

                "DROP INDEX IF EXISTS idx_doctransactionitem_t_action;;"
                "CREATE INDEX idx_doctransactionitem_t_action ON doctransactionitem (t_action);;"

                "DROP INDEX IF EXISTS idx_doctransactionitem_rd_doctransaction_id;;"
                "CREATE INDEX idx_doctransactionitem_rd_doctransaction_id ON doctransactionitem (rd_doctransaction_id);;"

                "DROP INDEX IF EXISTS idx_doctransactionitem_optimization;;"
                "CREATE INDEX idx_doctransactionitem_optimization ON doctransactionitem (rd_doctransaction_id, i_object_id, t_object_table, t_action, id);;"
                ;

        /**
        * This constant is used to initialized the data model (view creation)
        */
        const QString InitialDataModelView
        =
                "DROP VIEW IF EXISTS v_node;;"
                "CREATE VIEW  v_node AS SELECT * from node;;"

                "DROP VIEW IF EXISTS v_node_displayname;;"
                "CREATE VIEW v_node_displayname AS SELECT *, t_fullname AS t_displayname from node;;"
                ;

        err = SKGServices::executeSqliteOrder(this, InitialDataModelIndex);
        if (err.isSucceeded()) err = SKGServices::executeSqliteOrder(this, InitialDataModelView);
        if (err.isSucceeded()) err = SKGServices::executeSqliteOrder(this, InitialDataModelTrigger);
        if (err.isSucceeded()) {
                //Refresh dynamic triggers
                QRegExp rx_rd("rd_([^_]+)_([^_]+).*");
                QRegExp rx_rc("rc_([^_]+)_([^_]+).*");
                QStringList tables;
                err=SKGServices::getDistinctValues(this, "sqlite_master", "name", "type='table'", tables);
                int nb=tables.count();
                for (int i=0; err.isSucceeded() && i<nb; ++i) {
                        QString table=tables[i];
                        SKGStringListList attributes;
                        err=SKGServices::executeSelectSqliteOrder(this, "PRAGMA table_info("+table+");", attributes);
                        int nb2=attributes.count();
                        for (int j=1; err.isSucceeded() && j<nb2; ++j) { //Header is ignored
                                QString att=attributes.at(j).at(1);
                                if (rx_rd.indexIn(att)!=-1) {
                                        //Get parameters
                                        QString tab2=rx_rd.cap(1);
                                        QString att2 =rx_rd.cap(2);
                                        err = SKGServices::executeSqliteOrder(this, FOREIGNCONSTRAINTCASCADE(tab2,att2,table,att));
                                } else
                                        if (rx_rc.indexIn(att)!=-1) {
                                                //Get parameters
                                                QString tab2=rx_rc.cap(1);
                                                QString att2 =rx_rc.cap(2);
                                                err = SKGServices::executeSqliteOrder(this, FOREIGNCONSTRAINT(tab2,att2,table,att));
                                        }
                        }
                }
        }

        return err;
}

SKGError SKGDocument::migrate(bool& oMigrationDone)
{
        SKGError err;
        SKGTRACEINRC(5, "SKGDocument::migrate", err);
        oMigrationDone=false;

        err = beginTransaction("#INTERNAL#");
        if (getParameter("SKG_UNDO_MAX_DEPTH").isEmpty()) {
                err = setParameter("SKG_UNDO_MAX_DEPTH", SKGServices::intToString(SKG_UNDO_MAX_DEPTH));
        }

        if (getParameter("SKG_UNIQUE_ID").isEmpty()) {
                err = setParameter("SKG_UNIQUE_ID", QUuid::createUuid ().toString());
        }

        QString version = getParameter("SKG_DB_VERSION");
        QString initialversion = version;

        if (err.isSucceeded() && version.isEmpty()) {
                //First creation
                SKGTRACEL(10) << "Migration from 0 to 0.6" << endl;

                //Set new version
                version = "0.6";
                if (err.isSucceeded()) err = setParameter("SKG_DB_VERSION", version);

                //Set sqlite creation version
                SKGStringListList listTmp;
                if (err.isSucceeded()) err=SKGServices::executeSelectSqliteOrder(this, "select sqlite_version()", listTmp);
                if (err.isSucceeded() && listTmp.count()==2) err = setParameter("SKG_SQLITE_CREATION_VERSION", listTmp.at(1).at(0));
                oMigrationDone=true;
        }

        if (err.isSucceeded() && version == "0.1") {
                //Migration from version 0.1 to 0.2
                SKGTRACEL(10) << "Migration from 0.1 to 0.2" <<endl;

                // ==================================================================
                //Table doctransactionmsg
                QString sql=
                        "DROP TABLE IF EXISTS doctransactionmsg;;"
                        "CREATE TABLE doctransactionmsg ("
                        "id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT, "
                        "rd_doctransaction_id INTEGER NOT NULL,"
                        "t_message TEXT NOT NULL DEFAULT '');;";
                err = SKGServices::executeSqliteOrder(this, sql);

                //Set new version
                version="0.2";
                if (err.isSucceeded()) err=SKGDocument::setParameter ("SKG_DB_VERSION", version);
                oMigrationDone=true;
        }

        if (err.isSucceeded() && version == "0.2") {
                //Migration from version 0.2 to 0.3
                SKGTRACEL(10) << "Migration from 0.2 to 0.3" <<endl;

                err = SKGServices::executeSqliteOrder(this, "UPDATE node set f_sortorder=id");

                //Set new version
                version="0.3";
                if (err.isSucceeded()) err=SKGDocument::setParameter ("SKG_DB_VERSION", version);
                oMigrationDone=true;
        }

        if (err.isSucceeded() && version == "0.3") {
                //Migration from version 0.3 to 0.4
                SKGTRACEL(10) << "Migration from 0.3 to 0.4" <<endl;

                err = SKGServices::executeSqliteOrder(this, "ALTER TABLE node ADD COLUMN t_autostart VARCHAR(1) DEFAULT 'N' CHECK (t_autostart IN ('Y', 'N'))");
                if (err.isSucceeded()) err=SKGServices::executeSqliteOrder (this, "UPDATE node set t_autostart='Y' where t_name='"+tr("autostart")+'\'');

                //Set new version
                version="0.4";
                if (err.isSucceeded()) err=SKGDocument::setParameter ("SKG_DB_VERSION", version);
                oMigrationDone=true;
        }
        if (err.isSucceeded() && version == "0.4") {
                //Migration from version 0.4 to 0.5
                SKGTRACEL(10) << "Migration from 0.4 to 0.5" <<endl;

                err = SKGServices::executeSqliteOrder(this, "ALTER TABLE doctransactionmsg ADD COLUMN t_popup VARCHAR(1) DEFAULT 'Y' CHECK (t_popup IN ('Y', 'N'))");

                //Set new version
                version="0.5";
                if (err.isSucceeded()) err=SKGDocument::setParameter ("SKG_DB_VERSION", version);
                oMigrationDone=true;
        }
        if (err.isSucceeded() && version == "0.5") {
                //Migration from version 0.5 to 0.6
                SKGTRACEL(10) << "Migration from 0.5 to 0.6" <<endl;

                err=SKGServices::executeSqliteOrder (this, "UPDATE node set t_autostart='N' where t_autostart NOT IN ('Y', 'N')");

                //Set new version
                version="0.6";
                if (err.isSucceeded()) err=SKGDocument::setParameter ("SKG_DB_VERSION", version);
                oMigrationDone=true;
        }


        //Refresh views
        if (err.isSucceeded())  err = refreshViewsIndexesAndTriggers();

        //Set sqlite last version
        SKGStringListList listTmp;
        if (err.isSucceeded()) err=SKGServices::executeSelectSqliteOrder(this, "select sqlite_version()", listTmp);
        if (err.isSucceeded() && listTmp.count()==2) err = setParameter("SKG_SQLITE_LAST_VERSION", listTmp.at(1).at(0));


        if (err.isFailed()) err.addError(ERR_FAIL, tr("SKGDocument::migrate() failed. Migration database from version %1 to version %2 failed").arg(initialversion).arg(version));

        //close temporary transaction
        if (err.isSucceeded()) err=endTransaction(true);
        else  endTransaction(false);

        return err;
}

SKGError SKGDocument::createUndoRedoTemporaryTriggers() const
{
        SKGError err;
        SKGTRACEINRC(10, "SKGDocument::createUndoRedoTemporaryTriggers", err);

        //Create triggers
        QString transactionId=SKGServices::intToString(getTransactionToTreat(SKGDocument::UNDO));
        QStringList tables;
        err=SKGServices::getTablesList(this, tables);
        int nbTables=tables.count();
        for (int i=0; err.isSucceeded() && i<nbTables; ++i) {
                //Get table name
                QString table = tables[i];

                //Do we have to treat this table
                if (!SKGListNotUndoable.contains("T."+table)) {
                        //YES
                        //Get attributes name
                        QStringList attributes;
                        err=SKGServices::getAttributesList(this, table, attributes);

                        //Build sqlorder for update and insert
                        QString sqlorderForUpdate2;
                        QString sqlorderForInsert1;
                        QString sqlorderForInsert2;
                        int nbAttributes=attributes.count();
                        for (int j=0; err.isSucceeded() && j<nbAttributes; ++j) {
                                //Get attribute
                                QString att = attributes[j];

                                //Do we have to treat this attribute
                                if (!SKGListNotUndoable.contains("A."+table+'.'+att)) {

                                        //Build for update
                                        if (sqlorderForUpdate2.length() > 0)
                                                sqlorderForUpdate2 += ',';
                                        sqlorderForUpdate2 += att + "='||quote(old." + att + ")||'";

                                        //Build for insert part 1
                                        if (sqlorderForInsert1.length() > 0)
                                                sqlorderForInsert1 += ',';
                                        sqlorderForInsert1 += att;

                                        //Build for insert part 2
                                        if (sqlorderForInsert2.length() > 0)
                                                sqlorderForInsert2 += ',';
                                        sqlorderForInsert2 += "'||quote(old." + att + ")||'";
                                }
                        }

                        //Create specific triggers for the current transaction
                        err = SKGServices::executeSqliteOrder(this,
                                                              //DROP DELETE trigger
                                                              "DROP TRIGGER IF EXISTS UR_" + table + "_IN;;"

                                                              //Create DELETE trigger
                                                              "CREATE TEMP TRIGGER UR_" + table + "_IN "
                                                              "AFTER  INSERT ON " + table + " BEGIN "
                                                              "INSERT INTO doctransactionitem (rd_doctransaction_id, t_sqlorder,i_object_id,t_object_table,t_action) VALUES(" +transactionId+
                                                              ",'DELETE FROM " + table +
                                                              " WHERE id='||new.id,new.id,'" + table + "','D');END;;"

                                                              //DROP UPDATE trigger
                                                              "DROP TRIGGER IF EXISTS UR_" + table + "_UP;;"

                                                              //Create UPDATE trigger
                                                              "CREATE TEMP TRIGGER UR_" + table + "_UP "
                                                              "AFTER UPDATE ON " + table + " BEGIN "
                                                              "INSERT INTO doctransactionitem  (rd_doctransaction_id, t_sqlorder,i_object_id,t_object_table,t_action) VALUES(" +transactionId+
                                                              ",'UPDATE " + table +
                                                              " SET " + sqlorderForUpdate2 +
                                                              " WHERE id='||new.id,new.id,'" + table + "','U');END;;"

                                                              //DROP INSERT trigger
                                                              "DROP TRIGGER IF EXISTS UR_" + table + "_DE;;"

                                                              //Create INSERT trigger
                                                              "CREATE TEMP TRIGGER UR_" + table + "_DE "
                                                              "AFTER DELETE ON " + table +
                                                              " BEGIN "
                                                              "INSERT INTO doctransactionitem  (rd_doctransaction_id, t_sqlorder,i_object_id,t_object_table,t_action) VALUES(" +transactionId+
                                                              ",'INSERT INTO " + table +
                                                              '(' + sqlorderForInsert1 + ") VALUES(" + sqlorderForInsert2 + ")',old.id,'" + table + "','I'); END;;");
                }
        }
        return err;
}

QString SKGDocument::getParameter(const QString& iName) const
{
        SKGTRACEIN(10, "SKGDocument::getParameter");
        SKGTRACEL(10) << "Input parameter [iName]=[" << iName << ']' << endl;
        QString output = "";

        //Get parameter
        SKGObjectBase param;
        SKGError err = SKGObjectBase::getObject(this, "parameters", "t_name='"+SKGServices::stringToSqlString(iName)+"' AND t_uuid_parent='document'", param);
        if (err.isSucceeded()) {
                output = param.getAttribute("t_value");
        }
        return output;
}

SKGError SKGDocument::setParameter(const QString& iName, const QString& iValue) const
{
        SKGError err;
        SKGTRACEINRC(10, "SKGDocument::setParameter", err);
        SKGTRACEL(10) << "Input parameter [iName]=[" << iName << ']' << endl;
        SKGTRACEL(10) << "Input parameter [iValue]=[" << iValue << ']' << endl;

        SKGNamedObject param(this, "parameters");
        if (err.isSucceeded()) err = param.setName(iName);
        if (err.isSucceeded()) err = param.setAttribute("t_value", iValue);
        if (err.isSucceeded()) err = param.setAttribute("t_uuid_parent", "document");
        if (err.isSucceeded()) err = param.save();

        if (err.isFailed()) err.addError(ERR_FAIL, tr("SKGDocument::setParameter(%1,%2) failed").arg(iName).arg(iValue));

        return err;
}

SKGError SKGDocument::dump(int iMode) const
{
        SKGError err;
        if (getDatabase()) {
                //dump parameters
                SKGTRACE << "=== START DUMP ===" << endl;
                if (iMode&DUMPSQLITE) {
                        SKGTRACE << "=== DUMPSQLITE ===" << endl;
                        err.addError(SKGServices::dumpSelectSqliteOrder(this, "SELECT * FROM sqlite_master order by type"));

                        SKGTRACE << "=== DUMPSQLITE (TEMPORARY) ===" << endl;
                        err.addError(SKGServices::dumpSelectSqliteOrder(this, "SELECT * FROM sqlite_temp_master order by type"));
                }

                if (iMode&DUMPPARAMETERS) {
                        SKGTRACE << "=== DUMPPARAMETERS ===" << endl;
                        err.addError(SKGServices::dumpSelectSqliteOrder(this, "SELECT * FROM parameters order by id"));
                }

                if (iMode&DUMPNODES) {
                        SKGTRACE << "=== DUMPNODES ===" << endl;
                        err.addError(SKGServices::dumpSelectSqliteOrder(this, "SELECT * FROM node order by id"));
                }

                if (iMode&DUMPTRANSACTIONS) {
                        //dump transaction
                        SKGTRACE << "=== DUMPTRANSACTIONS ===" << endl;
                        err.addError(SKGServices::dumpSelectSqliteOrder(this, "SELECT * FROM doctransaction order by id"));

                        //dump transaction
                        SKGTRACE << "=== DUMPTRANSACTIONS (ITEMS) ===" << endl;
                        err.addError(SKGServices::dumpSelectSqliteOrder(this, "SELECT * FROM doctransactionitem order by rd_doctransaction_id, id"));
                }
                SKGTRACE << "=== END DUMP ===" << endl;
        }

        if (err.isFailed()) err.addError(ERR_FAIL, tr("SKGDocument::dump() failed"));
        return err;
}

QSqlDatabase* SKGDocument::getDatabase() const
{
        return currentDatabase;
}

SKGError SKGDocument::getConsolidatedView(const QString& iTable,
                const QString& iAsColumn,
                const QString& iAsRow,
                const QString& iAttribute,
                const QString& iOpAtt,
                const QString& iWhereClause,
                SKGStringListList& oTable) const
{
        SKGError err;
        SKGTRACEINRC(10, "SKGDocument::getConsolidatedView", err);
        SKGTRACEL(10) << "Input parameter [iTable]=[" << iTable << ']' << endl;
        SKGTRACEL(10) << "Input parameter [iAsColumn]=[" << iAsColumn << ']' << endl;
        SKGTRACEL(10) << "Input parameter [iAsRow]=[" << iAsRow << ']' << endl;
        SKGTRACEL(10) << "Input parameter [iAttribute]=[" << iAttribute << ']' << endl;
        SKGTRACEL(10) << "Input parameter [iOpAtt]=[" << iOpAtt << ']' << endl;
        SKGTRACEL(10) << "Input parameter [iWhereClause]=[" << iWhereClause << ']' << endl;

        //Mode
        int mode=0;
        if (!iAsColumn.isEmpty()) mode+=1;
        if (!iAsRow.isEmpty()) mode+=2;

        oTable.clear();
        oTable.push_back(QStringList());

        QStringList* titles=(QStringList*) &(oTable.at(0));

        if (mode==3) {
                titles->push_back(iAsRow+'/'+iAsColumn);
        } else {
                if (mode==1) {
                        titles->push_back(iAsColumn);

                        QStringList sums;
                        sums.push_back(tr("Sum"));
                        oTable.push_back(sums);
                } else {
                        if (mode==2) {
                                titles->push_back(iAsRow);
                                titles->push_back(tr("Sum"));
                        }
                }
        }

        //Create sqlorder
        QString att=iAsColumn;
        if (!att.isEmpty() && !iAsRow.isEmpty()) att+=',';
        att+=iAsRow;

        QString sort=iAsRow;
        if (!sort.isEmpty() && !iAsColumn.isEmpty()) sort+=',';
        sort+=iAsColumn;

        if (!att.isEmpty()) {
                QString sql="SELECT "+att+','+iOpAtt+'('+iAttribute+") FROM "+iTable;
                if (!iWhereClause.isEmpty()) sql+=" WHERE "+iWhereClause;
                if (!iOpAtt.isEmpty()) sql+=" GROUP BY "+att;
                sql+=" ORDER BY "+sort;

                QHash<QString, int> cols;
                QHash<QString, int> rows;

                SKGTRACEL(10) << "sqlorder=[" << sql << ']' << endl;
                SKGStringListList listTmp;
                err=SKGServices::executeSelectSqliteOrder(this, sql, listTmp);
                int nb=listTmp.count();
                for (int i=1; err.isSucceeded() && i<nb; ++i) { //Title is ignored
                        QStringList line=listTmp.at(i);
                        int rowindex=-1;
                        int colindex=-1;
                        if (mode>=2) {
                                QString rowname=line.at(mode==3 ? 1 : 0);

                                if (!rows.contains(rowname)) {
                                        QStringList r;
                                        r.push_back(rowname);
                                        int nbx=oTable[0].count();
                                        for (int j=1; j<nbx; ++j)
                                                r.push_back("0");

                                        oTable.push_back(r);

                                        rowindex=oTable.count()-1;
                                        rows.insert(rowname,rowindex);
                                } else rowindex=rows[rowname];
                        } else rowindex=1;

                        if (mode==1 || mode==3) {
                                QString colname=line.at(0);

                                if (!cols.contains(colname)) {
                                        //Search better position of this column
                                        colindex=-1;
                                        {
                                                QHashIterator<QString, int> cols_i(cols);
                                                while (cols_i.hasNext()) {
                                                        cols_i.next();
                                                        if (colname>cols_i.key() && cols_i.value()>colindex) colindex=cols_i.value();
                                                }
                                        }
                                        if (colindex==-1) colindex=1;
                                        else ++colindex;

                                        int nbx=oTable.count();
                                        for (int j=0; j<nbx; ++j) {
                                                if (j==0) oTable[j].insert(colindex,colname);
                                                else oTable[j].insert(colindex, "0");
                                        }

                                        {
                                                QHash<QString, int> tmp;
                                                QHashIterator<QString, int> cols_i(cols);
                                                while (cols_i.hasNext()) {
                                                        cols_i.next();
                                                        tmp.insert(cols_i.key(), cols_i.value()+(cols_i.value()>=colindex ? 1 : 0));
                                                }

                                                cols=tmp;
                                        }
                                        cols.insert(colname,colindex);

                                } else colindex=cols[colname];
                        } else colindex=1;

                        QString sum=line.at(mode==3 ? 2 : 1);

                        oTable[rowindex][colindex]=sum;
                }
        }

        return err;
}

QString SKGDocument::getDisplay(const QString& iString) const
{
        QString output=iString.toLower();

        if (output.endsWith("t_name")) output=tr("Name");
        else if (output.endsWith("d_date")) output=tr("Date");
        else if (output.endsWith("t_savestep")) output=tr("Save");
        return output;
}

void SKGDocument::addValueInCache(const QString& iKey, const QString& iValue)
{
        cache[iKey]=iValue;
}

QString SKGDocument::getCachedValue(const QString& iKey) const
{
        return cache[iKey];
}
#include "skgdocument.moc"

