/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
* This file is part of Skrooge and implements classes SKGTraces.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
*/
#include "skgtraces.h"
#include "skgerror.h"

#include <sys/time.h>

//===================================================================
int SKGTraces::SKGLevelTrace = (getenv("SKGTRACE") ? atoi(getenv("SKGTRACE")) : 0);
bool SKGTraces::SKGPerfo = (getenv("SKGTRACEPERFO") ? true : false);
QString SKGTraces::SKGIndentTrace = "##";
SKGPerfoMap     SKGTraces::SKGPerfoMethode;
SKGQStringStack     SKGTraces::SKGPerfoPathMethode;
//===================================================================
SKGTraces::SKGTraces(int iLevel, const QString& iName, SKGError* iRC)
{
        IFSKGTRACEL(iLevel) {
                Name = iName;
                Output = true;

                RC = iRC;
                SKGIndentTrace += "  ";
                SKGTRACE << ">" << Name << endl;
        }
        else Output = false;

        if (SKGPerfo) {
                Profiling=true;
                Name = iName;

                //Push the methode in the stack
                SKGTraces::SKGPerfoPathMethode.push(Name);

                //Pour les mesures de perfos
                struct timeval tv;
                struct timezone tz;
                gettimeofday(&tv, &tz);
                elapse = (static_cast<double>(1000.0 * tv.tv_sec)) + (static_cast<double>(tv.tv_usec / 1000.0));

                //Searching the key in the map
                it = SKGTraces::SKGPerfoMethode.find(Name);
                if (it == SKGTraces::SKGPerfoMethode.end()) {
                        //Not found ==> initialisation
                        SKGPerfoInfo init;
                        init.NbCall = 0;
                        init.Time = 0;
                        init.TimePropre = 0;
                        init.TimeMin = 99999999;
                        init.TimeMax = -1;

                        //Add the line
                        SKGTraces::SKGPerfoMethode[Name] = init;

                        //find again
                        it = SKGTraces::SKGPerfoMethode.find(Name);
                }
        } else {
                Profiling=false;
                elapse = -1;
        }
}

SKGTraces::~SKGTraces()
{
        //Get delta time
        if (elapse >= 0) {
                struct timeval tv;
                struct timezone tz;
                gettimeofday(&tv, &tz);
                elapse = (static_cast<double>(1000.0 * tv.tv_sec)) + (static_cast<double>(tv.tv_usec / 1000.0))- elapse;
        }

        if (Output) {
                SKGTRACESUITE << SKGTraces::SKGIndentTrace << "<" << Name;
                if (RC) {
                        SKGTRACESUITE << " RC=" << RC->getFullMessage();
                }
                if (Profiling) {
                        SKGTRACESUITE << " TIME=" << elapse << " ms";
                }
                SKGTRACESUITE << endl;
                SKGIndentTrace.resize(SKGIndentTrace.length() - 2);

                RC = NULL;
        }

        if (Profiling) {
                //Update values
                ++(it.value().NbCall);
                it.value().Time += elapse;
                it.value().TimePropre += elapse;
                if (elapse > it.value().TimeMax) it.value().TimeMax = elapse;
                if (elapse < it.value().TimeMin) it.value().TimeMin = elapse;

                if (!SKGTraces::SKGPerfoPathMethode.empty() && SKGTraces::SKGPerfoPathMethode.top() == Name) {

                        //Remove current methode from stack
                        SKGTraces::SKGPerfoPathMethode.pop();

                        //Get previous methode name
                        if (!SKGTraces::SKGPerfoPathMethode.empty()) {
                                QString previousMethode = SKGTraces::SKGPerfoPathMethode.top();

                                //Searching the key in the map
                                it = SKGTraces::SKGPerfoMethode.find(previousMethode);
                                if (it != SKGTraces::SKGPerfoMethode.end()) {
                                        it.value().TimePropre -= elapse;
                                }
                        }
                }
        }
}

void SKGTraces::exit(int rc)
{
        if (SKGPerfo) {
                SKGTRACESEPARATOR;
                SKGTRACE << "methode , nb call , millisecondes , moyenne , min , max , propre , moyenne propre" << endl << flush;

                while (!SKGTraces::SKGPerfoMethode.empty()) {

                        //Recheche du temps propre maximal
                        double maxtime = 0;
                        SKGPerfoMapIterator max;
                        SKGPerfoMapIterator it2;
                        for (it2 = SKGTraces::SKGPerfoMethode.begin() ; it2 != SKGTraces::SKGPerfoMethode.end(); ++it2) {
                                if (it2.value().TimePropre > maxtime) {
                                        maxtime = it2.value().TimePropre;
                                        max = it2;
                                }
                        }

                        //dump max
                        SKGTRACE << max.key()
                        << " , " << max.value().NbCall
                        << " , " << max.value().Time
                        << " , " << (max.value().Time) / (static_cast<double>(max.value().NbCall))
                        << " , " << max.value().TimeMin
                        << " , " << max.value().TimeMax
                        << " , " << max.value().TimePropre
                        << " , " << (max.value().TimePropre) / (static_cast<double>(max.value().NbCall))
                        << endl << flush;

                        //Remove it
                        SKGTraces::SKGPerfoMethode.erase(max);
                }
        }

        //Clear
        SKGTraces::SKGPerfoMethode.clear();

        if (rc == 0) {
                SKGTRACESEPARATOR;
                SKGTRACE << "SUCCEEDED" << endl << flush;
                SKGTRACESEPARATOR;
        } else {
                SKGTRACESEPARATOR;
                SKGTRACE << "FAILED" << endl << flush;
                SKGTRACESEPARATOR;
        }

        ::exit(rc);
}
