/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
* This file is part of Skrooge and defines classes SKGObjectModel.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgobjectmodel.h"
#include "skgdocumentbank.h"
#include "skgaccountobject.h"
#include "skgcategoryobject.h"
#include "skgoperationobject.h"
#include "skgrefundtrackerobject.h"
#include "skgrecurrentoperationobject.h"
#include "skgunitobject.h"
#include "skgunitvalueobject.h"
#include "skgtransactionmng.h"
#include "skgtraces.h"
#include "skgmainpanel.h"

#include <kicon.h>
#include <kstandarddirs.h>
#include <klocalizedstring.h>
#include <klocale.h>
#include <kconfiggroup.h>

#include <QApplication>
#include <QDir>
#include <QMimeData>
#include <QFont>
#include <qcolor.h>

SKGObjectModel::SKGObjectModel ( SKGDocumentBank* iDocument,
                                 const QString& iTable,
                                 const QString& iWhereClause,
                                 QObject *parent,
                                 const QString& iParentAttribute,
                                 bool iResetOnCreation )
                : SKGObjectModelBase ( iDocument, iTable, iWhereClause, parent, iParentAttribute, false )
{
        SKGTRACEIN ( 1, "SKGObjectModel::SKGObjectModel" );
        listSupportedDefault.clear();
        listSupportedDefault
        << "d_DUEDATE"
        << "d_DATE"
        << "d_date"
        << "t_PERIODNLS"
        << "i_auto_write_days"
        << "i_warn_days"
        << "i_NBRECURRENT"
        << "t_bookmarked"
        << "t_BANK"
        << "t_close"
        << "t_name"
        << "t_symbol"
        << "t_operationname"
        << "t_country"
        << "t_type"
        << "t_internet_code"
        << "t_ACCOUNT"
        << "t_bank_number"
        << "t_BANK_NUMBER"
        << "t_agency_number"
        << "t_number"
        << "i_number"
        << "t_agency_address"
        << "t_mode"
        << "t_payee"
        << "t_comment"
        << "t_REALCOMMENT"
        << "t_CATEGORY"
        << "t_REALCATEGORY"
        << "t_status"
        << "f_quantity"
//        << "f_QUANTITY"
//        << "f_REALQUANTITY"
//        << "t_UNIT"
        //  << "f_CURRENTAMOUNT_EXPENSE"
        //  << "f_CURRENTAMOUNT_INCOME"
        << "f_CURRENTAMOUNT"
        << "f_REALCURRENTAMOUNT"
        //  << "f_REALCURRENTAMOUNT_EXPENSE"
        //  << "f_REALCURRENTAMOUNT_INCOME"
        << "f_value"
        << "f_CHECKED"
        << "f_COMING_SOON"
        << "i_NBOPERATIONS"
        << "f_SUMCURRENTAMOUNT"
        << "i_SUMNBOPERATIONS"
        ;

        //Build schemas
        if ( getRealTable() =="operation" ) {
                SKGModelTemplate minimun;
                minimun.id="minimun";
                minimun.name=tr ( "Minimun" );
                minimun.icon="";
                minimun.schema="d_date;t_ACCOUNT;t_payee;t_status;f_REALCURRENTAMOUNT;f_CURRENTAMOUNT";
                listSchema.push_back ( minimun );

                SKGModelTemplate doubleColumn;
                doubleColumn.id="doublecolumn";
                doubleColumn.name=tr ( "Amount in 2 columns" );
                doubleColumn.icon="";
                doubleColumn.schema="d_date;i_NBRECURRENT;t_bookmarked;t_ACCOUNT;i_number;t_mode;t_payee;t_comment;t_REALCOMMENT;t_CATEGORY;t_REALCATEGORY;"
                                    "t_status;f_REALCURRENTAMOUNT_EXPENSE;f_REALCURRENTAMOUNT_INCOME;f_CURRENTAMOUNT_EXPENSE;f_CURRENTAMOUNT_INCOME";
                listSchema.push_back ( doubleColumn );
        } else if ( getRealTable() =="recurrentoperation" ) {
                listSchema.clear();

                SKGModelTemplate def;
                def.id="default";
                def.name=tr ( "Default" );
                def.icon="edit-undo";
                def.schema="d_DUEDATE;d_date;t_PERIODNLS;i_auto_write_days;i_warn_days;t_ACCOUNT;i_number;t_mode;t_payee;t_comment;t_CATEGORY;"
                           "t_status;f_CURRENTAMOUNT";
                listSchema.push_back ( def );

                SKGModelTemplate minimun;
                minimun.id="minimun";
                minimun.name=tr ( "Minimun" );
                minimun.icon="";
                minimun.schema="d_DUEDATE;d_date;t_PERIODNLS;i_auto_write_days;i_warn_days;t_ACCOUNT;t_payee;;"
                               "t_status;f_CURRENTAMOUNT";
                listSchema.push_back ( minimun );
        } else if ( getRealTable() =="account" ) {
                SKGModelTemplate minimun;
                minimun.id="minimun";
                minimun.name=tr ( "Minimun" );
                minimun.icon="";
                minimun.schema="t_BANK;t_close;t_name";
                listSchema.push_back ( minimun );

                SKGModelTemplate intermediate;
                intermediate.id="intermediate";
                intermediate.name=tr ( "Intermediate" );
                intermediate.icon="";
                intermediate.schema="t_BANK;t_close;t_name;t_type;f_CURRENTAMOUNT;f_CHECKED;f_COMING_SOON";
                listSchema.push_back ( intermediate );
        } else if ( getRealTable() =="category" ) {
                SKGModelTemplate minimun;
                minimun.id="minimun";
                minimun.name=tr ( "Minimun" );
                minimun.icon="";
                minimun.schema="t_name";
                listSchema.push_back ( minimun );

                SKGModelTemplate op;
                op.id="with_operations";
                op.name=tr ( "With operations" );
                op.icon="";
                op.schema="t_name;i_NBOPERATIONS;f_REALCURRENTAMOUNT";
                listSchema.push_back ( op );

                SKGModelTemplate op2;
                op2.id="with_cumulative_operations";
                op2.name=tr ( "With cumulative operations" );
                op2.icon="";
                op2.schema="t_name;i_SUMNBOPERATIONS;f_SUMCURRENTAMOUNT";
                listSchema.push_back ( op2 );
        } else if ( getRealTable() =="unit" ) {
                SKGModelTemplate minimun;
                minimun.id="minimun";
                minimun.name=tr ( "Minimun" );
                minimun.icon="";
                minimun.schema="t_name;t_symbol;t_type";
                listSchema.push_back ( minimun );
        }
        isResetRealyNeeded=iResetOnCreation;
        refresh();
}

void SKGObjectModel::buidCache()
{
        SKGObjectModelBase::buidCache();
        operationTable= ( getRealTable() =="operation" );
        refundTable= ( getRealTable() =="refund" );
        recurrentoperationTable= ( getRealTable() =="recurrentoperation" );
        accountTable= ( getRealTable() =="account" );
        unitTable= ( getRealTable() =="unit" );
        unitvalueTable= ( getRealTable() =="unitvalue" );

        if ( unitvalueTable ) {
                SKGUnitValueObject unitValObject=getObject ( index ( 0,0 ) );
                SKGUnitObject unitObject;
                unitValObject.getUnit ( unitObject );
                SKGUnitObject parent;
                unitObject.getUnit ( parent );
                if ( parent.exist() ) cacheUnit=parent.getSymbol();
        }

        if ( operationTable ) {
                //Read Setting
                KSharedConfigPtr config=KSharedConfig::openConfig ( "skrooge_operationrc" );
                KConfigGroup pref=config->group ( "skrooge_operation" );
                fontFutureOperations = pref.readEntry ( "fontFuture",QFont ( "DejaVu Sans", 10, -1, true ) );
                fontFutureOperationsColor = pref.readEntry ( "fontFutureColor",QColor ( Qt::gray ) );
        }
}

SKGObjectModel::~SKGObjectModel()
{
        SKGTRACEIN ( 1, "SKGObjectModel::~SKGObjectModel" );
}

QVariant SKGObjectModel::headerData ( int section, Qt::Orientation orientation, int role ) const
{
        _SKGTRACEIN ( 10, "SKGObjectModelBase::headerData" );

        if ( orientation==Qt::Horizontal ) {
                if ( role == Qt::DisplayRole ) {
                        QString att;
                        if ( section>=0 && section< listAttibutes.count() ) att=listAttibutes[section];
                        else att=SKGServices::intToString ( section );

                        if ( att=="t_bookmarked" || att=="i_NBRECURRENT" || att=="t_status" || att=="t_close" ) return "";
                } else if ( role == Qt::DecorationRole ) {
                        QString att;
                        if ( section>=0 && section< listAttibutes.count() ) att=listAttibutes[section];
                        else att=SKGServices::intToString ( section );

                        if ( att=="t_bookmarked" ) return KIcon ( "rating" );
                        else if ( att=="i_NBRECURRENT" ) return KIcon ( "chronometer" );
                        else if ( att=="t_status" ) return KIcon ( "dialog-ok" );
                        else if ( att=="t_close" ) return KIcon ( "window-close" );
                        else if ( att=="t_CATEGORY" || att=="t_REALCATEGORY" ) return KIcon ( "skrooge_category" );
                        else if ( att=="t_UNIT" ) return KIcon ( "skrooge_unit" );
                        else if ( att=="f_CURRENTAMOUNT_EXPENSE" || att=="f_REALCURRENTAMOUNT_EXPENSE" ) return KIcon ( "list-remove" );
                        else if ( att=="f_CURRENTAMOUNT_INCOME" || att=="f_REALCURRENTAMOUNT_INCOME" ) return KIcon ( "list-add" );
                }
        }
        return SKGObjectModelBase::headerData ( section, orientation, role );
}

QVariant SKGObjectModel::data ( const QModelIndex &index, int role ) const
{
        if ( !index.isValid() ) return QVariant();
        _SKGTRACEIN ( 10, "SKGObjectModel::data" );

        switch ( role ) {
        case Qt::DisplayRole:
        case Qt::EditRole:
        case Qt::UserRole: {
                SKGObjectBase* obj = getObjectPointer ( index );
                QString att=listAttibutes[index.column() ];
                QString val=obj->getAttribute ( att );
                if ( ( ( operationTable || recurrentoperationTable || refundTable) &&
                                ( att=="t_bookmarked" || att=="t_status" || att=="i_NBRECURRENT" ) ) ||
                                ( att=="t_close" ) ) {
                        if ( role==Qt::UserRole ) return val;
                        else return "";
                } else if ( accountTable && att=="t_type" ) {
                        if ( val=="C" ) return i18n ( "Current" );
                        else if ( val=="D" ) return i18n ( "Credit card" );
                        else if ( val=="A" ) return i18n ( "Assets" );
                        else if ( val=="I" ) return i18n ( "Investment" );
                        else return i18n ( "Other" );
                } else if ( unitTable && att=="t_type" ) {
                        if ( val=="1" ) return i18n ( "Primary currency" );
                        else if ( val=="2" ) return i18n ( "Secondary currency" );
                        else if ( val=="C" ) return i18n ( "Currency" );
                        else if ( val=="S" ) return i18n ( "Share" );
                        else if ( val=="I" ) return i18n ( "Index" );
                        else return i18n ( "Object" );
                } else if ( getAttributeType ( index.column() ) ==SKGObjectModelBase::DOUBLE ) {
                        double dval=SKGServices::stringToDouble ( val );
                        if ( role == Qt::DisplayRole ) {
                                if ( att=="f_CURRENTAMOUNT_INCOME" || att=="f_CURRENTAMOUNT_EXPENSE" ||
                                                att=="f_REALCURRENTAMOUNT_INCOME" || att=="f_REALCURRENTAMOUNT_EXPENSE" ) {
                                        double dval=SKGServices::stringToDouble ( val );
                                        if ( dval==0 ) return "";
                                }

                                QString unit=" ";
                                if ( QString::compare ( att, "f_QUANTITY", Qt::CaseInsensitive ) !=0 && QString::compare ( att, "f_REALQUANTITY", Qt::CaseInsensitive ) !=0 ) {
                                        unit= ( ( SKGDocumentBank* ) getDocument() )->getPrimaryUnit();
                                        if ( unitvalueTable && !cacheUnit.isEmpty() ) {
                                                unit=cacheUnit;
                                        }
                                }
                                return KGlobal::locale()->formatMoney ( dval, unit, 2 );
                        } else {
                                return dval;
                        }
                } else if ( getAttributeType ( index.column() ) ==SKGObjectModelBase::INTEGER ) {
                        if ( att=="i_number" && val=="0" ) return "";
                        return SKGServices::stringToInt ( val );
                }

                break;
        }
        case Qt::DecorationRole: {
                //decoration
                SKGObjectBase* obj = getObjectPointer ( index );
                QString att=listAttibutes[index.column() ];
                if ( accountTable && att=="t_BANK" ) {
                        QString t_icon=obj->getAttribute ( "t_icon" );
                        if ( t_icon.isEmpty() ) t_icon=obj->getAttribute ( "t_ICON" );
                        if ( !t_icon.isEmpty() ) {
                                QDir dirLogo ( KStandardDirs::locate ( "data", QString::fromLatin1 ( "skrooge/images/logo/" ) ) );
                                return qVariantFromValue ( QIcon ( dirLogo.absoluteFilePath ( t_icon ) ) );
                        }
                } else if ( getRealTable() =="category" && index.column() == 0 ) {
                        QString t_TYPEEXPENSE=obj->getAttribute ( "t_TYPEEXPENSE" );
                        QStringList overlay;
                        if ( t_TYPEEXPENSE==tr ( "Expenditure" ) ) overlay.push_back ( "list-remove" );
                        else if ( t_TYPEEXPENSE==tr ( "Income" ) ) overlay.push_back ( "list-add" );
                        return KIcon ( "skrooge_category", NULL, overlay );
                } else if ( operationTable || recurrentoperationTable ) {
                        if ( att=="t_mode" ) {
                                if ( obj->getAttribute ( "i_group_id" ) !="0" ) return KIcon ( "skrooge_transfer" );
                        } else if ( att=="t_CATEGORY" ) {
                                if ( SKGServices::stringToInt ( obj->getAttribute ( "i_NBSUBCATEGORY" ) ) >1 ) return KIcon ( "skrooge_split" );
                        } else if ( att=="i_NBRECURRENT" && operationTable ) {
                                if ( obj->getAttribute ( "i_NBRECURRENT" ) !="0" ) {
                                        QStringList overlay;
                                        overlay.push_back ( "rating" );
                                        return KIcon ( "chronometer", NULL, overlay );
                                } else if ( obj->getAttribute ( "r_recurrentoperation_id" ) !="0" ) return KIcon ( "chronometer" );
                        } else if ( att=="t_bookmarked" && obj->getAttribute ( "t_bookmarked" ) =="Y" ) return KIcon ( "rating" );
                } else if ( unitTable ) {
                        if ( att=="f_CURRENTAMOUNT" ) {
                                SKGUnitObject unit=*obj;
                                double annualchange=365*unit.getDailyChange();
                                if ( annualchange>=15 ) return KIcon ( "skrooge_much_more" );
                                else if ( annualchange<=-15 ) return KIcon ( "skrooge_much_less" );
                                else if ( annualchange>=5 ) return KIcon ( "skrooge_more" );
                                else if ( annualchange<=-5 ) return KIcon ( "skrooge_less" );
                        }
                }
                break;
        }
        case Qt::TextColorRole: {
                //Text color
                SKGObjectBase* obj = getObjectPointer ( index );
                QString att=listAttibutes[index.column() ];
                if ( recurrentoperationTable ) {
                        if ( att=="d_DUEDATE" && SKGServices::stringToTime ( obj->getAttribute ( "d_DUEDATE" ) ).date() <=QDate::currentDate() ) return qVariantFromValue ( QColor ( Qt::red ) );
                } else if ( operationTable ) {
                        if ( SKGServices::stringToTime ( obj->getAttribute ( "d_date" ) ).date() >QDate::currentDate() ) return qVariantFromValue ( fontFutureOperationsColor );
                }

                break;
        }
        case Qt::FontRole: {
                if ( operationTable ) {
                        SKGObjectBase* obj = getObjectPointer ( index );
                        if ( SKGServices::stringToTime ( obj->getAttribute ( "d_date" ) ).date() >QDate::currentDate() ) {
                                return fontFutureOperations;
                        }
                }
                return QVariant();
        }

        case Qt::CheckStateRole: {
                //CheckState
                SKGObjectBase* obj = getObjectPointer ( index );
                if ( obj ) {
                        QString att=listAttibutes[index.column() ];
                        if ( operationTable && att=="t_status" ) {
                                QString t_status=obj->getAttribute ( "t_status" );
                                if ( t_status=="P" ) return Qt::PartiallyChecked;
                                else if ( t_status=="C" ) return Qt::Checked;
                                else if ( t_status=="N" ) return Qt::Unchecked;
                        } else if ( att=="t_close" ) {
                                QString t_status=obj->getAttribute ( "t_close" );
                                if ( t_status=="Y" ) return Qt::Checked;
                                else return Qt::Unchecked;
                        } else  if ( recurrentoperationTable && att=="i_auto_write_days" ) {
                                QString t_auto_write=obj->getAttribute ( "t_auto_write" );
                                if ( t_auto_write=="Y" ) return Qt::Checked;
                                else return Qt::Unchecked;
                        } else  if ( recurrentoperationTable && att=="i_warn_days" ) {
                                QString t_auto_write=obj->getAttribute ( "t_warn" );
                                if ( t_auto_write=="Y" ) return Qt::Checked;
                                else return Qt::Unchecked;
                        }
                }
                break;
        }
        case Qt::ToolTipRole: {
                //Tooltip
                QString secondaryUnit= ( ( SKGDocumentBank* ) getDocument() )->getSecondaryUnit();
                if ( !secondaryUnit.isEmpty() ) {
                        QString att=listAttibutes[index.column() ];
                        if ( getAttributeType ( index.column() ) ==SKGObjectModelBase::DOUBLE && att!="f_QUANTITY" && att!="f_REALQUANTITY" ) {
                                QString toolTipString;
                                SKGObjectBase* obj = getObjectPointer ( index );
                                if ( obj ) {
                                        //Add secondary unit
                                        double val=SKGServices::stringToDouble ( obj->getAttribute ( att ) );
                                        double unitval= ( ( SKGDocumentBank* ) getDocument() )->getSecondaryUnitValue();
                                        if ( unitval )  toolTipString=KGlobal::locale()->formatMoney ( val/unitval, secondaryUnit, 2 );

                                        //Add original amount
                                        QString originalAmount=obj->getProperty ( "SKG_OP_ORIGINAL_AMOUNT" );
                                        if ( !originalAmount.isEmpty() ) {
                                                if ( !toolTipString.isEmpty() ) toolTipString+="\n\n";
                                                double val1=SKGServices::stringToDouble ( obj->getAttribute ( "f_CURRENTAMOUNT" ) );
                                                double val2=SKGServices::stringToDouble ( originalAmount );
                                                double gain= ( val2!=0 ? 100.0* ( val1-val2 ) /val2 : 0 );
                                                toolTipString+=i18n ( "Original amount=%1 (%2 %)",
                                                                      KGlobal::locale()->formatMoney ( val2, ( ( SKGDocumentBank* ) getDocument() )->getPrimaryUnit(), 2 ),
                                                                      gain );
                                        }

                                        if ( !toolTipString.isEmpty() ) return toolTipString;
                                }
                        }
                }
                break;
        }
        default: {
        }
        }
        return SKGObjectModelBase::data ( index, role );
}

bool SKGObjectModel::setData ( const QModelIndex & index, const QVariant& value, int role )
{
        if ( !index.isValid() ) return false;

        if ( role == Qt::CheckStateRole ) {
                SKGError err;
                if ( accountTable ) {
                        SKGAccountObject obj=getObject ( index );
                        SKGBEGINTRANSACTION ( *getDocument(),
                                              ( ( Qt::CheckState ) value.toInt() == Qt::Checked ? i18n ( "Close account [%1]", obj.getName() ) : i18n ( "Open account [%1]", obj.getName() ) ), err );
                        err=obj.setClosed ( ( Qt::CheckState ) value.toInt() == Qt::Checked );
                        if ( err.isSucceeded() ) err=obj.save();
                } else if ( refundTable ) {
                        SKGRefundTrackerObject obj=getObject ( index );
                        SKGBEGINTRANSACTION ( *getDocument(),
                                              ( ( Qt::CheckState ) value.toInt() == Qt::Checked ? i18n ( "Close tracker [%1]", obj.getName() ) : i18n ( "Open tracker [%1]", obj.getName() ) ), err );
                        err=obj.setClosed ( ( Qt::CheckState ) value.toInt() == Qt::Checked );
                        if ( err.isSucceeded() ) err=obj.save();
                } else if ( operationTable ) {
                        SKGOperationObject obj=getObject ( index );
                        SKGBEGINTRANSACTION ( *getDocument(), i18n ( "Change operation status" ), err );
                        SKGOperationObject::OperationStatus statusinitial=obj.getStatus();
                        SKGOperationObject::OperationStatus t_status=statusinitial;
                        if ( QApplication::keyboardModifiers() &Qt::ControlModifier ) {
                                //2747379: NONE ==> CHECKED, POINTED ==> CHECKED, CHECKED ==> CHECKED
                                t_status= SKGOperationObject::CHECKED ;
                                //t_status= ( t_status==SKGOperationObject::POINTED ? SKGOperationObject::NONE : ( t_status==SKGOperationObject::CHECKED ? SKGOperationObject::POINTED : SKGOperationObject::NONE ) );
                        } else {
                                //2747379: NONE ==> POINTED, POINTED ==> NONE, CHECKED ==> POINTED
                                t_status= ( t_status==SKGOperationObject::NONE ? SKGOperationObject::POINTED : ( t_status==SKGOperationObject::POINTED ? SKGOperationObject::NONE : SKGOperationObject::POINTED ) );
                                //t_status=(t_status==SKGOperationObject::POINTED ? SKGOperationObject::CHECKED : (t_status==SKGOperationObject::CHECKED ? SKGOperationObject::CHECKED : SKGOperationObject::POINTED ));
                        }
                        if ( t_status!=statusinitial ) {
                                err=obj.setStatus ( t_status );
                                if ( err.isSucceeded() ) err=obj.save();
                        }
                } else if ( recurrentoperationTable ) {
                        QString att=listAttibutes[index.column() ];

                        //Get the real object, not the object from the view
                        SKGObjectBase* objtmp=getObjectPointer ( index );
                        if ( objtmp ) {
                                SKGRecurrentOperationObject obj=SKGRecurrentOperationObject ( objtmp->getDocument(), objtmp->getID() );

                                SKGBEGINTRANSACTION ( *getDocument(), i18n ( "Recurrent operation update" ), err );
                                if ( att=="i_warn_days" ) {
                                        err=obj.warnEnabled ( !obj.isWarnEnabled() );
                                        if ( err.isSucceeded() ) err=obj.save();
                                } else if ( att=="i_auto_write_days" ) {
                                        err=obj.autoWriteEnabled ( !obj.isAutoWriteEnabled() );
                                        if ( err.isSucceeded() ) err=obj.save();
                                }
                        }
                }

                SKGMainPanel::displayErrorMessage ( err );
                return err.isSucceeded();
        }
        return QAbstractItemModel::setData ( index, value, role );
}

Qt::ItemFlags SKGObjectModel::flags ( const QModelIndex &index ) const
{
        _SKGTRACEIN ( 10, "SKGObjectModel::flags" );

        Qt::ItemFlags flags=SKGObjectModelBase::flags ( index );

        if ( getRealTable() =="category" ) {
                if ( index.isValid() ) flags|=Qt::ItemIsDragEnabled | Qt::ItemIsDropEnabled;
                else flags|=Qt::ItemIsDropEnabled;
        } else if ( index.isValid() && operationTable ) {
                QString att=listAttibutes[index.column() ];
                if ( att=="t_status" ) flags|=Qt::ItemIsUserCheckable;
        } else if ( index.isValid() && recurrentoperationTable ) {
                QString att=listAttibutes[index.column() ];
                if ( att=="i_warn_days" || att=="i_auto_write_days" ) flags|=Qt::ItemIsUserCheckable;
        } else if ( index.isValid() ) {
                QString att=listAttibutes[index.column() ];
                if ( att=="t_close" ) flags|=Qt::ItemIsUserCheckable;
        }

        return flags;
}

Qt::DropActions SKGObjectModel::supportedDragActions() const
{
        if ( getRealTable() =="category" ) return Qt::MoveAction;
        return SKGObjectModelBase::supportedDragActions();
}

Qt::DropActions SKGObjectModel::supportedDropActions() const
{
        if ( getRealTable() =="category" ) return Qt::MoveAction;
        return SKGObjectModelBase::supportedDropActions();
}

bool SKGObjectModel::dropMimeData ( const QMimeData *data,
                                    Qt::DropAction action,
                                    int row, int column,
                                    const QModelIndex &parent )
{
        if ( SKGObjectModelBase::dropMimeData ( data, action, row, column, parent ) ) return true;
        if ( action == Qt::IgnoreAction ) return true;
        if ( !data || !data->hasFormat ( "application/skrooge.category.ids" ) ) return false;
        if ( column > 0 ) return false;

        QByteArray encodedData = data->data ( "application/skrooge.category.ids" );
        QDataStream stream ( &encodedData, QIODevice::ReadOnly );
        QStringList newItems;

        SKGError err;
        SKGCategoryObject parentCategory;
        if ( parent.isValid() ) parentCategory=getObject ( parent );
        {
                SKGBEGINTRANSACTION ( *getDocument(), i18n ( "Move category" ), err );

                while ( !stream.atEnd() && err.isSucceeded() ) {
                        int o_id;
                        QString o_table;
                        stream >> o_table;
                        stream >> o_id;

                        SKGCategoryObject child ( getDocument(), o_id );
                        err=child.load();
                        if ( err.isSucceeded() ) {
                                if ( parent.isValid() ) err=child.setParentCategory ( parentCategory );
                                else err=child.removeParentCategory();
                        }
                        if ( err.isSucceeded() ) err=child.save();
                }
        }
        SKGMainPanel::displayErrorMessage ( err );
        return err.isSucceeded();
}

void SKGObjectModel::dataModified ( const QString& iTableName, int iIdTransaction )
{
        if ( getRealTable() ==iTableName || iTableName.isEmpty() || getRealTable() =="doctransaction" ) {
                SKGTRACEIN ( 1, "SKGObjectModel::dataModified" );
                if ( iTableName=="category" ) {
                        //Full refresh
                        isResetRealyNeeded=true;
                        refresh();
                } else {
                        SKGObjectModelBase::dataModified ( iTableName, iIdTransaction );
                }
        } else SKGObjectModelBase::dataModified ( iTableName, iIdTransaction );
}


#include "skgobjectmodel.moc"

