/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for file operation.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgfileplugin.h"
#include "skrooge_file.h"

#include "skgtraces.h"
#include "skgerror.h"
#include "skgbankincludes.h"
#include "skgmainpanel.h"

#include <kactioncollection.h>
#include <kstandardaction.h>
#include <krecentfilesaction.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <kinputdialog.h>
#include <ksplashscreen.h>

#include <QtGui/QWidget>
#include <QFile>

K_PLUGIN_FACTORY(SKGFilePluginFactory, registerPlugin<SKGFilePlugin>();)
K_EXPORT_PLUGIN(SKGFilePluginFactory("skrooge_file", "skrooge_file"))

SKGFilePlugin::SKGFilePlugin(QObject* iParent, const QVariantList& /*iArg*/) : SKGInterfacePlugin(iParent)
{
        SKGTRACEIN(10, "SKGFilePlugin::SKGFilePlugin");
}

SKGFilePlugin::~SKGFilePlugin()
{
        SKGTRACEIN(10, "SKGFilePlugin::~SKGFilePlugin");
        if (recentFiles) recentFiles->saveEntries(KConfigGroup(KGlobal::config(), "RecentFiles"));

        parent=NULL;
        currentBankDocument=NULL;
        recentFiles=NULL;
        saveAction=NULL;
        saveAsAction=NULL;
        passwordAction=NULL;
}

void SKGFilePlugin::setupActions(SKGMainPanel* iParent, SKGDocument* iDocument, const QStringList& iArgument)
{
        SKGTRACEIN(10, "SKGFilePlugin::setupActions");
        Q_UNUSED(iArgument);

        currentBankDocument=iDocument;
        parent=iParent;

        setComponentData( SKGFilePluginFactory::componentData() );
        setXMLFile("skrooge_file.rc");

        //Menu
        KAction* opennew=KStandardAction::openNew(this, SLOT(actionNew()), actionCollection());
        if (parent) parent->registedGlobalAction("file_new", opennew);

        KAction* open=KStandardAction::open(this, SLOT(actionOpen()), actionCollection());
        if (parent) parent->registedGlobalAction("file_open", open);

        saveAction=KStandardAction::save(this, SLOT(actionSave()), actionCollection());
        if (parent) parent->registedGlobalAction("file_save", saveAction);

        saveAsAction=KStandardAction::saveAs(this, SLOT(actionSaveAs()), actionCollection());
        if (parent) parent->registedGlobalAction("file_save_as", saveAsAction);

        passwordAction = new KAction(KIcon("document-encrypt"), i18n("&Change password..."), this);
        connect(passwordAction, SIGNAL(triggered(bool)), this, SLOT(actionChangePassword()));
        actionCollection()->addAction( QLatin1String("file_change_password"), passwordAction );
        passwordAction->setShortcut(Qt::CTRL+Qt::Key_K);

        if (parent) parent->registedGlobalAction("file_change_password", passwordAction);

        //Recent file
        recentFiles=KStandardAction::openRecent(this,SLOT(actionOpen(const KUrl)), actionCollection());
        if (recentFiles) recentFiles->loadEntries(KConfigGroup(KGlobal::config(), "RecentFiles"));

        //Get last argument
        connect(this, SIGNAL(loadFile(const KUrl)), this, SLOT(actionOpen(const KUrl)), Qt::QueuedConnection);

        int nbArg=iArgument.count();
        if (nbArg && QFile(iArgument.at(nbArg-1)).exists()) {
                QString filename=iArgument.at(nbArg-1);
                if (parent) {
                        KSplashScreen* splashScreen=parent->splashScreen();
                        if (splashScreen) splashScreen->showMessage(i18n("Opening file %1...", filename), Qt::AlignLeft, QColor(221, 130, 8));
                }
                emit loadFile(filename);
        } else {
                //Read Setting
                KSharedConfigPtr config=KSharedConfig::openConfig("skrooge_filerc");
                KConfigGroup pref=config->group("File");
                bool openlastfile = pref.readEntry("openlastfile",true);
                if (openlastfile) {
                        SKGTRACEIN(10, "SKGFilePlugin::openlastfile");
                        QString lastOpenedFile=pref.readPathEntry("lastfilepath",""); // krazy:exclude=doublequote_chars
                        if (!lastOpenedFile.isEmpty() && QFile(lastOpenedFile).exists()) {
                                if (parent) {
                                        KSplashScreen* splashScreen=parent->splashScreen();
                                        if (splashScreen) splashScreen->showMessage(i18n("Opening file %1...", lastOpenedFile), Qt::AlignLeft, QColor(221, 130, 8));
                                }
                                emit loadFile(lastOpenedFile);
                        }
                } else {
                        actionNew();
                }
        }
}

QWidget* SKGFilePlugin::getPreferenceWidget()
{
        SKGTRACEIN(10, "SKGFilePlugin::getPreferenceWidget");
        QWidget* widget=new QWidget();
        ui.setupUi(widget);

        return widget;
}

KConfigSkeleton* SKGFilePlugin::getPreferenceSkeleton()
{
        return skrooge_file::self();
}

void SKGFilePlugin::refresh()
{
        SKGTRACEIN(10, "SKGFilePlugin::refresh");

        //Refresh action status
        if (currentBankDocument) {
                bool test=(currentBankDocument->getDatabase()!=NULL);
                if (saveAction) saveAction->setEnabled(currentBankDocument->isFileModified());
                if (saveAsAction) saveAsAction->setEnabled(test);
                if (passwordAction) passwordAction->setEnabled(test);
        }
}

QString SKGFilePlugin::title() const
{
        return i18n("File");
}

QString SKGFilePlugin::icon() const
{
        return "document-save";
}

QString SKGFilePlugin::statusTip () const
{
        return i18n("File management (open, save, save as, ...)");
}

QString SKGFilePlugin::toolTip () const
{
        return i18n("File management");
}

QStringList SKGFilePlugin::tips() const
{
        QStringList output;
        output.push_back(i18n("<p>... the last opened file can be open automatically when skrooge is launched.</p>"));
        output.push_back(i18n("<p>... you can secure your document with a password.</p>"));
        return output;
}

int SKGFilePlugin::getOrder() const
{
        //Must be one of the first
        return 1;
}

void SKGFilePlugin::close()
{
        SKGTRACEIN(10, "SKGFilePlugin::close");
}

void SKGFilePlugin::actionNew()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGFilePlugin::actionNew",err);
        if (parent && currentBankDocument && parent->queryFileClose()) {
                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

                parent->closeAllTabs();

                err=currentBankDocument->initialize();

                KLocale* locale=KGlobal::locale();
                if (err.isSucceeded()) err=currentBankDocument->setLanguage(locale->language());
                if (err.isSucceeded()) {
                        SKGBEGINTRANSACTION(*currentBankDocument, i18n("Create default unit"), err);
                        if (err.isSucceeded()) {
                                //Create default unit
                                //Creation of PRIMARY unit
                                QString currencySymbol=locale->currencySymbol();
                                QString language=locale->language();
                                SKGTRACEL(10) << "language=" << language << ", currencySymbol=" << currencySymbol << endl;
                                if (language=="fr") {
                                        SKGUnitObject unit;
                                        err=SKGUnitObject::createCurrencyUnit((SKGDocumentBank*) currentBankDocument, "FRF", unit);
                                } else if (currencySymbol==QString::fromUtf8("€")) {
                                        SKGUnitObject unit;
                                        err=SKGUnitObject::createCurrencyUnit((SKGDocumentBank*) currentBankDocument, "EUR", unit);
                                } else if (currencySymbol==QString::fromUtf8("$")) {
                                        SKGUnitObject unit;
                                        err=SKGUnitObject::createCurrencyUnit((SKGDocumentBank*) currentBankDocument, "USD", unit);
                                } else {
                                        SKGUnitObject unit(currentBankDocument);
                                        if (err.isSucceeded()) err = unit.setName(currencySymbol);
                                        if (err.isSucceeded()) err = unit.setType(SKGUnitObject::PRIMARY);
                                        if (err.isSucceeded()) err = unit.setSymbol(locale->currencySymbol());
                                        if (err.isSucceeded()) err = unit.setCountry(locale->countryCodeToName (locale->country()));
                                        if (err.isSucceeded()) err = unit.save();

                                        //Creation of the value
                                        SKGUnitValueObject value;
                                        if (err.isSucceeded()) err = unit.addUnitValue(value);
                                        if (err.isSucceeded()) err = value.setDate(QDate(1950,1,1));
                                        if (err.isSucceeded()) err = value.setQuantity(1);
                                        if (err.isSucceeded()) err = value.save();
                                }
                        }
                }

                QApplication::restoreOverrideCursor();

                //status bar
                if (err.isSucceeded()) err=SKGError(0, i18n("Document successfully created."));
                else err.addError(ERR_FAIL, i18n("Document creation failed."));

                //Display error
                if (parent) parent->displayErrorMessage(err);
        }
}

void SKGFilePlugin::actionSave()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGFilePlugin::actionSave",err);
        if (currentBankDocument && parent) {
                if (currentBankDocument->getCurrentFileName().length()==0) {
                        actionSaveAs();
                } else {
                        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                        err=currentBankDocument->save();
                        QApplication::restoreOverrideCursor();

                        //Refresh
                        parent->refresh();

                        //status bar
                        if (err.isSucceeded()) err=SKGError(0, i18n("File successfully saved."));
                        else err.addError(ERR_FAIL, i18n("Cannot save file"));

                        //Display error
                        parent->displayErrorMessage(err);
                }
        }
}

void SKGFilePlugin::actionSaveAs()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGFilePlugin::actionSaveAs",err);
        if (currentBankDocument && parent) {
                QString fileName=SKGMainPanel::getSaveFileName(KUrl("kfiledialog:///SKG"), i18n("*.skg|Skrooge document") , parent);
                if (fileName.isEmpty()) return;

                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                err=currentBankDocument->saveAs(fileName, true);
                QApplication::restoreOverrideCursor();

                //Refresh
                parent->refresh();

                //status bar
                if (err.isSucceeded()) {
                        err=SKGError(0, i18n("File [%1] saved.", fileName));
                        //Add in recentFiles
                        if (recentFiles) {
                                recentFiles->addUrl(fileName);
                                recentFiles->saveEntries(KConfigGroup(KGlobal::config(), "RecentFiles"));
                        }
                        //Set as last open file in kcfg
                        KSharedConfigPtr config=KSharedConfig::openConfig("skrooge_filerc");
                        KConfigGroup pref=config->group("File");
                        pref.writePathEntry("lastfilepath",fileName);

                } else err.addError(ERR_FAIL, i18n("Failed to save [%1].", fileName));

                //Display error
                parent->displayErrorMessage(err);
        }
}

void SKGFilePlugin::actionOpen(const KUrl & iUrl)
{
        SKGError err;
        SKGTRACEINRC(10, "SKGFilePlugin::actionOpen",err);
        if (parent && currentBankDocument && parent->queryFileClose()) {

                QString fileName=iUrl.path();
                if (!fileName.length()) fileName=KFileDialog::getOpenFileName(KUrl("kfiledialog:///SKG"), "*.skg|"+i18n("Skrooge document") , parent);
                if (fileName.length()) {
                        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                        parent->closeAllTabs();
                        err=currentBankDocument->load(fileName);
                        QApplication::restoreOverrideCursor();

                        if (err.isFailed()) {
                                currentBankDocument->close();

                                //Open failed
                                //Password must be asked
                                QString additionnalMessage;
                                do {
                                        err=SKGError(0, "");
                                        KDialog* d=new KDialog(parent);
                                        QFrame* f=new QFrame(d);
                                        uipwd.setupUi(f);
                                        d->setMainWidget( f );
                                        uipwd.kPwd2->hide();
                                        uipwd.kLabel1->setText(i18n("Password:"));
                                        uipwd.kLabel2->hide();
                                        uipwd.kTitle->setText(additionnalMessage+i18n("This file seems to be protected.\nPlease enter the password."));
                                        uipwd.kTitle->setPixmap(KIcon("document-decrypt").pixmap(48, 48), KTitleWidget::ImageLeft);
                                        uipwd.kPwd1->setFocus();
                                        if (d->exec()==QDialog::Accepted) {
                                                KSplashScreen* splashScreen=parent->splashScreen();
                                                if (splashScreen) splashScreen->hide();

                                                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                                                err=currentBankDocument->load(fileName, uipwd.kPwd1->text());
                                                if (err.isFailed()) {
                                                        additionnalMessage=i18n("Wrong password.\n");
                                                }
                                                QApplication::restoreOverrideCursor();

                                                if (splashScreen) splashScreen->show();
                                        }
                                } while (err.isFailed());
                        }
                        if (err.isFailed()) err=currentBankDocument->initialize();

                        if (err.isSucceeded()) err=currentBankDocument->setLanguage(KGlobal::locale()->language());

                        //status bar
                        if (err.isSucceeded()) {
                                err=SKGError(0, i18n("File [%1] opened.", fileName));
                                //Add in recentFiles
                                if (recentFiles) {
                                        recentFiles->addUrl(fileName);
                                        recentFiles->saveEntries(KConfigGroup(KGlobal::config(), "RecentFiles"));
                                }
                                //Set as last open file in kcfg
                                KSharedConfigPtr config=KSharedConfig::openConfig("skrooge_filerc");
                                KConfigGroup pref=config->group("File");
                                pref.writePathEntry("lastfilepath",fileName);
                        } else  err.addError(ERR_FAIL, i18n("Failed to open [%1].", fileName));

                        //Display error
                        if (parent) parent->displayErrorMessage(err);
                }
        }
}

void SKGFilePlugin::actionChangePassword()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGFilePlugin::actionChangePassword",err);
        if (currentBankDocument) {
                KDialog* d=new KDialog(parent);
                QFrame* f=new QFrame(d);
                uipwd.setupUi(f);
                d->setMainWidget( f );
                uipwd.kPwd1->setFocus();
                uipwd.kTitle->setPixmap(KIcon("document-encrypt").pixmap(48, 48), KTitleWidget::ImageLeft);
                if (d->exec()==QDialog::Accepted) {
                        QString pwd1=uipwd.kPwd1->text();
                        QString pwd2=uipwd.kPwd2->text();
                        if (pwd1==pwd2) {
                                SKGBEGINTRANSACTION(*currentBankDocument, i18n("Change password"), err);
                                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                                err=currentBankDocument->changePassword(pwd1);
                                QApplication::restoreOverrideCursor();

                                //status
                                if (err.isSucceeded())  err=SKGError(0, i18n("Password changed."));
                                else err.addError(ERR_FAIL, i18n("Failed to change password."));
                        } else {
                                err=SKGError(ERR_FAIL, i18n("First and second passwords are different."));
                        }
                } else {
                        err=SKGError(0, i18n("Changing password was canceled."));
                }
                //Display error
                if (parent) parent->displayErrorMessage(err);
        }
}
#include "skgfileplugin.moc"
