/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin to manage scheduled operations.
 *
 * @author Stephane MANKOWSKI
 */
#include "skgscheduledpluginwidget.h"
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgdocumentbank.h"
#include "skgobjectmodel.h"
#include "skgsortfilterproxymodel.h"
#include "skgrecurrentoperationobject.h"
#include "skgtransactionmng.h"

#include <QDomDocument>
#include <QHeaderView>

SKGScheduledPluginWidget::SKGScheduledPluginWidget(SKGMainPanel* iParent, SKGDocumentBank* iDocument)
                : SKGTabWidget(iParent, iDocument)
{
        SKGTRACEIN(1, "SKGScheduledPluginWidget::SKGScheduledPluginWidget");

        ui.setupUi(this);

        //Build you panel here
        SKGObjectModel* modelview = new SKGObjectModel((SKGDocumentBank*) getDocument(), "v_recurrentoperation_display", "", this);
        SKGSortFilterProxyModel* modelproxy = new SKGSortFilterProxyModel(this);
        modelproxy->setSourceModel(modelview);
        modelproxy->setSortRole(Qt::UserRole);
        modelproxy->setDynamicSortFilter(true);

        ui.kView->setModel(modelproxy);
        ui.kView->setWindowTitle(i18n("Recurrent operations"));

        connect(ui.kView->selectionModel(), SIGNAL(selectionChanged(QItemSelection,QItemSelection)), this, SLOT(onSelectionChanged()));

        connect(modelview, SIGNAL(beforeReset()), ui.kView, SLOT(saveSelection()));
        connect(modelview, SIGNAL(afterReset()), ui.kView, SLOT(resetSelection()));

        //Add registered global action in contextual menu
        if (iParent) ui.kView->insertAction(0, iParent->getGlobalAction("edit_delete"));

        //Add Standard KDE Icons to buttons to Operations
        ui.kModifyBtn->setIcon(KIcon("dialog-ok-apply"));
        ui.kProcessBtn->setIcon(KIcon("system-run"));

        ui.kTitle->setPixmap(KIcon("dialog-information").pixmap(22, 22), KTitleWidget::ImageLeft);

        //Refresh
        connect((const QObject*) getDocument(), SIGNAL(tableModified(QString, int)), this, SLOT(dataModified(QString, int)), Qt::QueuedConnection );
        connect((const QObject*) getDocument(), SIGNAL(transactionSuccessfullyEnded(int)), this, SLOT(onSelectionChanged()), Qt::QueuedConnection);
        dataModified("", 0);
}

SKGScheduledPluginWidget::~SKGScheduledPluginWidget()
{
        SKGTRACEIN(1, "SKGScheduledPluginWidget::~SKGScheduledPluginWidget");
}

SKGObjectBase::SKGListSKGObjectBase SKGScheduledPluginWidget::getSelectedObjects()
{
        return ui.kView->getSelectedObjects();
}

int SKGScheduledPluginWidget::getNbSelectedObjects()
{
        return ui.kView->getNbSelectedObjects();
}

QString SKGScheduledPluginWidget::getState()
{
        SKGTRACEIN(10, "SKGScheduledPluginWidget::getState");
        QDomDocument doc("SKGML");
        QDomElement root = doc.createElement("parameters");
        doc.appendChild(root);

        //Memorize table settings
        root.setAttribute("view", ui.kView->getState());

        return doc.toString();
}

void SKGScheduledPluginWidget::setState(const QString& iState )
{
        SKGTRACEIN(10, "SKGScheduledPluginWidget::setState");
        QDomDocument doc("SKGML");
        doc.setContent(iState);
        QDomElement root = doc.documentElement();

        ui.kView->setState(root.attribute ( "view"));

        QSortFilterProxyModel* modelproxy = (QSortFilterProxyModel*) ui.kView->model();
        SKGObjectModel* modelview=(SKGObjectModel*) modelproxy->sourceModel ();
        if (modelview) {
                modelview->setFilter("1=0");
                modelview->setFilter("");
                modelview->refresh();
        }

        //Correction bug 2299394 vvv
        if (ui.kView->isAutoResized()) ui.kView->resizeColumnsToContents();
        //Correction bug 2299394 ^^^
}

QString SKGScheduledPluginWidget::getDefaultStateAttibute()
{
        return "SKGSCHEDULED_DEFAULT_PARAMETERS";
}

QWidget* SKGScheduledPluginWidget::getWidgetForPrint()
{
        return ui.kView;
}

void SKGScheduledPluginWidget::dataModified(const QString& iTableName, int iIdTransaction)
{
        SKGTRACEIN(1, "SKGScheduledPluginWidget::dataModified");
        if (iTableName=="recurrentoperation" || iTableName.isEmpty()) {
                //Correction bug 2299394 vvv
                if (ui.kView->isAutoResized()) ui.kView->resizeColumnsToContents();
                //Correction bug 2299394 ^^^
        }
}

void SKGScheduledPluginWidget::onDoubleClick()
{
        _SKGTRACEIN(10, "SKGScheduledPluginWidget::onDoubleClick");

        //Get selection
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
        if (selection.count()==1) {
                SKGRecurrentOperationObject recOp(selection.at(0));

                //Build where clause and title
                QString wc="((SELECT count(1) FROM v_recurrentoperation s WHERE s.rd_operation_id=v_operation_consolidated.id and s.id="+SKGServices::intToString(recOp.getID())+")>0) OR r_recurrentoperation_id="+SKGServices::intToString(recOp.getID());
                QString title=i18n("Recurrent operations");

                //Open
                if (QApplication::keyboardModifiers()&Qt::ControlModifier && QApplication::keyboardModifiers()&Qt::ShiftModifier) {
                        //Call debug plugin
                        QDomDocument doc("SKGML");
                        QDomElement root = doc.createElement("parameters");
                        doc.appendChild(root);
                        root.setAttribute("sqlOrder", "SELECT * from v_operation_consolidated WHERE "+wc);

                        getMainPanel()->setNewTabContent(getMainPanel()->getPluginByName("Skrooge debug plugin"), -1, doc.toString());
                } else {
                        //Call operation plugin
                        QDomDocument doc("SKGML");
                        doc.setContent ( getDocument()->getParameter("SKGOPERATION_DEFAULT_PARAMETERS") );
                        QDomElement root = doc.documentElement();
                        if (root.isNull()) {
                                root=doc.createElement("parameters");
                                doc.appendChild(root);
                        }

                        root.setAttribute("operationTable", "v_operation_consolidated");
                        root.setAttribute("operationWhereClause", wc);
                        root.setAttribute("title", title);
                        root.setAttribute("title_icon", "view-statistics");
                        root.setAttribute ( "currentPage", "-1" );

                        getMainPanel()->setNewTabContent(getMainPanel()->getPluginByName("Skrooge operation plugin"), -1, doc.toString());
                }
        }
}

void SKGScheduledPluginWidget::onSelectionChanged()
{
        SKGTRACEIN(10, "SKGScheduledPluginWidget::onSelectionChanged");

        int nb=getNbSelectedObjects();
        ui.kFirstOccurenceDate->setEnabled(nb<=1);
        ui.kModifyBtn->setEnabled(nb);
        ui.kProcessBtn->setEnabled(nb);

        if (nb==1) {
                SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
                SKGRecurrentOperationObject recOp=selection.at(0);

                ui.kFirstOccurenceDate->setDate(recOp.getDate());
                ui.kOnceEveryVal->setValue(recOp.getPeriodIncrement());
                ui.kOnceEveryUnit->setCurrentIndex((int) recOp.getPeriodUnit());
                ui.kRemindMeVal->setValue(recOp.getWarnDays());
                ui.kRemindMe->setCheckState(recOp.isWarnEnabled() ? Qt::Checked : Qt::Unchecked);
                ui.kAutoWriteVal->setValue(recOp.getAutoWriteDays());
                ui.kAutoWrite->setCheckState(recOp.isAutoWriteEnabled() ? Qt::Checked : Qt::Unchecked);
        }

        emit selectionChanged();
}

void SKGScheduledPluginWidget::onUpdate()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGScheduledPluginWidget::onUpdate",err);
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        {
                //Get Selection
                SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();

                int nb=selection.count();
                SKGBEGINPROGRESSTRANSACTION(*getDocument(), i18n("Recurrent operation update"), err, nb);
                for (int i=0; err.isSucceeded() && i<nb; ++i) {
                        //Get the real object, not the object from the view
                        SKGRecurrentOperationObject recOp=SKGRecurrentOperationObject(selection.at(i).getDocument(), selection.at(i).getID());

                        //Update it
                        if (nb==1) err=recOp.setDate(ui.kFirstOccurenceDate->date());
                        if (err.isSucceeded()) err=recOp.setPeriodIncrement(ui.kOnceEveryVal->value());
                        if (err.isSucceeded()) err=recOp.setPeriodUnit((SKGRecurrentOperationObject::PeriodUnit) ui.kOnceEveryUnit->currentIndex());
                        if (err.isSucceeded()) err=recOp.setWarnDays(ui.kRemindMeVal->value());
                        if (err.isSucceeded()) err=recOp.setAutoWriteDays(ui.kAutoWriteVal->value());
                        if (err.isSucceeded()) err=recOp.warnEnabled(ui.kRemindMe->checkState()==Qt::Checked);
                        if (err.isSucceeded()) err=recOp.autoWriteEnabled(ui.kAutoWrite->checkState()==Qt::Checked);
                        if (err.isSucceeded()) err=recOp.save();

                        if (err.isSucceeded()) err=getDocument()->stepForward(i+1);
                }
                //status bar
                if (err.isSucceeded()) err=SKGError(0, i18n("Recurrent operation updated."));
                else err.addError(ERR_FAIL, i18n("Update failed"));
        }
        QApplication::restoreOverrideCursor();

        //Display error
        getMainPanel()->displayErrorMessage(err);
}

void SKGScheduledPluginWidget::onProcess()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGScheduledPluginWidget::onProcess",err);
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        {
                //Get Selection
                SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();

                int nb=selection.count();
                SKGBEGINPROGRESSTRANSACTION(*getDocument(), i18n("Insert recurrent operations"), err, nb);
                for (int i=0; err.isSucceeded() && i<nb; ++i) {
                        //Get the real object, not the object from the view
                        SKGRecurrentOperationObject recOp=SKGRecurrentOperationObject(selection.at(i).getDocument(), selection.at(i).getID());

                        //Process it
                        int nbi=0;
                        err=recOp.process(nbi, true);
                        if (err.isSucceeded()) err=getDocument()->stepForward(i+1);
                }
                //status bar
                if (err.isSucceeded()) err=SKGError(0, i18n("Recurrent operation inserted."));
                else err.addError(ERR_FAIL, i18n("Insertion failed"));
        }
        QApplication::restoreOverrideCursor();

        //Display error
        getMainPanel()->displayErrorMessage(err);
}
#include "skgscheduledpluginwidget.moc"


