/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGTESTMACRO_H
#define SKGTESTMACRO_H
/** @file
 * This file contains macro for unit tests.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */

#include "skgtransactionmng.h"
#include "skgerror.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skgdocument.h"
#include "skgnodeobject.h"

#include <QFile>
#include <QDir>
#include <QCoreApplication>

/**
 * Return the test directory
 * @param iPath can be IN or OUT or REF
 * @return the test directory
 */
QString getTestPath(const QString& iPath)
{
        QString pPath =SKGServices::getenv(iPath);
        if (pPath.isEmpty()) {
                return QString("./tests/")+(iPath=="IN" ? "input": (iPath=="OUT" ? "output":"ref"));
        }
        return pPath;
}

/**
 * @def SKGINITTEST(SHOWERRONLY)
 * Initialise the test
 */
#ifndef Q_OS_WIN
#define SKGINITTEST(SHOWERRONLY) \
setenv("LC_ALL", "C", 1); \
setenv("KDEHOME", QFile::encodeName( QDir::homePath() + QLatin1String("/.kde-unit-test") ), 1); \
setenv("XDG_DATA_HOME", QFile::encodeName( QDir::homePath() + QLatin1String("/.kde-unit-test/xdg/local") ), 1); \
setenv("XDG_CONFIG_HOME", QFile::encodeName( QDir::homePath() + QLatin1String("/.kde-unit-test/xdg/config") ), 1); \
unsetenv("KDE_COLOR_DEBUG"); \
QFile::remove(QDir::homePath() + QLatin1String("/.kde-unit-test/share/config/qttestrc"));  \
QCoreApplication  app( argc, argv); \
app.setApplicationName( QLatin1String("qttest") ); \
SKGTRACESEPARATOR; \
SKGTRACE << "STARTING TEST" << endl << flush; \
SKGTRACESEPARATOR; \
bool showonlyfailures = SHOWERRONLY; \
int nberror = 0;  \
int nbcheck = 0;
#else
#define SKGINITTEST(SHOWERRONLY) \
qputenv("LC_ALL", "C"); \
qputenv("KDEHOME", QFile::encodeName( QDir::homePath() + QLatin1String("/.kde-unit-test") )); \
qputenv("XDG_DATA_HOME", QFile::encodeName( QDir::homePath() + QLatin1String("/.kde-unit-test/xdg/local") )); \
qputenv("XDG_CONFIG_HOME", QFile::encodeName( QDir::homePath() + QLatin1String("/.kde-unit-test/xdg/config") )); \
qputenv("KDE_COLOR_DEBUG", ""); \
QFile::remove(QDir::homePath() + QLatin1String("/.kde-unit-test/share/config/qttestrc"));  \
QCoreApplication  app( argc, argv); \
app.setApplicationName( QLatin1String("qttest") ); \
SKGTRACESEPARATOR; \
SKGTRACE << "STARTING TEST" << endl << flush; \
SKGTRACESEPARATOR; \
bool showonlyfailures = SHOWERRONLY; \
int nberror = 0;  \
int nbcheck = 0;
#endif

/*

*/
/**
 * @def SKGENDTEST()
 * Exit test
 */
#define SKGENDTEST()\
SKGTRACE << nberror << " errors / " << nbcheck << " checks =" << (100*(static_cast<double>(nberror)) / (static_cast<double>(nbcheck))) << "%" << endl;\
SKGTraces::dumpProfilingStatistics();\
return nberror;

/**
 * @def SKGRETURNTEST()
 * Return test
 */
#define SKGRETURNTEST()\
SKGTRACE << nberror << " errors / " << nbcheck << " checks =" << (100*(static_cast<double>(nberror)) / (static_cast<double>(nbcheck))) << "%" << endl;\
return nberror;

/**
 * @def SKGTEST(MESSAGE, RESULT, EXPECTEDRESULT)
 * To check the return of a method
 * Example of usage:
 * @code
 * SKGTEST("ERR:getHistoricalSize", err.getHistoricalSize(), 0);
 * @endcode
 */
#define SKGTEST(MESSAGE, RESULT, EXPECTEDRESULT) \
if(RESULT==EXPECTEDRESULT)\
{\
        if(!showonlyfailures) SKGTRACE << "Test [" << MESSAGE << "] : OK" << endl;\
}\
    else\
{\
        SKGTRACE << "!!! Test [" << MESSAGE << "] : KO in line " << __LINE__ << endl;\
        SKGTRACE << "     Expected: [" << EXPECTEDRESULT << ']' << endl;\
        SKGTRACE << "     Result  : [" << RESULT << ']' << endl;\
        ++nberror;\
}\
++nbcheck;

/**
 * @def SKGTESTERROR(MESSAGE, RESULT, EXPECTEDRESULT)
 * To check the return of a method
 * Example of usage:
 * @code
 * SKGTESTERROR("DOC:setParameter", document1.setParameter("ATT1", "VAL1"), true);
 * @endcode
 */
#define SKGTESTERROR(MESSAGE, RESULT, EXPECTEDRESULT) \
{ \
SKGError _err_=RESULT; \
if (_err_.isSucceeded() == EXPECTEDRESULT) { \
                if (!showonlyfailures) { \
                        SKGTRACE << "Test [" << MESSAGE << "] : OK" << endl; \
                        if (!EXPECTEDRESULT) { \
                                SKGTRACE << "     Error Message  :\n" << _err_.getFullMessageWithHistorical()  << endl; \
	}     \
}\
} else {\
                SKGTRACE << "!!! Test [" << MESSAGE << "] : KO in line " << __LINE__ << endl;\
                SKGTRACE << "     Expected: [" << (EXPECTEDRESULT ? "TRUE" : "FALSE") << ']' << endl;\
                SKGTRACE << "     Result  : [" << (_err_.isSucceeded() ? "TRUE" : "FALSE") << ']' << endl;\
                SKGTRACE << "     Error Message  :\n" << _err_.getFullMessageWithHistorical()  << endl;\
                ++nberror;\
}\
++nbcheck;\
}

#endif

