/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is plugin for advice.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgadviceboardwidget.h"
#include "skgdocumentbank.h"
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skginterfaceplugin.h"
#include "skgtransactionmng.h"

#include <KMenu>

#include <QDomDocument>
#include <QLayoutItem>
#include <QToolButton>
#include <QPushButton>

SKGAdviceBoardWidget::SKGAdviceBoardWidget ( SKGDocument* iDocument )
        : SKGWidget ( iDocument ), maxAdvice(7)
{
    SKGTRACEIN ( 10, "SKGAdviceBoardWidget::SKGAdviceBoardWidget" );

    ui.setupUi ( this );

    //Create menu
    setContextMenuPolicy ( Qt::ActionsContextMenu );

    KAction* menuResetAdvices=new KAction ( KIcon("edit-undo"), i18nc ("Noun, a user action", "Activate all advices" ), this );
    connect ( menuResetAdvices, SIGNAL ( triggered ( bool ) ), this, SLOT ( activateAllAdvices() ) );
    addAction ( menuResetAdvices );

    //Refresh
    connect ( ( const QObject* ) getDocument(), SIGNAL ( transactionSuccessfullyEnded(int) ), this, SLOT ( dataModified () ), Qt::QueuedConnection );
}

SKGAdviceBoardWidget::~SKGAdviceBoardWidget()
{
    SKGTRACEIN ( 10, "SKGAdviceBoardWidget::~SKGAdviceBoardWidget" );
}

QString SKGAdviceBoardWidget::getState()
{
    QDomDocument doc ( "SKGML" );
    QDomElement root = doc.createElement ( "parameters" );
    doc.appendChild ( root );
    root.setAttribute ( "maxAdvice", SKGServices::intToString(maxAdvice) );
    return doc.toString();
}

void SKGAdviceBoardWidget::setState ( const QString& iState )
{
    QDomDocument doc ( "SKGML" );
    doc.setContent ( iState );
    QDomElement root = doc.documentElement();

    QString maxAdviceS=root.attribute ( "maxAdvice" );
    if ( !maxAdviceS.isEmpty() ) maxAdvice=SKGServices::stringToInt(maxAdviceS);

    dataModified ();
}

void SKGAdviceBoardWidget::dataModified ()
{
    SKGTRACEIN ( 10, "SKGAdviceBoardWidget::dataModified" );

    //Remove all item of the layout
    QLayoutItem *child=NULL;
    while ( ( child = ui.layout->takeAt ( 0 ) ) != 0 )
    {
        QWidget* w=child->widget();
        if (w) delete w;
        delete child;
    }

    //Get list of ignored advices
    QString currentMonth=QDate::currentDate().toString ( "yyyy-MM" );
    QStringList ignoredAdvices=getDocument()->getParameters("advices","t_value='I' OR t_value='I_"+currentMonth+'\'');

    //Build the list of all advices by requesting all plugins
    QList< SKGInterfacePlugin::SKGAdvice > globalAdvicesList;
    int index=0;
    while ( index>=0 )
    {
        SKGInterfacePlugin* plugin=SKGMainPanel::getMainPanel()->getPluginByIndex ( index );
        if ( plugin )
        {
            foreach(const SKGInterfacePlugin::SKGAdvice& ad, plugin->advices())
            {
                if (!ignoredAdvices.contains(ad.uuid)) globalAdvicesList.push_back(ad);
            }
        }
        else index=-2;
        ++index;
    }

    //Fill layout
    bool atLeastOne=false;
    int nbDisplayedAdvice=0;
    for (int i=0; i<maxAdvice && globalAdvicesList.count()>0; ++i) //globalAdvicesList is modified in the loop
    {
        //Search most important advice
        int priority=-99;
        int pos=-99;
        int nb=globalAdvicesList.count();
        for (int j=0; j<nb; ++j)
        {
            SKGInterfacePlugin::SKGAdvice ad=globalAdvicesList.at(j);
            if (ad.priority>priority)
            {
                pos=j;
                priority=ad.priority;
            }
        }

        //Get advice
        SKGInterfacePlugin::SKGAdvice ad=globalAdvicesList.at(pos);

        //Create icon
        QString iconName=(ad.priority==-1 ? "dialog-information" : ad.priority>=8 ? "security-low" : ad.priority<=4 ? "security-high" : "security-medium");
        QString toolTip=i18n("<p><b>Priority %1:</b>%2</p>", SKGServices::intToString(priority), ad.longMessage);

        //Add ignored action
        ad.autoCorrection.push_back(i18nc("Dismiss an advice provided", "Dismiss"));
        ad.autoCorrection.push_back(i18nc("Dismiss an advice provided", "Dismiss during current month"));

        int nbSolution=ad.autoCorrection.count();

        //Build button
        QStringList overlays;
        overlays.push_back(nbSolution>2 ? "system-run" : "edit-delete");
        QToolButton*  icon= new QToolButton(this);
        icon->setIcon(KIcon(iconName, NULL, overlays));
        icon->setIconSize(QSize(22, 22));
        icon->setMaximumSize(QSize(22, 22));
        icon->setCursor(Qt::PointingHandCursor);
        icon->setAutoRaise(true);

        KMenu* menu=new KMenu ( this );
        menu->setIcon ( icon->icon());
        for (int k=0; k<nbSolution; ++k)
        {
            QAction* act=menu->addAction(KIcon(k<nbSolution-2 ?  "system-run" : "edit-delete"), ad.autoCorrection.at(k), this, SLOT ( adviceClicked() ));
            if (act)
            {
                act->setProperty("id", ad.uuid);
                act->setProperty("solution", k==nbSolution-2 ? -2 : k==nbSolution-1 ? -1 : k);
            }
        }

        icon->setMenu ( menu );
        icon->setPopupMode ( QToolButton::InstantPopup );

        icon->setToolTip(toolTip);

        //Create text
        QLabel* label = new QLabel(this);
        label->setText(ad.shortMessage);
        label->setToolTip(toolTip);

        //Add them
        ui.layout->addRow(icon, label);

        //Remove item from list
        globalAdvicesList.removeAt(pos);

        atLeastOne=true;
        ++nbDisplayedAdvice;
    }

    //Add more
    if (globalAdvicesList.count())
    {
        QPushButton*  more= new QPushButton(this);
        more->setIcon(KIcon("arrow-down-double"));
        more->setIconSize(QSize(22, 22));
        more->setMaximumSize(QSize(22, 22));
        more->setCursor(Qt::PointingHandCursor);
        connect ( more, SIGNAL ( clicked(bool)), this, SLOT ( moreAdvice () ), Qt::QueuedConnection );

        ui.layout->addRow(more, new QLabel(this));
    }
    else if (nbDisplayedAdvice>7)
    {
        //Add less
        QPushButton*  less= new QPushButton(this);
        less->setIcon(KIcon("arrow-up-double"));
        less->setIconSize(QSize(22, 22));
        less->setMaximumSize(QSize(22, 22));
        less->setCursor(Qt::PointingHandCursor);
        connect ( less, SIGNAL ( clicked(bool)), this, SLOT ( lessAdvice () ), Qt::QueuedConnection );

        ui.layout->addRow(less, new QLabel(this));
    }


    //Add spacer
    ui.layout->addItem ( new QSpacerItem ( 20, 40, QSizePolicy::Minimum, QSizePolicy::Expanding ) );
}

void SKGAdviceBoardWidget::moreAdvice()
{
    maxAdvice=9999999;
    dataModified();
}

void SKGAdviceBoardWidget::lessAdvice()
{
    maxAdvice=7;
    dataModified();
}

void SKGAdviceBoardWidget::activateAllAdvices()
{
    SKGError err;
    {
        SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action","Activate all advices" ), err );

        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        err=SKGServices::executeSqliteOrder(getDocument(), "DELETE FROM parameters WHERE t_uuid_parent='advices'");
        QApplication::restoreOverrideCursor();
    }

    //status bar
    if ( err.isSucceeded() ) err=SKGError(0, i18nc("Successful message after an user action", "Advices activated." ) );
    else err.addError ( ERR_FAIL, i18nc("Error message",  "Advices activation failed" ) );

    SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

void SKGAdviceBoardWidget::adviceClicked()
{
    //Get advice identifier
    QString uuid=sender()->property("id").toString();
    if (!uuid.isEmpty())
    {
        //Get solution clicker
        int solution=sender()->property("solution").toInt();

        if (solution<0)
        {
            //We have to ignore this advice
            SKGError err;
            {
                SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action","Dismiss advice" ), err );

                QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                QString currentMonth=QDate::currentDate().toString ( "yyyy-MM" );

                //Create dismiss
                err=getDocument()->setParameter(uuid, solution==-2 ? "I" : "I_"+currentMonth, QVariant(), "advices");

                //Delete useless dismiss
                if (err.isSucceeded())err=SKGServices::executeSqliteOrder(getDocument(), "DELETE FROM parameters WHERE t_uuid_parent='advices' AND t_value like 'I_%' AND t_value!='I_"+currentMonth+'\'');

                QApplication::restoreOverrideCursor();
            }

            //status bar
            if ( err.isSucceeded() ) err=SKGError(0, i18nc("Successful message after an user action", "Advice dismissed." ) );
            else err.addError ( ERR_FAIL, i18nc("Error message",  "Advice dismiss failed" ) );
        }
        else
        {
            //Execute the advice correction on all plugin
            int index=0;
            while ( index>=0 )
            {
                SKGInterfacePlugin* plugin=SKGMainPanel::getMainPanel()->getPluginByIndex ( index );
                if ( plugin )
                {
                    SKGError err=plugin->executeAdviceCorrection(uuid, solution);
                    if (err.isSucceeded() || err.getReturnCode()!=ERR_NOTIMPL)
                    {
                        //The correction has been done or failed. This is the end.
                        index=-2;
                    }
                }
                else index=-2;
                ++index;
            }
        }
    }
}

#include "skgadviceboardwidget.moc"
