/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is a plugin for bookmarks management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgbookmarkplugin.h"
#include "skgtraces.h"
#include "skgerror.h"
#include "skgmainpanel.h"
#include "skgservices.h"
#include "skgnodeobject.h"
#include "skgtransactionmng.h"
#include "skgbookmarkplugindockwidget.h"

#include <QtGui/QWidget>

#include <kactioncollection.h>
#include <kstandarddirs.h>
#include <ktoolbarpopupaction.h>

/**
 * This plugin factory.
 */
K_PLUGIN_FACTORY ( SKGBookmarkPluginFactory, registerPlugin<SKGBookmarkPlugin>(); )
/**
 * This plugin export.
 */
K_EXPORT_PLUGIN ( SKGBookmarkPluginFactory ( "skg_bookmark", "skg_bookmark" ) )

SKGBookmarkPlugin::SKGBookmarkPlugin ( QObject* iParent, const QVariantList& iArg ) :
        SKGInterfacePlugin ( iParent ), dockWidget ( NULL )
{
    Q_UNUSED ( iArg );
    SKGTRACEIN ( 10, "SKGBookmarkPlugin::SKGBookmarkPlugin" );
}

SKGBookmarkPlugin::~SKGBookmarkPlugin()
{
    SKGTRACEIN ( 10, "SKGBookmarkPlugin::~SKGBookmarkPlugin" );
    currentDocument=NULL;
    dockWidget=NULL;
    importStdBookmarkAction=NULL;
    bookmarkAction=NULL;
    bookmarkMenu=NULL;
}

bool SKGBookmarkPlugin::setupActions ( SKGDocument* iDocument, const QStringList& iArgument )
{
    SKGTRACEIN ( 10, "SKGBookmarkPlugin::setupActions" );
    Q_UNUSED ( iArgument );

    currentDocument=iDocument;


    setComponentData ( SKGBookmarkPluginFactory::componentData() );
    setXMLFile ( "skg_bookmark.rc" );

    dockWidget = new QDockWidget ( SKGMainPanel::getMainPanel() );
    dockWidget->setObjectName ( QString::fromUtf8 ( "skg_bookmark_docwidget" ) );
    dockWidget->setAllowedAreas ( Qt::LeftDockWidgetArea|Qt::RightDockWidgetArea );
    dockWidget->setWindowTitle ( title() );
    dockWidget->setWidget ( new SKGBookmarkPluginDockWidget ( currentDocument ) );

    // add action to control hide / display of Bookmarks
    dockWidget->toggleViewAction()->setShortcut ( Qt::SHIFT+Qt::Key_F10 );
    actionCollection()->addAction ( "view_bookmarks", dockWidget->toggleViewAction() );

    //Import bookmarks
    QStringList overlaybookmarks;
    overlaybookmarks.push_back ( "rating" );

    importStdBookmarkAction = new KAction ( KIcon ( "document-import", NULL, overlaybookmarks ), i18nc ("Verb", "Import standard bookmarks" ), this );
    connect ( importStdBookmarkAction, SIGNAL ( triggered ( bool ) ), SLOT ( importStandardBookmarks() ) );
    actionCollection()->addAction ( QLatin1String ( "import_standard_bookmarks" ), importStdBookmarkAction );
    if ( SKGMainPanel::getMainPanel() ) SKGMainPanel::getMainPanel()->registedGlobalAction ( "import_standard_bookmarks", importStdBookmarkAction );

    //
    KAction* goHome=KStandardAction::home ( this, SLOT ( goHome() ), actionCollection() );
    if ( SKGMainPanel::getMainPanel() ) SKGMainPanel::getMainPanel()->registedGlobalAction ( "go_home", goHome );

    //Bookmark
    bookmarkAction = new KToolBarPopupAction ( KIcon ( "bookmark-new-list" ), i18nc ( "Verb, action to display bookmarks", "Bookmarks" ), this );
    connect ( bookmarkAction, SIGNAL ( triggered(Qt::MouseButtons,Qt::KeyboardModifiers)), this, SLOT ( actionOpenBookmark() ) );
    actionCollection()->addAction ( QLatin1String ( "edit_bookmark" ), bookmarkAction );
    bookmarkMenu= static_cast<KMenu*>(bookmarkAction->menu());
    if (bookmarkMenu)
    {
        bookmarkMenu->setProperty ("id", 0 );
        connect ( bookmarkMenu , SIGNAL ( aboutToShow() ), this, SLOT ( onShowBookmarkMenu() ) );
    }

    bookmarkAction->setStickyMenu ( false );
    bookmarkAction->setDelayed(false);
    bookmarkAction->setData ( 0 );

    //bookmarkAction=KStandardAction::redo(this, SLOT(actionRedo()), actionCollection());
    if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit-bookmark", bookmarkAction );

    return true;
}

void SKGBookmarkPlugin::refresh()
{
    SKGTRACEIN ( 10, "SKGBookmarkPlugin::refresh" );
    if ( dockWidget )
    {
        SKGBookmarkPluginDockWidget* p= static_cast<SKGBookmarkPluginDockWidget*>(dockWidget->widget());
        if ( p ) p->refresh();
    }

    if ( currentDocument )
    {
        bool test= ( currentDocument->getDatabase() !=NULL );
        if ( importStdBookmarkAction ) importStdBookmarkAction->setEnabled ( test );

        //Automatic bookmarks creation
        if ( currentDocument->getDatabase() !=NULL )
        {
            QString doc_id=currentDocument->getUniqueIdentifier();
            if ( docUniqueIdentifier!=doc_id )
            {
                docUniqueIdentifier=doc_id;

                int nbNode=0;
                SKGError err=SKGObjectBase::getNbObjects ( currentDocument, "node", "", nbNode );
                if ( err.isSucceeded() && nbNode==0 )
                {
                    importStandardBookmarks();
                }

                //Automatic open of autostart bookmark
                if ( err.isSucceeded() ) goHome();
            }
        }
    }
}

QString SKGBookmarkPlugin::title() const
{
    return i18nc ( "Noun, a bookmark as in a webbrowser bookmark","Bookmark" );
}

QStringList SKGBookmarkPlugin::tips() const
{
    QStringList output;
    output.push_back(i18nc("Description of a tips", "<p>... some bookmarks can be opened automatically when the application is launched.</p>" ) );
    output.push_back(i18nc("Description of a tips", "<p>... bookmarks can be reorganized by drag & drop.</p>" ) );
    output.push_back(i18nc("Description of a tips", "<p>... a double click on a folder of bookmarks will open all the bookmarks it contains.</p>" ) );
    output.push_back(i18nc("Description of a tips", "<p>... you can import standard bookmarks.</p>" ) );
    return output;
}

int SKGBookmarkPlugin::getOrder() const
{
    return 3;
}

void SKGBookmarkPlugin::close()
{
    SKGTRACEIN ( 10, "SKGBookmarkPlugin::close" );
}

QDockWidget* SKGBookmarkPlugin::getDockWidget()
{
    return dockWidget;
}

void SKGBookmarkPlugin::importStandardBookmarks()
{
    SKGTRACEIN ( 10, "SKGBookmarkPlugin::importStandardBookmarks" );
    SKGError err;
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    SKGBEGINTRANSACTION ( *currentDocument, i18nc("Noun, name of the user action", "Import standard bookmarks" ), err );

    //TODO: Must be generic
    QStringList bks;
    bks << i18nc ( "Noun, bookmark name", "Dashboard" ) +"|Y|\"Dashboard plugin\";\"Dashboard\";\"go-home\";\"<!DOCTYPE SKGML><parameters zoomPosition=\"\"0\"\"> <ITEM-1 zoom=\"\"0\"\" index=\"\"0\"\" name=\"\"Skrooge bank plugin\"\" state=\"\"&lt;!DOCTYPE SKGML>&#xa;&lt;parameters menuOther=&quot;Y&quot; menuCurrent=&quot;Y&quot; menuFavorite=&quot;N&quot; menuAssets=&quot;Y&quot; menuInvestment=&quot;Y&quot; menuCreditCard=&quot;Y&quot;/>&#xa;\"\"/> <ITEM-2 zoom=\"\"0\"\" index=\"\"0\"\" name=\"\"Skrooge operation plugin\"\" state=\"\"&lt;!DOCTYPE SKGML>&#xa;&lt;parameters menuTransfert=&quot;Y&quot;/>&#xa;\"\"/> <ITEM-3 zoom=\"\"0\"\" index=\"\"1\"\" name=\"\"Skrooge operation plugin\"\" state=\"\"\"\"/> <ITEM-4 zoom=\"\"0\"\" index=\"\"0\"\" name=\"\"Skrooge scheduled plugin\"\" state=\"\"\"\"/> <ITEM-5 zoom=\"\"0\"\" index=\"\"0\"\" name=\"\"Skrooge search plugin\"\" state=\"\"&lt;!DOCTYPE SKGML>&#xa;&lt;parameters menuFavorite=&quot;N&quot;/>&#xa;\"\"/> <ITEM-6 zoom=\"\"0\"\" index=\"\"0\"\" name=\"\"Skrooge unit plugin\"\" state=\"\"&lt;!DOCTYPE SKGML>&#xa;&lt;parameters menuShares=&quot;Y&quot; menuSharesOwnedOnly=&quot;N&quot; menuIndexes=&quot;Y&quot;/>&#xa;\"\"/> <ITEM-7 zoom=\"\"0\"\" index=\"\"0\"\" name=\"\"Skrooge calculator plugin\"\" state=\"\"\"\"/></parameters>\""
    << i18nc ( "Noun, bookmark name", "Report > Income vs Expenditure on 12 last months" ) +"|N|\"Skrooge report plugin\";\"Rapport\";\"view-statistics\";\"<!DOCTYPE SKGML><parameters representation=\"\"both\"\" expenses=\"\"Y\"\" tableAndGraphState=\"\"&lt;!DOCTYPE SKGML>&#xa;&lt;parameters graphMode=&quot;1&quot; graphicViewState=&quot;&amp;lt;!DOCTYPE SKGML>&amp;#xa;&amp;lt;parameters isToolBarVisible=&amp;quot;Y&amp;quot; antialiasing=&amp;quot;Y&amp;quot;/>&amp;#xa;&quot; splitterState=&quot;000000ff00000000000000020000025b0000032901000000060100000001&quot; bottomToolBarVisible=&quot;Y&quot; filter=&quot;&quot; allPositive=&quot;Y&quot; linearRegressionVisible=&quot;Y&quot; sortColumn=&quot;0&quot; limitVisible=&quot;Y&quot; sortOrder=&quot;0&quot;/>&#xa;\"\" mode=\"\"0\"\" period=\"\"3\"\" incomes=\"\"Y\"\" interval=\"\"2\"\" lines=\"\"t_TYPEEXPENSENLS\"\" nbLevelLines=\"\"0\"\" nbLevelColumns=\"\"0\"\" forecast=\"\"0\"\" columns=\"\"d_DATEMONTH\"\" nb_intervals=\"\"12\"\" forecastValue=\"\"20\"\" transfers=\"\"N\"\" currentPage=\"\"0\"\"/>\""
    << i18nc ( "Noun, bookmark name", "Report > Pie categories in 12 last months" ) +"|N|\"Skrooge report plugin\";\"Rapport\";\"view-statistics\";\"<!DOCTYPE SKGML><parameters representation=\"\"both\"\" expenses=\"\"Y\"\" tableAndGraphState=\"\"&lt;!DOCTYPE SKGML>&#xa;&lt;parameters graphMode=&quot;2&quot; graphicViewState=&quot;&amp;lt;!DOCTYPE SKGML>&amp;#xa;&amp;lt;parameters isToolBarVisible=&amp;quot;Y&amp;quot; antialiasing=&amp;quot;Y&amp;quot;/>&amp;#xa;&quot; splitterState=&quot;000000ff00000000000000020000025b0000032901000000060100000001&quot; bottomToolBarVisible=&quot;Y&quot; filter=&quot;&quot; allPositive=&quot;Y&quot; linearRegressionVisible=&quot;Y&quot; sortColumn=&quot;-1&quot; limitVisible=&quot;Y&quot; sortOrder=&quot;0&quot;/>&#xa;\"\" mode=\"\"0\"\" period=\"\"3\"\" incomes=\"\"Y\"\" interval=\"\"2\"\" lines=\"\"t_REALCATEGORY\"\" nbLevelLines=\"\"0\"\" nbLevelColumns=\"\"0\"\" forecast=\"\"0\"\" columns=\"\"d_DATEMONTH\"\" nb_intervals=\"\"12\"\" forecastValue=\"\"20\"\" transfers=\"\"N\"\" currentPage=\"\"0\"\"/>\""
    << i18nc ( "Noun, bookmark name", "Report > History" ) +"|N|\"Skrooge report plugin\";\"Rapport\";\"view-statistics\";\"<!DOCTYPE SKGML><parameters representation=\"\"both\"\" expenses=\"\"Y\"\" tableAndGraphState=\"\"&lt;!DOCTYPE SKGML>&#xa;&lt;parameters graphMode=&quot;1&quot; graphicViewState=&quot;&amp;lt;!DOCTYPE SKGML>&amp;#xa;&amp;lt;parameters isToolBarVisible=&amp;quot;Y&amp;quot; antialiasing=&amp;quot;Y&amp;quot;/>&amp;#xa;&quot; splitterState=&quot;000000ff0000000000000002000002b50000032601000000060100000001&quot; bottomToolBarVisible=&quot;Y&quot; filter=&quot;&quot; allPositive=&quot;N&quot; linearRegressionVisible=&quot;Y&quot; sortColumn=&quot;-1&quot; limitVisible=&quot;Y&quot; sortOrder=&quot;0&quot;/>&#xa;\"\" mode=\"\"1\"\" period=\"\"0\"\" incomes=\"\"Y\"\" interval=\"\"2\"\" lines=\"\"t_ACCOUNTTYPE\"\" nbLevelLines=\"\"0\"\" nbLevelColumns=\"\"0\"\" forecast=\"\"0\"\" columns=\"\"d_DATEMONTH\"\" nb_intervals=\"\"1\"\" forecastValue=\"\"20\"\" transfers=\"\"Y\"\" currentPage=\"\"0\"\"/>\"";


    foreach ( const QString& bk, bks )
    {
        QStringList line=SKGServices::splitCSVLine ( bk, '|', false );
        if ( line.count() ==3 )
        {
            SKGNodeObject node;
            err=SKGNodeObject::createPathNode ( currentDocument, line.at ( 0 ), node );
            if ( err.isSucceeded() ) err=node.setAutoStart ( line.at ( 1 ) =="Y" );
            if ( err.isSucceeded() ) err=node.setData ( line.at ( 2 ) );
            if ( err.isSucceeded() ) err=node.save();
        }
    }

    QApplication::restoreOverrideCursor();

    //status
    if ( err.isSucceeded() )  err=SKGError(0, i18nc("Successful message after an user action", "Standard bookmarks imported." ) );
    else err.addError ( ERR_FAIL, i18nc("Error message",  "Import standard bookmarks failed" ) );

    //Display error
    if ( SKGMainPanel::getMainPanel() ) SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

void SKGBookmarkPlugin::goHome()
{
    if ( SKGMainPanel::getMainPanel() ) SKGMainPanel::getMainPanel()->closeAllPages();

    SKGObjectBase::SKGListSKGObjectBase oNodeList;
    SKGObjectBase::getObjects ( currentDocument, "v_node", "t_autostart='Y' ORDER BY f_sortorder, t_name", oNodeList );
    int nbAutoStartedBookmarks = oNodeList.count();
    for ( int i=0; i<nbAutoStartedBookmarks; i++ )
    {
        SKGNodeObject autostarted_bookmark = oNodeList.at ( i );
        autostarted_bookmark.load();
        SKGTRACEIN ( 10, "autostarting bookmark : "+autostarted_bookmark.getName() );
        SKGBookmarkPluginDockWidget::openBookmark ( autostarted_bookmark, i>0 );

    }

}

void SKGBookmarkPlugin::actionOpenBookmark(Qt::MouseButtons iMouse, Qt::KeyboardModifiers iKeyboard)
{
    QAction* callerAction=dynamic_cast<QAction*>(this->sender());
    if ( callerAction ) {
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        SKGNodeObject node(currentDocument, callerAction->data().toInt());
        SKGBookmarkPluginDockWidget::openBookmark(node, iMouse&Qt::MidButton || iKeyboard&Qt::ControlModifier);
        QApplication::restoreOverrideCursor();
    }
}

void SKGBookmarkPlugin::onShowBookmarkMenu()
{
    KMenu* callerMenu=dynamic_cast<KMenu*>(this->sender());
    if ( callerMenu ) {
        //Remove previous menu
        callerMenu->clear();

        //Build query
        QString wc="r_node_id=0 OR r_node_id IS NULL OR r_node_id=''";
        int idParent=callerMenu->property("id").toInt();
        if (idParent!=0) wc="r_node_id="+SKGServices::intToString(idParent);

        //Build new menu
        SKGObjectBase::SKGListSKGObjectBase listNode;
        SKGObjectBase::getObjects ( currentDocument,"v_node", wc+" ORDER BY f_sortorder, t_name", listNode );
        int nb=listNode.count();
        for ( int i=0; i<nb; ++i ) {
            SKGNodeObject node=listNode.at(i);
            if (node.isFolder())
            {
                //This is a folder
                KMenu* menu=new KMenu(callerMenu);
                if (menu)
                {
                    callerMenu->addMenu(menu);
                    menu->setTitle(node.getName());
                    menu->setIcon(node.getIcon());
                    menu->setProperty ("id", node.getID() );
                    connect ( menu , SIGNAL ( aboutToShow() ), this, SLOT ( onShowBookmarkMenu() ) );
                }
            }
            else
            {
                //This is a bookmark
                KAction* act=new KAction(callerMenu);
                if ( act ) {
                    callerMenu->addAction(act);
                    act->setText(node.getName());
                    act->setIcon(node.getIcon());
                    act->setData ( node.getID() );
                    connect ( act, SIGNAL ( triggered(Qt::MouseButtons,Qt::KeyboardModifiers)), this, SLOT ( actionOpenBookmark(Qt::MouseButtons,Qt::KeyboardModifiers) ) );
                }
            }
        }
    }
}


#include "skgbookmarkplugin.moc"
