/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is a plugin for bookmarks management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgbookmarkplugindockwidget.h"
#include "skgtraces.h"
#include "skgdocument.h"
#include "skgobjectmodelbase.h"
#include "skgmainpanel.h"
#include "skgsortfilterproxymodel.h"
#include "skgnodeobject.h"
#include "skginterfaceplugin.h"
#include "skgtransactionmng.h"

#include <kmenu.h>

#include <QHeaderView>
#include <QResizeEvent>

bool SKGBookmarkPluginDockWidget::middleClick=false;

SKGBookmarkPluginDockWidget::SKGBookmarkPluginDockWidget(SKGDocument* iDocument)
        : SKGTabPage(iDocument)
{
    SKGTRACEIN(1, "SKGBookmarkPluginDockWidget::SKGBookmarkPluginDockWidget");

    ui.setupUi(this);

    ui.kBookmarksList->setEditTriggers(QAbstractItemView::NoEditTriggers);

    //Build contextual menu
    ui.kBookmarksList->setContextMenuPolicy(Qt::CustomContextMenu);

    mainMenu = new KMenu(ui.kBookmarksList);
    actDelete = mainMenu->addAction(KIcon("list-remove"), i18nc("Verb, delete an item","Delete"));
    connect(actDelete, SIGNAL(triggered(bool)), this, SLOT(onRemoveBookmark()));

    actRename = mainMenu->addAction(KIcon("edit-rename"), i18nc("Verb, change the name of an item","Rename"));
    connect(actRename, SIGNAL(triggered(bool)), this, SLOT(onRenameBookmark()));

    mainMenu->addSeparator();

    actAddBookmarkGroup = mainMenu->addAction(KIcon("folder-new"), i18nc("Verb","Add bookmark group"));
    connect(actAddBookmarkGroup, SIGNAL(triggered(bool)), this, SLOT(onAddBookmarkGroup()));

    actAddBookmark = mainMenu->addAction(KIcon("list-add"), i18nc("Verb","Bookmark current page"));
    connect(actAddBookmark, SIGNAL(triggered(bool)), this, SLOT(onAddBookmark()));

    actSetAutostart = mainMenu->addAction(KIcon("user-online"), i18nc("Verb, automatically load when the application is started","Autostart"));
    connect(actSetAutostart, SIGNAL(triggered(bool)), this, SLOT(onSetAutostart()));

    actUnsetAutostart = mainMenu->addAction(KIcon("user-offline"), i18nc("Verb","Remove Autostart"));
    connect(actUnsetAutostart, SIGNAL(triggered(bool)), this, SLOT(onUnsetAutostart()));

    connect(ui.kBookmarksList, SIGNAL(customContextMenuRequested(const QPoint & ) ),this,SLOT(showMenu( const QPoint& )));

    //Add model
    SKGObjectModelBase* modelview = new SKGObjectModelBase(getDocument(), "v_node", "1=1 ORDER BY f_sortorder, t_name", this, "r_node_id");
    SKGSortFilterProxyModel* modelproxy = new SKGSortFilterProxyModel(this);
    modelproxy->setSourceModel(modelview);
    ui.kBookmarksList->setModel(modelproxy);

    ui.kFilterEdit->setProxy(modelproxy);

    connect(ui.kBookmarksList->selectionModel(), SIGNAL(selectionChanged(QItemSelection,QItemSelection)), this, SLOT(refresh()));
    connect(ui.kBookmarksList, SIGNAL(pressed(QModelIndex)), this, SLOT(onBeforeOpenBookmark()));
    connect(ui.kBookmarksList, SIGNAL(clicked(QModelIndex)), this, SLOT(onOpenBookmark(QModelIndex)));
    connect(ui.kBookmarksList, SIGNAL(doubleClicked(QModelIndex)), this, SLOT(onOpenBookmarkFolder(QModelIndex)));

    connect(modelview, SIGNAL(beforeReset()), ui.kBookmarksList, SLOT(saveSelection()));
    connect(modelview, SIGNAL(afterReset()), ui.kBookmarksList, SLOT(resetSelection()));

    ui.kBookmarksList->setTextResizable(false);
}

SKGBookmarkPluginDockWidget::~SKGBookmarkPluginDockWidget()
{
    SKGTRACEIN(1, "SKGBookmarkPluginDockWidget::~SKGBookmarkPluginDockWidget");
    mainMenu=NULL;
    actDelete=NULL;
    actRename=NULL;
    actAddBookmark=NULL;
    actAddBookmarkGroup=NULL;
    actSetAutostart=NULL;
    actUnsetAutostart=NULL;
}

QWidget* SKGBookmarkPluginDockWidget::mainWidget() const
{
    return ui.kBookmarksList;
}

void SKGBookmarkPluginDockWidget::showMenu(const QPoint& pos)
{
    if (mainMenu) mainMenu->popup(ui.kBookmarksList->mapToGlobal(pos));
}

void SKGBookmarkPluginDockWidget::onSetAutostart()
{
    SKGTRACEIN(10, "SKGBookmarkPluginDockWidget::onSetAutoStart");
    setAutostart("Y");
}

void SKGBookmarkPluginDockWidget::onUnsetAutostart()
{
    SKGTRACEIN(10, "SKGBookmarkPluginDockWidget::onSetAutoStart");
    setAutostart("N");
}

void SKGBookmarkPluginDockWidget::setAutostart(const QString& value)
{
    SKGTRACEIN(10, "SKGBookmarkPluginDockWidget::setAutostart");
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

    SKGObjectBase::SKGListSKGObjectBase selectedBookmarks = getSelectedObjects();

    SKGError err;

    //foreach selected bookmark, set the t_autostart attribute to 'y' or 'n'
    {
        int nb=selectedBookmarks.count();
        SKGBEGINPROGRESSTRANSACTION(*getDocument(), value == "Y" ? i18nc("Noun, name of the user action", "Autostart bookmarks") : i18nc("Noun, name of the user action", "Do not Autostart bookmarks"), err, nb);
        for (int i = 0 ; err.isSucceeded() && i<nb  ; ++i) {
            SKGNodeObject bookmark = selectedBookmarks.at(i);
            err = bookmark.setAttribute("t_autostart",value);
            if (err.isSucceeded())  err=bookmark.save();
            if (err.isSucceeded())  err=getDocument()->stepForward(i+1);
        }
    }
    QApplication::restoreOverrideCursor();

    //status bar
    if (err.isSucceeded()) err=SKGError(0, value == "Y" ? i18nc("Successful message after an user action", "Autostart bookmarks") : i18nc("Successful message after an user action", "Do not Autostart bookmarks"));
    SKGMainPanel::getMainPanel()->displayErrorMessage(err);

}


void SKGBookmarkPluginDockWidget::openBookmark(const SKGNodeObject& iNode, bool iFirstInNewPage)
{
    SKGTRACEIN(1, "SKGBookmarkPluginDockWidget::openBookmark");
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

    QList<SKGNodeObject> nodes;
    nodes.push_back(iNode);

    int setForcusOn=0;
    int nbTab=SKGMainPanel::getMainPanel()->countPages();
    if (nbTab) {
        setForcusOn=nbTab-1;
        if (middleClick) {
            setForcusOn=nbTab;
        } else {
            setForcusOn=SKGMainPanel::getMainPanel()->currentPageIndex();
        }
    }

    int tabNumberForNextOpen=(middleClick || iFirstInNewPage ? -1 : SKGMainPanel::getMainPanel()->currentPageIndex());
    for (int i=0; i<nodes.count(); ++i) { //WARNING: This list is modified dynamically
        SKGNodeObject selectedNode=nodes.at(i);
        QStringList data=SKGServices::splitCSVLine(selectedNode.getData());
        if (data.count()>3) {
            //This bookmark has information ==> we open it
            SKGMainPanel::getMainPanel()->openPage(SKGMainPanel::getMainPanel()->getPluginByName(data[0]),tabNumberForNextOpen, data[3], selectedNode.getName(), SKGServices::intToString(selectedNode.getID()));
            selectedNode.opened = true;
            tabNumberForNextOpen=-1;
        } else {
            //This bookmark is a folder ==> we open children by recursion
            SKGObjectBase::SKGListSKGObjectBase children;
            selectedNode.getNodes(children);
            foreach(const SKGObjectBase& item, children) {
                nodes.push_back(item);
            }
        }
    }

    //Set focus on first page
    SKGMainPanel::getMainPanel()->setCurrentPage(setForcusOn);

    QApplication::restoreOverrideCursor();
}

void SKGBookmarkPluginDockWidget::onBeforeOpenBookmark()
{
    middleClick=QApplication::keyboardModifiers() &Qt::ControlModifier || QApplication::mouseButtons () &Qt::MidButton;
}

void SKGBookmarkPluginDockWidget::onOpenBookmark(const QModelIndex & index )
{
    SKGTRACEIN(1, "SKGBookmarkPluginDockWidget::onOpenBookmark");
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
    QSortFilterProxyModel* proxyModel=(QSortFilterProxyModel*) ui.kBookmarksList->model();
    SKGObjectModelBase* model=(SKGObjectModelBase*) proxyModel->sourceModel();
    if (model) {
        SKGNodeObject node=model->getObject(proxyModel->mapToSource(index));
        bool isFolder=node.getData().isEmpty();
        if (!isFolder) openBookmark(node);
    }
    QApplication::restoreOverrideCursor();
}

void SKGBookmarkPluginDockWidget::onOpenBookmarkFolder(const QModelIndex & index )
{
    SKGTRACEIN(1, "SKGBookmarkPluginDockWidget::onOpenBookmarkFolder");
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
    QSortFilterProxyModel* proxyModel=(QSortFilterProxyModel*) ui.kBookmarksList->model();
    SKGObjectModelBase* model=(SKGObjectModelBase*) proxyModel->sourceModel();
    if (model) {
        SKGNodeObject node=model->getObject(proxyModel->mapToSource(index));
        bool isFolder=node.getData().isEmpty();
        if (isFolder) openBookmark(node);
    }
    QApplication::restoreOverrideCursor();
}

void SKGBookmarkPluginDockWidget::onAddBookmarkGroup()
{
    SKGTRACEIN(1, "SKGBookmarkPluginDockWidget::onAddBookmarkGroup");
    SKGError err;
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
    SKGNodeObject node;
    {
        //Get current selection name
        QString name;
        SKGObjectBase::SKGListSKGObjectBase bookmarks=getSelectedObjects();
        if (bookmarks.count()) {
            SKGNodeObject parentNode=bookmarks.at(0);
            if (!parentNode.getData().isEmpty())
            {
                //This is not a folder, we have to take the parent
                SKGNodeObject parentNodeTmp;
                parentNode.getParentNode(parentNodeTmp);
                parentNode=parentNodeTmp;
            }
            name=parentNode.getFullName();
        }

        //Add current name
        if (!name.isEmpty()) name+=OBJECTSEPARATOR;
        name+=i18nc("Default name for bookmark", "New bookmark");

        //Create bookmark folder
        SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Bookmark folder creation '%1'", name), err);
        err=SKGNodeObject::createPathNode(getDocument(), name, node, true);
    }
    QApplication::restoreOverrideCursor();

    //status bar
    if (err.isSucceeded())
    {
        ui.kBookmarksList->selectObject(node.getUniqueID());
        err=SKGError(0, i18nc("Successful message after an user action", "Bookmark group created"));
    }
    SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGBookmarkPluginDockWidget::onAddBookmark()
{
    SKGTRACEIN(1, "SKGBookmarkPluginDockWidget::onAddBookmark");
    SKGError err;
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

    //Get current page
    SKGNodeObject node;
    SKGTabPage* currentPage=SKGMainPanel::getMainPanel()->currentPage();
    if (currentPage) {
        //Get current selection name
        QString name;
        SKGObjectBase::SKGListSKGObjectBase bookmarks=getSelectedObjects();
        if (bookmarks.count()) {
            SKGNodeObject parentNode=bookmarks.at(0);
            if (!parentNode.getData().isEmpty())
            {
                //This is not a folder, we have to take the parent
                SKGNodeObject parentNodeTmp;
                parentNode.getParentNode(parentNodeTmp);
                parentNode=parentNodeTmp;
            }
            name=parentNode.getFullName();
        }

        //Add current name
        if (!name.isEmpty()) name+=OBJECTSEPARATOR;

        QString defaultName=currentPage->objectName();
        QString defaultIcon;
        SKGInterfacePlugin* plugin=SKGMainPanel::getMainPanel()->getPluginByName(defaultName);
        if (plugin) {
            defaultName=plugin->title();
            defaultIcon=plugin->icon();
        }
        name+=defaultName;

        //Create bookmark
        SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Bookmark creation '%1'", name), err);
        err=SKGNodeObject::createPathNode(getDocument(), name, node, true);
        if (err.isSucceeded()) {
            QString value=SKGServices::stringToCsv(currentPage->objectName())+';'+
                          SKGServices::stringToCsv(defaultName)+';'+
                          SKGServices::stringToCsv(defaultIcon)+';'+
                          SKGServices::stringToCsv(currentPage->getState());

            err=node.setData(value);
            if (err.isSucceeded()) err=node.save();
        }
    }

    QApplication::restoreOverrideCursor();

    //status bar
    if (err.isSucceeded())
    {
        ui.kBookmarksList->selectObject(node.getUniqueID());
        err=SKGError(0, i18nc("Successful message after an user action", "Bookmark created"));
    }
    SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGBookmarkPluginDockWidget::onRenameBookmark()
{
    SKGTRACEIN(1, "SKGBookmarkPluginDockWidget::onRenameBookmark");

    SKGObjectBase::SKGListSKGObjectBase selection;
    QItemSelectionModel *selModel=ui.kBookmarksList->selectionModel();
    QSortFilterProxyModel* proxyModel=(QSortFilterProxyModel*) ui.kBookmarksList->model();
    if (proxyModel && selModel) {
        SKGObjectModelBase* model=(SKGObjectModelBase*) proxyModel->sourceModel();
        if (model) {
            QModelIndexList indexes=selModel->selectedRows();
            if (indexes.count()==1) {
                ui.kBookmarksList->edit ( indexes.at(0) );
            }
        }
    }
}

void SKGBookmarkPluginDockWidget::onRemoveBookmark()
{
    SKGTRACEIN(1, "SKGBookmarkPluginDockWidget::onRemoveBookmark");
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

    SKGError err;
    {
        SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Bookmark delete"), err);
        //Get current selection
        SKGObjectBase::SKGListSKGObjectBase bookmarks=getSelectedObjects();
        int nb=bookmarks.count();
        for (int i=0; i<nb && err.isSucceeded(); ++i) {
            SKGNodeObject node=bookmarks.at(i);
            err=node.remove();
        }
    }

    QApplication::restoreOverrideCursor();

    //status bar
    if (err.isSucceeded()) err=SKGError(0, i18nc("Successful message after an user action", "Bookmark deleted"));
    SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGBookmarkPluginDockWidget::onBookmarkEditorChanged()
{
    SKGTRACEIN(10, "SKGBookmarkPluginDockWidget::onBookmarkEditorChanged");

    //Enable bookmark buttons
    int nbSelectedObjects=getNbSelectedObjects();
    bool testAdd=getDocument()->getDatabase()!=NULL && nbSelectedObjects<=1;
    if (actAddBookmarkGroup) actAddBookmarkGroup->setEnabled(testAdd);
    if (actAddBookmark) actAddBookmark->setEnabled(testAdd && SKGMainPanel::getMainPanel()->currentPageIndex()>=0);
    if (actDelete) actDelete->setEnabled(nbSelectedObjects>0);
    if (actRename) actRename->setEnabled(nbSelectedObjects==1);
}

void SKGBookmarkPluginDockWidget::refresh()
{
    SKGTRACEIN(10, "SKGBookmarkPluginDockWidget::refresh");
    //Bookmarks
    SKGObjectBase::SKGListSKGObjectBase bookmarks=getSelectedObjects();
    int nbSelectedObjects=bookmarks.count();
    if (nbSelectedObjects==1) {
        SKGNodeObject node=bookmarks.at(0);
        if (actSetAutostart) actSetAutostart->setEnabled(node.getAttribute("t_autostart")!="Y");
        if (actUnsetAutostart) actUnsetAutostart->setEnabled(node.getAttribute("t_autostart")=="Y");
    }

    //Set current selection of dock
    /*SKGTabPage* cPage= SKGMainPanel::getMainPanel()->currentPage();
    if(cPage)
    {
      //Get uuid of current page
      QString id=cPage->getBookmarkID();
      if(!id.isEmpty())
      {
        bool previous=ui.kBookmarksList->blockSignals ( true );
        ui.kBookmarksList->clearSelection();
        ui.kBookmarksList->selectObject(id+"-node");
        ui.kBookmarksList->blockSignals ( previous );
      }
    } */

    onBookmarkEditorChanged();
}

void SKGBookmarkPluginDockWidget::resizeEvent(QResizeEvent *event)
{
    if (event) {
        QSize newSize=event->size ();

        //Compute icon size
        int s=qMax(qMin(newSize.width()/5,64),16);
        ui.kBookmarksList->setIconSize(QSize(s, s));
    }

    QWidget::resizeEvent(event);
}

#include "skgbookmarkplugindockwidget.moc"
