/***************************************************************************
 *   Copyright (C) 2010 by SIDDHARTH SHARMA siddharth.kde@gmail.com        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/

/** @file
* This file is Plasma Applet for Account Skroogoid
*
* @author Siddharth SHARMA
* @author Guillaume DE BURE
*/

#include "accountskroogoid.h"
#include "skgdocumentbank.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skgaccountobject.h"
#include "skgunitobject.h"
#include "skgmainpanel.h"

#include <QGraphicsLinearLayout>

#include <KPushButton>
#include <Plasma/ToolButton>
#include <kmessagebox.h>

AccountSkroogoid::AccountSkroogoid(QObject *parent, const QVariantList &args)
    : Plasma::Applet(parent, args)
{
    // This signalMapper will be used to open the account tab when clicking on the
    // corresponding button
    signalMapper = new QSignalMapper(this);
    connect(signalMapper, SIGNAL(mapped(QString)), this, SLOT(openAccountTab(QString)));
}

void AccountSkroogoid::init()
{

    setAspectRatioMode(Plasma::IgnoreAspectRatio);

    // This is the layout for the whole widget
    widgetLayout = new QGraphicsLinearLayout(Qt::Vertical);
    setLayout(widgetLayout);

    // Get the current document
    SKGMainPanel* panel = SKGMainPanel::getMainPanel();
    doc = qobject_cast<SKGDocumentBank*> (panel->getDocument());

    // Get the primary unit
    primary = doc->getPrimaryUnit();

    // Load the dataengine that contains data we need
    Plasma::DataEngine* engine = dataEngine("skgdataengine");
    if(!engine->isValid()) {
        KMessageBox::information(panel, "engine not found");
    }

    // Initialize some variables
    double sumTypeV1 = 0;
    double sumV1 = 0;
    QString currentType;

    // Get data from the dataengine
    engine->connectSource("Accounts", this);
    int nb = listData.count();

    for(int i = 0; i < nb; ++i) {
        // Account name
        QString account = listData.keys().at(i);

        QStringList values = listData[account];

        // Account type
        QString type = values.at(0);

        // Amount of money
        double v1 = SKGServices::stringToDouble(values.at(1));

        // Account closed
        bool closed = (values.at(2) == "Y");

        // Display different sections depending on account types
        if(type != currentType) {
            if(!currentType.isEmpty()) {
                addTotalLine(i18nc("The sum of several numerical values", "Total of %1", currentType), sumTypeV1);
            }

            sumTypeV1 = 0;
            currentType = type;

            // Create a label for the account type
            Plasma::Label* accountTypeLabel = new Plasma::Label();
            accountTypeLabel->setText(currentType);
            accountTypeLabel->setAlignment(Qt::AlignHCenter);

            // Use an italic font
            QFont accountTypeFont =  accountTypeLabel->font();
            accountTypeFont.setItalic(true);
            accountTypeLabel->setFont(accountTypeFont);
            widgetLayout->addItem(accountTypeLabel);

        }
        if(!closed || abs(v1) > 0.1) {    // Only display non closed accounts
            addAccountLine(account, v1);
        }
        sumTypeV1 += v1;
        sumV1 += v1;
    }

    // If the account is the last, we must add a line for the total per type
    addTotalLine(i18nc("The sum of several numerical values", "Total of %1", currentType), sumTypeV1);

    // Add an empty label for a visual separation
    Plasma::Label* emptyLabel = new Plasma::Label();
    widgetLayout->addItem(emptyLabel);

    // Add a line for grand total
    QFont* totalFont = new QFont();
    totalFont->setBold(true);
    addTotalLine(i18nc("The sum of several numerical values", "Total"), sumV1, totalFont);
    delete totalFont;

    widgetLayout->addStretch(20);

}


AccountSkroogoid::~AccountSkroogoid()
{
}


void AccountSkroogoid::dataUpdated(const QString sourceName, const Plasma::DataEngine::Data& engineData)
{

    if(sourceName != "Accounts") {
        return;
    }

    for(int i = 0 ; i < engineData.keys().count() ; i++) {
        const QString accountName = engineData.keys().at(i);
        QVariant accountValuesVariant = engineData.value(accountName);
        QStringList accountValues = accountValuesVariant.value<QStringList>();

        listData[accountName] = accountValues;

    }
}


QGraphicsLinearLayout* AccountSkroogoid::addAccountLine(QString label, double value)
{

    // This is the layout for the line
    QGraphicsLinearLayout* lineLayout = new QGraphicsLinearLayout(Qt::Horizontal);
    widgetLayout->addItem(lineLayout);

    // Create a button for the account.
    Plasma::ToolButton* button = new Plasma::ToolButton();
    button->setText(label);
    lineLayout->addItem(button);

    // Open the account in a new tab when clicking on the account
    connect(button, SIGNAL(clicked()), signalMapper, SLOT(map()));
    signalMapper->setMapping(button, label);

    // Add a spacer for cleaner layout
    lineLayout->addStretch();

    // Create label for the value
    Plasma::Label* valueLabel = createValueLabel(value);
    lineLayout->addItem(valueLabel);

    return lineLayout;
}

QGraphicsLinearLayout* AccountSkroogoid::addTotalLine(QString label, double value, QFont* labelFont)
{

    // This is the line layout
    QGraphicsLinearLayout* lineLayout = new QGraphicsLinearLayout(Qt::Horizontal);
    widgetLayout->addItem(lineLayout);

    // Create a label for the line
    Plasma::Label* lineLabel = new Plasma::Label();
    lineLabel->setText(label);

    // Apply the font if it is defined
    if(labelFont) {
        lineLabel->setFont(*labelFont);
    }

    lineLayout->addItem(lineLabel);

    // Add a spacer for cleaner layout
    lineLayout->addStretch();

    // Set up the label for the value
    Plasma::Label* valueLabel = createValueLabel(value);
    lineLayout->addItem(valueLabel);

    return lineLayout;

}

Plasma::Label* AccountSkroogoid::createValueLabel(double value)
{
    // Get the locale, used for formatting with the correct money style
    KLocale* locale = KGlobal::locale();

    // Create a label
    Plasma::Label* valueLabel = new Plasma::Label();
    valueLabel->setText(locale->formatMoney(value, primary.Symbol, primary.NbDecimal));

    // If the value is negative, use a red font.
    if(value <= 0) {
        QPalette valueLabelPalette = valueLabel->palette();
        QBrush brush(Qt::red);
        brush.setStyle(Qt::SolidPattern);
        valueLabelPalette.setBrush(QPalette::All, QPalette::WindowText, brush);
        valueLabel->setPalette(valueLabelPalette);
    }

    return valueLabel;

}


void AccountSkroogoid::openAccountTab(QString accountName)
{
    // Create a dom document, containing relevant parameters
    QDomDocument params("SKGML");

    // Get default parameters from the current Skrooge document
    params.setContent(doc->getParameter("SKGOPERATION_DEFAULT_PARAMETERS"));
    QDomElement root = params.documentElement();
    if(root.isNull()) {
        root = params.createElement("parameters");
        params.appendChild(root);
    }

    // Add an attribute account with the account name
    root.setAttribute("account", accountName);

    // And finally, open a new tab for this account
    SKGMainPanel::getMainPanel()->openPage(SKGMainPanel::getMainPanel()->getPluginByName("Skrooge operation plugin"), -1, params.toString());

}


#include "accountskroogoid.moc"
