/***************************************************************************
 *   Copyright (C) 2010 by SIDDHARTH SHARMA siddharth.kde@gmail.com        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/

/** @file
* This file is Plasma Applet for Account Skroogoid
*
* @author Siddharth SHARMA
*/

#include "adviceskroogoid.h"
#include "skgdocumentbank.h"
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skginterfaceplugin.h"
#include "skgtransactionmng.h"

#include <QGraphicsLinearLayout>
#include <QGraphicsProxyWidget>
#include <QGraphicsGridLayout>
#include <QToolButton>
#include <QPushButton>
#include <QAction>
#include <QApplication>
#include <KMenu>



AdviceSkroogoid::AdviceSkroogoid(QObject *parent, const QVariantList &args)
    : Plasma::PopupApplet(parent, args)
{
    QGraphicsProxyWidget* pw = new QGraphicsProxyWidget();
    setAspectRatioMode(Plasma::IgnoreAspectRatio);
    resize(300, 300);
    displayLabel = new Plasma::Label();
    gl_layout = new QGraphicsLinearLayout();
    grid = new QGraphicsGridLayout();
    gl_layout->addItem(grid);
    setLayout(gl_layout);
    SKGMainPanel* panel = SKGMainPanel::getMainPanel();
    QStringList ignoredAdvices = panel->getDocument()->getParameters("advices");

    QList< SKGInterfacePlugin::SKGAdvice > globalAdvicesList;
    int index = 0;
    while(index >= 0) {
        SKGInterfacePlugin* plugin = SKGMainPanel::getMainPanel()->getPluginByIndex(index);
        if(plugin) {
            foreach(const SKGInterfacePlugin::SKGAdvice & ad, plugin->advices()) {
                if(!ignoredAdvices.contains(ad.uuid)) globalAdvicesList.push_back(ad);
            }
        } else index = -2;
        ++index;
    }
    //Fill layout
    bool atLeastOne = false;
    int nbDisplayedAdvice = 0;
    for(int i = 0; i < maxAdvice && globalAdvicesList.count() > 0; ++i) {  //globalAdvicesList is modified in the loop
        //Search most important advice
        int priority = -99;
        int pos = -99;
        int nb = globalAdvicesList.count();
        for(int j = 0; j < nb; ++j) {
            SKGInterfacePlugin::SKGAdvice ad = globalAdvicesList.at(j);
            if(ad.priority > priority) {
                pos = j;
                priority = ad.priority;
            }
        }

        //Get advice
        SKGInterfacePlugin::SKGAdvice ad = globalAdvicesList.at(pos);

        //Create icon
        QString iconName = (ad.priority == -1 ? "dialog-information" : ad.priority >= 8 ? "security-low" : ad.priority <= 4 ? "security-high" : "security-medium");
        QString toolTip = i18n("<p><b>Priority %1:</b>%2</p>", SKGServices::intToString(priority), ad.longMessage);

        //Add ignored action
        ad.autoCorrection.push_back(i18nc("Dismiss an advice provided by Skrooge", "Dismiss"));

        int nbSolution = ad.autoCorrection.count();

        //Build button
        QStringList overlays;
        overlays.push_back(nbSolution > 1 ? "system-run" : "edit-delete");
        QToolButton*  icon = new QToolButton();
        icon->setIcon(KIcon(iconName, NULL, overlays));
        icon->setIconSize(QSize(22, 22));
        icon->setMaximumSize(QSize(22, 22));
        icon->setCursor(Qt::PointingHandCursor);
        icon->setAutoRaise(true);
        pw->setWidget(icon);

        KMenu* menu = new KMenu();
        menu->setIcon(icon->icon());
        for(int k = 0; k < nbSolution; ++k) {
            QAction* act = menu->addAction(KIcon(k < nbSolution - 1 ?  "system-run" : "edit-delete"), ad.autoCorrection.at(k), this, SLOT(adviceClicked()));
            act->setProperty("id", ad.uuid);
            act->setProperty("solution", k < nbSolution - 1 ? k : -1);
        }

        icon->setMenu(menu);
        icon->setPopupMode(QToolButton::InstantPopup);

        icon->setToolTip(toolTip);

        //Create text
        Plasma::Label* label = new Plasma::Label();
        label->setText(ad.shortMessage);
        label->setToolTip(toolTip);
        // gl_layout->addItem(label);

        //Add them
        //   ui.layout->addRow(icon, label);

        //Remove item from list
        globalAdvicesList.removeAt(pos);

        atLeastOne = true;
        ++nbDisplayedAdvice;
        //  gl_layout->addItem(pw);
    }
    //Add more
    if(globalAdvicesList.count()) {
        QGraphicsProxyWidget* pw_more = new QGraphicsProxyWidget();
        QPushButton*  more = new QPushButton();
        more->setIcon(KIcon("arrow-down-double"));
        more->setIconSize(QSize(22, 22));
        more->setMaximumSize(QSize(22, 22));
        more->setCursor(Qt::PointingHandCursor);
        connect(more, SIGNAL(clicked(bool)), this, SLOT(moreAdvice()), Qt::QueuedConnection);
        pw_more->setWidget(more);
        // gl_layout->addItem(pw_more);

    } else if(nbDisplayedAdvice > 7) {
        //Add less
        QGraphicsProxyWidget* pw_less = new QGraphicsProxyWidget();
        QPushButton*  less = new QPushButton();
        less->setIcon(KIcon("arrow-up-double"));
        less->setIconSize(QSize(22, 22));
        less->setMaximumSize(QSize(22, 22));
        less->setCursor(Qt::PointingHandCursor);
        connect(less, SIGNAL(clicked(bool)), this, SLOT(lessAdvice()), Qt::QueuedConnection);
        pw_less->setWidget(less);
        // gl_layout->addItem(pw_less);

    }


    //Add spacer
    // ui.layout->addItem ( new QSpacerItem ( 20, 40, QSizePolicy::Minimum, QSizePolicy::Expanding ) );
}
void  AdviceSkroogoid::moreAdvice()
{
    maxAdvice = 9999999;
    //  dataModified();
}

void  AdviceSkroogoid::lessAdvice()
{
    maxAdvice = 7;
    //  dataModified();
}
void AdviceSkroogoid::activateAllAdvices()
{
    SKGMainPanel* panel = SKGMainPanel::getMainPanel();
    SKGError err;
    {
        SKGBEGINTRANSACTION(*panel->getDocument(), i18nc("Noun, name of the user action", "Activate all advices"), err);
        err = panel->getDocument()->executeSqliteOrder("DELETE FROM parameters WHERE t_uuid_parent='advices'");
    }

    //status bar
    if(!err) err = SKGError(0, i18nc("Successful message after an user action", "Advices activated."));
    else err.addError(ERR_FAIL, i18nc("Error message",  "Advices activation failed"));

    SKGMainPanel::displayErrorMessage(err);
}

void AdviceSkroogoid::adviceClicked()
{
    SKGMainPanel* panel = SKGMainPanel::getMainPanel();
    //Get advice identifier
    QString uuid = sender()->property("id").toString();
    if(!uuid.isEmpty()) {
        //Get solution clicker
        int solution = sender()->property("solution").toInt();

        if(solution == -1) {
            //We have to ignore this advice
            SKGError err;
            {
                SKGBEGINTRANSACTION(*panel->getDocument(), i18nc("Noun, name of the user action", "Dismiss advice"), err);
                err = panel->getDocument()->setParameter(uuid, "I", QVariant(), "advices");
            }

            //status bar
            if(!err) err = SKGError(0, i18nc("Successful message after an user action", "Advice dismissed."));
            else err.addError(ERR_FAIL, i18nc("Error message",  "Advice dismiss failed"));
        } else {
            //Execute the advice correction on all plugin
            int index = 0;
            while(index >= 0) {
                SKGInterfacePlugin* plugin = SKGMainPanel::getMainPanel()->getPluginByIndex(index);
                if(plugin) {
                    SKGError err = plugin->executeAdviceCorrection(uuid, solution);
                    if(!err || err.getReturnCode() != ERR_NOTIMPL) {
                        //The correction has been done or failed. This is the end.
                        index = -2;
                    }
                } else index = -2;
                ++index;
            }
        }
    }
}

AdviceSkroogoid::~AdviceSkroogoid()
{
}


#include "adviceskroogoid.moc"
