/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGERROR_H
#define SKGERROR_H
/** @file
 * This file defines classes SKGError and macros.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include <qstring.h>
#include <qobject.h>

#include "skgbasemodeler_export.h"
#include "skgdefine.h"

/**
 * To facilitate the error management
 */
#define IFKO(ERROR) \
    if (Q_UNLIKELY(ERROR))

/**
 * To facilitate the error management
 */
#define IFOK(ERROR) \
    if (Q_LIKELY(!ERROR))

/**
 * To facilitate the error management
 */
#define IFOKDO(ERROR, ACTION) \
    IFOK(ERROR) {ERROR = ACTION;}

/**
* This class manages errors
*/
class SKGBASEMODELER_EXPORT SKGError final : public QObject
{
    Q_OBJECT
    /**
     * Return code of the error
     */
    Q_PROPERTY(int returnCode READ getReturnCode WRITE setReturnCode)
    /**
     * Message of the error
     */
    Q_PROPERTY(QString message READ getMessage WRITE setMessage)
    /**
     * Action of the error
     */
    Q_PROPERTY(QString action READ getAction WRITE setAction)
    /**
     * To know if it is a success of the error
     */
    Q_PROPERTY(bool succeeded READ isSucceeded)
    /**
     * To know if it is a failure of the error
     */
    Q_PROPERTY(bool failed READ isFailed)
public:
    /**
    * Constructor
    */
    explicit SKGError();

    /**
    * Copy constructor
    * @param iError the error to copy
    */
    SKGError(const SKGError& iError);

    /**
    * Move constructor
    * @param iError the error to copy
    */
    SKGError(SKGError&& iError) noexcept;

    /**
    * Constructor
    * @param iRc the error code
    * @param iMessage the error message
    * @param iAction the error action. This is normally a recovery action (example: skg://file_save, http://google.com, ...)
    */
    SKGError(int iRc, const QString& iMessage, const QString& iAction = QString());

    /**
    * Destructor
    */
    ~SKGError();

    /**
    * Operator affectation
    * @param iError the error to copy
    */
    SKGError& operator= (const SKGError& iError);

    /**
    * To know if this is an error or not. Equivalent to @see isSucceeded
    * @return true or false
    */
    bool operator!() const;

    /**
    * To know if this is an error or not. Equivalent to @see isFailed
    * @return true or false
    */
    operator bool() const;

public Q_SLOTS:
    /**
    * To know if it is an error or not
    * @return true: It is an error
    *         false: It is not an error (it could be a warning)
    */
    bool isFailed() const;

    /**
    * To know if it is an error or not
    * @return true: It is not an error (it could be a warning)
    *         false: It is an error
    */
    bool isSucceeded() const;

    /**
    * To know if it is a warning or not
    * @return true: It is a warning
    *         false: It is not a warning
    */
    bool isWarning() const;

    /**
    * Return the return code associated to this error
    * @return 0 : It is not an error (SUCCEEDED)
    *         <0: It is just a warning (SUCCEEDED)
    *         >0: It is not error (FAILED)
    */
    int getReturnCode() const;

    /**
    * Set the return code associated to this error
    * @param iReturnCode the return code
    *         0 : It is not an error (SUCCEEDED)
    *         <0: It is just a warning (SUCCEEDED)
    *         >0: It is not error (FAILED)
    * @return itself to facilitate usage
    */
    SKGError& setReturnCode(int iReturnCode);

    /**
    * Set the message associated to this error
    * @param iMessage the message
    * @return itself to facilitate usage
    */
    SKGError& setMessage(const QString&   iMessage);

    /**
    * Return the message associated to this error
    * @return the message
    */
    QString getMessage() const;

    /**
    * Return the full message associated to this error
    * @return the full message
    */
    QString getFullMessage() const;

    /**
    * Return the full message with historical associated to this error
    * @return the full message
    */
    QString getFullMessageWithHistorical() const;

    /**
     * Return the size of the historical associated to this error
     * @return the size
     */
    int getHistoricalSize() const;

    /**
    * Set the action associated to this error
    * @param iAction the action
    * @return itself to facilitate usage
    */
    SKGError& setAction(const QString& iAction);

    /**
    * Return the action associated to this error
    * @return the action
    */
    QString getAction() const;

    /**
    * Add a new historical message to the current error.
    * @param iRc the error code
    * @param iMessage the error message
    * @param iAction the error action. This is normally a recovery action (example: skg://file_save, http://google.com, ...)
    * @return itself to facilitate usage
    */
    SKGError& addError(int iRc, const QString& iMessage, const QString& iAction = QString());

    /**
    * Add a new historical message to the current error.
    * @param iError the error
    * @return itself to facilitate usage
    */
    SKGError& addError(const SKGError& iError);

    /**
    * Return previous error associated to this SKGError in the historical
    * @return previous error (null if not exist)
    * WARNING: this pointer mustn't be deleted
    */
    SKGError* getPreviousError() const;

private:
    /**
    * the return code of the error
    *         0 : It is not an error (SUCCEEDED)
    *         <0: It is just a warning (SUCCEEDED)
    *         >0: It is not error (FAILED)
    */
    int m_rc;

    /**
     * the message of the error
     */
    QString m_message;

    /**
     * the action of the error
     */
    QString m_action;

    /**
     * the previous error on this branch
     */
    SKGError* m_previousError;
};
#endif  // SKGERROR_H
