/*
 *  Algebraic operations on the finite field GF(2^m)
 *
 *
 * 
 * 	Copyright (C) 2004-2006  Manuel Pancorbo Castro
 * 
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version 2
 *	of the License, or (at your option) any later version.
 * 
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 * 
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *	Manuel Pancorbo Castro <mpancorbo@gmail.com>
 * 
 * 
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#define _GFLIB_C
#include "gflib.h"

static gf_poly R /*, ci, fi, gi, ti*/;
static int GF_Init = 0;

/*unsigned long int  sqrsqr[256];*/  /** a^4 **/
/*
#ifdef FULLTABLES
static unsigned short int tab[256][256];
#else
static unsigned short int tab[16][16];
#endif

static unsigned short int tab[16][16];
*/

#if (defined FULLTABLES)  && (DIGIT_BIT==28)
gf_unit  sqr[1<<14];
#else
static unsigned short int  sqr[256];
#endif

static int tables_builded = 0;

static void build_tables(void)
{
	long int i, ii, jj;
	gf_unit c;
	#if (defined FULLTABLES)  && (DIGIT_BIT==28)
	long int lim = (1 << 14);
	#else
	long int lim = 256;
	#endif
	/*
	#ifdef FULLTABLES
	int lim = 256;
	#else
	int lim = 16;
	#endif
	*/
	#if 0
	int lim = 16;

	for(i = 0; i < lim; ++i){
		for(j = 0; j < lim; ++j){
			c = 0; ii = i; jj = j;
			while(ii){
				if(ii & 1) c ^= jj;
				ii >>= 1;
				jj <<= 1;
			}
			tab[i][j] = c;
			/*fprintf(stderr, "%x,", c);*/
		}
		/*fprintf(stderr, "\n");*/
	}
	#endif
	
	for(i = 0; i < lim; ++i){
		/** Sqr **/
		jj = 1, ii = i, c = 0;
		while(ii){
			if(ii & 1) c ^= jj;
			jj <<= 2;
			ii >>= 1;
		}
		sqr[i] = c;
	}
	#if 0
	for(i = 0; i < 256; ++i){
		c = sqr[i];
		sqrsqr[i] = (sqr[c >> 8] << 16) | sqr[c & 0xff] ;
	}
	#endif
	/*fprintf(stderr, "sqr: %d\n", sqr[255]);*/
}


/*int gf2m_Open(gf_poly *a)*/
gf_poly gf2m_Open()
{
	gf_poly a;
	
	if( (a = malloc(sizeof(mp_int))) == NULL ) return NULL;
	if(mp_init_size(a, 2*GF_M / DIGIT_BIT + 4)) return NULL;
	return a;
}

ERROR gf2m_Init(void)
/* Sets global 'R' with reduction trinomial */
/* Sets auxiliar variables */
{
	if(GF_Init) return 0;
	
	gfOpen(R);
	if(R == NULL) return 1;
	gf2m_Set(R, 1);
	mp_mul_2d(R, GF_M - GF_T, R);
	mp_add_d(R, 1, R);
	mp_mul_2d(R, GF_T, R);
	mp_add_d(R, 1, R);
	

	if(!tables_builded){
		build_tables();

		tables_builded = 1;
	}

	GF_Init = 1;
	return 0;
}

void gf2m_Quit(void)
{
	gfClear(R);
	
	GF_Init = 0;
}

int gf2m_Deg(gf_poly a)
{
	register int deg;
	register mp_digit c;
	int m = USED(a);
	
	assert(a != NULL);
	assert(gfExists(a));
	if(mp_iszero(a)) return -1;
	/*while(!DIGIT(a, m - 1)) m--;*/
	c = DIGIT(a, m-1);
	deg = (m-1) * DIGIT_BIT;
	while(c){
		++deg;
		c >>= 1;
	}
	return deg - 1; 

}

void gf2m_Clear(gf_poly a)
{	
	assert( a != NULL );
	assert(gfExists(a));
	gf2m_Zero(a); mp_clear(a);
	free(a);
}

void gf2m_Pack(gf_poly a, gfPoint b)
{
	memset(b, 0, GF_SIZE);
	if (gf_iszero(a)) return;
	gfPut(a, b + GF_SIZE - gfSize(a));

}

void gf2m_SmallAdd(gf_poly a, gf_unit b)
{
	assert( (a != NULL) );
	assert( gfExists(a) );
	
	if(!b) return;
	
	if(gf_iszero(a)){
		*(a->dp) ^= b;
		a->used++;
	}
	else{
		*(a->dp) ^= b;
		mp_clamp(a);
	}
}


#if (GF_T < DIGIT_BIT)
void gf2m_Reduce(gf_poly p)
	/* reduces p mod the irreducible trinomial X^191 + Y^9 + 1 */
{
	assert(GF_Init);
	assert(p != NULL);
	assert(gfExists(p));
	
	const int digit_offset = GF_M/DIGIT_BIT;
	const int bit_offset_M0 = GF_M % DIGIT_BIT;
	const int bit_offset_MT = (GF_M-GF_T) % DIGIT_BIT;
	register mp_digit val, *ptr, mask;
	
	ptr = &(p->dp[p->used-1]);
	
	while(p->used > digit_offset + 1){
		if((val = *ptr)){
			mask = (val >> bit_offset_MT);
			mask ^= (val >> bit_offset_M0);
			*(ptr - digit_offset) ^= mask;
			mask = (val << (DIGIT_BIT - bit_offset_MT));
			mask ^= (val << (DIGIT_BIT - bit_offset_M0));
			mask &= MP_MASK;
			*(ptr - digit_offset -1) ^= mask;
			*ptr = 0;
		}
		ptr--;
		p->used--;
	}
	
	if(p->used == digit_offset + 1){
		mask = ((1 << bit_offset_M0) - 1) ^ MP_MASK;
		if((val = *ptr & mask)){
			*ptr &= (mask ^ MP_MASK);
			mask = (val >> bit_offset_MT);
			mask ^= (val >> bit_offset_M0);
			*(ptr - digit_offset) ^= mask;
		}
	}
	while(!(*ptr) && p->used){ptr--, p->used--;}	
}

#elif (GF_T % DIGIT_BIT == 0)
void gf2m_Reduce(gf_poly p)
	/* reduces p mod the irreducible trinomial X^191 + Y^140 + 1 */
{
	assert(GF_Init);
	assert(p != NULL);
	assert(gfExists(p));

	const int digit_offset_M0 = GF_M/DIGIT_BIT;
	const int digit_offset_MT = (GF_M-GF_T) / DIGIT_BIT;
	const int bit_offset_M0 = GF_M % DIGIT_BIT;
	
	register mp_digit val, *ptr, mask;
	
	ptr = &(p->dp[p->used-1]);
	while(p->used > digit_offset_M0 + 1){
		if((val = *ptr)){
			mask = (val >> bit_offset_M0);
			*(ptr - digit_offset_M0) ^= mask;
			*(ptr - digit_offset_MT) ^= mask;
			mask = (val << (DIGIT_BIT - bit_offset_M0)) & MP_MASK;
			*(ptr - digit_offset_M0 -1) ^= mask;
			*(ptr - digit_offset_MT -1) ^= mask;
			*ptr = 0;
		}
		ptr--;
		p->used--;
	}
	if(p->used == digit_offset_M0 + 1){
		mask = ((1 << bit_offset_M0) - 1) ^ MP_MASK;
		if((val = *ptr & mask)){
			*ptr &= (mask ^ MP_MASK);
			mask = (val >> bit_offset_M0);
			*(ptr - digit_offset_M0) ^= mask;
			*(ptr - digit_offset_MT) ^= mask;
			
		}
	}
	while(!(*ptr) && p->used){ptr--, p->used--;}	
}
#elif (GF_M - GF_T > DIGIT_BIT)
void gf2m_Reduce(gf_poly p)
	/* reduces p mod the irreducible trinomial X^191 + Y^T + 1 */
{
	assert(GF_Init);
	assert(p != NULL);
	assert(gfExists(p));

	const int digit_offset_M0 = GF_M/DIGIT_BIT;
	const int digit_offset_MT = (GF_M-GF_T) / DIGIT_BIT;
	const int bit_offset_M0 = GF_M % DIGIT_BIT;
	const int bit_offset_MT = (GF_M-GF_T) % DIGIT_BIT;
	
	register mp_digit val, *ptr, mask;
	
	ptr = &(p->dp[p->used-1]);
	while(p->used > digit_offset_M0 + 1){
		if((val = *ptr)){
			mask = (val >> bit_offset_M0);
			*(ptr - digit_offset_M0) ^= mask;
			mask = (val >> bit_offset_MT);
			*(ptr - digit_offset_MT) ^= mask;
			mask = (val << (DIGIT_BIT - bit_offset_M0)) & MP_MASK;
			*(ptr - digit_offset_M0 -1) ^= mask;
			mask = (val << (DIGIT_BIT - bit_offset_MT)) & MP_MASK;
			*(ptr - digit_offset_MT -1) ^= mask;
			*ptr = 0;
		}
		ptr--;
		p->used--;
	}
	if(p->used == digit_offset_M0 + 1){
		mask = ((1 << bit_offset_M0) - 1) ^ MP_MASK;
		if((val = *ptr & mask)){
			*ptr &= (mask ^ MP_MASK);
			mask = (val >> bit_offset_M0);
			*(ptr - digit_offset_M0) ^= mask;
			mask = (val >> bit_offset_MT);
			*(ptr - digit_offset_MT) ^= mask;
			
		}
	}
	while(!(*ptr) && p->used){ptr--, p->used--;}	
}

#else
	#error "Unsupported trinomial"

#endif



mp_word gf2m_SmallMult(register gf_unit a, gf_unit b)
	/*** returns a*b ***/
{
	mp_word c = 0;
	register gf_unit bit = (gf_unit) 1 << (gf_unit) (DIGIT_BIT - 1);
	
	while(bit){
		c <<= 1;
		if(a & bit) c ^= b;
			
		bit >>= 1;
	}
	
	return c;
}


void _gf2m_Multiply (gf_poly s, /*const*/ gf_poly p, /*const*/ gf_poly q)
	/** Sets r = p*q (no reduction) **/
	/** It computes products of size DIGIT_BIT **/
{
	
	int deg_p, deg_q, sum;
	register int i, j, k;
	mp_word c;
	gf_poly r;
		
	assert( (s != NULL) && (p != NULL) && (q != NULL) );
	assert(gfExists(s) && gfExists(p) && gfExists(q));	
	
	
	deg_p = p->used;
	deg_q = q->used;
	sum = deg_p + deg_q - 1;
	
	gfOpen(r);
	gf2m_Zero(r);
	for(k = 0; k < sum; ++k){
		
		c = 0;
		i = k - deg_q + 1;
		for(i = ((i < 0)? 0:i), j = k - i; (i < deg_p) && (j >= 0); ++i, --j)
			c ^= gf2m_SmallMult(p->dp[i], q->dp[j]);
		
		r->dp[k] ^= (c & MP_MASK);
		r->dp[k+1] ^= (c >> (gf_unit)DIGIT_BIT);
		r->used += 2;
		/*mp_clamp(r);*/
	}
	r->used++; mp_clamp(r);
	gfCopy(s, r);
	gfClear(r);
}


void gf2m_Multiply (gf_poly r, /*const*/ gf_poly p, /*const*/ gf_poly q)
	/** Sets r = p*q (mod R) **/
{
	_gf2m_Multiply (r, p, q);
	gf2m_Reduce(r);
}

#if (defined FULLTABLES)  && (DIGIT_BIT==28)
void _gf2m_Square(gf_poly r, /*const */ gf_poly p)
	/* sets r = p^2 No reduction */
	/* only if DIGIT_BIT == 28 **/
{
	register int i;
	register gf_unit c;
	const gf_unit mask = (1 << 14) - 1;
	gf_poly s;
	
	assert( (r != NULL) && (p != NULL) );
	assert(gfExists(r) && gfExists(p));
	if(!tables_builded){
		build_tables();
		
		tables_builded = 1;
	}
	gfOpen(s);gfZero(s);
	for(i = 0; i < p->used; ++i){
		c = p->dp[i];
		s->dp[s->used++] = sqr[c & mask];
		s->dp[s->used++] = sqr[c >> 14];
	}
	s->used++; mp_clamp(s);
	gfCopy(r, s); gfClear(s);
}
#else
void _gf2m_Square(gf_poly r, /*const */ gf_poly p)
	/* sets r = p^2 No reduction */
{
	unsigned char buf[2*GF_SIZE];
	int deg;
	register int i;
	register gf_unit c;
	
	assert( (r != NULL) && (p != NULL) );
	assert(gfExists(r) && gfExists(p));
	if(!tables_builded){
		build_tables();
		
		tables_builded = 1;
	}
	
	memset(buf, 0, 2*GF_SIZE);
	deg = gfSize(p);
	gfPut(p, buf);
	
	gf2m_Zero(r);
	
	for(i = 0; i < deg; ++i){
		mp_mul_2d(r, 16, r);
		c = sqr[buf[i]];
		
		gf2m_SmallAdd(r, c);
		
	}
}

#endif

void gf2m_Square(gf_poly r, /*const */ gf_poly p)
	/* sets r = p^2 (mod R) */
{
	_gf2m_Square(r, p);
	gf2m_Reduce(r);
}


#ifdef GFLIB_TEST
/*** Old inversion algorithm ***/
ERROR gf2m_Divide2 (gf_poly b,/* const */ gf_poly p, /* const */ gf_poly a)
	/* sets b := (p / a) mod (x^GF_M + x^GF_T + 1) */
	/* warning: a, b and p must not overlap! */
	/* 'p' can be null. Then, it will be performed a^-1 mod R */
{
	assert (GF_Init != 0);	
	assert( (b != NULL) && (a != NULL) );
	assert(gfExists(b) && gfExists(a));
	assert ((b != a) && (p != a));
	
	if(mp_iszero(a)) return 1;
	
	register int j;
	gf_poly ci, fi, gi, ti;
	
	gfOpen(ci); gfOpen(fi); gfOpen(gi);
	gfOpen(ti);
	/* initialize b := p; ci := 0; fi := a; gi := x^GF_M + x^GF_T + 1: */
	if(p == NULL){ /** Interprets 'only inversion of a' **/
		gf2m_Set(b, 1);
	}
	else {
		gfCopy(b, p);
	}
	gf2m_Zero(ci);
	gf2m_Copy(fi, a), gf2m_Copy(gi, R);
	/*int count = 0;*/
	
	for(;;){
		if(mp_cmp_d(fi, 1) == MP_EQ){
			gfClear(ci), gfClear(fi);
			gfClear(gi), gfClear(ti);
			
			if (p != NULL) gfReduce(b);
			return 0;
		}
		if((j = gf2m_Deg(fi) - gf2m_Deg(gi)) < 0){
			j = -j;
			goto SWAP_FG; 
		}
SWAP_GF:
		/*++count;*/
		if(j){
			mp_mul_2d(gi, j, ti);
			gf2m_Add(fi, ti, fi);
			mp_mul_2d(ci, j, ti);
			
			gf2m_Add(b, ti, b);
		}
		else{
			gf2m_Add(fi, gi, fi);
			gf2m_Add(b, ci, b);
		}
	}
	for(;;){
		if(mp_cmp_d(gi, 1) == MP_EQ){
			if (p != NULL) gfReduce(ci);
			gf2m_Copy(b, ci);
			gfClear(ci), gfClear(fi);
			gfClear(gi), gfClear(ti);
			
			return 0;
		}
		if((j = gf2m_Deg(gi) - gf2m_Deg(fi)) < 0){
			j = -j;
			goto SWAP_GF; 
		}
		
SWAP_FG:
		/*++count;*/
		if(j){
			mp_mul_2d(fi, j, ti);
			gf2m_Add(gi, ti, gi);
			mp_mul_2d(b, j, ti);
			gf2m_Add(ci, ti, ci);
		}
		else{
			gf2m_Add(gi, fi, gi);
			gf2m_Add(ci, b, ci);
		}
	}
}

#endif

/*** New inversion algorithm (a bit faster than previous ***/
ERROR gf2m_Divide (gf_poly b,/* const */ gf_poly p, /* const */ gf_poly a)
	/* sets b := (p / a) mod (x^GF_M + x^GF_T + 1) */
	/* warning: a, b and p must not overlap! */
	/* 'p' can be null. Then, it will be performed a^-1 mod R */
{
	assert (GF_Init != 0);	
	assert( (b != NULL) && (a != NULL) );
	assert(gfExists(b) && gfExists(a));
	assert ((b != a) && (p != a));
	
	if(mp_iszero(a)) return 1;
	
	const int M_digit_offset = (GF_M - 1)/ DIGIT_BIT,
			  T_digit_offset = (GF_T - 1)/ DIGIT_BIT,
			  M_bit_offset = (GF_M - 1) % DIGIT_BIT,
			  T_bit_offset = (GF_T - 1) % DIGIT_BIT;
	const gf_unit lo = (1 << T_bit_offset);
	const gf_unit hi = (1 << M_bit_offset);
	int flag;
	register gf_poly u, v, g2, g1 = b /** Alias of b **/;
	
	gfOpen(g2); gfOpen(u); gfOpen(v);
	
	/* initialize g1 := p; g2 := 0; u := a; v := x^GF_M + x^GF_T + 1: */
	if(p == NULL){ /** Interprets 'only inversion of a' **/
		gf2m_Set(g1, 1);
	}
	else {
		gfCopy(g1, p);
	}
	gf2m_Zero(g2);
	gf2m_Copy(u, a), gf2m_Copy(v, R);
	/*int count = 0;*/

	while( (mp_cmp_d(u, 1) != MP_EQ) && (mp_cmp_d(v, 1) != MP_EQ)){
		while(!mp_isodd(u)){
			mp_div_2(u, u);
			flag = mp_isodd(g1);
			mp_div_2(g1, g1);
			if(flag){
				g1->dp[M_digit_offset] ^= hi;	
				g1->dp[T_digit_offset] ^= lo;
				g1->used = M_digit_offset + 1;
			}
		}
		while(!mp_isodd(v)){
			mp_div_2(v, v);
			flag = mp_isodd(g2);
			mp_div_2(g2, g2);
			if(flag){
				g2->dp[M_digit_offset] ^= hi;	
				g2->dp[T_digit_offset] ^= lo;
				g2->used = M_digit_offset + 1;
			}
		}
		if(mp_cmp_mag(u, v) == MP_GT){
			gf2m_Add(u, u, v);
			gf2m_Add(g1, g1, g2);
		}
		else{
			gf2m_Add(v, u, v);
			gf2m_Add(g2, g1, g2);
		}
	}
	if(mp_cmp_d(v, 1) == MP_EQ) gf2m_Copy(g1, g2);
	gfClear(g2), gfClear(u), gfClear(v);
	
	return 0;
	
}

void gf2m_SquareRoot (gf_poly p, gf_unit b)
	/* sets p := sqrt(b) = b^(2^(GF_M-1)) */
{
	register int i;
	gf_poly q;
	
	assert(p != NULL);
	assert(gfExists(p)); 
	assert (GF_Init != 0);
	if(!b){
		gf2m_Zero(p);
		return;
	}
	
	gfOpen(q);
	
	gf2m_Set(p, b);
	i = GF_M - 1;
	while (i) {
		gf2m_Square (q, p);
		gf2m_Square (p, q);
		i -= 2;
	}
	gfClear(q);
} /* gf2m_SquareRoot */

#ifndef TRACE_MASK
int gf2m_Trace(/*const*/ gf_poly p)
{
	int tr = 0;
	gf_poly t;

	assert (GF_Init != 0);
	assert( p != NULL );
	assert(gfExists(p)); 
	gfOpen(t);
	
	mp_div_2d(p, GF_TM0, t, NULL);
	
	tr ^= mp_isodd(t);
	#ifdef GF_TM1
	while(!mp_iszero(t)){
		mp_div_2d(t, GF_TM1, t, NULL);
		tr ^= mp_isodd(t);
	}
	#endif
	gfClear(t);
	return tr;
}  /* gf2m_Trace */



ERROR gf2m_QuadSolve (gf_poly p, /*const*/ gf_poly beta)
	/* sets p to a solution of p^2 + p = beta */
{
	register int i;
	
	assert (GF_Init != 0);
	assert( (beta != NULL) && (p != NULL) );
	assert(gfExists(beta) && gfExists(p));
	assert (p != beta);
	if (gfTrace (beta) != 0) {
		return 1; /* no solution */
	}

	gf2m_Copy (p, beta);
	for (i = 0; i < GF_M/2; i++) {
		gf2m_Square (p, p);
		gf2m_Square (p, p);
		gf2m_Add (p, p, beta);
	}
	return 0;
} /* gfQuadSolve */
#endif

void gf2m_Random(gf_poly a)
	/* Only for testing purposes */
{
	int i;
	assert( a != NULL);
	assert(gfExists(a)); 
	gf2m_Zero(a);
	for(i = 0; i < 192/DIGIT_BIT + 1; ++i){
		mp_mul_2d(a, DIGIT_BIT, a);
		mp_add_d(a, rand() & MP_MASK, a);
	}
	gf2m_Reduce(a);
}

void gf2m_Print (FILE *fOut, /*const*/ char *tag, /*const*/ gf_poly p)
	/* prints tag and the contents of p to file fOut */
{
	char buf[256];
	mp_tohex(p, buf);
	fprintf (fOut, "%s = %s\n", tag, buf);

}

#if defined GFLIB_TEST || defined TRACE_MASK

int gf2m_SlowTrace (/*const*/ gf_poly p)
	/* slowly evaluates to the trace of p (or an error code) */
{
	int i;
	gf_poly t;

	assert (GF_Init != 0);
	gfOpen(t);
	gf2m_Copy (t, p);
	for (i = 1; i < GF_M; i++) {
		gf2m_Square (t, t);
		gf2m_Add (t, t, p);
	}

	i = (t->used != 0);
	
	gfClear(t);
	return i;
} /* gfSlowTrace */

#endif

#ifdef GFLIB_TEST

main(int argc, char ** argv)
{

	int i, test_count
		,afail = 0
		,mfail = 0
		,dfail = 0
		,sfail = 0
		,ifail = 0
		,rfail = 0
		,tfail = 0
		,qfail = 0
		,gfail = 0
		,dvfail = 0
		;
	gf_poly f, g, h, x, y, z;
	gf_unit b;
	clock_t elapsed, elap_quad = 0L, elap_mul = 0, 
			elap_inv = 0, elap_inv2 = 0, elap_sqr = 0;
	ulong64 TOGGLE = (1LL << DIGIT_BIT) - 1;
	
	if(argc > 1){
		test_count = atoi(argv[1]);
	}
	else test_count = 10;
	
	printf("Bits por dígito %d\n", DIGIT_BIT);
	gfInit();
	gfPrint(stderr, "R", R);
	
	srand ((unsigned)(time(NULL) % 65521U));
	printf ("Executing %d field self tests.", test_count);
	/*printf("Parameters:\n Curve: %d\nField (2^%d)(2^%d)\n", LaCurva, GF_L, GF_K);*/
	printf("Bits: %d\nTrinomial: %d\n", GF_M, GF_T);
	gfOpen(f); gfOpen(g); gfOpen(h);
	gfOpen(x); gfOpen(y); gfOpen(z);
	elapsed = -clock ();
	
	for (i = 0; i < test_count; i++) {
		gfRandom (f);
		gfRandom (g);
		gfRandom (h);
		
		/* addition test: f+g = g+f */
		gfAdd (x, f, g); 
		gfAdd (y, g, f);
		
		/*gfPrint(stderr, "sumando 1:\n", f);
		gfPrint(stderr, "sumando 2:\n", g);
		gfPrint(stderr, "resultado:\n", x);*/
		if (!gfEqual (x, y)) {
			afail++;
			/* printf ("Addition test #%d failed!\n", i); */
		}
		
		/* multiplication test: f*g = g*f */
		/*mp_set(f, 1);
		mp_clamp(f);*/
		
		
		elap_mul -= clock();
		gf2m_Multiply (x, f, g);
		gf2m_Multiply (y, g, f);
		elap_mul += clock();
		if (!gfEqual (x, y)) {
			fprintf(stderr, "Tamaño mp_word = %d\n", sizeof(mp_word));
			gfPrint(stderr, "Verdadero", x);
			gfPrint(stderr, "Prueba   ", y);
			
			mfail++;
			/*fprintf(stderr, "Elementos f: %d\n\n", f->K);
			fprintf(stderr, "Elementos g: %d\n\n", g->K);*/
			/* printf ("Multiplication test #%d failed!\n", i); */
			/*fprintf(stderr, "Prueba de multiplicar: %lx x %lx = %llx\n",
			b = (1 << DIGIT_BIT ) - 1;
				1, b, gf2m_SmallMult(b, 1));*/
		}
		
		/* distribution test: f*(g+h) = f*g + f*h */
		elap_mul -= clock();
		gfMultiply (x, f, g);
		gfMultiply (y, f, h);
		elap_mul += clock();
		gfAdd (y, x, y);
		gfAdd (z, g, h);
		elap_mul -= clock();
		gfMultiply (x, f, z);
		elap_mul += clock();
		if (!gfEqual (x, y)) {
			dfail++;
			/*gfPrint(stderr, "f", f);
			gfPrint(stderr, "g", g);
			gfPrint(stderr, "h", h);
			gfPrint(stderr, "f*(g+h)", x);
			fprintf(stderr, "Elementos: %d\n\n", x->K);
			gfPrint(stderr, "f*g+f*h", y);
			fprintf(stderr, "Elementos: %d\n\n", y->K);
			return -1; */
			/* printf ("Distribution test #%d failed!\n", i); */
		}
		
		/* squaring test: f^2 = f*f */
		gfZero(x), gfZero(y);
		/*mp_set(f, 0x1234567);
		mp_mul_2d(f, 10, f);*/
		elap_sqr -= clock();
		_gf2m_Square (x, f);
		gfReduce(x);
		elap_sqr += clock();
		elap_mul -= clock();
		_gf2m_Multiply (y, f, f);
		gfReduce(y);
		elap_mul += clock();
		if (!gfEqual (x, y)) {
			sfail++;
			gfPrint(stderr, "f", f);
			gfPrint(stderr, "f^2", x);
			gfPrint(stderr, "f*f", y);
			fprintf(stderr, "Último dígito: %lx\n", y->dp[y->used-1]);
			fprintf(stderr, "Siguiente dígito: %lx\n", y->dp[y->used]);
			/*return -1;*/
			/* printf ("Squaring test #%d failed:\n", i); */
		}
		#if 0
		/** Division test **/
		mp_div_2d(f, GF_M/2, f, NULL);
		gf2m_div(g, f, x, y);
		gfPrint(stderr, "Dividendo", g);
		gfPrint(stderr, "Divisor", f);
		gfPrint(stderr, "Cociente", x);
		gfPrint(stderr, "Resto", y);
		gfMultiply(f, x, f);
		gfAdd(f, y, f);
		gfPrint(stderr, "Resultado", f);
			
		if(!gfEqual(f, g)){
			dvfail++;
			gfPrint(stderr, "Original", g);
			gfPrint(stderr, "Resultado", f);
		}
		#endif
		/* inversion test: g*(f/g) = f */
		if (!mp_iszero(g)) {
			/*gf2m_Set(g, 2);*/
			gfReduce(g); 
			elap_inv -= clock();
			gf2m_Divide (x, f, g); /* x = f/g */
			elap_inv += clock();
			elap_inv2 -= clock();
			gf2m_Divide2 (y, f, x); /* y = f/x = f*g/f = g */
			elap_inv2 += clock();
			/*gfReduce(f);*/
			/*fprintf(stderr, "Grado de f: %d\n", gfDeg(f));*/
			
			/*gfMultiply(z, g, x); *//* z= x*g =? f */
			if (!gfEqual (g, y)) {
			/*
				gfPrint(stderr, "Original", g);
				gfPrint(stderr, "Resultado", x);
				ifail++;
			*/
				printf ("Inversion test #%d failed!\n", i); 
			}
		}
		/* square root test: sqrt(b)^2 = b */
		b =  rand () & TOGGLE;
		gf2m_Zero(z);
		if (b) {
			gf2m_Set(z, b);
		} else {
			gf2m_Zero(z);
		}
		gfSquareRoot (y, b);
		gfSquare (x, y);
		
		if (!gfEqual (x, z)) {
			rfail++;
			gfPrint(stderr, "Original", z);
			gfPrint(stderr, "Raiz cuadrada", y);
			fprintf(stderr, "Grado: %d\n", gfDeg(y));
			gfPrint(stderr, "y*y", x);
			return -1;
			/* printf ("Square root test #%d failed!\n", i); */
		}
		
		/* trace test: slow tr(f) = tr(f) */
		if (gfTrace (f) != gfSlowTrace (f)) {
			tfail++;
			/* printf ("Trace test #%d failed!\n", i); */
		}
		
		/* quadratic equation solution test: x^2 + x = f (where tr(f) = 0)*/
		if (gfTrace (f) == 0) {
			/*fprintf(stderr, "Traza no nula\n");*/
			elap_quad -= clock ();
			gfQuadSolve (x, f);
			elap_quad += clock ();
			elap_sqr -= clock();
			gfSquare (y, x);
			elap_sqr += clock();
			gfAdd (y, y, x);
			if (!gfEqual (y, f)) {
				qfail++;
				/* printf ("Quadratic equation test #%d failed!\n", i); */
			}
		}
		
		
	
	}
	elapsed += clock ();
	
	gfQuit();
	                   
	
	printf (" done, elapsed time = %.2f s.\n", (float)elapsed/CLOCKS_PER_SEC);
	printf (" multiplication time = %.4f s.\n", (float)elap_mul/CLOCKS_PER_SEC/6.);
	printf (" squaring time = %.4f s.\n", (float)elap_sqr/CLOCKS_PER_SEC/2.);
	printf (" quad- solving time = %.4f s.\n", (float)elap_quad/CLOCKS_PER_SEC);
	printf (" inversion time = %.4f s.\n", (float)elap_inv/CLOCKS_PER_SEC);
	printf (" old inversion time = %.4f s.\n", (float)elap_inv2/CLOCKS_PER_SEC);
	if (gfail) printf ("---> %d degree checks failed <---\n", gfail);
	if (afail) printf ("---> %d additions failed <---\n", afail);
	if (mfail) printf ("---> %d multiplications failed <---\n", mfail);
	if (dfail) printf ("---> %d distributions failed <---\n", dfail);
	if (sfail) printf ("---> %d squarings failed <---\n", sfail);
	if (ifail) printf ("---> %d inversions failed <---\n", ifail);
	if (rfail) printf ("---> %d square roots failed <---\n", rfail);
	if (tfail) printf ("---> %d traces failed <---\n", tfail);
	/*if (qfail) printf ("---> %d quadratic equations failed <---\n", qfail);*/
	gfClear(x), gfClear(y), gfClear(z), gfClear(f), gfClear(g), gfClear(h); 
	return afail || mfail || dfail || dvfail || sfail || ifail || rfail || tfail || qfail;
}

#elif defined TRACE_MASK

main()
{
	gf_poly x;
	int tr, i;
	
	gfOpen(x);
	gf2m_Set(x, 1);
	
	gfInit();
	printf("Polinomio: x^%d + x^%d + 1\n", GF_M, GF_T);
	for(i = 0; i < GF_M; ++i){
		/*tr = gf2m_SlowTrace(x);*/
		if((tr = gf2m_SlowTrace(x)))
			printf("Elemento: %d\tTraza: %d\n", i, tr);
		mp_mul_2(x,x);
	}
	gfClear(x);
	gfQuit();
}
#endif

