type assoc =
    Non_assoc
  | Left
  | Right
  | Assoc

type op =
    { prec: int;
      assoc: assoc;
      printer: unit -> unit }

type expr =
    Atom of string
  | Unary of op * expr
  | Binary of op * expr * expr
  | Match of string * expr * string

let min_prec = 0

let prec_of {prec = prec} = max min_prec prec
let assoc_of {assoc = assoc} = assoc
let printer_of {printer = printer} = printer

let arg_prec side assoc prec =
  if assoc = side || assoc = Assoc then
    prec
  else
    prec + 1

let rec unparse_expr level expr =
  match expr with
    Atom str ->
      Format.print_string str
  | Unary(op, arg) ->
      let prec = prec_of op in
      if level > prec then
	parenthesize expr
      else begin
	printer_of op ();
	unparse_expr prec arg
      end
  | Binary(op, arg1, arg2) ->
      let prec = prec_of op in
      if level > prec then
	parenthesize expr
      else
	let assoc = assoc_of op in
	unparse_expr (arg_prec Left assoc prec) arg1;
	printer_of op ();
	unparse_expr (arg_prec Right assoc prec) arg2
  | Match(opener, expr, closer) ->
      Format.open_box (String.length opener);
      Format.print_string opener;
      unparse_expr min_prec expr;
      Format.print_string closer;
      Format.close_box()

and parenthesize expr =
  Format.open_box 2;
  Format.print_string "( ";
  unparse_expr min_prec expr;
  Format.print_string " )";
  Format.close_box()

let unparse expr =
  Format.open_box 0;
  unparse_expr min_prec expr;
  Format.close_box()
