%{

#include <stdio.h>
#include <stdlib.h>
#include "symbol.h"
#include "xmalloc.h"
#include "intern.h"
#include "parser.h"
#include "scanner.h"

/* Symbols for evaluating booleans */

struct symbol {
  int defined;
  int value;
};

void assign(symbol_t symbol, int value)
{
  if (!symbol->defined) {
    symbol->defined = 1;
    symbol->value = value;
  }
}

int fetch(symbol_t symbol)
{
  /* Should check defined field */
  return symbol->value;
}

symbol_t
make_symbol(const char *name)
{
  symbol_t symbol = xmalloc(sizeof(struct symbol));
  symbol->defined = 0;
  symbol->value = 0;
  return symbol;
}

/* Turn off printer */

static int am_muted = 0;

void
mute(int val)
{
  am_muted = val;
}

int
muted()
{
  return am_muted;
}

/* A printer that removes extra output so as to minimize the
   difference between the input and the output. */

static int blanks = 0;		/* Space character count */
static int nls = 0;		/* New line character count */

/* Print text and set blanks and nls appropriately */
void
print_text(const char *text)
{
  if (am_muted)
    return;
  for (;;) {
    int ch = *text++;
    if (!ch)
      return;
    if (ch == ' ')
      blanks++;
    else if (ch == '\n') {
      blanks = 0;		/* Drop spaces at line end */
      nls++;
    }
    else {
      if (nls > 2)
	nls = 2;	/* drop extra blank lines */
      for (; nls > 0; nls--) putc('\n', stdout);
      for (; blanks > 0; blanks--) putc(' ', stdout);
      putc(ch, stdout);
    }
  }
}

/* Maybe flush last newline */
void
print_end()
{
  if (nls > 0)
    putc('\n', stdout);
}

void
print_fresh_line()
{
  if (!nls)
    print_text("\n");
}

void
print_token()
{
  print_text(yytext);
}

/* For error reporting... */

static int lineno = 1;

%}

ID		[A-Za-z_][A-Za-z_0-9]*

%%
" "|"\t"        { print_token(); }
"\n"            { lineno++; print_token(); }
#[^\n]*		{ lineno++; return OTHER; }
"=="            { return EQ; }
"!="            { return NEQ; }
"!"             { return NOT; }
"&&"            { return AND; }
"^"             { return XOR; }
"||"            { return OR; }
";"             { return yytext[0]; }
"("             { return yytext[0]; }
")"             { return yytext[0]; }
"{"             { return yytext[0]; }
"}"             { return yytext[0]; }
if|IF           { return IF; }
true|TRUE       { return TRUTH; }
false|FALSE     { return FALSEHOOD; }
else|ELSE       { return ELSE; }
bool|BOOL       { return BOOL; }
{ID}		{ yylval.symbol = intern(yytext); return ID; }
.               { return OTHER; }

%%

static const char *filename = "-";
static char **argv;

int
setfile(char **args)
{
  if (!*args) {
    argv = args;
    yyin = stdin;
    return 0;
  }
  else {
    filename = *args;
    argv = args + 1;
    if (filename[0] == '-' && !filename[1])
      yyin = stdin;
    else
      yyin = fopen(filename, "r");
    if (yyin)
      return 0;
    perror(filename);
    return -1;
  }
}

int
yywrap()
{
  if (!*argv)
    return 1;
  else {
    filename = *argv;
    lineno = 1;
    argv += 1;
    if (filename[0] == '-' && !filename[1])
      yyin = stdin;
    else
      yyin = fopen(filename, "r");
    if (yyin)
      return 0;
    perror(filename);
    exit(1);
  }
}

int
yyerror(const char *msg)
{
  return fprintf(stderr, "%s:%d: %s at token %s\n",
                 filename, lineno, msg, yytext);
}
