(* Security context formulas *)
type 'atoms contexts =
    Ctx_true
  | Ctx_false
  | Ctx_types of 'atoms
  | Ctx_roles of 'atoms
  | Ctx_users of 'atoms
  | Ctx_not of 'atoms contexts
  | Ctx_and of 'atoms contexts * 'atoms contexts
  | Ctx_or of 'atoms contexts * 'atoms contexts
  | Ctx_imply of 'atoms contexts * 'atoms contexts
  | Ctx_iff of 'atoms contexts * 'atoms contexts

(* Action formulas *)
type 'atoms actions =
    Act_true
  | Act_false
  | Act_classes of 'atoms
  | Act_permissions of 'atoms
  | Act_not of 'atoms actions
  | Act_and of 'atoms actions * 'atoms actions
  | Act_or of 'atoms actions * 'atoms actions
  | Act_imply of 'atoms actions * 'atoms actions
  | Act_iff of 'atoms actions * 'atoms actions

(* Transition formulas *)
type 'atoms transitions =
    Tran_true
  | Tran_false
  | Tran_classes of 'atoms
  | Tran_permissions of 'atoms
  | Tran_types of 'atoms
  | Next_types of 'atoms
  | Same_types				(* t = t' *)
  | Tran_roles of 'atoms
  | Next_roles of 'atoms
  | Same_roles				(* r = r' *)
  | Tran_users of 'atoms
  | Next_users of 'atoms
  | Same_users				(* u = u' *)
  | Tran_not of 'atoms transitions
  | Tran_and of 'atoms transitions * 'atoms transitions
  | Tran_or of 'atoms transitions * 'atoms transitions
  | Tran_imply of 'atoms transitions * 'atoms transitions
  | Tran_iff of 'atoms transitions * 'atoms transitions

(* Diagrams *)
type 'atoms diagram =
    ('atoms contexts * 'atoms actions * bool) list * (* arrows *)
      'atoms contexts *			(* final state *)
      'atoms contexts * 'atoms actions	(* exceptions *)

(* An information flow policy labeled transition system *)
type 'atoms lts = {
    types : 'atoms;		        (* All security context types *)
    roles : 'atoms;		        (* All security context roles *)
    users : 'atoms;		        (* All security context users *)
    classes : 'atoms;			(* All action classes *)
    permissions : 'atoms;		(* All action permissions *)
    initial : 'atoms contexts;	        (* The initial security contexts *)
    transition : 'atoms transitions;	(* The transition relation *)
    specifications : 'atoms transitions list (* Specifications to be checked *)
      (* The specifications are generated from neverallow and the like. *)
  }

open Unparse

let class_var_name = "c"
let permission_var_name = "p"
let type_var_name = "t"
let role_var_name = "r"
let user_var_name = "u"
let next var_name = var_name ^ "'"
let next_type_var_name = (next type_var_name)
let next_role_var_name = (next role_var_name)
let next_user_var_name = (next user_var_name)

(* Unparser operators *)

let mk_true = Atom "TRUE"
let mk_false = Atom "FALSE"
let mk_atom str = Atom str

let comma_op =
  { prec = 0;
    assoc = Assoc;
    printer = fun () ->
      Format.print_string ",";
      Format.print_space()
  }

let mk_comma l r =
  Binary (comma_op, l, r)

let semicolon_op =
  { prec = 0;
    assoc = Assoc;
    printer = fun () ->
      Format.print_string ";";
      Format.print_space()
  }

let mk_semicolon l r =
  Binary (semicolon_op, l, r)

let break_op =
  { prec = 0;
    assoc = Assoc;
    printer = fun () ->
      Format.print_space()
  }

let mk_break l r =
  Binary (break_op, l, r)

let imply_op =
  { prec = 1;
    assoc = Right;
    printer = fun () ->
      Format.print_space();
      Format.print_string "-> "
  }

let mk_imply l r =
  Binary (imply_op, l, r)

let iff_op =
  { prec = 2;
    assoc = Non_assoc;			(* check this *)
    printer = fun () ->
      Format.print_space();
      Format.print_string "<-> "
  }

let mk_iff l r =
  Binary (iff_op, l, r)

let or_op =
  { prec = 3;
    assoc = Assoc;
    printer = fun () ->
      Format.print_space();
      Format.print_string "| ";
  }

let mk_or l r =
  if l = mk_false then
    r
  else if r = mk_false then
    l
  else
    Binary (or_op, l, r)

let and_op =
  { prec = 4;
    assoc = Assoc;
    printer = fun () ->
      Format.print_space();
      Format.print_string "& "
  }

let mk_and l r =
  if l = mk_true then
    r
  else if r = mk_true then
    l
  else
    Binary (and_op, l, r)

(* Note the low precedence of not.  This is how NuSMV does it. *)
let not_op =
  { prec = 5;
    assoc = Right;
    printer = fun () ->
      Format.print_string "!"
  }

let mk_not r =
  Unary (not_op, r)

let eq_op =
  { prec = 7;
    assoc = Non_assoc;
    printer = fun () ->
      Format.print_string " = "
  }

let mk_eq l r =
  Binary(eq_op, Atom l, Atom r)

let neq_op =
  { prec = 7;
    assoc = Non_assoc;
    printer = fun () ->
      Format.print_string " != "
  }

let mk_neq l r =
  Binary(neq_op, Atom l, Atom r)

let in_op =
  { prec = 8;
    assoc = Non_assoc;
    printer = fun () ->
      Format.print_string " : "
  }

let not_in_op =
  { prec = 8;
    assoc = Non_assoc;
    printer = fun () ->
      Format.print_string " !: "
  }

let is_length_one l =
  match l with
    [] -> false
  | _ :: l' -> l' = []

let mk_set s =
  let rec loop u s =
    match s with
      [] -> u
    | e :: s ->
	loop (mk_comma u (Atom e)) s in
  match s with
    [] -> Atom "{}"
  | e :: s ->
      Match ("{", loop (Atom e) s, "}")

let mk_in l r =
  if r = [] then
    mk_false
  else if is_length_one r then
    mk_eq l (List.hd r)
  else
    Binary (in_op, Atom l, mk_set r)

let mk_not_in l r =
  if r = [] then
    mk_true
  else if is_length_one r then
    mk_neq l (List.hd r)
  else
    Binary (not_in_op, Atom l, mk_set r)

let rec unparse_contexts ctx =
  match ctx with
    Ctx_true -> mk_true
  | Ctx_false -> mk_false
  | Ctx_types set -> mk_in type_var_name set
  | Ctx_roles set -> mk_in role_var_name set
  | Ctx_users set -> mk_in user_var_name set
  | Ctx_not (Ctx_not ctx) -> unparse_contexts ctx
  | Ctx_not (Ctx_true) -> mk_false
  | Ctx_not (Ctx_false) -> mk_true
  | Ctx_not (Ctx_types set) -> mk_not_in type_var_name set
  | Ctx_not (Ctx_roles set) -> mk_not_in role_var_name set
  | Ctx_not (Ctx_users set) -> mk_not_in user_var_name set
  | Ctx_not arg -> mk_not (unparse_contexts arg)
  | Ctx_and (arg1, arg2) ->
      mk_and (unparse_contexts arg1) (unparse_contexts arg2)
  | Ctx_or (arg1, arg2) ->
      mk_or (unparse_contexts arg1) (unparse_contexts arg2)
  | Ctx_imply (arg1, arg2) ->
      mk_imply (unparse_contexts arg1) (unparse_contexts arg2)
  | Ctx_iff (arg1, arg2) ->
      mk_iff (unparse_contexts arg1) (unparse_contexts arg2)

let print_contexts ctx =
  unparse (unparse_contexts ctx)

let rec unparse_actions act =
  match act with
    Act_true -> mk_true
  | Act_false -> mk_false
  | Act_classes set -> mk_in class_var_name set
  | Act_permissions set -> mk_in permission_var_name set
  | Act_not (Act_not act) -> unparse_actions act
  | Act_not (Act_true) -> mk_false
  | Act_not (Act_false) -> mk_true
  | Act_not (Act_classes set) -> mk_not_in class_var_name set
  | Act_not (Act_permissions set) -> mk_not_in permission_var_name set
  | Act_not arg -> mk_not (unparse_actions arg)
  | Act_and (arg1, arg2) ->
      mk_and (unparse_actions arg1) (unparse_actions arg2)
  | Act_or (arg1, arg2) ->
      mk_or (unparse_actions arg1) (unparse_actions arg2)
  | Act_imply (arg1, arg2) ->
      mk_imply (unparse_actions arg1) (unparse_actions arg2)
  | Act_iff (arg1, arg2) ->
      mk_iff (unparse_actions arg1) (unparse_actions arg2)

let print_actions act =
  unparse (unparse_actions act)

let rec unparse_transitions tran =
  match tran with
    Tran_true -> mk_true
  | Tran_false -> mk_false
  | Tran_classes set -> mk_in class_var_name set
  | Tran_permissions set -> mk_in permission_var_name set
  | Tran_types set -> mk_in type_var_name set
  | Next_types set -> mk_in next_type_var_name set
  | Same_types -> mk_eq type_var_name next_type_var_name
  | Tran_roles set -> mk_in role_var_name set
  | Next_roles set -> mk_in next_role_var_name set
  | Same_roles -> mk_eq role_var_name next_role_var_name
  | Tran_users set -> mk_in user_var_name set
  | Next_users set -> mk_in next_user_var_name set
  | Same_users -> mk_eq user_var_name next_user_var_name
  | Tran_not (Tran_not tran) -> unparse_transitions tran
  | Tran_not (Tran_true) -> mk_false
  | Tran_not (Tran_false) -> mk_true
  | Tran_not (Tran_classes set) -> mk_not_in class_var_name set
  | Tran_not (Tran_permissions set) -> mk_not_in permission_var_name set
  | Tran_not (Tran_types set) -> mk_not_in type_var_name set
  | Tran_not (Next_types set) -> mk_not_in next_type_var_name set
  | Tran_not (Same_types) -> mk_neq type_var_name next_type_var_name
  | Tran_not (Tran_roles set) -> mk_not_in role_var_name set
  | Tran_not (Next_roles set) -> mk_not_in next_role_var_name set
  | Tran_not (Same_roles) -> mk_neq role_var_name next_role_var_name
  | Tran_not (Tran_users set) -> mk_not_in user_var_name set
  | Tran_not (Next_users set) -> mk_not_in next_user_var_name set
  | Tran_not (Same_users) -> mk_neq user_var_name next_user_var_name
  | Tran_not arg -> mk_not (unparse_transitions arg)
  | Tran_and (arg1, arg2) ->
      mk_and (unparse_transitions arg1) (unparse_transitions arg2)
  | Tran_or (arg1, arg2) ->
      mk_or (unparse_transitions arg1) (unparse_transitions arg2)
  | Tran_imply (arg1, arg2) ->
      mk_imply (unparse_transitions arg1) (unparse_transitions arg2)
  | Tran_iff (arg1, arg2) ->
      mk_iff (unparse_transitions arg1) (unparse_transitions arg2)

let print_transitions tran =
  unparse (unparse_transitions tran)

let unparse_arrow (ctx, act, more) =
  let ctx = unparse_contexts ctx in
  let act = unparse_actions act in
  let act =
    if more then
      Match ("", act, "+")
    else
      act in
  mk_comma ctx act

let unparse_arrows s =
  let rec loop u s =
    match s with
      [] -> u
    | e :: s ->
	loop (mk_semicolon u (unparse_arrow e)) s in
  match s with
    [] -> Atom "[]"
  | e :: s ->
      Match ("[", loop (unparse_arrow e) s, "]")

let unparse_exceptions (last, ex_ctx, ex_act) =
  let last = unparse_contexts last in
  match ex_ctx, ex_act with
    Ctx_false, Act_false -> last
  | _ ->
      let ex_ctx = unparse_contexts ex_ctx in
      let ex_act = unparse_actions ex_act in
      mk_break last
	(Match ("[", mk_semicolon ex_ctx ex_act, "]"))

let unparse_diagram (arrows, last, ex_ctx, ex_act) =
  mk_break
    (unparse_arrows arrows)
    (unparse_exceptions (last, ex_ctx, ex_act))

let print_diagram diagram =
  unparse (unparse_diagram diagram)

let emit_section title =
  Format.print_newline();
  Format.print_string "# ";
  Format.print_string title;
  Format.print_newline()

let print_section_start name =
  Format.print_newline();
  Format.print_string " ";
  Format.print_string name;
  Format.print_newline();
  Format.print_newline();
  Format.print_string "  "

let emit_state_var var_name var_type =
  Format.open_box 2;
  Format.print_string " STATE ";
  Format.print_string var_name;
  Format.print_string ":";
  Format.print_space();
  unparse (mk_set var_type);
  Format.close_box();
  Format.print_newline()

let emit_action_var var_name var_type =
  Format.open_box 2;
  Format.print_string " ACTION ";
  Format.print_string var_name;
  Format.print_string ":";
  Format.print_space();
  unparse (mk_set var_type);
  Format.close_box();
  Format.print_newline()

let emit_decls
    {types = types;
     roles = roles;
     users = users;
     classes = classes;
     permissions = permissions} =
  emit_state_var type_var_name types;
  emit_state_var role_var_name roles;
  emit_state_var user_var_name users;
  emit_action_var class_var_name classes;
  emit_action_var permission_var_name permissions

let emit_init contexts =
  print_section_start "INIT";
  print_contexts contexts;
  Format.print_newline()

let emit_trans transitions =
  print_section_start "TRANS";
  print_transitions transitions;
  Format.print_newline()

let emit_spec transitions =
  print_section_start "SPEC";
  print_transitions transitions;
  Format.print_newline()

let print_lts lts =
  emit_decls lts;
  emit_section "Initial States";
  emit_init lts.initial;
  emit_section "Transition Relation";
  emit_trans lts.transition;
  let specs = lts.specifications in
  if specs <> [] then
    begin
      emit_section "Specifications";
      List.iter emit_spec specs
    end
