/*
  Converts LTS and diagrams into SMV.
  Copyright (C) 2005 The MITRE Corporation

  Author: John D. Ramsdell

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <obstack.h>
#include <slat/xmalloc.h>
#include <slat/symbol.h>
#include <slat/intern.h>
#include <slat/formula.h>
#include <slat/action.h>
#include "gensmv.h"

extern int errno;
extern int yyparse(void);

/* An implementation of a symbol */

/* Allocation using obstack */

#define obstack_chunk_alloc xmalloc
#define obstack_chunk_free free

static struct obstack stack[1];

void
symbol_init(void)
{
  obstack_init(stack);
}

struct symbol
{
  const char sym[1];
};

symbol_t
make_symbol(const char *name)
{
  return (symbol_t)obstack_copy0(stack, name, strlen(name));
}

const char *
symbol_name(symbol_t symbol)
{
  return symbol->sym;
}

/* Parser actions */

static lts_t the_lts;

void
parsed_lts(lts_t lts)
{
  the_lts = lts;
}

static diagram_list_t the_diagrams;

void
parsed_diagrams(diagram_list_t diagrams)
{
  the_diagrams = diagrams;
}

static tran_t the_tran;

void
parsed_transition(tran_t tran)
{
  the_tran = tran;
}

/* File opening and reading */

static int
lts2smv(char *output, char *lts, char *diagrams)
{
  intern_init();
  symbol_init();
  formula_stack(stack);

  if (lts) {
    if (setfile(!strcmp("-", lts) ? 0 : lts) == -1) {
      fprintf(stderr, "Cannot open %s\n", lts);
      return 1;
    }
    int i = yyparse();
    if (i)
      return i;
    if (!the_lts) {
      fprintf(stderr, "Failed to find LTS in input file %s\n", lts);
      return 1;
    }
  }

  if (diagrams) {
    if (setfile(!strcmp("-", diagrams) ? 0 : diagrams) == -1) {
      fprintf(stderr, "Cannot open %s\n", diagrams);
      return 1;
    }
    int i = yyparse();
    if (i)
      return i;
    if (!the_diagrams) {
      fprintf(stderr, "Failed to find diagrams in input file %s\n", diagrams);
      return 1;
    }
  }

  intern_free();

  if (output) {
    if (!freopen(output, "w", stdout)) {
      char *msg = strerror(errno);
      fprintf(stderr, "Failed while opening %s: %s\n", output, msg);
      return 1;
    }
  }

  return gensmv(the_lts, the_diagrams);
}

/* Command line argument handling. */

#ifdef PACKAGE
const char package[] = PACKAGE;
#else
const char *package = (const char *)0;
#endif

#ifdef VERSION
const char version[] = VERSION;
#else
const char version[] = "Version information not available";
#endif

static void
print_version(const char *program)
{
  if (package)
    program = package;
  fprintf(stderr, "Package: %s %s\n", program, version);
}

static void
usage(const char *prog)
{
  fprintf(stderr,
	  "Usage: %s [options] [lts [diagrams]]\n"
	  "Options:\n"
	  "  -o file -- output to file (default is standard output)\n"
	  "  -v      -- print version information\n"
	  "  -h      -- print this message\n"
	  "Use - as a file name to specify standard input\n"
	  "Use the empty string for the lts to read only the diagrams\n",
	  prog);
}

int
main(int argc, char **argv)
{
  extern char *optarg;
  extern int optind;

  char *output = 0;
  char *lts = 0;
  char *diagrams = 0;

  for (;;) {
    int c = getopt(argc, argv, "o:vh");
    if (c == -1)
      break;
    switch (c) {
    case 'o':
      output = optarg;
      break;
    case 'v':
      print_version(argv[0]);
      return 0;
    case 'h':
      usage(argv[0]);
      return 0;
    default:
      usage(argv[0]);
      return 1;
    }
  }

  switch (argc - optind) {
  case 2:
    diagrams = argv[optind + 1];
    /* fall thru okay */
  case 1:
    lts = argv[optind];
    /* fall thru okay */
  case 0:
    break;
  default:
    fprintf(stderr, "Bad arg count\n");
    usage(argv[0]);
    return 1;
  }

  if (diagrams) {
    if (lts && !strcmp("", lts))
      lts = 0;
    if (lts && !strcmp(lts, diagrams)) {
      fprintf(stderr, "Both input files are the same\n");
      return 1;
    }
  }
  else if (!lts)
    lts = "-";

  return lts2smv(output, lts, diagrams);
}
