/*
 * The Sleuth Kit
 *
 * $Date: 2005/09/02 23:34:04 $
 *
 * Brian Carrier [carrier@sleuthkit.org]
 * Copyright (c) 2004-2005 Brian Carrier.  All rights reserved
 *
 * gpt: GUID Partition Tables 
 *
 *
 * This software is distributed under the Common Public License 1.0
 *
 */

#include "mm_tools.h"
#include "gpt.h"
#include "dos.h"


/* 
 * Process the partition table at the sector address 
 * 
 * It is loaded into the internal sorted list 
 */
static uint8_t
gpt_load_table(MM_INFO * mm, uint8_t test)
{
    gpt_head head;
    gpt_entry *ent;
    dos_sect dos_part;
    unsigned int i, a;
    uint32_t ent_size;
    char *safe_str, *head_str, *tab_str, *ent_buf;
    if (verbose)
	fprintf(stderr, "gpt_load_table: Sector: %" PRIuDADDR "\n",
		mm->sect_offset);

    if (mm_read_block_nobuf
	(mm, (char *) &dos_part, sizeof(dos_part), mm->sect_offset)
	!= sizeof(dos_part)) {
	if (test) {
	    fprintf(stderr,
		    "Error reading DOS safety partition tablein Sector: %"
		    PRIuDADDR "\n", mm->sect_offset);
	    return 1;
	}
	else {
	    error("Error reading DOS safety partition tablein Sector: %"
		  PRIuDADDR "\n", mm->sect_offset);
	}
    }

    /* Sanity Check */
    if (guessu16(mm, dos_part.magic, DOS_MAGIC)) {
	if (test)
	    return 1;
	else
	    error("Missing DOS safety partition (invalid magic) (Sector: %"
		  PRIuDADDR ")\n", mm->sect_offset);
    }

    if (dos_part.ptable[0].ptype != GPT_DOS_TYPE) {
	if (test)
	    return 1;
	else
	    error
		("Missing DOS safety partition (invalid type in table: %d)",
		 dos_part.ptable[0].ptype);
    }
    safe_str = mymalloc(16);
    snprintf(safe_str, 16, "Safety Table");
    mm_part_add(mm, (DADDR_T) 0, (DADDR_T) 1, MM_TYPE_DESC, safe_str, -1,
		-1);


    /* Read the GPT header */
    if (mm_read_block_nobuf
	(mm, (char *) &head, sizeof(head),
	 mm->sect_offset + 1) != sizeof(head)) {
	if (test) {
	    error("Error reading GPT Header structure in Sector: %"
		  PRIuDADDR "\n", mm->sect_offset + 1);
	    return 1;
	}
	else {
	    error("Error reading GPT Header structure in Sector: %"
		  PRIuDADDR "\n", mm->sect_offset + 1);
	}
    }


    if (getu64(mm, &head.signature) != GPT_HEAD_SIG) {
	if (test)
	    return 1;
	else
	    error("Invalid signature in GPT Header: %" PRIx64 "\n",
		  getu64(mm, &head.signature));
    }

    head_str = mymalloc(16);
    snprintf(head_str, 16, "GPT Header");
    mm_part_add(mm, (DADDR_T) 1,
		(DADDR_T) ((getu32(mm, &head.head_size_b) + 511) / 512),
		MM_TYPE_DESC, head_str, -1, -1);

    /* Allocate a buffer for each table entry */
    ent_size = getu32(mm, &head.tab_size_b);
    if (ent_size < sizeof(gpt_entry)) {
	if (test)
	    return 1;
	else
	    error("Header reports partition entry size of %" PRIu32 "\n",
		  ent_size);
    }

    tab_str = mymalloc(20);
    snprintf(tab_str, 20, "Partition Table");
    mm_part_add(mm, (DADDR_T) getu64(mm, &head.tab_start_lba),
		(DADDR_T) ((ent_size * getu32(mm, &head.tab_num_ent) +
			    511) / 512), MM_TYPE_DESC, tab_str, -1, -1);


    /* Process the partition table */
    ent_buf = mymalloc(mm->block_size);

    i = 0;
    for (a = 0; i < getu32(mm, &head.tab_num_ent); a++) {
	char *name;

	/* Read a sector */
	if (mm_read_block_nobuf(mm, ent_buf, mm->block_size,
				getu64(mm,
				       &head.tab_start_lba) + a) !=
	    mm->block_size) {
	    if (test) {
		fprintf(stderr,
			"Error reading GPT partition table sector : %"
			PRIuDADDR "\n", getu64(mm,
					       &head.tab_start_lba) + a);
		return 1;
	    }
	    else {
		error("Error reading GPT partition table sector : %"
		      PRIuDADDR "\n", getu64(mm, &head.tab_start_lba) + a);
	    }
	}

	/* Process the sector */
	ent = (gpt_entry *) ent_buf;
	for (; (uintptr_t) ent < (uintptr_t) ent_buf + mm->block_size &&
	     i < getu32(mm, &head.tab_num_ent); ent++ && i++) {

	    if (verbose)
		fprintf(stderr,
			"gpt_load: %d  Starting Sector: %" PRIu64
			"  End: %" PRIu64 " Flag: %" PRIx64 "\n", i,
			getu64(mm, ent->start_lba), getu64(mm,
							   ent->end_lba),
			getu64(mm, ent->flags));


	    if (getu64(mm, ent->start_lba) == 0)
		continue;


	    name = mymalloc(72);
	    uni2ascii((char *) ent->name, 72, name, 72);
	    mm_part_add(mm, (DADDR_T) getu64(mm, ent->start_lba),
			(DADDR_T) (getu64(mm, ent->end_lba) -
				   getu64(mm, ent->start_lba) + 1),
			MM_TYPE_VOL, name, -1, i);
	}
    }

    return 0;
}


/* 
 * Walk the partitions that have already been loaded during _open
 *
 */
void
gpt_part_walk(MM_INFO * mm, PNUM_T start, PNUM_T last, int flags,
	      MM_PART_WALK_FN action, char *ptr)
{
    MM_PART *part;
    unsigned int cnt = 0;

    if (start < mm->first_part || start > mm->last_part)
	error("Invalid starting partition: %" PRIuPNUM "", start);

    if (last < mm->first_part || last > mm->last_part)
	error("Invalid ending partition: %" PRIuPNUM "", last);

    part = mm->part_list;
    while ((part != NULL) && (cnt <= last)) {

	if (cnt >= start)
	    action(mm, cnt, part, 0, ptr);

	part = part->next;
	cnt++;
    }

    return;
}

void
gpt_close(MM_INFO * mm)
{
    mm_part_free(mm);
    free(mm);
}

MM_INFO *
gpt_open(IMG_INFO * img_info, DADDR_T offset, uint8_t test)
{
    MM_INFO *mm = (MM_INFO *) mymalloc(sizeof(*mm));

    mm->img_info = img_info;
    mm->mmtype = MM_GPT;
    mm->str_type = "GUID Partition Table";

    /* If an offset was given, then use that too */
    mm->sect_offset = offset + GPT_PART_OFFSET;

    /* inititialize settings */
    mm->part_list = NULL;
    mm->first_part = mm->last_part = 0;
    mm->endian = 0;
    mm->dev_bsize = 512;
    mm->block_size = 512;

    /* Assign functions */
    mm->part_walk = gpt_part_walk;
    mm->close = gpt_close;

    /* Load the partitions into the sorted list */
    if (gpt_load_table(mm, test)) {
	gpt_close(mm);
	return NULL;
    }

    /* fill in the sorted list with the 'unknown' values */
    mm_part_unused(mm);

    return mm;
}
