%  vwdraw.sl: this file is part of the VWhere SLgtk guilet {{{
%
%  Copyright (c) 2003-2005 Massachusetts Institute of Technology
%  Copyright (C) 2002 Michael S. Noble <mnoble@space.mit.edu>
%
% }}}

% Front matter {{{
private variable FilterDescriptor = struct {
	plotd,			% GtkPlotDescriptor
	zoom,			% fractional zoom
	hshift, vshift,		% fractional shift left/right/up/down
	regions,		% list of regions applied to plot
	drawc,			% drawing context
	prefs,			% Plot-specific preferences
	x,y			% x,y vectors
};

private variable DrawContext = struct {
	drawable,		% Gdk window to be drawn upon
	gc,			% graphics context for drawing
	x,y,			% (x,y) = coordinates of last button press
	motion_id,		% sig handlers on parent widget of drawable
	bpress_id,
	expose_id,
	destroy
};

% }}}

% Polygon drawing functions {{{
private define fd_drawc_destroy(filtd)
{
   if (filtd.drawc != NULL) {
	variable pd = filtd.plotd;
	if (filtd.drawc.motion_id != NULL)
	   g_signal_handler_disconnect(pd.canvas,filtd.drawc.motion_id);
	if (filtd.drawc.bpress_id != NULL)
	   g_signal_handler_disconnect(pd.canvas,filtd.drawc.bpress_id);
	filtd.drawc = NULL;
   }
}

private define poly_refresh_segment(canvas, event, poly)
{
   % Erase previous segment
   variable dc = poly.filtd.drawc;
   gdk_draw_line(dc.drawable, dc.gc, poly.x[-1], poly.y[-1], dc.x, dc.y);

   % Draw new segment [avoid get_pointer() trip to X server, use event x,y]
   gdk_draw_line(dc.drawable, dc.gc, poly.x[-1], poly.y[-1], event.x, event.y);
   dc.x = event.x;
   dc.y = event.y;

   return TRUE;					% mark event as handled
}

private define poly_next_click(event, poly)
{
   variable canvas = poly.filtd.plotd.canvas;

   switch (event.button)
   { case 1:

	poly.x = [poly.x, event.x];
	poly.y = [poly.y, event.y];
	return TRUE;				% mark event as handled
   }
   { case 2:
 
 	if (length(poly.x) > 2) {		% exclude vacuous polygons

	   variable i, x1, x2, y1, y2;

	   poly.x = typecast(poly.x,Double_Type);
	   poly.y = typecast(poly.y,Double_Type);

	   % Poly now closed: convert pixel-space coords to canvas-space ...
	   for (i=0; i < length(poly.x); i++) {
	      gtk_plot_canvas_get_position(canvas,
				int(poly.x[i]),int(poly.y[i]),&x1,&y1);
	      poly.x[i] = x1;
	      poly.y[i] = y1;
	   }

	   poly.widget = _gtk_plot_canvas_put_polygon(canvas, poly.x, poly.y,
			GTK_PLOT_LINE_SOLID, 0, gdk_blue,  NULL, FALSE);

	   % ... define its bounding box, and add poly to the plot region set
	   region_set_bbox(poly);
	   region_list_append(poly.filtd.regions,poly);
	}
   }

   % Both Mouse2 and Mouse3 terminate polygon drawing
   fd_drawc_destroy(poly.filtd);
   gtk_plot_canvas_refresh(canvas);
   return FALSE;				% mark event as unhandled
}

private define poly_expose(canvas, event, poly, drawc)
{
   g_signal_handler_disconnect(canvas,drawc.expose_id);

   % Now we may safely begin drawing/redrawing polygon segment(s)
   drawc.bpress_id = g_signal_connect_swapped (canvas, "button_press_event",
					&poly_next_click, poly);

   drawc.motion_id = g_signal_connect (canvas, "motion_notify_event",
					&poly_refresh_segment, poly);

   return FALSE;		% give next expose handler a chance, too
}

private define poly_first_click(event, vwd)
{
   variable x,y, poly = region_new(SHAPE_POLYGON,vwd.filtd,NULL);
   variable canvas = vwd.filtd.plotd.canvas, dc = @DrawContext;

   dc.drawable = gtk_widget_get_window(canvas);
   dc.gc = gdk_gc_new(dc.drawable);
   gdk_gc_set_function(dc.gc,GDK_INVERT);

   % Intercept next expose event, for proper segment drawing on slow X servers
   dc.expose_id = g_signal_connect( canvas, "expose_event",
					&poly_expose, poly, dc);

   poly.x = [event.x];		% Convenient to keep poly coords in pixel
   poly.y = [event.y];		% space until poly is fully drawn
   dc.x = event.x;
   dc.y = event.y;
   dc.destroy = &fd_drawc_destroy;
   poly.filtd.drawc = dc;
}
% }}}

% Canvas {{{

private define canvas_rect_drag(canvas, x1, y1, x2, y2, vwd)
{
   % Don't add single points, lines, or other uselessly tiny regions
   if (orelse {abs(x1 - x2) < 0.005} {abs(y1 - y2) < 0.005} )
	return;

   % Ensure that (x1,y1) is top left, (x2,y2) is bottom right (canvas coords)
   variable t;
   if (x1 > x2) { t = x2; x2 = x1; x1 = t; }
   if (y1 > y2) { t = y2; y2 = y1; y1 = t; }

   variable regwidget, type = vwd.tbox.curr;
   switch(type)
	{ case SHAPE_RECTANGLE:

		regwidget = gtk_plot_canvas_put_rectangle(canvas,x1,y1,x2,y2,
			GTK_PLOT_LINE_SOLID, 0, gdk_blue, NULL,
			GTK_PLOT_BORDER_LINE, FALSE);
	}
	{ case SHAPE_ELLIPSE:

		regwidget = gtk_plot_canvas_put_ellipse(canvas,x1,y1,x2,y2,
			GTK_PLOT_LINE_SOLID, 0, gdk_blue, NULL, FALSE);
	}
	{ case TB_ZOOM_FIT:

		variable new_zoom = zoom_in(vwd.filtd.zoom);
		if (new_zoom != vwd.filtd.zoom) {

		   variable plotd = vwd.filtd.plotd;
		   (x1, y1) = _gtk_plot_transform(plotd, x1, y1, 1);
		   (x2, y2) = _gtk_plot_transform(plotd, x2, y2, 1);
		   _gtk_plot_set_xrange(plotd, x1, x2);
		   _gtk_plot_set_yrange(plotd, y2, y1);

		   gtk_plot_canvas_unselect(canvas);	% remove rubberband box
		   transform_regions(vwd);
		   vwd.filtd.zoom = new_zoom;
		   display_zoom(vwd);
		}

		return;
	}
	{ return; }

   variable region = region_new(type,vwd.filtd,regwidget);

   % At this point the rect or ellipse region is fully drawn, and so can
   % be added to the plot region set and have its bounding box defined
   region_list_append(vwd.filtd.regions,region);
   region_set_bbox(region);

   gtk_plot_canvas_unselect(canvas);	% remove rubberband box
   gtk_plot_canvas_refresh(canvas);
}

private define canvas_changed(canvas, vwd)
{
   if (vwd.active_child == NULL)
      return;

   variable reg = region_widget_to_struct(vwd.filtd.regions,vwd.active_child);
   if (reg != NULL)
      region_set_bbox(reg);

   vwd.active_child = NULL;
}

private define canvas_select(canvas, event, child, vwd)
{
   % veto any selection whilst drawing 
   if (vwd.filtd.drawc != NULL) return FALSE;

   switch (gtk_plot_canvas_get_child_type(child))
	{ case GTK_PLOT_CANVAS_PLOT:
	   
		if (vwd.tbox.curr == SHAPE_POLYGON)
		   poly_first_click(event, vwd);
	}
	{ case GTK_PLOT_CANVAS_NONE:

		if (vwd.tbox.curr != SHAPE_POLYGON) {
		   display_coords(vwd);
		   return TRUE;			% permit the selection
		}

		poly_first_click(event, vwd);
	}
	{ case GTK_PLOT_CANVAS_ELLIPSE	 or
	  case GTK_PLOT_CANVAS_RECTANGLE or
	  case GTK_PLOT_CANVAS_POLYGON   or
	  case GTK_PLOT_CANVAS_DATA:

		display_coords(vwd);
		return TRUE;			% permit the selection
	}

   return FALSE;				% veto the selection
}

private define canvas_delete(canvas,child,vwd) { return TRUE; }

private define canvas_motion(canvas,event,vwd)
{
   display_coords(vwd);
   return FALSE;			% propagate event, mark as unhandled
}

private define canvas_move_resize(canvas, child, dummy1, dummy2, vwd)
{
   switch (gtk_plot_canvas_get_child_type(child))
	{ case GTK_PLOT_CANVAS_ELLIPSE   or
	  case GTK_PLOT_CANVAS_RECTANGLE or
	  case GTK_PLOT_CANVAS_POLYGON:

		vwd.active_child = child;
	 	return TRUE;
	}
   return FALSE;
}

private define canvas_keypress(canvas, key, vwd)
{
   switch(key.keyval)
	{ case GDK_Delete or case GDK_KP_Delete or case GDK_BackSpace:

	   variable active_item = gtk_plot_canvas_get_active_item(canvas);
	   variable child = gtk_plot_canvas_get_child_data(active_item);

	   if (child != NULL) {
		gtk_plot_canvas_cancel_action(canvas);
		if (gtk_plot_canvas_remove_child(canvas,child)) {

		   switch (gtk_plot_canvas_get_child_type(child))
			{ case GTK_PLOT_CANVAS_ELLIPSE or
			  case GTK_PLOT_CANVAS_RECTANGLE or
			  case GTK_PLOT_CANVAS_POLYGON:

			     	variable rlist = vwd.filtd.regions;
				region_list_delete(rlist,
					region_widget_to_struct(rlist, child));

				_gtk_plot_redraw(vwd.filtd.plotd);
			}
		}
	   }
	   return TRUE;
	}
   return FALSE;
}

private define canvas_bpress(canvas, event, vwd)
{
   if (event.button == 3) { prefs_select_full(vwd); return TRUE; }
   return FALSE;
}
% }}}

% Point drawing {{{
private define maximize_axes_ranges(filtd)
{
   variable plotd = filtd.plotd;
   variable xmin = min(filtd.x), xmax = max(filtd.x);
   variable ymin = min(filtd.y), ymax = max(filtd.y);
   _gtk_plot_set_xrange(plotd, xmin, xmax);
   _gtk_plot_set_yrange(plotd, ymin, ymax);
   
   variable axis = plotd.dset.axes[0];
   axis.min = xmin; axis.max = xmax; axis.range = (xmax - xmin);
   axis = plotd.dset.axes[1];
   axis.min = ymin; axis.max = ymax; axis.range = (ymax - ymin);
}

define draw_points(vwd, xexpr, yexpr)
{
   variable fd, pd;				% filter and plot descriptors
   variable pp, fp = vwd.fprefs;		% plot and filter preferences

   if (xexpr == NULL and yexpr == NULL) {
	fd = vwd.filtd;				% drawing on existing plot pane
	pd = fd.plotd;
	pp = fd.prefs;
	xexpr = _gtk_plot_axis_get_title(pd.plot,  GTK_PLOT_AXIS_BOTTOM);
	yexpr = _gtk_plot_axis_get_title(pd.plot,  GTK_PLOT_AXIS_LEFT);
   }
   else
   {
	variable x = vwd.vectors[xexpr];
	variable y = vwd.vectors[yexpr];

	if (not (length(x) or length(y))) {
	   _error_dialog("empty plot","X or Y axis data is zero-length");
	   return 0;
	}

	fd = @FilterDescriptor;			% drawing on new plot pane
	fd.x = x;
	fd.y = y;
	pd   = NULL;

	if (vwd.filtd == NULL) {
	   fd.zoom    = 1.0;
	   fd.hshift  = 0.0;
	   fd.vshift  = 0.0;
	}
	else {
	   fd.zoom    = vwd.filtd.zoom;		% default new plots to
	   fd.hshift  = vwd.filtd.hshift;	% current zoom/shift
	   fd.vshift  = vwd.filtd.vshift;
	}

	fd.regions = @List;
	pp = plot_prefs_new();
	fd.prefs   = pp;
   }

   % Incrementally filter points before drawing, if requested
   variable incl_x, incl_y, excl_x = NULL, excl_y = NULL, filter = NULL;
   variable selected, num_selected, filtered;
   if (fp.incrfilt.value)
	filter = apply_filters(vwd);		% may also return NULL

   if (filter == NULL) {
	incl_x = fd.x;
	incl_y = fd.y;
	num_selected = vwd.veclen;
   }
   else {
	selected = where(filter);
	incl_x = fd.x[ selected ];
	incl_y = fd.y[ selected ];
	num_selected = length(selected);
   }
   if (num_selected == 0 and not fp.drawbg.value) {
	_error_dialog("empty plot","All points have been filtered, and "+
	      "you've\nswitched off background point drawing");
	return 0;
   }

   % If requested, plot the background (filtered) points (if any)
   if (andelse {fp.drawbg.value} {num_selected != vwd.veclen} ) {

	if (pp.align.value) {
	   excl_x = fd.x;
	   excl_y = fd.y;
	}
	else {
	   filtered = where(not(filter));
	   excl_x = fd.x[filtered];
	   excl_y = fd.y[filtered];
	}

	% In aligned mode draw foreground and background with same style
	variable linestyle;
	if (pp.align.value)
	   linestyle = pp.fgstyle.value;
	else
	   linestyle = pp.bgstyle.value;

	% Connect background (filtered) points with line color == symbol color
	if (pd == NULL)
	   pd = _gtk_plot(excl_x,excl_y, pp.bgcolor.value, linestyle);
	else
	   _gtk_oplot(pd, excl_x,excl_y, pp.bgcolor.value, linestyle);

	_gtk_plot_set_symbol(pd.dset, pp.sym.value, pp.bgcolor.value,
					pp.symsize.value, pp.symstyle.value);
   }

   % Now overplot the points which pass filters.  Using this ordering
   % ensures that filtered pts (when drawn) never eclipse selected pts
   if (num_selected) {

	if (pd == NULL) 
	   pd = _gtk_plot(incl_x, incl_y, gdk_black, pp.fgstyle.value);
	else
	   _gtk_oplot(pd, incl_x, incl_y, gdk_black, pp.fgstyle.value);

	_gtk_plot_set_symbol(pd.dset, pp.sym.value, pp.fgcolor.value,
					pp.symsize.value, pp.symstyle.value);

	% Ensure secondary, tertiary, etc plot panes match existing dimensions
	if (vwd.filtd != NULL) {

	   variable w, h, win = gtk_widget_get_window(vwd.filtd.plotd.canvas);
	   if (win != NULL) {
		gdk_drawable_get_size(win,&w,&h);
		gtk_plot_canvas_set_size(pd.canvas,w,h);
	   }
	}
   }

   _gtk_plot_set_labels(pd, xexpr, yexpr);

   vwd.filtd = fd;		% Reset current filter descriptor to this
   fd.plotd = pd;		% Rset current plot descriptor to this

   % It is visually convenient to draw selected (foreground) points relative
   % to the full range of the original dataset.  This happens automatically
   % when filtered (background) points are also drawn (because, by definition,
   % unfiltered + filtered == entire dataset).  When background is not drawn
   % the axes ranges will shrink to fit the foreground (or not), per whether
   % the fullrange filtering preference is off (or on).

   if (andelse {num_selected != vwd.veclen} {fp.fullrange.value})
	maximize_axes_ranges(fd);

   set_viewpoint(vwd);

   % Install custom signal handlers on the canvas
   () = g_signal_connect(pd.canvas,"changed", &canvas_changed,vwd);
   () = g_signal_connect(pd.canvas,"select_region", &canvas_rect_drag,vwd);
   () = g_signal_connect(pd.canvas,"select_item", &canvas_select,vwd);
   () = g_signal_connect(pd.canvas,"delete_item", &canvas_delete,vwd);
   () = g_signal_connect(pd.canvas,"move_item", &canvas_move_resize, vwd);
   () = g_signal_connect(pd.canvas,"resize_item", &canvas_move_resize, vwd);
   () = g_signal_connect(pd.canvas,"key_press_event", &canvas_keypress,vwd);
   () = g_signal_connect(pd.canvas,"motion_notify_event", &canvas_motion,vwd);
   () = g_signal_connect(pd.canvas,"button_press_event", &canvas_bpress, vwd);

   % Finally, display the plot(s) in the current pane
   gtk_widget_show_all(pd.canvas);

   return 1;
}
% }}}
