%  vwutils.sl: this file is part of the VWhere SLgtk guilet {{{
%
%  Copyright (C) 2003-2005 Massachusetts Institute of Technology
%  Copyright (C) 2002 Michael S. Noble <mnoble@space.mit.edu>
% 
% }}}

% Front matter {{{ 
#ifexists require
require("toolbox");
require("gtkplot");
#else
() = evalfile("toolbox");
() = evalfile("gtkplot");
#endif

public variable _vwhere_version = 100302;
public variable _vwhere_version_string = "1.3.2";
variable window_title = "VWhere "+ _vwhere_version_string;
% }}}

define echo_status() % {{{
{
   variable vwd, message,retval;
   if (_NARGS == 3) retval = ();
   (vwd,message) = ();
   () = gtk_statusbar_push(vwd.status,1,message);
   if (_NARGS == 3) return retval;
} % }}}

% Expression evaluation {{{
define evaluate(vwd, e, namespace)
{
   variable ad = _auto_declare, err;
   variable e2 = sprintf("_auto_declare=1; %s; _auto_declare = %d;", e, ad);

#ifeval  _slang_version < 20000
   ERROR_BLOCK
   {
	_clear_error();
	_toggle_error_hook();
	_error_dialog("",sprintf("could not evaluate expression: %s",e));
	return 0;
   }

   % Custom error hook, to enable recovery from normally unrecoverable errs
   _toggle_error_hook();
   if (namespace != "")
	eval(e2, namespace);
   else
	eval(e2);
   _toggle_error_hook();
#else
   try (err) { eval(e2, namespace); }
   catch AnyError: {
	_error_dialog("", sprintf("could not evaluate expression: %s\n%s",
		 						e, err.descr));
	return 0;
   }
#endif

   return 1;
}
% }}}

define has_background(vwd) % {{{
{
   if (vwd.filtd == NULL) return 0;
   return (length(vwd.filtd.plotd.dsets) > 1);
} % }}}

() = evalfile("vwprefs", vw_private_ns);

%  Regions {{{

variable SHAPE_RECTANGLE = "Rectangle";
variable SHAPE_ELLIPSE   = "Ellipse";
variable SHAPE_POLYGON   = "Polygon";

define _inside_rectangle(x,y,x1,y1,x2,y2)
{
   variable xin = (x >= x1 and x <= x2);
   variable yin = (y >= y1 and y <= y2);

   return (xin and yin);
}

define _inside_circle(x,y,h,k,r)
{
   variable z = (x - h) ^ 2 + (y - k)^2;
   return (z <= r^2);
}

define _inside_ellipse(x,y,h,k,r1,r2)
{
   if (r1 == r2)
      return _inside_circle(x,y,h,k,r1);

   variable xp = (x - h) / r1;
   variable yp = (y - k) / r2;
   return ((xp^2 + yp^2) <= 1.0);
}

define _inside_polygon(x,y,xp,yp)
{
   return __inside_polygon(x,y,xp,yp);
}

#ifnexists List
typedef struct { head, tail } List;
#endif
variable Region = struct {
	type,			% rectangle, ellipse, etc
	widget,			% underlying Gtk widget visualizing the region
	filtd,			% filter descriptor for plot containing region
	x,y,			% points defining the region
	bbox,			% region bounding box (in dataspace coords)
	prev,			% previous region in list
	next			% next region in list
};

define region_list_append(list,region)	% O(1) insertion
{
   if (list.head == NULL) {
	list.head = region;
	list.tail = region;
   }
   else {
	region.prev = list.tail;
	list.tail.next = region;
	list.tail = region;
   }
}
define region_list_delete(list,region)	% O(1) removal
{
   if (region.prev != NULL)
	region.prev.next = region.next;
   else
	list.head = region.next;

   if (region.next == NULL) list.tail = region.prev;
}

define region_set_bbox(region)
{
   % Define bounding box, in plot data space, for moving/scaling/filtering
   variable x1, y1, x2, y2, pd = region.filtd.plotd, canvas = pd.canvas;
   () = gtk_plot_canvas_get_child_position(canvas,
					region.widget, &x1, &y1, &x2, &y2);
   gtk_plot_canvas_get_pixel(canvas,x1,y1,&x1,&y1);
   gtk_plot_canvas_get_pixel(canvas,x2,y2,&x2,&y2);
   gtk_plot_get_point(pd.plot,x1,y1,&x1,&y1);
   gtk_plot_get_point(pd.plot,x2,y2,&x2,&y2);
   region.bbox = [x1, y1, x2, y2];
}

define region_new(type,filtd,widget)
{
   variable region = @Region;
   region.type     = type;
   region.widget   = widget;
   region.filtd	   = filtd;

   return region;
}

define region_widget_to_struct(list,widget)
{
   foreach(list.head) {
	variable region = ();
	if (__eqs_gtkopaque(widget,region.widget))
	   return region;
   }
   return NULL;
}

private variable cursors = [
	gdk_cursor_new(GDK_LEFT_PTR),
	gdk_cursor_new(GDK_CROSSHAIR),
	gdk_cursor_new(GDK_WATCH)
];
private variable num_cursors = length(cursors);
define set_cursor(widget,cursor)
{
   variable window = gtk_widget_get_window(widget);
   if (window != NULL) {
	if (cursor < 1)
	   cursor = 1;
	else if (cursor > num_cursors)
	   cursor = num_cursors;
	gdk_window_set_cursor(window, cursors[cursor-1]);
   }
}

define transform_regions(vwd)
{
   variable fd = vwd.filtd, pd = fd.plotd, canvas = pd.canvas;

   set_cursor(vwd.plotwin, 3);	% shouldn't this require
   set_cursor(vwd.panes, 3);		% only 1 function call?
   set_cursor(pd.canvas, 3);

   gtk_plot_canvas_freeze(canvas);		% avoid multiple redraws
   variable region, bbox, x1, y1, x2, y2;
   foreach(fd.regions.head) {
	region = ();
	bbox = region.bbox;
	(x1,y1) = _gtk_plot_transform(pd,bbox[0],bbox[1],-1);
	(x2,y2) = _gtk_plot_transform(pd,bbox[2],bbox[3],-1);
	gtk_plot_canvas_child_move_resize(canvas,region.widget,x1,y1,x2,y2);
   }
   gtk_plot_canvas_thaw(canvas);
   _gtk_plot_redraw(pd);

   set_cursor(vwd.plotwin, 1);
   set_cursor(vwd.panes, 1);
   set_cursor(pd.canvas, 1);
}

define transform_axis(plotd, which, shift, zoom, func)
{
   % Display a shifted centroid 1/2, 1/4, 1/8, ... size of original view
   variable axis = plotd.dset.axes[which];

   shift = shift * axis.range;
   variable zoom_delta = (1 - zoom) * (axis.range / 2);

   (@func)(plotd, axis.min + shift + zoom_delta, axis.max + shift - zoom_delta);
}

define display_zoom(vwd)
{
   gtk_label_set_text(vwd.zoom_area,sprintf("Zoom: %3.3f",1/vwd.filtd.zoom));
}

define display_coords(vwd)
{
   variable x, y, pd = vwd.filtd.plotd;
   gtk_widget_get_pointer(pd.canvas, &x, &y);
   gtk_plot_get_point(pd.plot, x, y, &x, &y);
   gtk_label_set_text(vwd.coords_area, sprintf("( %5.3f, %5.3f )", x, y));
}

define set_viewpoint(vwd)
{
   if (vwd.plotwin == NULL) return;

   % Transform the plot (and any regions) accordingly
   variable fd = vwd.filtd, pd = fd.plotd;
   transform_axis(pd, 0, fd.hshift, fd.zoom, &_gtk_plot_set_xrange);
   transform_axis(pd, 1, fd.vshift, fd.zoom, &_gtk_plot_set_yrange);
   transform_regions(vwd);
   display_zoom(vwd);
}

define region_translate_coords(reg)
{

   reg = @reg;

   switch(reg.type)
	{ case SHAPE_RECTANGLE or case SHAPE_ELLIPSE:
	   % Bounding box already specificies data-relative coords, so just
	   % ensure (x1,y1) is bottom left, (x2,y2) top right (data coords)
	   % This is done b/c in screen coords (pixel or canvas) y increases
	   % from top to bottom, but in data-relative coords it's the reverse.
	   reg.x = [ reg.bbox[0], reg.bbox[2] ];
	   reg.y = [ reg.bbox[3], reg.bbox[1] ];
	}
	{ case SHAPE_POLYGON: 

	   % Poly may have been moved/scaled since being laid, but vertices
	   % (poly.x, poly.y, in canvas space) are still relative to the
	   % initial pos, so recenter/scale them relative to current pos.
	   variable x = min(reg.x), x2 = max(reg.x), w = (x2 - x);
	   variable y = min(reg.y), y2 = max(reg.y), h = (y2 - y);

	   variable bbx1, bbx2, bby1, bby2;
	   variable pd = reg.filtd.plotd, canvas = pd.canvas;

	   () = gtk_plot_canvas_get_child_position(canvas,reg.widget,
						&bbx1, &bby1, &bbx2, &bby2);

	   variable xscale = (bbx2 - bbx1) / w;
	   variable yscale = (bby2 - bby1) / h;
	   reg.x = bbx1 + (reg.x - x) * xscale;
	   reg.y = bby1 + (reg.y - y) * yscale;

	   % Finally, map the canvas-space coordinates to data space
	   variable i, nvertices = length(reg.x);
	   for (i=0; i < nvertices; i++) {
		gtk_plot_canvas_get_pixel(canvas, reg.x[i], reg.y[i], &x, &y);
		gtk_plot_get_point(pd.plot, x, y, &x, &y);
		reg.x[i] = x;
		reg.y[i] = y;
	   }
	}

   return reg;
}

private variable VW_AND  = 0;
private variable VW_OR   = 1;

define apply_filters(vwd)
{
   variable nplots = length(vwd.filtds), result = NULL;
   foreach (vwd.filtds) {

	variable filtd = (), x = filtd.x , y = filtd.y, plot_indices = NULL;

	foreach(filtd.regions.head) {

	   variable region = (), region_indices;
	   region = region_translate_coords(region);

	   switch(region.type)
	   { case SHAPE_RECTANGLE:

		region_indices = _inside_rectangle(x, y, region.x[0],
				region.y[0], region.x[1], region.y[1]);
	   }
	   { case SHAPE_ELLIPSE: 

		variable r1 = abs(region.x[1] - region.x[0]) / 2.0;
		variable r2 = abs(region.y[1] - region.y[0]) / 2.0;
		region_indices = _inside_ellipse(x,y,
				region.x[0] + r1, region.y[0] + r2, r1, r2);
	   }
	   { case SHAPE_POLYGON:

		region_indices = _inside_polygon(x, y, region.x, region.y);
	   }

	   if (plot_indices == NULL) 
		plot_indices = region_indices;		% conserve memory
	   else
		plot_indices = (plot_indices or region_indices);
	}

	if (result == NULL) {
	    result = plot_indices;			% conserve memory
	    if (nplots == 1) break;
	    continue;
	}

	if (plot_indices == NULL)			% any contribution
	   continue;					% from this plot?

	switch(vwd.fprefs.combine.value)
	   { case VW_AND: result = (result and plot_indices); }
	   { case VW_OR:  result = (result or plot_indices); }
	   { error("Illegal method specified for combining plot filters"); }
   }

   return result;
}
% }}}

%  Printing {{{

private define print_cb(ctx, vwd)
{
   variable pd = vwd.filtd.plotd, fname;

   if (ctx.dest == 0)
	fname = _tmp_file_name("vwhere");
   else
	fname = ctx.fname;

   % Generate an encapsulated postscript file
   () = gtk_plot_canvas_export_ps(pd.canvas, fname, ctx.orient, TRUE, ctx.size);

   if (ctx.dest == 0) {
	() = system( sprintf("%s %s",ctx.cmd, fname));
	() = remove(fname);
   }
}

private define print(vwd)
{
   if (vwd.print_ctx == NULL) {
	vwd.print_ctx = @GtkPrintContext;
	vwd.print_ctx.title = "VWhere Print Dialog";
	vwd.print_ctx.fname = "vwhere.ps";
   }
   _print_dialog(vwd.print_ctx, &print_cb, vwd);
}
% }}}

%  Toolbox {{{

private variable MAX_ZOOM = 512.0;
define zoom_in(current_zoom)
{
   return max( [current_zoom / 2.0, 1/ MAX_ZOOM] );
}

define zoom_out(current_zoom)
{
   return min( [current_zoom * 2.0, MAX_ZOOM ] );
}

static define help(dummy)
{
   _info_window(window_title+" Help", _get_slgtk_doc_string("vwhere"));
}

variable TB_ZOOM_OUT	= "ZoomOut";
variable TB_ZOOM_IN	= "ZoomIn";
variable TB_ZOOM_FIT	= "ZoomToFit";
variable TB_NORMAL	= "NormalView";
variable TB_SH_LEFT	= "ScrollLeft";
variable TB_SH_RIGHT	= "ScrollRight";
variable TB_SH_UP	= "ScrollUp";
variable TB_SH_DOWN	= "ScrollDown";
variable TB_LINESTYLE	= "CurrentPlotLineStyles";
variable TB_PREFS	= "Preferences";
variable TB_PRINT	= "Print";
variable TB_HELP	= "Help";

define tb_choose(tb,vwd)
{
   variable fd = vwd.filtd, pd = fd.plotd, canvas = pd.canvas;
   if (fd.drawc != NULL) (@fd.drawc.destroy) (fd);
   variable shift = 0.1 * fd.zoom;

   % User may pan up to 400% in either direction, or zoom in/out up to 16x
   % Each zoom in takes centroid of half
   switch(tb.curr)
	{ case TB_SH_LEFT   : fd.hshift = max( [fd.hshift - shift, -4.0] ); }
	{ case TB_SH_RIGHT  : fd.hshift = min( [fd.hshift + shift,  4.0] ); }
	{ case TB_SH_UP     : fd.vshift = min( [fd.vshift + shift,  4.0] ); }
	{ case TB_SH_DOWN   : fd.vshift = max( [fd.vshift - shift, -4.0] ); }
	{ case TB_ZOOM_IN   : fd.zoom = zoom_in(fd.zoom); }
	{ case TB_ZOOM_OUT  : fd.zoom = zoom_out(fd.zoom); }
	{ case TB_NORMAL    :

		if (fd.zoom == 1) return;
	   	fd.zoom = 1.0; fd.hshift = 0.0; fd.vshift = 0.0;
	}
	{ case TB_ZOOM_FIT  : set_cursor(pd.plot, 2); return; }
	{ case TB_PREFS     : prefs_select_full(vwd); return; }
	{ case TB_PRINT	    : print(vwd); return; }
	{ case TB_LINESTYLE : prefs_select_mini(vwd); return; }
	{ case TB_HELP      : help(NULL); return; }
	{ set_cursor(pd.plot, 1); return; }

   set_viewpoint(vwd);
}

define make_toolbox(cbfunc,data)
{
   variable tbox = toolbox_new(GTK_ORIENTATION_HORIZONTAL);

   toolbox_add_button(tbox,SLGTK_STOCK_RECTANGLE,SHAPE_RECTANGLE, cbfunc, data);
   toolbox_add_button(tbox,SLGTK_STOCK_ELLIPSE,SHAPE_ELLIPSE, cbfunc, data);
   toolbox_add_button(tbox,SLGTK_STOCK_POLYGON,SHAPE_POLYGON, cbfunc, data);
   toolbox_add_button(tbox,GTK_STOCK_ZOOM_FIT, TB_ZOOM_FIT, cbfunc, data);
   toolbox_set_focus_default(tbox,FALSE);
   toolbox_add_button(tbox,GTK_STOCK_ZOOM_IN, TB_ZOOM_IN, cbfunc, data);
   toolbox_add_button(tbox,GTK_STOCK_ZOOM_OUT, TB_ZOOM_OUT, cbfunc, data);
   toolbox_add_button(tbox,GTK_STOCK_ZOOM_100, TB_NORMAL, cbfunc, data);
   toolbox_add_button(tbox,GTK_STOCK_GO_BACK, TB_SH_LEFT, cbfunc, data);
   toolbox_add_button(tbox,GTK_STOCK_GO_FORWARD, TB_SH_RIGHT, cbfunc, data);
   toolbox_add_button(tbox,GTK_STOCK_GO_UP, TB_SH_UP, cbfunc, data);
   toolbox_add_button(tbox,GTK_STOCK_GO_DOWN, TB_SH_DOWN, cbfunc, data);
   toolbox_add_button(tbox,SLGTK_STOCK_LINESEG, TB_LINESTYLE,
	 					cbfunc, data, "pressed");
   toolbox_add_button(tbox,GTK_STOCK_PREFERENCES, TB_PREFS, cbfunc, data);
   toolbox_add_button(tbox,GTK_STOCK_PRINT, TB_PRINT, cbfunc, data);
   toolbox_add_button(tbox,GTK_STOCK_DIALOG_QUESTION, TB_HELP, cbfunc, data);
   return tbox;
}
% }}}
